package com.crytek.jira.plugins.crashHandler.signature;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.not;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Matchers.anyString;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.junit.Before;
import org.junit.Test;

import com.atlassian.jira.config.ConstantsManager;
import com.crytek.jira.plugins.crashHandler.SignatureInfo;
import com.crytek.jira.plugins.crashHandler.Utils;
import com.onresolve.jira.crytek.settings.Settings;

public class EditDistanceComparerUnitTest
{
	private static final int AN_ID = 3;
	private static int DEFAULT_COST = 1;

	private EditDistanceComparer comparer;
	private SignatureInfo signature;
	private static final String TEN_A = "a a a a a a a a a a";

	private boolean EXPENSIVE_TESTS;

	@Before
	public void setUp()
	{
		EXPENSIVE_TESTS = false;
	}

	@Test
	public void levenshteinDistance_WithAddDelete()
	{

		double result = levenshteinDistanceFromString("a", "a b b");

		assertTrue(Utils.equals(2., result));
	}

	@Test
	public void levenshteinDistance_WithReplace()
	{
		double result = levenshteinDistanceFromString("a a b b", "a a a a");
		assertTrue(Utils.equals(2, result));
	}

	@Test
	public void levenshteinDistance_WithEmpty_MaximumDistance()
	{
		double expectedDistance = DEFAULT_COST * 10d;

		double result = levenshteinDistanceFromString("", TEN_A);
		assertTrue(Utils.equals(expectedDistance, result));

		result = levenshteinDistanceFromString(TEN_A, "");
		assertTrue(Utils.equals(expectedDistance, result));
	}

	@Test
	public void levenshteinDistance_WithSameString_Transitivity()
	{
		int expectedDistance = 0;

		double result = levenshteinDistanceFromString(TEN_A, TEN_A);
		assertTrue(Utils.equals(expectedDistance, result));
	}

	@Test
	public void levenshteinDistance_ShouldBeCommutative()
	{
		String stringA = "a a a b c c c";
		String stringB = "a a b c c c a";

		double ab = levenshteinDistanceFromString(stringA, stringB);
		double ba = levenshteinDistanceFromString(stringB, stringA);

		assertTrue(Utils.equals(ab, ba));
	}

	@Test
	public void rateSimilarity_withHalfString_half()
	{
		String stackA = "a a a b b b";
		String stackB = "a a a a a a";
		double result = rateSimilarity_FromString(stackA, stackB);
		assertTrue(Utils.equals(0.5, result));
	}

	@Test
	public void rateSimilarity_withTenth_ninety()
	{
		String stackA = "a a a a a a a a a a a a a a a a a a a a";
		String stackB = "b a a a a a a a a a a a a a a a a a a b";
		double result = rateSimilarity_FromString(stackA, stackB);
		assertTrue(Utils.equals(0.9, result));
	}

	@Test
	public void rateSimilarity_withEmptyString_zero()
	{
		double result = rateSimilarity_FromString(TEN_A, "");
		assertTrue(Utils.equals(0.0, result));

		result = rateSimilarity_FromString("", "");
		assertTrue(Utils.equals(0.0, result));
	}

	@Test
	public void rateSimilarity_withNinety_tenthHaveApproximationError()
	{
		String stackA = "a a a a a a a a a a a a a a a a a a a a";
		String stackB = "a b b b b b b b b b b b b b b b b b b a";
		double result = rateSimilarity_FromString(stackA, stackB);
		assertThat(0.1, not(equalTo(result)));

		// Testing Utils
		assertTrue(Utils.equals(0.1, result));
		assertFalse(Utils.greaterThan(0.1, result));
		assertFalse(Utils.lessThan(0.1, result));
	}

	public double rateSimilarity_FromString(String stackA, String stackB)
	{
		comparer = getComparerForString(stackA);
		signature = getSignatureForCallstack(stackB);

		// test commutativity
		double firstComparison = comparer.rateSimilarity(signature);

		comparer = getComparerForString(stackB);
		signature = getSignatureForCallstack(stackA);

		assertThat(firstComparison, equalTo(comparer.rateSimilarity(signature)));
		return firstComparison;
	}

	public double levenshteinDistanceFromString(String stringA, String stringB)
	{
		comparer = getComparerForString(stringA);
		signature = getSignatureForCallstack(stringB);

		// test commutativity
		double firstDistance = comparer.levenshteinDistance(signature.getCrashSignatureLines());

		comparer = getComparerForString(stringB);
		signature = getSignatureForCallstack(stringA);

		assertThat(firstDistance, equalTo(comparer.levenshteinDistance(signature.getCrashSignatureLines())));

		OptimizedEditDistanceComparer oComparer = getOptimizedComparerForString(stringB);
		assertThat(comparer.levenshteinDistance(signature.getCrashSignatureLines()), equalTo(oComparer.optimizedLevenshteinDistance(signature
				.getCrashSignatureLines())));
		return firstDistance;
	}

	public EditDistanceComparer getComparerForString(String callStack)
	{
		SignatureInfo signatureForCallStack = getSignatureForCallstack(callStack);
		return new EditDistanceComparer(signatureForCallStack);
	}

	public OptimizedEditDistanceComparer getOptimizedComparerForString(String callstack)
	{
		SignatureInfo signatureForCallStack = getSignatureForCallstack(callstack);
		ConstantsManager constantsManager = mock(ConstantsManager.class);
		when(constantsManager.getPriorityName(anyString())).thenReturn("");
		Settings settings = Settings.getDefaultSettings(constantsManager);
		return new OptimizedEditDistanceComparer(signatureForCallStack, settings);
	}

	public SignatureInfo getSignatureForCallstack(String callStack)
	{
		return new SignatureInfo(AN_ID, "", callStack);
	}

	// it's about 11. milliseconds per loop;
	@Test
	public void levenshteinDistance_Performance()
	{
		if (!EXPENSIVE_TESTS)
		{
			return;
		}
		String serie1 = "8 10 94 80 70 27 44 14 83 7 31 37 62 99 61 33 94 29 26 33 97 25 88 65 39 23 62 37 32 64 3 61 24 52 55 83 50 52 18 76 29 9 89 83 9 35 59 11 18 15 58 48 51 0 13 77 50 29 99 31 36 52 57 27 78 9 1 42 48 73 15 70 84 52 11 88 14 27 18 79 16 11 66 12 62 91 54 88 58 38 69 87 41 25 99 12 3 86 11 72 ";
		serie1 = serie1 + serie1 + serie1 + serie1;
		String serie2 = "45 5 51 82 44 40 94 26 73 1 15 75 29 25 28 38 33 64 96 10 48 17 73 7 12 26 95 85 3 37 1 70 48 44 34 49 53 21 68 47 48 85 50 80 68 6 53 49 39 39 92 18 80 67 99 28 47 35 74 68 41 13 37 82 77 91 32 90 54 12 39 64 57 82 0 27 46 51 17 10 51 68 27 21 25 64 87 72 65 70 9 45 26 38 44 20 98 55 26 7 ";
		serie2 = serie2 + serie2 + serie2 + serie2;
		comparer = getComparerForString(serie1);
		signature = getSignatureForCallstack(serie2);
		for (int i = 0; i < 100; i++)
		{
			comparer.rateSimilarity(signature);
		}
	}

	@Test
	public void levenshteinDistance_PerformanceOneHundred()
	{
		if (!EXPENSIVE_TESTS)
		{
			return;
		}
		String serie1 = "8 10 94 80 70 27 44 14 83 7 31 37 62 99 61 33 94 29 26 33 97 25 88 65 39 23 62 37 32 64 3 61 24 52 55 83 50 52 18 76 29 9 89 83 9 35 59 11 18 15 58 48 51 0 13 77 50 29 99 31 36 52 57 27 78 9 1 42 48 73 15 70 84 52 11 88 14 27 18 79 16 11 66 12 62 91 54 88 58 38 69 87 41 25 99 12 3 86 11 72 ";
		String serie2 = "45 5 51 82 44 40 94 26 73 1 15 75 29 25 28 38 33 64 96 10 48 17 73 7 12 26 95 85 3 37 1 70 48 44 34 49 53 21 68 47 48 85 50 80 68 6 53 49 39 39 92 18 80 67 99 28 47 35 74 68 41 13 37 82 77 91 32 90 54 12 39 64 57 82 0 27 46 51 17 10 51 68 27 21 25 64 87 72 65 70 9 45 26 38 44 20 98 55 26 7 ";

		comparer = getComparerForString(serie1);
		signature = getSignatureForCallstack(serie2);
		for (int i = 0; i < 100; i++)
		{
			comparer.rateSimilarity(signature);
		}

	}

	@Test
	public void oLevenshteinDistance_Performance()
	{
		if (!EXPENSIVE_TESTS)
		{
			return;
		}
		String serie1 = "8 10 94 80 70 27 44 14 83 7 31 37 62 99 61 33 94 29 26 33 97 25 88 65 39 23 62 37 32 64 3 61 24 52 55 83 50 52 18 76 29 9 89 83 9 35 59 11 18 15 58 48 51 0 13 77 50 29 99 31 36 52 57 27 78 9 1 42 48 73 15 70 84 52 11 88 14 27 18 79 16 11 66 12 62 91 54 88 58 38 69 87 41 25 99 12 3 86 11 72 ";
		serie1 = serie1 + serie1 + serie1 + serie1;
		String serie2 = "45 5 51 82 44 40 94 26 73 1 15 75 29 25 28 38 33 64 96 10 48 17 73 7 12 26 95 85 3 37 1 70 48 44 34 49 53 21 68 47 48 85 50 80 68 6 53 49 39 39 92 18 80 67 99 28 47 35 74 68 41 13 37 82 77 91 32 90 54 12 39 64 57 82 0 27 46 51 17 10 51 68 27 21 25 64 87 72 65 70 9 45 26 38 44 20 98 55 26 7 ";
		serie2 = serie2 + serie2 + serie2 + serie2;
		OptimizedEditDistanceComparer oComparer = getOptimizedComparerForString(serie1);
		signature = getSignatureForCallstack(serie2);
		for (int i = 0; i < 1; i++)
		{
			oComparer.rateSimilarity(signature);
		}
		equalTo(getOptimizedComparerForString(serie1).rateSimilarity(signature));
	}

	@Test
	public void PrefixComparer_Performance()
	{
		if (!EXPENSIVE_TESTS)
		{
			return;
		}
		String serie1 = "8 10 94 80 70 27 44 14 83 7 31 37 62 99 61 33 94 29 26 33 97 25 88 65 39 23 62 37 32 64 3 61 24 52 55 83 50 52 18 76 29 9 89 83 9 35 59 11 18 15 58 48 51 0 13 77 50 29 99 31 36 52 57 27 78 9 1 42 48 73 15 70 84 52 11 88 14 27 18 79 16 11 66 12 62 91 54 88 58 38 69 87 41 25 99 12 3 86 11 72";
		serie1 = serie1 + serie1 + serie1 + serie1 + "a";
		String serie2 = serie1 + serie1 + serie1 + serie1 + "b";
		PrefixComparer comparer = new PrefixComparer(new SignatureInfo(4, "", serie1));

		for (int i = 0; i < 10000; i++)
		{
			signature = getSignatureForCallstack(serie2);
			comparer.rateSimilarity(signature);
		}
	}
}
