﻿using System;
using System.IO;
using System.Windows.Forms;
using CrashHandler;
using System.Text.RegularExpressions;

namespace Ps3CrashHandler
{
    public class Ps3CommandLineTools 
    {
        private const String ELF_OUTPUT = "CallStackDecoder.elf";

        private const String errorLogName = "error.log";
        public const String newErrorLogName = "PS3error.log";
        private const String CallStackTraceHeading = "Call Stack Trace:";
        private const String PS3_BIN = "ps3bin.exe";
        private const String PPU_LV2_ADDR2LINE = "-i {0} -a2l {1}";

        private const String SEPARATOR_AND_VALUE = "\\s*(.*)$";
        private const String ADDRESS_LINE_MATCHER = "^Address:" + SEPARATOR_AND_VALUE;
        private const String FILE_NAME_MATCHER = "^File Name:" + SEPARATOR_AND_VALUE;
        private const String LINE_NUMBER_MATCHER = "^Line Number:" + SEPARATOR_AND_VALUE;
        private const String FUNCTION_NAME_MATCHER = "^Symbol:" + SEPARATOR_AND_VALUE;

        private const String FAKE_LOG_HEADER = "Logged at Friday, April 03, 2009 16:23:41\r\nFileVersion: 1.0.0.1\r\nProductVersion: 1.0.0.1\r\n";




        private static String ToolsLocation()
        {
            return Path.GetDirectoryName(Application.ExecutablePath);
        }

        public static String DecodeCallstack(String selfLocation)
        {
            String errorLogLocation = Path.GetDirectoryName(selfLocation);
            String errorLogContent = Util.GetFileText(Path.Combine(errorLogLocation, errorLogName));

            String fullCallStack = StringHelper.GetLinesBetweenPatterns(errorLogContent, CallStackTraceHeading, "^\\s*$", StringHelper.LineOptions.exclusive);
            String[] callStackLines = StringHelper.GetAllLines(fullCallStack);

            for(int index = 0; index < callStackLines.Length; ++index)
            {
                callStackLines[index] = DecodeCallStackLine(callStackLines[index], selfLocation);
            }

            String errorLogHeader = StringHelper.GetLinesBetweenPatterns(errorLogContent, ".*", CallStackTraceHeading, StringHelper.LineOptions.includeBoth);
            String translatedCallStack = String.Join(Environment.NewLine, callStackLines);

            String translatedErrorlog = String.Format("{0}{1}{2}{3}",FAKE_LOG_HEADER, errorLogHeader, Environment.NewLine, translatedCallStack);
            String outputErrorlogLocation = Path.Combine(errorLogLocation, newErrorLogName);
            BackupOldLogs(outputErrorlogLocation);
            File.WriteAllText(outputErrorlogLocation, translatedErrorlog);
            return translatedErrorlog;
        }

        private static String DecodeCallStackLine(String line, String selfLocation){
            Match match = Regex.Match(line, @"(\d*\))\s*(0x[0-9a-fA-F]*)");
            String callStackLineNumber = match.Groups[1].Value;
            String functionAddr = match.Groups[2].Value;
            
            String ps3binArgs = String.Format(PPU_LV2_ADDR2LINE, selfLocation, functionAddr);
            CommandLine shell = new CommandLine( Path.Combine(ToolsLocation(),PS3_BIN), ps3binArgs);

            shell.Run();

            if(shell.IsError()){
                throw new Exception(shell.Error); 
            }
            return String.Format("{0} {1}", callStackLineNumber, TranslateOutput(shell.Output));
        }

        private static String TranslateOutput(String output){
            String functionName = GetMatchingGroup(output, FUNCTION_NAME_MATCHER);
            if(String.IsNullOrEmpty(functionName) || functionName.StartsWith("??")){
                return GetMatchingGroup(output, ADDRESS_LINE_MATCHER);
            }
            String filename = GetMatchingGroup(output, FILE_NAME_MATCHER);
            String lineNumber = GetMatchingGroup(output, LINE_NUMBER_MATCHER);
            return String.Format("{0} [{1}:{2}]", functionName, filename, lineNumber);

        }

        private static String GetMatchingGroup(String output, String lineMatcher){
            Match match = Regex.Match(output, lineMatcher, RegexOptions.Multiline);
            if(match.Groups.Count < 1 ){
                return String.Empty;
            }
            return match.Groups[1].Value.Trim();
        }

        private static void BackupOldLogs(String pathname){
            try{
                if(File.Exists(pathname)){
                    DateTime write = File.GetLastWriteTime(pathname);
                    File.Move(pathname, pathname + ".bak" + write.Ticks);
                }
            } catch(Exception){
                Console.Out.WriteLine("couldn't backup old file, overwriting it...");
            }
        }



    }
}
