﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Dia2Lib;
using System.IO;
using CrashHandler;

// The logic here has been extracted from the "CallStackDisplay" example in the "Xbox samples"
// The logic to interrogate the symbol server has been removed

namespace Xbox360CrashHandler
{
    public class DiaAccess
    {
        //private readonly Dia2Lib.IDiaDataSource source;
        private List<Module> modules;
        private Log m_log;
        public DiaAccess(Log log){
            modules = new List<Module>();
            m_log = log;
        }

        public bool LoadPdb(ulong baseAddress, uint size, XboxSignature signature){

            m_log.Info("Trying to load " + signature.PdbFilename + "(" + signature.PdbLocation + ")");
            if(!File.Exists(signature.PdbLocation)){
                m_log.Info("[Warning] File not found: " + signature.PdbLocation);
                return false;
            }
            try
            {
                Dia2Lib.IDiaDataSource source = new Dia2Lib.DiaSource();
                Guid guid = signature.Guid;
                source.loadAndValidateDataFromPdb(signature.PdbLocation, ref guid, 0, signature.Age);

                IDiaSession session;
                source.openSession(out session);
                session.loadAddress = baseAddress;
                modules.Add(new Module(baseAddress, size, session));
            }
            catch(System.Runtime.InteropServices.COMException e)
            {
                LogComException(e, "Loading:" + signature.PdbLocation);
                return false;
            }
            catch(Exception e)
            {
                m_log.Warn("An error occurred looking for " + signature.PdbFilename + "(" + signature.PdbLocation + ")" + e.Message);
                return false;
            }

            m_log.Info("[info]" + signature.PdbFilename + " Loaded successfully!");
            return true;
        }

        private void LogComException(System.Runtime.InteropServices.COMException e, String when)
        {
            if (Enum.IsDefined(typeof(DiaErrorCodes), (uint)e.ErrorCode))
            {
                String errorCodeName = Enum.GetName(typeof(DiaErrorCodes), (uint)e.ErrorCode);
                m_log.Warn("Com exception: " + errorCodeName + " = 0x" + e.ErrorCode.ToString("X") + " "+ when);
            }
            else
            {
                m_log.Warn("<Unknown> Com exception:" + e.ErrorCode.ToString("X") + " " + when);
                foreach(String key in e.Data.Keys){
                    m_log.Warn("key: " + key + " value:" + e.Data[key]);
                }
                throw e;
            }
        }

        public XboxCrashLineInfo DecodeSymbol(ulong address)
        {
            Module module = GetModuleIndex(address);
            if(module == null){
                m_log.Info("[Info] Couldn't find a module for the address " + address.ToString("X"));
                return null;
            }
            try
            {
                IDiaSession session = module.Session;
                IDiaSymbol functionSymbol;
                session.findSymbolByVA(address, SymTagEnum.SymTagFunction, out functionSymbol);

                if(functionSymbol == null){
                    return null;
                }
                String functionName = functionSymbol.name;
               
                IDiaEnumLineNumbers functionOccurrences;
                uint instructionSize = 4;
                session.findLinesByVA(address, instructionSize, out functionOccurrences);

                const uint linesToFetch = 1;
                IDiaLineNumber functionLineNumber;
                uint fechedLines;

                functionOccurrences.Next(linesToFetch, out functionLineNumber, out fechedLines);
                if (fechedLines == 0)
                {
                    return null;
                }

                uint lineNumber = functionLineNumber.lineNumber;

                String filename = functionLineNumber.sourceFile.fileName;
                m_log.Info(address.ToString("X") + " = " + functionName);
                return new XboxCrashLineInfo(functionName, lineNumber, filename, address);
            } 
            catch(System.Runtime.InteropServices.COMException e){
                LogComException(e, "Loading function name for address: " +address.ToString("X"));
                return null;
            }
            catch(Exception e)
            {
                m_log.Warn("Couldn't load function name for address: " + address.ToString("X") + " An exception occurred: " + e.Message);
                return null;
            }
            
        }

        private Module GetModuleIndex(ulong address){
            foreach(Module module in modules){
                if( address > module.BaseAddress && address < module.BaseAddress + module.Size){
                    return module;
                }
            }
            return null;
        }

        private class Module
        {
            public Module(ulong baseAddress, uint size, IDiaSession session){
                this.baseAddress = baseAddress;
                this.size = size;
                this.session = session;
            }
            readonly ulong baseAddress;
            public ulong BaseAddress
            {
                get { return baseAddress; }
            }

            readonly uint size;
            public uint Size
            {
                get { return size; }
            }

            readonly IDiaSession session;
            public IDiaSession Session
            {
                get { return session; }
            }
        }

        //These error codes are defined in dia2.h (located in %VSInstallDir%\dia sdk\include\dia2.h)
        private enum DiaErrorCodes :uint{
            E_INVALIDARG = 0x80070057,
            E_OUTOFMEMORY  = 0x8007000e,
            E_PDB_OK = 0x806d0001,
            E_PDB_USAGE = 0x806d0002,
            E_PDB_OUT_OF_MEMORY = 0x806d0003,
            E_PDB_FILE_SYSTEM = 0x806d0004,
            E_PDB_NOT_FOUND = 0x806d0005,
            E_PDB_INVALID_SIG = 0x806d0006,
            E_PDB_INVALID_AGE = 0x806d0007,
            E_PDB_PRECOMP_REQUIRED = 0x806d0008,
            E_PDB_OUT_OF_TI = 0x806d0009,
            E_PDB_NOT_IMPLEMENTED = 0x806d000a,
            E_PDB_V1_PDB = 0x806d000b,
            E_PDB_FORMAT = 0x806d000c,
            E_PDB_LIMIT = 0x806d000d,
            E_PDB_CORRUPT = 0x806d000e,
            E_PDB_TI16 = 0x806d000f,
            E_PDB_ACCESS_DENIED = 0x806d0010,
            E_PDB_ILLEGAL_TYPE_EDIT = 0x806d0011,
            E_PDB_INVALID_EXECUTABLE = 0x806d0012,
            E_PDB_DBG_NOT_FOUND = 0x806d0013,
            E_PDB_NO_DEBUG_INFO = 0x806d0014,
            E_PDB_INVALID_EXE_TIMESTAMP = 0x806d0015,
            E_PDB_RESERVED = 0x806d0016,
            E_PDB_DEBUG_INFO_NOT_IN_PDB = 0x806d0017,
            E_PDB_SYMSRV_BAD_CACHE_PATH = 0x806d0018,
            E_PDB_SYMSRV_CACHE_FULL = 0x806d0019,
            E_PDB_MAX = 0x806d001a,
            E_UNEXPECTED = 0x8000ffff
        }
    }
}
