﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using DebugMonitor;
using CrashHandler;
using System.IO;

namespace Xbox360CrashHandler
{
    public class XboxCrashFilesLoader
    {
        private readonly Log m_log;
        private readonly ExceptionHandler m_exceptionHandler;
        private readonly XboxFilesystem m_xboxFilesystem;
        private readonly Settings m_settings;
        private readonly String m_logFilename;
        private String m_xboxBackupLocation;
        private Dictionary<String, FileAttribute> m_filesCopied;
        private const String BACKUP_NAME_FORMAT = "_dd_MM_yy_(hh_mm_ss)";

        public XboxCrashFilesLoader(Log logger, ExceptionHandler exceptionHandler, Settings settings, String logFilename, XboxFilesystem xboxFilesystem)
        {
            m_log = logger;
            m_exceptionHandler = exceptionHandler;
            m_settings = settings;
            m_logFilename = logFilename;
            m_xboxFilesystem = xboxFilesystem;
            m_filesCopied = new Dictionary<String, FileAttribute>();
            if (Directory.Exists(XboxCrashHandler.LOG_FOLDER))
                Directory.Delete(XboxCrashHandler.LOG_FOLDER, true);
            Directory.CreateDirectory(XboxCrashHandler.LOG_FOLDER);
        }

        public String LoadLastCrash()
        {
            m_log.Info("Loading xbox data");
            FileAttribute latestErrorLog;

            try
            {
                latestErrorLog = LoadErrorLog();
                LoadGameLog(latestErrorLog);
                LoadDump(latestErrorLog);

                
            }
            catch (Exception e)
            {
                m_log.Popup("Failed loading one or more files, if you want to see the files I managed to copy go on " + XboxCrashHandler.LOG_FOLDER + Environment.NewLine + " It's possible that latest crash didn't create an error.log or there are no recent crashes");
                m_exceptionHandler.HandleFailedSubmit("Failed loading one or more files: " + e.Message );
                return null;
            }

            LoadExtraAttachments(latestErrorLog);
            return latestErrorLog.Path;           
        }

        private void LoadExtraAttachments(FileAttribute latestErrorLog)
        {
            foreach (String filename in m_settings.ExtraAttachments)
            {
                try
                {
                    m_log.Info("Looking for: " + filename);
                    FileAttribute copiedFile = FindRelatedFileFromSameFolder(filename, latestErrorLog);

                    CopyFile(filename, copiedFile);
                } 
                catch (Exception e)
                {
                    m_log.Info("An error occurred loading " + filename + Environment.NewLine + e.Message);
                }
            }
        }

        private void CopyFile(String filename, FileAttribute fileToCopy)
        {
            bool copyResult = m_xboxFilesystem.CopyAndRename(fileToCopy, XboxCrashHandler.LOG_FOLDER, filename);

            LogCopyResult(filename, fileToCopy, copyResult);

            if (copyResult)
            {
                m_filesCopied.Add(filename, fileToCopy);
            } 
            else 
            {
                throw new Exception("Failed to load " +filename);
            }
        }

        private FileAttribute LoadErrorLog()
        {
            m_log.Info("Looking for: " + m_settings.ErrorLogName);
            FileAttribute latestErrorLog = m_xboxFilesystem.FindLatestFileByName(m_settings.ErrorLogName, 2);
            CopyFile(m_settings.ErrorLogName, latestErrorLog);

            TimeSpan errorLogAge = (DateTime.Now - (DateTime) latestErrorLog.Modified);

            if(errorLogAge.TotalHours > m_settings.MaximumErrorLogAgeInHours)
            {
                m_log.Popup("The crash found is " + Util.PrettyTimespan(errorLogAge) + " old"+ Environment.NewLine + "It's located in: " + latestErrorLog.Path);
            }
            
            m_xboxBackupLocation = Path.Combine(latestErrorLog.Path, m_settings.XboxBackupFolder);
            return latestErrorLog;
        }

        private void LoadGameLog(FileAttribute errorlog){
            m_log.Info("Looking for: " + m_logFilename);
            FileAttribute gameLog = m_xboxFilesystem.GetEarlierFile(errorlog.Path, m_logFilename.Replace(".", ".*\\."), (DateTime)errorlog.Modified);
            CopyFile(m_logFilename, gameLog);
        }

        private void LoadDump(FileAttribute errorlog){
            m_log.Info("Looking for: error.dmp");
            
            FileAttribute dump = m_xboxFilesystem.GetEarlierFile("dumps", ".*\\.dmp", (DateTime)errorlog.Modified);
            try{
                CopyFile("error.dmp", dump);
            }
            catch{
                m_log.Warn("Couldn't load a dump file for the crash");
            }
        }

        private void LogCopyResult(String filename, FileAttribute fileAttribute, bool result)
        {
            if (result)
            {
                m_log.Info("Successfully loaded: " + filename + "(" + fileAttribute.Modified + ")" + Environment.NewLine + "Located in " + Path.Combine(fileAttribute.Path, fileAttribute.Name));
                
            }
            else
            {
                m_log.Info("[ERROR] Couldn't find: " + filename + " for the last crash ");
            }
        }

        private FileAttribute FindRelatedFileFromSameFolder(String filename, FileAttribute referenceFile)
        {
            FileAttribute fileToCopy = m_xboxFilesystem.GetFileInFolder(referenceFile.Path, filename);

            if(fileToCopy == null){
                return null;
            }
            double attachmentDifferenceInMinutes  = ((DateTime)referenceFile.Modified - (DateTime)fileToCopy.Modified ).TotalMinutes;
            if (attachmentDifferenceInMinutes < m_settings.SameFileTimeToleranceInMinutes && attachmentDifferenceInMinutes > -m_settings.SameFileTimeToleranceInMinutes)
            {
                return fileToCopy;
            }
            return null;
        }

        public void MoveCopiedFilesToBackupFolder(String buildNumber)
        {
            m_xboxFilesystem.CreateDirectory(m_xboxBackupLocation);

            String date = DateTime.Now.ToString(BACKUP_NAME_FORMAT);
            foreach(KeyValuePair<String, FileAttribute> fileDefinition in m_filesCopied)
            {
                FileAttribute file = fileDefinition.Value;
                String from = Path.Combine(file.Path, file.Name);
                String backupName = Path.GetFileNameWithoutExtension(fileDefinition.Key) + "_Build(" + buildNumber + ")" + date + Path.GetExtension(fileDefinition.Key);

                String to = Path.Combine(m_xboxBackupLocation, backupName);
                bool copyResult  = m_xboxFilesystem.MoveFile(from, to);
                if (copyResult)
                {
                    m_log.Info("Backed up " + file.Name + " to: " + to);
                }
                else
                {
                    m_log.Warn("Failed to backup" + file.Name + " to: " + to);
                }
            }
        }
    }
}
