﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using DebugMonitor;
using CrashHandler;
using PcCrashHandler = CrashHandler.CrashHandler;

namespace Xbox360CrashHandler
{
    public class XboxCrashHandler : PcCrashHandler
    {
        public static readonly String TEMP_FOLDER = Path.Combine(Path.GetTempPath(), "XBoxCrashHandler");
        public static readonly String LOG_FOLDER = Path.Combine(TEMP_FOLDER, "log");
        public static readonly String PDB_FOLDER = Path.Combine(TEMP_FOLDER, "pdb");
        private readonly XboxCrashFilesLoader m_xboxCrashFilesLoader;
        private readonly XboxFilesystem m_xboxFilesystem;
        
        [STAThread]
        public static void Main(String[] args)
        {
            if (!Directory.Exists(PDB_FOLDER))
                Directory.CreateDirectory(PDB_FOLDER);

            if (!Directory.Exists(LOG_FOLDER))
                Directory.CreateDirectory(LOG_FOLDER);

            String errorLogArgument = null;
            IDictionary<String,String> commandLineArguments = Util.GetNamedArgumentsWithProperties(args);
            if (commandLineArguments.ContainsKey(PcCrashHandler.errorLogNameArgument))
            {
                errorLogArgument = commandLineArguments[PcCrashHandler.errorLogNameArgument];
            }

            PcCrashHandler handler = new XboxCrashHandler(LOG_FOLDER, "Game.log", errorLogArgument);
            handler.HandleGameCrash();

            Environment.Exit(0);
        }

        public XboxCrashHandler(String folder, String logFilename, String errorLog):base(folder, logFilename, errorLog){

            try
            {
                m_xboxFilesystem = new XboxFilesystem();
            }
            catch (Exception e)
            {
                m_log.Popup("Cannot Connect to the default xbox (is it turned on?):" + Environment.NewLine + "Exception:" + e.Message);
                System.Environment.Exit(0);
            }
            this.m_xboxCrashFilesLoader = new XboxCrashFilesLoader(m_log, m_exceptionHandler, m_settings, logFilename, m_xboxFilesystem);
            m_crashInfo.OperatingSystemInfo = LoadXboxOsInfo();
            m_crashInfo.Platform = m_settings.PlatformXbox;
        }

	//in this way the default behaviour (ie. loading the directx info) is overwritten for xbox
        protected override void RetrievePlatformAttachments()
        {
            
        }

        private String LoadXboxOsInfo()
        {
            XboxSystemInfo info = new XboxSystemInfo();
            String fullDetails = Environment.NewLine;
            if(info.IsValid())
            {
                fullDetails += "Kernel Version: " + info.KernelVersion + Environment.NewLine;
                fullDetails += "Base Kernel Version: " + info.BaseKernelVersion + Environment.NewLine;
                fullDetails += "Xdk Version: " + info.XdkVersion + Environment.NewLine;
            }
            else
            {
                fullDetails = "<no Xbox system info found>";
            }
            return fullDetails;
        }
        protected override void RetrieveLogFiles()
        {
            try
            {
                String pathToError = m_xboxCrashFilesLoader.LoadLastCrash();
                String errorLogLocation = Path.Combine(LOG_FOLDER, m_settings.ErrorLogName);
                
                XboxCallStackDecoder decoder = new XboxCallStackDecoder(errorLogLocation, m_log, pathToError,m_settings, m_exceptionHandler, m_xboxFilesystem);
                System.IO.File.WriteAllText(errorLogLocation, decoder.GetCallStackTrace());
            }
            catch (System.Exception ex)
            {
                m_log.Info(ex.StackTrace);
                m_exceptionHandler.HandleFailedSubmit("error trying to retrieve log files", ex );
            }
            
        }

        public override JiraIssueData ReportIssue()
        {
            JiraIssueData result = base.ReportIssue();
            if (result != null)
            {
                m_xboxCrashFilesLoader.MoveCopiedFilesToBackupFolder(m_crashInfo.ProductVersion);
            }
            return result;
        }
    }
}
