﻿using System;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using DebugMonitor;

namespace Xbox360CrashHandler
{
    public class XboxFilesystem
    {
        private static string ROOT_FOLDER = @"E:\";
        private DebugMonitor.Connection m_xboxConnection;
        private FileManagement m_fileManagement;

        public XboxFilesystem()
        {
            m_xboxConnection = new DebugMonitor.Connection();
            if (!m_xboxConnection.ConnectionEsablished())
            {
                throw new Exception("Cannot connect to the default Xbox");
            }
            m_fileManagement = new FileManagement();
        }

        public XboxFilesystem(String xboxName)
        {
            m_xboxConnection = new DebugMonitor.Connection(xboxName);
        }

        public FileAttribute GetFileInFolder(String folder, String filename)
        {
            List<FileAttribute> filesInFolder = GetFileList(folder);
            foreach (FileAttribute file in filesInFolder)
            {
                if (file.Name.Equals(filename))
                {
                    return file;
                }
            }
            return null;
        }

        public List<FileAttribute> GetFileList(String folder)
        {
            if(!m_xboxConnection.ConnectionEsablished())
            {
                throw new Exception("Couldn't connect to the xbox");
            }

            List<FileAttribute> result = new List<FileAttribute>();
            DebugMonitor.WalkDirectory walkDirectory = new DebugMonitor.WalkDirectory(folder);
            foreach (DebugMonitor.FileAttribute attribute in walkDirectory)
            {
                result.Add(attribute);
            }
            return result;
        }

        public FileAttribute GetEarlierFile(String path, String fileNameRegex, DateTime referenceDate)
        {
            FileAttribute result = null;
            List<FileAttribute> relatedFiles = GetAllFiles(path, 1);
            foreach (FileAttribute file in relatedFiles)
            {
                if (Regex.IsMatch(file.Name, fileNameRegex, RegexOptions.IgnoreCase) && (DateTime)file.Modified >= referenceDate)
                {
                    if (result == null || (DateTime)file.Modified < (DateTime)result.Modified)
                    {
                        result = file;
                    }
                }
            }
            return result;
        }

        public List<FileAttribute> GetAllFiles(String path, int maxDepth)
        {
            Stack<String> missedDirectories = new Stack<String>();
            List<FileAttribute> result = new List<FileAttribute>();
            String absolutePath = GetRootedPath(path);
            
            missedDirectories.Push(absolutePath);
            while(missedDirectories.Count > 0)
            {
                String currentDirectory = missedDirectories.Pop();
                List<FileAttribute> directoryContent = GetFileList(currentDirectory);
                foreach (FileAttribute content in directoryContent)
                {
                    if(content.IsDirectory)
                    {
                        String fullPath = Path.Combine(currentDirectory,content.Name);
                        if (CalculateRelativeDepth(absolutePath, fullPath) <= maxDepth)
                        {
                            missedDirectories.Push(fullPath);
                        }
                    } 
                    else 
                    {
                        result.Add(content);
                    }
                }
            }
            return result;
        }

        public bool ReceiveFile(FileAttribute from, String to)
        {
            if (from == null || to == null)
            {
                return false;
            }

            String xboxOrigin = Path.Combine(from.Path, from.Name);
            String filesystemDestination = Path.Combine(to, from.Name);
            return ReceiveFile(xboxOrigin, filesystemDestination);
        }

        public bool ReceiveFile(String from, String to)
        {
            if (File.Exists(to))
            {
                File.Delete(to);
            }

            return m_fileManagement.ReceiveFile(from, to);
        }

        public bool MoveFile(String from, String to)
        {
            String absoluteXboxFromPath = GetRootedPath(from);
            String absoluteXboxToPath = GetRootedPath(to);
            return m_fileManagement.RenameFile(absoluteXboxFromPath, absoluteXboxToPath);
        }

        public bool CreateDirectory(String xboxPath)
        {
            String absoluteXboxPath = GetRootedPath(xboxPath);

            return m_fileManagement.CreateDirectory(absoluteXboxPath);
        }

        private String GetRootedPath(String path)
        {
            return Path.IsPathRooted(path) ? path : Path.Combine(ROOT_FOLDER, path);
        }
        

        public FileAttribute FindLatestFileByName(String fileName, int levels)
        {
            List<FileAttribute> files = GetAllFiles(String.Empty, levels);

            DateTime latestDate = DateTime.MinValue;
            FileAttribute result = null;
            foreach (FileAttribute file in files)
            {
                if (((DateTime)file.Modified) > latestDate && file.Name.Trim().ToLower().Equals(fileName.ToLower()))
                {
                    result = file;
                    latestDate = (DateTime)file.Modified;
                }
            }
            return result;
        }

        private int CalculateRelativeDepth(String basePath, String fullPath)
        {
            String relativePath = fullPath.Replace(basePath, "");
            return relativePath.Split(new String[]{"\\"}, StringSplitOptions.RemoveEmptyEntries).Length;
        }

        public bool CopyAndRename(FileAttribute closestMatch, String path, String filename)
        {
            bool success = false;
            if (closestMatch != null)
            {
                success = ReceiveFile(closestMatch, path);
                File.Move(Path.Combine(path, closestMatch.Name), Path.Combine(path, filename));
            }

            return success;
        }
    }
}
