#include "StdAfx.h"
#include "PathHelpers.h"

// Returns position of last extension in last name (string::npos if not found)
// note: returns string::npos for names starting from '.' and having no
// '.' later (for example 'aaa/.ccc', 'a:.abc', '.rc')
static size_t findExtensionPosition(const string& path)
{
	const size_t dotPos = path.rfind('.');
	if (dotPos == string::npos)
	{
		return string::npos;
	}

	const size_t separatorPos = path.find_last_of("\\/:");
	if (separatorPos != string::npos)
	{
		if (separatorPos + 1 >= dotPos)
		{
			return string::npos;
		}		
	}
	else if (dotPos == 0)
	{
		return string::npos;
	}

	return dotPos + 1;
}

string PathHelpers::FindExtension(const string& path)
{
	const size_t extPos = findExtensionPosition(path);
	return (extPos == string::npos) ? string() : path.substr(extPos, string::npos);
}

string PathHelpers::ReplaceExtension(const string& path, const string& newExtension)
{
	if (path.empty())
	{
		return string();
	}

	if (newExtension.empty())
	{
		return RemoveExtension(path);
	}

	const char last = path[path.length() - 1];
	if ((last == '\\') || (last == '/') || (last == ':') || (last == '.'))
	{
		return path;
	}

	const size_t extPos = findExtensionPosition(path);
	return ((extPos == string::npos) ? path + "." : path.substr(0, extPos)) + newExtension;
}

string PathHelpers::RemoveExtension(const string& path)
{
	const size_t extPos = findExtensionPosition(path);
	return (extPos == string::npos) ? path : path.substr(0, extPos - 1);
}

string PathHelpers::GetDirectory(const string& path)
{
	int slashPos;
	for (slashPos = int(path.size()) - 1; slashPos >= 0 && strchr("/\\", path[slashPos]) == 0; --slashPos);
	if (slashPos == -1)
		return "";
	return path.substr(0, slashPos);
}

string PathHelpers::GetFilename(const string& path)
{
	int slashPos;
	for (slashPos = int(path.size()) - 1; slashPos >= 0 && strchr("/\\", path[slashPos]) == 0; --slashPos);
	if (slashPos == -1)
		return path;
	return path.substr(slashPos + 1, string::npos);
}

string PathHelpers::AddSeparator(const string& path)
{
	if (path.empty())
		return string();
	int last = path[path.length() - 1];
	if (last != '/' && last != '\\')
		return path + '\\';
	return path;
}

string PathHelpers::RemoveSeparator(const string& path)
{
	if (path.empty())
		return string();
	int last = path[path.length() - 1];
	if (last != '/' && last != '\\')
		return path;
	return path.substr(0, path.length() - 1);
}

string PathHelpers::Join(const string& path1, const string& path2)
{
	if (path1.empty())
		return path2;
	if (!path2.empty() && (path2[0] == '/' || path2[0] == '\\'))
	{
		return PathHelpers::AddSeparator(path1) + path2.substr(1); // Skip first seprator
	}
	return PathHelpers::AddSeparator(path1) + path2;
}

string PathHelpers::GetParentDirectory(const string& path)
{
	string pathWithoutSlash = PathHelpers::RemoveSeparator(path);
	return PathHelpers::GetDirectory(path);
}

bool PathHelpers::MatchWildcard(const string& text, const string& wildcard)
{
	const char* pString = &text[0], *pWildcard = &wildcard[0];
	// skip the obviously the same starting substring
	while (*pWildcard && *pWildcard != '*' && *pWildcard != '?')
		if (*pString != *pWildcard)
			return false; // must be exact match unless there's a wildcard character in the wildcard string
		else
			++pString, ++pWildcard;

	if (!*pString)
	{
		// this will only match if there are no non-wild characters in the wildcard
		for(; *pWildcard; ++pWildcard)
			if (*pWildcard != '*' && *pWildcard != '?')
				return false;
		return true;
	}

	switch(*pWildcard)
	{
	case '\0':
		return false; // the only way to match them after the leading non-wildcard characters is !*pString, which was already checked

		// we have a wildcard with wild character at the start.
	case '*':
		{
			// merge consecutive ? and *, since they are equivalent to a single *
			while (*pWildcard == '*' || *pWildcard == '?')
				++pWildcard;

			if (!*pWildcard)
				return true; // the rest of the string doesn't matter: the wildcard ends with *

			for (; *pString; ++pString)
				if (MatchWildcard(pString, pWildcard))
					return true;

			return false;
		}
	case '?':
		return MatchWildcard(pString+1, pWildcard + 1) || MatchWildcard(pString, pWildcard+1);
	default:
		assert (0);
		return false;
	}
}

bool PathHelpers::IsRelative(const string& path)
{
	bool isRelative = true;
	isRelative = isRelative && (path.size() < 2 || path[1] != ':'); // Does path start with drive specifier?
	isRelative = isRelative && (path.size() < 2 || path[0] != '\\' || path[1] != '\\'); // Is UNC path?
	isRelative = isRelative && (path.size() < 1 || (path[0] != '\\' && path[0] != '/')); // Starts with slash?
	return isRelative;
}

string PathHelpers::ToUnixPath(const string& path)
{
	string s(path);
	std::replace(s.begin(), s.end(), '\\', '/');
	return s;
}

string PathHelpers::ToDosPath(const string& path)
{
	string s(path);
	std::replace(s.begin(), s.end(), '/', '\\' );
	return s;
}


#if !defined(CRY_STRING)
// Returns position of last extension in last name (wstring::npos if not found)
// note: returns wstring::npos for names starting from '.' and having no
// '.' later (for example 'aaa/.ccc', 'a:.abc', '.rc')
static size_t findExtensionPosition(const wstring& path)
{
	const size_t dotPos = path.rfind(L'.');
	if (dotPos == wstring::npos)
	{
		return wstring::npos;
	}

	const size_t separatorPos = path.find_last_of(L"\\/:");
	if (separatorPos != wstring::npos)
	{
		if (separatorPos + 1 >= dotPos)
		{
			return wstring::npos;
		}		
	}
	else if (dotPos == 0)
	{
		return wstring::npos;
	}

	return dotPos + 1;
}

wstring PathHelpers::FindExtension(const wstring& path)
{
	const size_t extPos = findExtensionPosition(path);
	return (extPos == wstring::npos) ? wstring() : path.substr(extPos, wstring::npos);
}

wstring PathHelpers::ReplaceExtension(const wstring& path, const wstring& newExtension)
{
	if (path.empty())
	{
		return wstring();
	}

	if (newExtension.empty())
	{
		return RemoveExtension(path);
	}

	const wchar_t last = path[path.length() - 1];
	if ((last == L'\\') || (last == L'/') || (last == L':') || (last == L'.'))
	{
		return path;
	}

	const size_t extPos = findExtensionPosition(path);
	return ((extPos == wstring::npos) ? path + L"." : path.substr(0, extPos)) + newExtension;
}

wstring PathHelpers::RemoveExtension(const wstring& path)
{
	const size_t extPos = findExtensionPosition(path);
	return (extPos == wstring::npos) ? path : path.substr(0, extPos - 1);
}

wstring PathHelpers::GetDirectory(const wstring& path)
{
	int slashPos;
	for (slashPos = int(path.size()) - 1; slashPos >= 0 && wcschr(L"/\\", path[slashPos]) == 0; --slashPos);
	if (slashPos == -1)
		return wstring();
	return path.substr(0, slashPos);
}

wstring PathHelpers::GetFilename(const wstring& path)
{
	int slashPos;
	for (slashPos = int(path.size()) - 1; slashPos >= 0 && wcschr(L"/\\", path[slashPos]) == 0; --slashPos);
	if (slashPos == -1)
		return path;
	return path.substr(slashPos + 1, wstring::npos);
}

wstring PathHelpers::AddSeparator(const wstring& path)
{
	if (path.empty())
		return wstring();
	const wchar_t last = path[path.length() - 1];
	if (last != L'/' && last != L'\\')
		return path + L'\\';
	return path;
}

wstring PathHelpers::RemoveSeparator(const wstring& path)
{
	if (path.empty())
		return wstring();
	const wchar_t last = path[path.length() - 1];
	if (last != L'/' && last != L'\\')
		return path;
	return path.substr(0, path.length() - 1);
}

wstring PathHelpers::Join(const wstring& path1, const wstring& path2)
{
	if (path1.empty())
		return path2;
	if (!path2.empty() && (path2[0] == L'/' || path2[0] == L'\\'))
	{
		return PathHelpers::AddSeparator(path1) + path2.substr(1); // Skip first separator
	}
	return PathHelpers::AddSeparator(path1) + path2;
}

wstring PathHelpers::GetParentDirectory(const wstring& path)
{
	wstring pathWithoutSlash = PathHelpers::RemoveSeparator(path);
	return PathHelpers::GetDirectory(path);
}

bool PathHelpers::MatchWildcard(const wstring& text, const wstring& wildcard)
{
	const wchar_t* pString = &text[0], *pWildcard = &wildcard[0];
	// skip the obviously the same starting substring
	while (*pWildcard && *pWildcard != L'*' && *pWildcard != L'?')
		if (*pString != *pWildcard)
			return false; // must be exact match unless there's a wildcard character in the wildcard wstring
		else
			++pString, ++pWildcard;

	if (!*pString)
	{
		// this will only match if there are no non-wild characters in the wildcard
		for(; *pWildcard; ++pWildcard)
			if (*pWildcard != L'*' && *pWildcard != L'?')
				return false;
		return true;
	}

	switch (*pWildcard)
	{
	case L'\0':
		return false; // the only way to match them after the leading non-wildcard characters is !*pString, which was already checked

		// we have a wildcard with wild character at the start.
	case L'*':
		{
			// merge consecutive ? and *, since they are equivalent to a single *
			while (*pWildcard == L'*' || *pWildcard == L'?')
				++pWildcard;

			if (!*pWildcard)
				return true; // the rest of the wstring doesn't matter: the wildcard ends with *

			for (; *pString; ++pString)
				if (MatchWildcard(pString, pWildcard))
					return true;

			return false;
		}
	case L'?':
		return MatchWildcard(pString+1, pWildcard + 1) || MatchWildcard(pString, pWildcard+1);
	default:
		assert(0);
		return false;
	}
}

bool PathHelpers::IsRelative(const wstring& path)
{
	bool isRelative = true;
	isRelative = isRelative && (path.size() < 2 || path[1] != L':'); // Does path start with drive specifier?
	isRelative = isRelative && (path.size() < 2 || path[0] != L'\\' || path[1] != L'\\'); // Is UNC path?
	isRelative = isRelative && (path.size() < 1 || (path[0] != L'\\' && path[0] != L'/')); // Starts with slash?
	return isRelative;
}

wstring PathHelpers::ToUnixPath(const wstring& path)
{
	wstring s(path);
	std::replace(s.begin(), s.end(), L'\\', L'/');
	return s;
}

wstring PathHelpers::ToDosPath(const wstring& path)
{
	wstring s(path);
	std::replace(s.begin(), s.end(), L'/', L'\\');
	return s;
}

#endif //!defined(CRY_STRING)
