#include "StdAfx.h"
#include "StringHelpers.h"
#include <cctype>
#include <algorithm>

bool StringHelpers::Equals(const string& str, const string& pattern)
{
	if (str.length() != pattern.length())
	{
		return false;
	}
	return std::memcmp(str.data(), pattern.data(), pattern.length()) == 0;
}

bool StringHelpers::EqualsIgnoreCase(const string& str, const string& pattern)
{
	if (str.length() != pattern.length())
	{
		return false;
	}
	return memicmp(str.data(), pattern.data(), pattern.length()) == 0;
}

bool StringHelpers::StartsWith(const string& str, const string& pattern)
{
	if (str.length() < pattern.length())
	{
		return false;
	}
	return std::memcmp(str.data(), pattern.data(), pattern.length()) == 0;
}

bool StringHelpers::StartsWithIgnoreCase(const string& str, const string& pattern)
{
	if (str.length() < pattern.length())
	{
		return false;
	}
	return memicmp(str.data(), pattern.data(), pattern.length()) == 0;
}

bool StringHelpers::EndsWith(const string& str, const string& pattern)
{
	if (str.length() < pattern.length())
	{
		return false;
	}
	return std::memcmp(str.data() + str.length() - pattern.length(), pattern.data(), pattern.length()) == 0;
}

bool StringHelpers::EndsWithIgnoreCase(const string& str, const string& pattern)
{
	if (str.length() < pattern.length())
	{
		return false;
	}
	return memicmp(str.data() + str.length() - pattern.length(), pattern.data(), pattern.length()) == 0;
}

bool StringHelpers::MatchesWildcardsIgnoreCase(const string& str, const string& wildcards)
{
	const char* savedStr;
	const char* savedWild;

	const char* pStr = str.c_str();
	const char* pWild = wildcards.c_str();

	while ((*pStr) && (*pWild != '*')) 
	{
		if ((tolower(*pWild) != tolower(*pStr)) && (*pWild != '?'))
		{
			return false;
		}
		++pWild;
		++pStr;
	}

	while (*pStr) 
	{
		if (*pWild == '*') 
		{
			if (!*++pWild) 
			{
				return true;
			}
			savedWild = pWild;
			savedStr = pStr + 1;
		} 
		else if ((tolower(*pWild) == tolower(*pStr)) || (*pWild == '?')) 
		{
			++pWild;
			++pStr;
		} 
		else 
		{
			pWild = savedWild;
			pStr = savedStr++;
		}
	}

	while (*pWild == '*') 
	{
		++pWild;
	}

	return *pWild == 0;
}

bool StringHelpers::MatchesWildcardsIgnoreCaseExt(const string& str, const string& wildcards, std::vector<string>& wildcardMatches)
{
	const char* savedStrBegin;
	const char* savedStrEnd;
	const char* savedWild;
	size_t savedWildCount;

	const char* pStr = str.c_str();
	const char* pWild = wildcards.c_str();
	size_t wildCount = 0;

	while ((*pStr) && (*pWild != '*')) 
	{
		if (*pWild == '?')
		{
			wildcardMatches.push_back(string(pStr, pStr + 1));
			++wildCount;
		}
		else if (tolower(*pWild) != tolower(*pStr))
		{
			wildcardMatches.resize(wildcardMatches.size() - wildCount);
			return false;
		}
		++pWild;
		++pStr;
	}

	while (*pStr) 
	{
		if (*pWild == '*') 
		{
			if (!pWild[1]) 
			{
				wildcardMatches.push_back(string(pStr));
				return true;
			}
			savedWild = pWild++;
			savedStrBegin = pStr;
			savedStrEnd = pStr;
			savedWildCount = wildCount;
			wildcardMatches.push_back(string(savedStrBegin, savedStrEnd));
			++wildCount;
		} 
		else if (*pWild == '?')
		{
			wildcardMatches.push_back(string(pStr, pStr + 1));
			++wildCount;
			++pWild;
			++pStr;
		} 
		else if (tolower(*pWild) == tolower(*pStr)) 
		{
			++pWild;
			++pStr;
		} 
		else 
		{
			wildcardMatches.resize(wildcardMatches.size() - (wildCount - savedWildCount));
			wildCount = savedWildCount;
			++savedStrEnd;
			pWild = savedWild + 1;
			pStr = savedStrEnd;
			wildcardMatches.push_back(string(savedStrBegin, savedStrEnd));
			++wildCount;
		}
	}

	while (*pWild == '*') 
	{
		++pWild;
		wildcardMatches.push_back(string());
		++wildCount;
	}

	if (*pWild)
	{
		wildcardMatches.resize(wildcardMatches.size() - wildCount);
		return false;
	}

	return true;
}

string StringHelpers::TrimLeft(const string& s)
{
	const size_t first = s.find_first_not_of(" \r\t");
	return (first == s.npos) ? string() : s.substr(first);
}

string StringHelpers::TrimRight(const string& s)
{
	const size_t last = s.find_last_not_of(" \r\t");
	return (last == s.npos) ? s : s.substr(0,last+1);
}

string StringHelpers::Trim(const string& s)
{
	return TrimLeft(TrimRight(s));
}

string StringHelpers::RemoveDuplicateSpaces(const string& s)
{
	string res;
	bool spaceFound = false;

	for (size_t i = 0, n = s.length(); i < n; ++i)
	{
		if ((s[i]==' ') || (s[i]=='\r') || (s[i]=='\t'))
		{
			spaceFound = true;
		}
		else
		{
			if (spaceFound)
			{
				res += ' ';
				spaceFound = false;
			}
			res += s[i];
		}
	}

	if (spaceFound)
	{
		res += ' ';
	}
	return res;
}

string StringHelpers::MakeLowerCase(const string& s)
{
	string copy = s;
	std::transform(copy.begin(), copy.end(), copy.begin(), tolower);
	return copy;
}

string StringHelpers::MakeUpperCase(const string& s)
{
	string copy = s;
	std::transform(copy.begin(), copy.end(), copy.begin(), toupper);
	return copy;
}

string StringHelpers::Replace(const string& s, char oldChar, char newChar)
{
	string copy = s;
	std::replace(copy.begin(), copy.end(), oldChar, newChar);
	return copy;
}

void StringHelpers::ConvertStringByRef(string& out, const string& in)
{
	out = in;
}

void StringHelpers::Split(const string& str, const string& separator, bool bReturnEmptyPartsToo, std::vector<string>& outParts)
{
	if (str.empty())
	{
		return;
	}

	if (separator.empty())
	{
		for (size_t i = 0; i < str.length(); ++i)
		{
			outParts.push_back(str.substr(i, 1));
		}
		return;		
	}

	size_t partStart = 0;

	for (;;)
	{
		const size_t pos = str.find(separator, partStart);
		if (pos == string::npos)
		{
			break;
		}
		if (bReturnEmptyPartsToo || (partStart < pos))
		{
			outParts.push_back(str.substr(partStart, pos - partStart));
		}
		partStart = pos + separator.length();
	}

	if (bReturnEmptyPartsToo || (partStart < str.length()))
	{
		outParts.push_back(str.substr(partStart, str.length() - partStart));
	}
}


#if !defined(CRY_STRING)
bool StringHelpers::Equals(const wstring& str, const wstring& pattern)
{
	if (str.length() != pattern.length())
	{
		return false;
	}
	return std::memcmp(str.data(), pattern.data(), pattern.length()*sizeof(wstring::value_type)) == 0;
}

bool StringHelpers::EqualsIgnoreCase(const wstring& str, const wstring& pattern)
{
	const size_t patternLength = pattern.length();
	if (str.length() != patternLength)
	{
		return false;
	}
	for (size_t i = 0; i < patternLength; ++i)
	{
		if (towlower(str[i]) != towlower(pattern[i]))
		{
			return false;
		}
	}
	return true;
}

bool StringHelpers::StartsWith(const wstring& str, const wstring& pattern)
{
	if (str.length() < pattern.length())
	{
		return false;
	}
	return std::memcmp(str.data(), pattern.data(), pattern.length()*sizeof(wstring::value_type)) == 0;
}

bool StringHelpers::StartsWithIgnoreCase(const wstring& str, const wstring& pattern)
{
	const size_t patternLength = pattern.length();
	if (str.length() < patternLength)
	{
		return false;
	}
	for (size_t i = 0; i < patternLength; ++i)
	{
		if (towlower(str[i]) != towlower(pattern[i]))
		{
			return false;
		}
	}
	return true;
}

bool StringHelpers::EndsWith(const wstring& str, const wstring& pattern)
{
	if (str.length() < pattern.length())
	{
		return false;
	}
	return std::memcmp(str.data() + str.length() - pattern.length(), pattern.data(), pattern.length()*sizeof(wstring::value_type)) == 0;
}

bool StringHelpers::EndsWithIgnoreCase(const wstring& str, const wstring& pattern)
{
	const size_t patternLength = pattern.length();
	if (str.length() < patternLength)
	{
		return false;
	}
	for (size_t i = str.length() - patternLength, j = 0; i < patternLength; ++i, ++j)
	{
		if (towlower(str[i]) != towlower(pattern[j]))
		{
			return false;
		}
	}
	return true;
}

bool StringHelpers::MatchesWildcardsIgnoreCase(const wstring& str, const wstring& wildcards)
{
	const wchar_t* savedStr;
	const wchar_t* savedWild;

	const wchar_t* pStr = str.c_str();
	const wchar_t* pWild = wildcards.c_str();

	while ((*pStr) && (*pWild != L'*')) 
	{
		if ((towlower(*pWild) != towlower(*pStr)) && (*pWild != L'?'))
		{
			return false;
		}
		++pWild;
		++pStr;
	}

	while (*pStr) 
	{
		if (*pWild == L'*') 
		{
			if (!*++pWild) 
			{
				return true;
			}
			savedWild = pWild;
			savedStr = pStr + 1;
		} 
		else if ((towlower(*pWild) == towlower(*pStr)) || (*pWild == L'?')) 
		{
			++pWild;
			++pStr;
		} 
		else 
		{
			pWild = savedWild;
			pStr = savedStr++;
		}
	}

	while (*pWild == L'*') 
	{
		++pWild;
	}

	return *pWild == L'\0';
}

bool StringHelpers::MatchesWildcardsIgnoreCaseExt(const wstring& str, const wstring& wildcards, std::vector<wstring>& wildcardMatches)
{
	const wchar_t* savedStrBegin;
	const wchar_t* savedStrEnd;
	const wchar_t* savedWild;
	size_t savedWildCount;

	const wchar_t* pStr = str.c_str();
	const wchar_t* pWild = wildcards.c_str();
	size_t wildCount = 0;

	while ((*pStr) && (*pWild != L'*')) 
	{
		if (*pWild == L'?')
		{
			wildcardMatches.push_back(wstring(pStr, pStr + 1));
			++wildCount;
		}
		else if (towlower(*pWild) != towlower(*pStr))
		{
			wildcardMatches.resize(wildcardMatches.size() - wildCount);
			return false;
		}
		++pWild;
		++pStr;
	}

	while (*pStr) 
	{
		if (*pWild == L'*') 
		{
			if (!pWild[1]) 
			{
				wildcardMatches.push_back(wstring(pStr));
				return true;
			}
			savedWild = pWild++;
			savedStrBegin = pStr;
			savedStrEnd = pStr;
			savedWildCount = wildCount;
			wildcardMatches.push_back(wstring(savedStrBegin, savedStrEnd));
			++wildCount;
		} 
		else if (*pWild == L'?')
		{
			wildcardMatches.push_back(wstring(pStr, pStr + 1));
			++wildCount;
			++pWild;
			++pStr;
		} 
		else if (towlower(*pWild) == towlower(*pStr)) 
		{
			++pWild;
			++pStr;
		} 
		else 
		{
			wildcardMatches.resize(wildcardMatches.size() - (wildCount - savedWildCount));
			wildCount = savedWildCount;
			++savedStrEnd;
			pWild = savedWild + 1;
			pStr = savedStrEnd;
			wildcardMatches.push_back(wstring(savedStrBegin, savedStrEnd));
			++wildCount;
		}
	}

	while (*pWild == L'*') 
	{
		++pWild;
		wildcardMatches.push_back(wstring());
		++wildCount;
	}

	if (*pWild)
	{
		wildcardMatches.resize(wildcardMatches.size() - wildCount);
		return false;
	}

	return true;
}

wstring StringHelpers::TrimLeft(const wstring& s)
{
	const size_t first = s.find_first_not_of(L" \r\t");
	return (first == s.npos) ? wstring() : s.substr(first);
}

wstring StringHelpers::TrimRight(const wstring& s)
{
	const size_t last = s.find_last_not_of(L" \r\t");
	return (last == s.npos) ? s : s.substr(0,last+1);
}

wstring StringHelpers::Trim(const wstring& s)
{
	return TrimLeft(TrimRight(s));
}

wstring StringHelpers::RemoveDuplicateSpaces(const wstring& s)
{
	wstring res;
	bool spaceFound = false;

	for (size_t i = 0, n = s.length(); i < n; ++i)
	{
		if ((s[i]==' ') || (s[i]=='\r') || (s[i]=='\t'))
		{
			spaceFound = true;
		}
		else
		{
			if (spaceFound)
			{
				res += ' ';
				spaceFound = false;
			}
			res += s[i];
		}
	}

	if (spaceFound)
	{
		res += ' ';
	}
	return res;
}

wstring StringHelpers::MakeLowerCase(const wstring& s)
{
	wstring copy = s;
	std::transform(copy.begin(), copy.end(), copy.begin(), towlower);
	return copy;
}

wstring StringHelpers::MakeUpperCase(const wstring& s)
{
	wstring copy = s;
	std::transform(copy.begin(), copy.end(), copy.begin(), towupper);
	return copy;
}

wstring StringHelpers::Replace(const wstring& s, wchar_t oldChar, wchar_t newChar)
{
	wstring copy = s;
	std::replace(copy.begin(), copy.end(), oldChar, newChar);
	return copy;
}

void StringHelpers::ConvertStringByRef(wstring& out, const string& in)
{
	std::vector<wchar_t> buf(in.length() + 1);
	if (mbstowcs(&buf[0], in.c_str(), buf.size()) < 0)
	{
		out = L"";
	}
	else
	{
		out = &buf[0];
	}
}

void StringHelpers::ConvertStringByRef(string& out, const wstring& in)
{
	std::vector<char> buf(4 * (in.length()+1));
	if (wcstombs(&buf[0], in.c_str(), buf.size()) < 0)
	{
		out = "";
	}
	else
	{
		out = &buf[0];
	}
}

void StringHelpers::ConvertStringByRef(wstring& out, const wstring& in)
{
	out = in;
}

void StringHelpers::Split(const wstring& str, const wstring& separator, bool bReturnEmptyPartsToo, std::vector<wstring>& outParts)
{
	if (str.empty())
	{
		return;
	}

	if (separator.empty())
	{
		for (size_t i = 0; i < str.length(); ++i)
		{
			outParts.push_back(str.substr(i, 1));
		}
		return;		
	}

	size_t partStart = 0;

	for (;;)
	{
		const size_t pos = str.find(separator, partStart);
		if (pos == wstring::npos)
		{
			break;
		}
		if (bReturnEmptyPartsToo || (partStart < pos))
		{
			outParts.push_back(str.substr(partStart, pos - partStart));
		}
		partStart = pos + separator.length();
	}

	if (bReturnEmptyPartsToo || (partStart < str.length()))
	{
		outParts.push_back(str.substr(partStart, str.length() - partStart));
	}
}

#endif //!defined(CRY_STRING)
