#include "../StdTypes.hpp"
#ifdef _MSC_VER
#include <process.h>
#include <direct.h>
#endif
#ifdef UNIX
#include <pthread.h>
#endif
#include "../Error.hpp"
#include "../STLHelper.hpp"
#include "../tinyxml/tinyxml.h"
#include "CrySimpleSock.hpp"
#include "CrySimpleJobCompile.hpp"
#include "CrySimpleServer.hpp"
#include "CrySimpleCache.hpp"
#include "ShaderList.hpp"

#define MAX_COMPILER_WAIT_TIME (60*1000)

volatile long CCrySimpleJobCompile::m_GlobalCompileTasks			= 0;
volatile long CCrySimpleJobCompile::m_RemoteServerID					= 0;
volatile long long CCrySimpleJobCompile::m_GlobalCompileTime	=	0;

struct STimer
{
	__int64 m_freq;
	STimer()
	{
		QueryPerformanceFrequency((LARGE_INTEGER *)&m_freq);
	}
	__int64 GetTime() const
	{
		__int64 t;
		QueryPerformanceCounter((LARGE_INTEGER *)&t);
		return t;
	}

	double TimeToSeconds( __int64 t )
	{
		return ((double)t)/m_freq;
	}
};


STimer g_Timer;

CCrySimpleJobCompile::CCrySimpleJobCompile(uint32_t requestIP,std::vector<uint8_t>& rVec,size_t Size,const TiXmlElement* pElement,EProtocolVersion Version):
CCrySimpleJobCache(requestIP,rVec,Size)
{
	InterlockedIncrement(&m_GlobalCompileTasks);
	if(State()==ECSJS_CACHEHIT)
	{
		State(ECSJS_DONE);
		return;
	}

	if(!SEnviropment::Instance().m_FallbackServer.empty() && m_GlobalCompileTasks > SEnviropment::Instance().m_FallbackTreshold)
	{
		tdEntryVec ServerVec;
		CSTLHelper::Tokenize(ServerVec,SEnviropment::Instance().m_FallbackServer,";");
		uint32_t Idx=m_RemoteServerID++;
		uint32_t Count=(uint32_t)ServerVec.size();
		std::string Server	=	ServerVec[Idx%Count];
		printf("  Remote Compile on %s ...\n",Server.c_str());
		CCrySimpleSock Sock(Server,SEnviropment::Instance().m_port);
		if(Sock.Valid())
		{
			Sock.Forward(rVec);
			std::vector<uint8_t> Tmp;
			if(Sock.Backward(Tmp))
			{
				rVec	=	Tmp;
				if(Tmp.size()<=4 || (Version==EPV_V002 && Tmp[4]!=ECSJS_DONE))
				{
					Error("failed to compile request");
					return;
				}
				State(ECSJS_DONE);
				//printf("done\n");
			}
			else
				printf("failed, fallback to local\n");
		}
		else
			printf("failed, fallback to local\n");
	}
	if(State()==ECSJS_NONE)
	{
		if(!Compile(pElement,rVec) || rVec.size() == 0)
		{
			Error("failed to compile request");
			return;
		}

		tdDataVector rDataRaw;
		rDataRaw.swap(rVec);
		if (!CSTLHelper::Compress( rDataRaw,rVec ))
		{
			Error("failed to compress request");
			return;
		}
		State(ECSJS_DONE);
	}

	// Cache compiled data
	const char* pCaching	=	pElement->Attribute("Caching");
	if(State()!=ECSJS_ERROR && (!pCaching || std::string(pCaching)=="1"))
		CCrySimpleCache::Instance().Add(HashID(),rVec);
}

CCrySimpleJobCompile::~CCrySimpleJobCompile()
{
	InterlockedDecrement(&m_GlobalCompileTasks);
}

bool CCrySimpleJobCompile::Compile(const TiXmlElement* pElement,std::vector<uint8_t>& rVec)
{
	const char* pProfile			=	pElement->Attribute("Profile");
	const char* pProgram			=	pElement->Attribute("Program");
	const char* pEntry				=	pElement->Attribute("Entry");
	const char* pCompileFlags	=	pElement->Attribute("CompileFlags");

	const char* pShaderRequestLine =	pElement->Attribute("ShaderRequest");
	
	if(!pProfile)
	{
		CrySimple_ERROR("failed to extract Profile of the request");
		return false;
	}
	if(!pProgram)
	{
		CrySimple_ERROR("failed to extract Program of the request");
		return false;
	}
	if(!pEntry)
	{
		CrySimple_ERROR("failed to extract Entry of the request");
		return false;
	}
	if(!pCompileFlags)
	{
		CrySimple_ERROR("failed to extract Compile+Flags of the request");
		return false;
	}

	static long volatile nTmpCounter = 0;

	InterlockedIncrement( &nTmpCounter );

	char tmpstr[64];
	sprintf( tmpstr,"%d",nTmpCounter );

	const std::string TmpIn	=	SEnviropment::Instance().m_Temp+tmpstr+".In";
	const std::string TmpOut=	SEnviropment::Instance().m_Temp+tmpstr+".Out";
	CSTLHelper::ToFile(TmpIn,std::vector<uint8_t>(pProgram,&pProgram[strlen(pProgram)]));

	char BuildCmd[1024];
	sprintf(BuildCmd,pCompileFlags,pEntry,pProfile,TmpOut.c_str(),TmpIn.c_str());

	const std::string Cmd=SEnviropment::Instance().m_Compiler+BuildCmd;

	__int64 t0 = g_Timer.GetTime();

	std::string outError;
	if(!Execute(Cmd,outError))
	{
		unsigned char* nIP = (unsigned char*) &RequestIP();
		char sIP[128];
		sprintf(sIP, "%d.%d.%d.%d", nIP[0], nIP[1], nIP[2], nIP[3]);

		const char* pPlatform						= pElement->Attribute("Platform");
		std::string errorString("Shader compile error");
		if(pPlatform)
			errorString += std::string("Platform:            ") + pPlatform + "\n";
		if(pShaderRequestLine)
			errorString += std::string("Shader request line: ") + pShaderRequestLine + "\n";
		errorString += std::string("Request IP:          ") + sIP + "\n";
		errorString += std::string("Target profile:      ") + pProfile + "\n";
		errorString += std::string("Entry function:      ") + pEntry + "\n";
		errorString += std::string("Reported error(s):   ") + "\n";
		errorString += outError + "\n";

		remove(TmpIn.c_str());
		remove(TmpOut.c_str());

		CryCompiler_ERROR("Shader compile error",errorString);
		return false;
	}

	if (!CSTLHelper::FromFile(TmpOut,rVec))
	{
		remove(TmpIn.c_str());
		remove(TmpOut.c_str());
		CrySimple_ERROR(std::string("Could not read: ")+TmpOut);
		return false;
	}
	remove(TmpIn.c_str());
	remove(TmpOut.c_str());

	__int64 t1 = g_Timer.GetTime();
	__int64 dt = t1-t0;
#ifdef _WIN64
	InterlockedAdd64( &m_GlobalCompileTime,dt );
#endif

	int millis = (int)(g_Timer.TimeToSeconds(dt) * 1000.0);
	int secondsTotal = (int)g_Timer.TimeToSeconds(m_GlobalCompileTime);
	printf( "  Compiled [%4dms|%dms] (%s) %s\n",millis,secondsTotal,pProfile,pEntry );

	return true;
}

