#include "../StdTypes.hpp"
#include "../Error.hpp"
#include "DXPSReflect.hpp"
#include "../Server/DXPSServer.hpp"
#include "../DXPSHelper.hpp"
#include "../LZSS/DXPSLZSS.hpp"
#include <assert.h>
#ifdef _MSC_VER
#include <io.h>
#include <Windows.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/locking.h>
#include <share.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#endif

//#define GPAD_CG_DEBUG

#define MAX_COMPILER_WAIT_TIME (60*1000)

/*
#define refract(incident,normal,IOR)
	normalize( 
	          (incident) +
	          (  dot(-(incident),(normal)) -
	             sqrt( max(	1.f - IOR*(	1.f -
																		dot(-(incident),(normal))*dot(-(incident),(normal))
																	)
												 ,
													0.f)
										)
					  ) * (normal) )
*/

const std::string SHADERPATCH_CENTROID = "\n\
#define 	TEXCOORD0_centroid 	TEXCOORD0\n\
#define 	TEXCOORD1_centroid 	TEXCOORD1\n\
#define 	TEXCOORD2_centroid 	TEXCOORD2\n\
#define 	TEXCOORD3_centroid 	TEXCOORD3\n\
#define 	TEXCOORD4_centroid 	TEXCOORD4\n\
#define 	TEXCOORD5_centroid 	TEXCOORD5\n\
#define 	TEXCOORD6_centroid 	TEXCOORD6\n\
#define 	TEXCOORD7_centroid 	TEXCOORD7\n\
#define		VPOS	WPOS\n";


const std::string SHADERPATCH_REFRACT = "\n\
#define refract(incident,normal,IOR) normalize((incident)+(dot(-(incident),(normal))-sqrt(max(1.f-IOR*(1.f-dot(-(incident),(normal))*dot(-(incident),(normal))),0.f)))*(normal))\n";
/*
const std::string SHADERPATCH_REFRACT ="\n\
float3 refract( float3 incident, float3 normal, float IOR)\n\
{\n\
	float IdotN = dot( -incident, normal );\n\
	float cosSqr = 1 - IOR*(1 - IdotN*IdotN);\n\
	cosSqr =  cosSqr<0.f?0.f:sqrt( cosSqr );\n\
	return  normalize( incident + (IdotN - cosSqr) * normal ); \n\
}\n\
float2 refract( float2 in, float2 nrm, float IOR)\n\
{\n\
	return refract(float3(in.x,in.y,0.f),float3(nrm.x,nrm.y,0.f),IOR).xy;\n\
}\n\
";
/*
half3 refract( half3 in, half3 nrm, float IOR)\n\
{\n\
	return refract(float3(in.x,in.y,in.z),float3(nrm.x,nrm.y,nrm.z),IOR);\n\
}\n\
half2 refract( half2 in,half2 nrm, float IOR)\n\
{\n\
  float3 Ret=refract(float3(in.x,in.y,0.f),float3(nrm.x,nrm.y,0.f),IOR);\n\
	return Ret.xy;\n\
}\n\
*/

void CDXPSReflect::PatchPowAbs(std::vector<uint8_t>& rProgram)
{
//	const uint32_t KEY	=	'p'|('o'<<8)|('w'<<16)|('('<<24);
	const uint32_t KEY	=	'(wop';

	std::vector<uint8_t> Tmp;
	Tmp.reserve(rProgram.size());

	for(size_t a=0,Size=rProgram.size();a<Size;)
	{
//		char Text[2]={rProgram[a],0};
//		OutputDebugString(Text);
		if(a<Size-sizeof(KEY) && KEY==*reinterpret_cast<uint32_t*>(&rProgram[a]))
		{
			Tmp.push_back('p');
			Tmp.push_back('o');
			Tmp.push_back('w');
			Tmp.push_back('(');
//			Tmp.push_back('a');
//			Tmp.push_back('b');
//			Tmp.push_back('s');
//			Tmp.push_back('(');
			Tmp.push_back('0');
			Tmp.push_back('.');
			Tmp.push_back('0');
			Tmp.push_back('0');
			Tmp.push_back('1');
			Tmp.push_back('f');
			Tmp.push_back('+');
			a+=4;

		}
		else
			Tmp.push_back(rProgram[a++]);
	}

	rProgram	=	Tmp;
}

void CDXPSReflect::PatchCentroid(std::vector<uint8_t>& rProgram)
{
	std::string Prog=SHADERPATCH_CENTROID+std::string((const char*)&rProgram[0],rProgram.size());;
	rProgram	=	std::vector<uint8_t>(Prog.c_str(),&Prog.c_str()[Prog.size()]);
}

void CDXPSReflect::PatchRefraction(std::vector<uint8_t>& rProgram)
{
	std::string Prog=SHADERPATCH_REFRACT+std::string((const char*)&rProgram[0],rProgram.size());;
	rProgram	=	std::vector<uint8_t>(Prog.c_str(),&Prog.c_str()[Prog.size()]);
}

void CDXPSReflect::PatchToBigOfANumeric(std::vector<uint8_t>& rProgram)
{
	std::string Prog((char*)&rProgram[0],rProgram.size());
//	const uint32_t SemanticNameCount	=	2;
//	const std::string SemanticNames[SemanticNameCount]	=	{"POSITION","COLOR"};
	const uint32_t SemanticNameCount	=	1;
	const std::string SemanticNames[SemanticNameCount]	=	{"POSITION"};

	//seek semantic to be replaced
	for(uint32_t a=0;a<SemanticNameCount;a++)
		for(uint32_t b=a==0?1:2;b<16;b++)
		{
			char Name[1024];
			sprintf(Name,"%s%d",SemanticNames[a].c_str(),b);
			const size_t	SemPos	=	Prog.find(Name);
			if(SemPos!=std::string::npos)
			{
				const size_t	SemScopeBegin	=	Prog.rfind("{",SemPos);
				const size_t	SemScopeEnd		=	Prog.find("}",SemPos);
				assert(SemScopeBegin!=std::string::npos);
				assert(SemScopeEnd!=std::string::npos);

				//seek unused textcoordsemantic

				char Replacement[1024];
				bool Found=false;
				for(uint32_t c=0;c<8 && !Found;c++)
				{
					sprintf(Replacement,"TEXCOORD%d",c);
					size_t Pt=Prog.find(Replacement,SemScopeBegin);
					Found	=	(Pt==std::string::npos || Pt>SemScopeEnd);
				}
				if(Found)
				{
#ifdef _DEBUG
					OutputDebugString(Prog.c_str());
#endif
					std::vector<uint8_t> Tmp;
					CDXPSHelper::Replace(Tmp,rProgram,Name,Replacement);
					rProgram	=	Tmp;
					Prog	=	std::string((char*)&rProgram[0],rProgram.size());
#ifdef _DEBUG
					OutputDebugString(Prog.c_str());
#endif
					m_SemanticPatches[Replacement]=Name;
				}
			}
		}

}

CDXPSReflect::CDXPSReflect(const std::string& rEntry,const bool VertexShader,const std::string& rOutFile,const std::string& rInFile)
{
	m_ShaderFileName			=	rInFile+".patched";
	m_BinaryOutput				=	rInFile+".bin";
	m_ShaderHalfStripped	=	rInFile+".bin2";
	m_ShaderDisassembled	=	rInFile+".asm";
	m_ShaderStripped			=	rInFile+".bin3";
	m_ShaderCache					=	rOutFile;
	m_VertexShader				=	VertexShader;
	m_Entry								=	rEntry;

	const uint32_t ID=0;

	std::vector<uint8_t> Program,Out;
	CDXPSHelper::FromFile(rInFile,Program);

	ParsePragmas(Program);

	PatchToBigOfANumeric(Program);
	PatchCentroid(Program);
	PatchRefraction(Program);

	CDXPSHelper::ToFile(m_ShaderFileName,Program);

	uint32 Best		=	0;

	if(m_Pragma.Iterations())
	{
		g_Success	=	Compile(Best,false);

		uint64_t Score	=	0;
		const uint64_t	TresHold	=	12000000000;
		if(g_Success && !m_VertexShader)
		{
			Score	=	Profile(Best);
			if(Score)
			for(uint32 a=0;a<m_Pragma.Iterations() && Score<TresHold;a++)	//sony suggests 11 to 19 for high probability to get the best shader
			{
				const uint64_t CurrentScore	=	Profile(a);
				if(CurrentScore>Score)
				{
					Best	=	a;
					Score	=	CurrentScore;
				}
			}
		}

	}

	if(	Compile(Best,true)	&&
			HalfStrip()	&&
			Disasseble()	&&
			Strip()	&&
			Reflect(Program))
	{
		g_Success	=	true;	//just relevant for cmdline compilation

		CDXPSHelper::FromFile(m_ShaderCache,Out);

#if defined(DXPS_LZSS_COMPRESS)
		{
			std::vector<uint8_t> Best,Test;
			const uint32_t ID=0;
			CDXPSLZSS<4>::Encode(Best,Out,ID);
#if !defined(_DEBUG)
			CDXPSLZSS<6>::Encode(Test,Out,ID);	if(Test.size()<Best.size())Best	=	Test;
			CDXPSLZSS<8>::Encode(Test,Out,ID);	if(Test.size()<Best.size())Best	=	Test;
			CDXPSLZSS<10>::Encode(Test,Out,ID);	if(Test.size()<Best.size())Best	=	Test;
			CDXPSLZSS<12>::Encode(Test,Out,ID);	if(Test.size()<Best.size())Best	=	Test;
			CDXPSLZSS<14>::Encode(Test,Out,ID);	if(Test.size()<Best.size())Best	=	Test;
#endif
			Out	=	Best;
		}
#endif
		uint32_t Size	=	(uint32_t)Out.size();
		Out.resize(Size+4);
		memmove(&Out[4],&Out[0],Size);
		CDXPSHelper::EndianSwizzleU32(Size);
		*reinterpret_cast<uint32_t*>(&Out[0])	=	Size;
		CDXPSHelper::ToFile(m_ShaderCache,Out);
	}
	else
	{
		g_Success	=	false;	//just relevant for cmdline compilation
	}

  // // for debugging
	remove(m_ShaderFileName.c_str());
	remove(m_BinaryOutput.c_str());

  //remove(m_ShaderFileName.c_str());
//  remove(m_BinaryOutput.c_str());
	remove(m_ShaderHalfStripped.c_str());
	remove(m_ShaderDisassembled.c_str());
	remove(m_ShaderStripped.c_str());
}

void CDXPSReflect::ParsePragmas(std::vector<uint8_t>& rProgram)
{
	std::string Prog((char*)&rProgram[0],rProgram.size());
	tdEntryVec Toks;
	CDXPSHelper::Tokenize(Toks,Prog,"\n");
	for(size_t a=0;a<Toks.size();a++)
	{
		const std::string::size_type Pt0=Toks[a].find("#pragma");
		if(Pt0==std::string::npos)
			continue;

	//while(1)
	//{
	//	int a=0;
	//}

		const char* pStr	=	Toks[a].c_str();
		while(*pStr && *pStr!='\"')
		{
			pStr++;
		}
		if(!*pStr)//no quotation found?
			continue;
		pStr++;

		size_t Len=strlen(pStr);
		while(Len>0 && pStr[Len]!='\"')
		{
			Len--;
		}

		if(!Len)	//end-quotation not found
			continue;

		std::string Param(pStr,Len);

		const std::string::size_type Pt1=Toks[a].find("sce-cgc");
		const std::string::size_type Pt2=Toks[a].find("dxps");
		if(Pt1!=std::string::npos)
		{
			m_PragmaParams	+=	Param+" ";
//      printf("%s\n", m_PragmaParams.c_str());
		}
		else
		if(Pt2!=std::string::npos)
		{
			tdEntryVec Tok;
			CDXPSHelper::Tokenize(Tok,Param,"=");
			if(Tok.size()!=2)
				continue;
			if(Tok[0]=="iterations")
			{
				uint32_t Count=0;
				sscanf(Tok[1].c_str(),"%d",&Count);
				m_Pragma.Iterations(Count);
			}

		}
		else
			continue;
		Toks[a]	=	"";
	}
  
  // for debugging
  //std::string Out = std::string("//") + m_PragmaParams +  std::string("\n\n\n");

  std::string Out;
	for(size_t a=0;a<Toks.size();a++)
		Out+=Toks[a]+'\n';


	rProgram	=	std::vector<uint8_t>(Out.c_str(),&Out.c_str()[Out.size()]);
}

bool CDXPSReflect::Execute(const std::string& rFileName,const std::string& rParams,std::string &outError)
{
#ifdef _MSC_VER
	const std::string Cmd	=	rFileName+" "+rParams;
	bool	Ret	= false;
	DWORD	ExitCode	=	0;

	STARTUPINFO StartupInfo;
	PROCESS_INFORMATION ProcessInfo;
	memset(&StartupInfo, 0, sizeof(StartupInfo));
	memset(&ProcessInfo, 0, sizeof(ProcessInfo));
	StartupInfo.cb = sizeof(StartupInfo);

	
	std::string Path="";
	std::string::size_type Pt = Cmd.find_first_of(' ');
	if(Pt!=std::string::npos)
	{
		std::string First	=	std::string(Cmd.c_str(),Pt);
		std::string::size_type Pt2 = First.find_last_of('/');
		if(Pt2!=std::string::npos)
			Path	=	std::string(First.c_str(),Pt2);
		else
			Pt	=	std::string::npos;
	}

	HANDLE hReadErr, hWriteErr;

	{
		CreatePipe(&hReadErr, &hWriteErr, NULL, 0);
		SetHandleInformation(hWriteErr, HANDLE_FLAG_INHERIT, HANDLE_FLAG_INHERIT);

		StartupInfo.hStdInput = GetStdHandle(STD_INPUT_HANDLE);
		StartupInfo.hStdOutput = hWriteErr;//GetStdHandle(STD_OUTPUT_HANDLE);
		StartupInfo.hStdError = hWriteErr;
		StartupInfo.dwFlags |= STARTF_USESTDHANDLES;

		if(CreateProcess(NULL,(char*)Cmd.c_str(),0,0, TRUE,CREATE_DEFAULT_ERROR_MODE,0,Pt!=std::string::npos?Path.c_str():0,&StartupInfo,&ProcessInfo) != false)
		{
			std::string error;

			DWORD waitResult = 0;
			HANDLE waitHandles[] = { ProcessInfo.hProcess, hReadErr };
			while(true)
			{
				//waitResult = WaitForMultipleObjects(sizeof(waitHandles) / sizeof(waitHandles[0]), waitHandles, FALSE, MAX_COMPILER_WAIT_TIME);
				waitResult = WaitForSingleObject(ProcessInfo.hProcess, 1000 );
				if (waitResult == WAIT_FAILED)
					break;

				DWORD bytesRead, bytesAvailable;
				while(PeekNamedPipe(hReadErr, NULL, 0, NULL, &bytesAvailable, NULL) && bytesAvailable)
				{
					char buff[4096];
					ReadFile(hReadErr, buff, sizeof(buff)-1, &bytesRead, 0);
					buff[bytesRead] = '\0';
					error += buff;
				}

				//if (waitResult == WAIT_OBJECT_0 || waitResult == WAIT_TIMEOUT)
					//break;

				if (waitResult == WAIT_OBJECT_0)
					break;
			}

			//if (waitResult != WAIT_TIMEOUT)
			{
				GetExitCodeProcess(ProcessInfo.hProcess,&ExitCode);
				if (ExitCode)
				{
					Ret = false;
					outError = error;
				}
				else
				{
					Ret = true;
				}
			}
			/*
			else
			{
				Ret = false;
				outError = std::string("Timed out executing: ") + Cmd;
				TerminateProcess(ProcessInfo.hProcess, 1);
			}
			*/

			CloseHandle(ProcessInfo.hProcess);
			CloseHandle(ProcessInfo.hThread);
		}

		CloseHandle(hReadErr);
		if (hWriteErr)
			CloseHandle(hWriteErr);
	}

	return Ret;
#endif

#ifdef UNIX
	return system((rFileName + " " + rParams).c_str()) == 0;
#endif
}

uint64_t CDXPSReflect::Profile(uint32 randomSched)
{
	//compile
	std::string Param;
	if(m_VertexShader)
		return 0;

	const std::string PFile	=	m_BinaryOutput+".txt";

	char Seed[1024];
	_itoa(randomSched,Seed,10);
  
#ifdef GPAD_CG_DEBUG
  Param	=	std::string("-trunc -capture -type cg_fp -cgcopt \"-po randomSeed=1234 -po randomSched=")+Seed+" -O3 -signtex 0xffff -nobcolor -fuse-nrmh" + m_PragmaParams+" \" ";// -nofastprecision  -nofastmath";
#else
  Param	=	std::string("-trunc -type cg_fp -cgcopt \"-po randomSeed=1234 -po randomSched=")+Seed+" -O3 -signtex 0xffff -nobcolor -fuse-nrmh" + m_PragmaParams+" \" ";// -nofastprecision  -nofastmath";
#endif

/*	Param	=	std::string("-trunc -type cg_fp -cgcopt \"-po randomSched=")+Seed+(m_ForceFP32?" ":" -po OutColorPrec=fp16 ")+"-texformat 0 RGBA8 \
-texformat 1 RGBA8 \
-texformat 2 RGBA8 \
-texformat 3 RGBA8 \
-texformat 4 RGBA8 \
-texformat 5 RGBA8 \
-texformat 6 RGBA8 \
-texformat 7 RGBA8 \
-signtex 0xffff  \
-nobcolor \
-O3 \" ";// -nofastprecision  -nofastmath";
*/
	Param +=	" -function ";
	Param +=	m_Entry;

	Param +=	" -o ";
	Param	+=	PFile;

	Param +=	" ";
	Param	+=	m_ShaderFileName;

	std::string Err;
	if(!Execute(SEnviropment::Instance().m_ProfilerFileName,Param,Err))
		return 0;
	std::vector<uint8_t> Vec;
	CDXPSHelper::FromFile(PFile,Vec);
	const std::string DisAssembly(reinterpret_cast<char*>(&Vec[0]),Vec.size());

	uint64_t Pixel=0;
	uint64_t Regs=0;
	{
		const std::string Match	=	" r regs, ";
		const std::string::size_type Pt=DisAssembly.find(Match);
		if(Pt==std::string::npos)
			return 0;

		const char* pStr	=	&DisAssembly.c_str()[Pt+Match.size()];
		while(*pStr>='0' &&	*pStr<='9')
		{
			uint64_t Tmp=*pStr++-'0';
			Pixel=Pixel*10+Tmp;
		}
	}
	{
		const std::string Match	=	" cycles, ";
		const std::string::size_type Pt=DisAssembly.find(Match);
		if(Pt==std::string::npos)
			return 0;

		const char* pStr	=	&DisAssembly.c_str()[Pt+Match.size()];
		while(*pStr>='0' &&	*pStr<='9')
		{
			uint64_t Tmp=*pStr++-'0';
			Regs=Regs*10+Tmp;
		}
	}
	if(!Regs)
		return 0;

	return Pixel/((Regs+1)/2);
/*
	//analyse instruction and register count
	uint32 InstructionCount	=	0;
	uint32 RegCount	=	0;

	std::vector<uint8_t> Vec;
	CDXPSHelper::FromFile(m_ShaderDisassembled,Vec);
	const std::string DisAssembly(reinterpret_cast<char*>(&Vec[0]),Vec.size());

	const std::string Match1	=	"# instructionCount ";
	const std::string Match2	=	"# registerCount ";

	const std::string::size_type Pt1=DisAssembly.find(Match1);
	if(Pt1==std::string::npos)
		return 0;

	const std::string::size_type Pt2=DisAssembly.find(Match2);
	if(Pt2==std::string::npos)
		return 0;

	const char* pStr	=	&DisAssembly.c_str()[Pt1+Match1.size()];
	while(*pStr>='0' &&	*pStr<='9')
		InstructionCount=InstructionCount*10+*pStr++-'0';
	
	pStr	=	&DisAssembly.c_str()[Pt2+Match2.size()];
	while(*pStr>='0' &&	*pStr<='9')
		RegCount=RegCount*10+*pStr++-'0';


	if(!RegCount)
		return 0;
	return InstructionCount*((RegCount+1)/2);//combining instruction count and available pipelines for this binary
*/
}

bool CDXPSReflect::Compile(uint32 randomSched,bool ErrorStream)
{
	//compile
	std::string Param;
	if(m_VertexShader)
#ifdef GPAD_CG_DEBUG
		Param	=	" -p sce_vp_rsx -capture -po MaxLocalParams=327 -contalloc ";
#else
    Param	=	" -p sce_vp_rsx -po MaxLocalParams=327 -contalloc ";
#endif
	else
	{
		char Seed[1024];
		_itoa(randomSched,Seed,10);
/*		Param	=	std::string("-p sce_fp_rsx -po randomSched=")+Seed+
(m_ForceFP32?" ":" -po OutColorPrec=fp16 ")+
"-texformat 0 RGBA8 \
-texformat 1 RGBA8 \
-texformat 2 RGBA8 \
-texformat 3 RGBA8 \
-texformat 4 RGBA8 \
-texformat 5 RGBA8 \
-texformat 6 RGBA8 \
-texformat 7 RGBA8 \
-signtex 0xffff \
-nobcolor \
-O3 ";// -nofastprecision  -nofastmath";*/

#ifdef GPAD_CG_DEBUG
    if( randomSched )
      Param	=	std::string("-p sce_fp_rsx -capture -po randomSeed=1234 -po randomSched=")+Seed+" -O3 -signtex 0xffff -nobcolor -fuse-nrmh "+m_PragmaParams;   
    else
      Param	=	std::string("-p sce_fp_rsx -capture -O3 -signtex 0xffff -nobcolor -fuse-nrmh ")+m_PragmaParams;   
#else
    if( randomSched )
      Param	=	std::string("-p sce_fp_rsx -po randomSeed=1234 -po randomSched=")+Seed+" -O3 -signtex 0xffff -nobcolor -fuse-nrmh "+m_PragmaParams;   
    else
      Param	=	std::string("-p sce_fp_rsx -O3 -signtex 0xffff -nobcolor -fuse-nrmh ")+m_PragmaParams;   
#endif
                                             
    // randomSched usage seems generating more cycles on some cases ?
    //Param	=	std::string("-p sce_fp_rsx ")+" -O3 -signtex 0xffff -nobcolor -fuse-nrmh "+m_PragmaParams;   
	}

	Param +=	" -e ";
	Param +=	m_Entry;

	Param +=	" -o ";
	Param	+=	m_BinaryOutput;

	Param +=	" ";
	Param	+=	m_ShaderFileName;

	std::string Err;
	const bool Ret =	Execute(SEnviropment::Instance().m_CompilerFileName,Param,Err);
	DWORD	Written=0;
	if(ErrorStream)
//		WriteFile(GetStdHandle(STD_ERROR_HANDLE), Err.c_str(), static_cast<DWORD>(Err.size()), &Written, NULL);
		fprintf(stderr,Err.c_str());
//	fprintf(stderr,"testerror\n");
	return Ret;
}

bool CDXPSReflect::HalfStrip()
{
	std::string Param;
	Param =	" -param -o ";
	Param	+=	m_ShaderHalfStripped;

	Param +=	" ";
	Param	+=	m_BinaryOutput;

	std::string Err;
	return Execute(SEnviropment::Instance().m_StripperFileName,Param,Err);
}

bool CDXPSReflect::Disasseble()
{
#ifdef _DEBUG
	{
		std::string Param;
		Param =	" -o ";
		Param	+=	m_ShaderDisassembled+".dbg";

		Param +=	" ";
		
		Param	+=	m_BinaryOutput;

		std::string Err;
		Execute(SEnviropment::Instance().m_DisassemblerFileName,Param,Err);
	}
#endif
	std::string Param;
	Param =	" -o ";
	Param	+=	m_ShaderDisassembled;

	Param +=	" ";
	
//	Param	+=	m_BinaryOutput;
	Param	+=	m_ShaderHalfStripped;

	std::string Err;
	return Execute(SEnviropment::Instance().m_DisassemblerFileName,Param,Err);
}

bool CDXPSReflect::Strip()
{
	std::string Param;
//	Param =	"  -name -semantic -param -varying -nodefault -sampler -o ";
	Param =	"  -a -o ";
	Param	+=	m_ShaderStripped;

	Param +=	" ";
	Param	+=	m_BinaryOutput;

	std::string Err;
	return Execute(SEnviropment::Instance().m_StripperFileName,Param,Err);
}

bool CDXPSReflect::Reflect(std::vector<uint8_t>& rProgram)
{
//	m_Reflection.Reflect(m_ShaderCache,m_BinaryOutput,m_ShaderDisassembled,rProgram,m_SemanticPatches);
	m_Reflection.Reflect(m_ShaderCache,m_ShaderStripped,m_ShaderDisassembled,rProgram,m_SemanticPatches);
	return true;
}

