#include "stdafx.h"


#pragma message (">>> include lib: d3d9.lib")
#pragma comment (lib, "d3d9.lib")
#pragma message (">>> include lib: d3dx9.lib")
#pragma comment (lib, "d3dx9.lib")


static const tchar c_version[] = _T("v1.01");
static const float c_defDepthBufferScale = 1024.0f;


template <typename T>
struct AutoRelease
{
	AutoRelease(T*& pObj) : m_pObj(pObj) {}
	~AutoRelease()
	{
		if (m_pObj)
			m_pObj->Release();
	}

	T*& m_pObj;
};


static bool DepthBufferToMask(IDirect3DDevice9* pD3DDevice, const tchar* infile, const tchar* outfile, float maskThreshold, float depthBufferScale)
{
	printf("\""PASTE_STR"\" -> \""PASTE_STR"\" ...", infile, outfile);

	D3DXIMAGE_INFO imgInfo;
	if (FAILED(D3DXGetImageInfoFromFile(infile, &imgInfo)))
	{
		printf("\nError: Failed to get image info!\n");
		return false;
	}

	if (imgInfo.Format != D3DFMT_A32B32G32R32F && imgInfo.ImageFileFormat != D3DXIFF_HDR)
	{
		printf("\nError: Invalid image file passed!\n");
		return false;
	}

	IDirect3DTexture9* pSrcTexture = 0;
	AutoRelease<IDirect3DTexture9> relSrcTex(pSrcTexture);
	if (FAILED(D3DXCreateTextureFromFileEx(pD3DDevice, infile, imgInfo.Width, imgInfo.Height,
		1, 0, imgInfo.Format, D3DPOOL_SYSTEMMEM, D3DX_DEFAULT, D3DX_DEFAULT, 0, 0, 0, &pSrcTexture)))
	{
		printf("\nError: Failed to load image data!\n");
		return false;
	}

	D3DSURFACE_DESC desc;
	pSrcTexture->GetLevelDesc(0, &desc);

	IDirect3DTexture9* pDstTexture;
	AutoRelease<IDirect3DTexture9> relDstTex(pDstTexture);
	if (FAILED(pD3DDevice->CreateTexture(imgInfo.Width, imgInfo.Height, 1, 0, D3DFMT_L8, D3DPOOL_SYSTEMMEM, &pDstTexture, 0)))
	{
		printf("\nError: Failed to create destination image!\n");
		return false;
	}

	bool dataConverted = false;
	D3DLOCKED_RECT src;
	if (SUCCEEDED(pSrcTexture->LockRect(0, &src, 0, 0)))
	{
		D3DLOCKED_RECT dst;
		if (SUCCEEDED(pDstTexture->LockRect(0, &dst, 0, 0)))
		{
			const float invDepthBufferScale = 1.0f / depthBufferScale;

			for (unsigned int y=0; y<imgInfo.Height; ++y)
			{
				float* pSrc = (float*) ((size_t) src.pBits + y * src.Pitch);
				unsigned char* pDst = (unsigned char*) ((size_t) dst.pBits + y * dst.Pitch);

				for (unsigned int x=0; x<imgInfo.Width; ++x, ++pDst, pSrc+=4)
					*pDst = pSrc[0] * invDepthBufferScale < maskThreshold ? 0 : 255;
			}

			dataConverted = true;
			pDstTexture->UnlockRect(0);
		}
		pSrcTexture->UnlockRect(0);
	}

	if (!dataConverted)
	{
		printf("\nError: Failed to generate mask image!\n");
		return false;
	}

	if (FAILED(D3DXSaveTextureToFile(outfile, D3DXIFF_PNG, pDstTexture, 0)))
	{
		printf("\nError: Failed to save mask image!\n");
		return false;
	}

	printf(" done.\n");
	return true;
}


static tstring GetBaseDir(const tchar* searchPath)
{
	tstring baseDir = searchPath;

	size_t i = baseDir.size();
	while (i)
	{
		if (baseDir[i-1] == '/' || baseDir[i-1] == '\\')
		{
			baseDir.resize(i);
			return baseDir;
		}
		else
			--i;
	}

	return tstring(_T(""));
}


static tstring GetOutFile(const tchar* infile)
{
	tstring outfile = infile;

	size_t i = outfile.size();
	while (i)
	{
		if (outfile[i-1] == '.')
		{
			outfile.resize(i);
			outfile += T("png");
			return outfile;
		}
		else
			--i;
	}

	outfile += T(".png");
	return outfile;
}


static bool BatchProcessFiles(IDirect3DDevice9* pD3DDevice, const tchar* searchPath, float maskThreshold, float depthBufferScale, bool breakOnError)
{
	tstring baseDir = GetBaseDir(searchPath);

	_tfinddata64_t fileInfo;
	intptr_t hFile = _tfindfirst64(searchPath, &fileInfo);

	bool successful = true;
	if (hFile != -1L)
	{
		do
		{
			if ((fileInfo.attrib & _A_SUBDIR) == 0)
			{
				tstring infile = baseDir + fileInfo.name;
				tstring outfile = GetOutFile(infile.c_str());

				successful = DepthBufferToMask(pD3DDevice, infile.c_str(), outfile.c_str(), maskThreshold, depthBufferScale);
				if (!successful && breakOnError)
				{
					printf("Error occurred, skip processing of remaining files!\n");
					return false;
				}
			}
		} while (_tfindnext64(hFile, &fileInfo) == 0);
		_findclose(hFile);
	}

	return successful;
}


int _tmain(int argc, tchar* argv[])
{
	if (argc < 3)
	{
		printf("DepthBufferToMask "PASTE_STR", Copyright (C) Crytek 2009\n\n", c_version);
		printf("Usage: DepthBufferToMask.exe inFile maskThreshold [depthBufferScale] [breakOnError]\n\n");
		printf(" inFile : Input depth buffer image in HDR format captured with CryEngine.\n");
		printf("          Wild cards are supported. Output file will use name of input file\n");
		printf("          and be stored in PNG format.\n\n");
		printf(" maskThreshold : Linear depth value in range [0..1]. Depth buffer values below\n");
		printf("                 the threshold will result in a black pixel mask. Otherwise the\n");
		printf("                 pixel mask will be white.\n\n");
		printf(" depthBufferScale : Value by which the depth buffer in inFile was previously\n");
		printf("                    scaled. Needed to normalize depth values for comparison\n");
		printf("                    with maskThreshold. Default value is %.4g.\n\n", c_defDepthBufferScale);
		printf(" breakOnError : When set to 1, processing of remaining files will be skipped\n");
		printf("                if mask generation for one of them fails. Default value is 0.\n");
		return 1;
	}

	tchar* infile = argv[1];

	float maskThreshold = (float) tatof(argv[2]);
	if (maskThreshold < 0 || maskThreshold > 1)
	{
		float origMaskThreshold = maskThreshold;
		maskThreshold = maskThreshold < 0 ? 0 : maskThreshold > 1 ? 1 : maskThreshold;
		printf("Warning: Passed-in mask threshold of %.4g clamped to %.4g.\n", origMaskThreshold, maskThreshold);
	}

	float depthBufferScale = argc > 3 ? (float) tatof(argv[3]) : c_defDepthBufferScale;
	if (depthBufferScale < 1e-4f)
	{
		printf("Warning: Passed-in depth buffer scale value of %.4g is invalid, use default.\n", depthBufferScale, c_defDepthBufferScale);
		depthBufferScale = c_defDepthBufferScale;
	}

	bool breakOnError = argc > 4 ? tatoi(argv[4]) != 0 : false;

	IDirect3D9* pD3D = Direct3DCreate9(D3D_SDK_VERSION);
	AutoRelease<IDirect3D9> relD3D(pD3D);
	if (!pD3D)
	{
		printf("Error: Couldn't initialize D3D!\n");
		return -1;
	}

	D3DPRESENT_PARAMETERS presentParams;
	memset(&presentParams, 0, sizeof(presentParams));
	presentParams.Windowed = TRUE;
	presentParams.SwapEffect = D3DSWAPEFFECT_COPY;
	presentParams.BackBufferWidth = 1;
	presentParams.BackBufferHeight = 1;
	presentParams.BackBufferFormat = D3DFMT_UNKNOWN;

	IDirect3DDevice9* pD3DDevice = 0;
	AutoRelease<IDirect3DDevice9> relD3DDev(pD3DDevice);
	if (FAILED(pD3D->CreateDevice(D3DADAPTER_DEFAULT, D3DDEVTYPE_NULLREF, GetConsoleWindow(), D3DCREATE_SOFTWARE_VERTEXPROCESSING, &presentParams, &pD3DDevice)))
	{
		printf("Error: Failed to create D3D null ref device!\n");
		return -1;
	}

	printf("Creating mask(s) for \""PASTE_STR"\"...\n  Mask threshold: %.4g\n  Depth buffer scale: %.4g\n  Break on error: %d\n\n", infile, maskThreshold, depthBufferScale, breakOnError ? 1 : 0);
	return BatchProcessFiles(pD3DDevice, infile, maskThreshold, depthBufferScale, breakOnError) ? 0 : -1;
}

