#include <vector>
#include <string>
#include <math.h>

using namespace std;

const double sqrt_pi = 1.7724538509055160272981674833411;

double poly(double x,int deg,double *A)
{      
	double t;
	t = A[deg];
	while (deg--)
		t = t*x + A[deg];
	return t;
}

double erf(double x)
{
	if (x<0)
		return -erf(-x);

	double A[] = {1.0, 0.0705230784, 0.0422820123, 0.0092705272,
		0.0001520143, 0.0002765672, 0.0000430638};
	return 1.0 - 1.0/(pow(poly(x, 6, A), 16));
} 

double inverf(double y, double err = 1e-10)
{
	if (y<0)
		return -inverf(-y);

	double mn = 0.0f, mx = 1.0f;
	while (erf(mx)<y)
	{
		mn = mx;
		mx *= 2.0f;
	}
	while (fabs(erf((mx+mn)*0.5)-y) > err)
	{
		double x = (mn + mx) * 0.5;
		if (erf(x) < y)
			mn = x;
		else
			mx = x;
	}
	return (mx + mn) * 0.5;
}

struct Gaussian
{
	Gaussian(double aa, double bb, double cc) : a(aa), b(bb), c(cc) {}
	double a, b, c;
	double operator()( double x )
	{
		return a * exp(-b*(x-c)*(x-c));
	}
};

double integrate_no_c_0( const Gaussian& f, double p )
{
	double sqrt_b = sqrt(f.b);
	return f.a * sqrt_pi / 2 / sqrt_b * erf( sqrt_b * p );
}

double integrate_no_c( const Gaussian& f, double p, double q )
{
	return integrate_no_c_0( f, q ) - integrate_no_c_0( f, p );
}

double integrate( const Gaussian& f, double p, double q )
{
	// start by centering the expression
	Gaussian f2(f.a, f.b, 0);
	return integrate_no_c( f2, p-f.c, q-f.c );
}

double integrate( const Gaussian& f )
{
	return f.a * sqrt_pi / sqrt(f.b);
}

bool aliases( const Gaussian& g, unsigned sz )
{
	unsigned last = 0;
	for (unsigned i=1; i<sz; i++)
	{
		double y = integrate(g, 0, i);
		if (y > 4294967295.49)
			return true;
		unsigned iy = unsigned(y+0.5);
		if (iy == last)
			return true;
		last = iy;
	}
	return false;
}

Gaussian mkgaussian( unsigned sz, double percentile, double err = 1e-10 )
{
	printf("// mkgaussian sz:%d\n", sz);
	double mn = 0, mx = 1;
	Gaussian f(1,0,0);
	while (true)
	{
		f.b = mx;
		double tot = 0.5*integrate(f);
		double in = integrate(f,0,sz);
		double pct = in/tot;
		if (pct < percentile)
		{
			mn = mx;
			mx *= 2;
		}
		else
			break;
	}
	while (true)
	{
		f.b = 0.5*(mx+mn);
		double tot = 0.5*integrate(f);
		double in = integrate(f,0,sz);
		double pct = in/tot;
		if (fabs(pct - percentile) > err)
		{
			if (pct < percentile)
				mn = f.b;
			else
				mx = f.b;
		}
		else
			break;
	}
	printf("// b = %.15lf\n", f.b);
	return f;
}

Gaussian dealias( const Gaussian& g, unsigned sz, double err = 1e-10 )
{
	Gaussian f = g;
	double mn = 0, mx = 1;
	f.a = mx;
	while (aliases(f, sz))
	{
		mn = mx;
		mx *= 2;
		f.a = mx;
	}
	while (true)
	{
		f.a = 0.5*(mn+mx);
		bool alias = aliases(f, sz);
		if (!alias && (mx-mn) < err)
			break;
		if (alias)
			mn = f.a;
		else
			mx = f.a;
	}
	printf("// a = %.15lf\n", f.a);
	return f;
}

typedef vector<unsigned> UVec;

void fillvec( const Gaussian& g, UVec& v, int sz )
{
	v.resize(0);
	v.reserve(sz);
	for (int i=0; i<sz; i++)
		v.push_back( unsigned(integrate(g, 0, i)+0.5) );
}

unsigned countdelts( const UVec& v, const UVec& startp )
{
	unsigned delts = 0;
	for (unsigned i=1; i<v.size(); i++)
	{
		unsigned inc;
		for (inc = startp.size(); inc > 2 && startp[inc-2] <= i; inc--)
			;
		inc--;
		unsigned diff = v[i-1] + inc;
		int delt = v[i] - diff;
		if (delt != 0)
			delts ++;
	}
	return delts;
}

void getdelts( UVec& delts, UVec& szs, const UVec& v, const UVec& startp, int start, int stop )
{
	for (int i=start+1; i<stop; i++)
	{
		unsigned inc;
		for (inc = startp.size(); inc > 2 && startp[inc-2] <= i; inc--)
			;
		inc--;
		unsigned diff = v[i-1] + inc;
		int delt = v[i] - diff;
		if (delt != 0)
		{
			delts.push_back(i-start);
			szs.push_back(delt);
		}
	}
}

void basicfitlines( const UVec& v, UVec& startp )
{
	startp.resize(0);
	unsigned maxinc = v[1] - v[0];
	for (unsigned i=2; i<v.size(); i++)
	{
		if (v[i] - v[i-1] > maxinc)
			maxinc = v[i] - v[i-1];
	}
	startp.resize(maxinc+1);
	startp[maxinc] = 0;
	startp[0] = v.size();
	int lastinc = maxinc;
	int lastdiff = 0;
	for (unsigned i=2; i<v.size(); i++)
	{
		unsigned est = v[i-1] + lastinc;
		int diff = v[i] - est;
		if (diff < 0 && lastdiff < 0)
		{
			lastinc += diff;
			startp[lastinc] = i;
		}
		else
		{
			lastdiff = diff;
		}
	}
	for (unsigned i=1; i<startp.size()-1; i++)
	{
		if (!startp[i])
		{
			startp[i] = startp[i+1];
			startp[i+1] --;
		}
	}
}

void dumpstartp( const UVec& v, int mind )
{
	printf( "//" );
	for (int i=0; i<v.size(); i++)
		printf(" %d", v[i]);
	printf(" [%d]\n", mind);
}

void mindelts( const UVec& v, UVec& startp )
{
	dumpstartp(startp, countdelts(v,startp));
	vector<unsigned> order;
	order.reserve(startp.size() - 2);
	for (unsigned i=0; i<startp.size()-2; i++)
		order.push_back(i+1);
	bool shrank;
	do 
	{
		shrank = false;
		for (unsigned i=0; i<order.size(); i++)
		{
			int a = rand() % order.size();
			int b;
			do { b = rand() % order.size(); } while (a==b);
			swap(order[a], order[b]);
		}
		for (unsigned i=0; i<order.size(); i++)
		{
			int slot = rand() % (startp.size() - 2) + 1;
			int mn0 = startp[slot+1]+1;
			int mx0 = startp[slot-1]-1;
			unsigned old = startp[slot];
			int mn = max(mn0, int(old-15));
			int mx = min(mx0, int(old+15));
			int mind = countdelts(v, startp);
			unsigned minval = old;
			for (int test = mn; test <= mx; test++)
			{
				if (test == old)
					continue;
				startp[slot] = test;
				int d = countdelts(v, startp);
				if (d < mind)
				{
					mind = d;
					minval = test;
					shrank = true;
				}
			}
			startp[slot] = minval;
			if (old != minval)
				dumpstartp(startp, mind);
		}
	} while(shrank);
}

void fitlines( const UVec& v, UVec& startp )
{
	basicfitlines(v,startp);
	mindelts(v,startp);
}

string tabs( int depth )
{
	string s;
	while (depth--)
		s += '\t';
	return s;
}

#define TABS(n) tabs(n).c_str()

void output_lowsym0( const Gaussian& f, const UVec& v, const UVec& startp, int p, int depth )
{
	printf("%slow0 = %d;\n", TABS(depth), v[startp[p+1]]);
	printf("%ssym0 = %d;\n", TABS(depth), v[startp[p]] - v[startp[p+1]]);
}

void output_dec( const Gaussian& f, const UVec& v, const UVec& startp, int p, int depth )
{
	printf("// %d\n", p);
	output_lowsym0( f, v, startp, p, depth );
	printf("%sbase = %d;\n", TABS(depth), startp[p+1]);
	printf("%ssym1 = %d;\n", TABS(depth), p+1);
}

void output_enc( const Gaussian& f, const UVec& v, const UVec& startp, int p, int depth )
{
	output_lowsym0( f, v, startp, p, depth );
	printf("%slow1 = (x-%d)*%d;\n", TABS(depth), startp[p+1], p+1);
	printf("%ssym1 = %d;\n", TABS(depth), p+1);
}

void output_slot(const Gaussian& f, const UVec& v, const UVec& startp, int p, int depth)
{
	printf("%sslot = %d;\n", TABS(depth), p+1);
}

void output_btree( const Gaussian& f, const UVec& v, const UVec& startp, int start, int stop, const char * sym, int depth, bool gt, bool usep, bool brace, void (*line)( const Gaussian& f, const UVec& v, const UVec& startp, int p, int depth ) )
{
	switch (stop - start)
	{
	case 0:
		break;
	case 1:
		line(f,v,startp,start,depth);
		break;
	default:
		{
			double closestToHalf = 1e40;
			int closestToHalfIndex = -1;
			for (int i=start+1; i<=stop-1; i++)
			{
				double left = integrate( f, startp[start], startp[i] );
				double right = integrate( f, startp[i], startp[stop] );
				double diff = fabs(right-left);
				if (diff < closestToHalf)
				{
					closestToHalfIndex = i;
					closestToHalf = diff;
				}
			}
			int val;
			if (usep)
				val = startp[closestToHalfIndex];
			else
				val = v[startp[closestToHalfIndex]];
			printf("%sif (%s%s%d)\n", TABS(depth), sym, gt?">":"<", val);
			if (brace)
				printf("%s{\n", TABS(depth));
			if (gt)
				output_btree( f, v, startp, start, closestToHalfIndex, sym, depth+1, gt, usep, brace, line );
			else
				output_btree( f, v, startp, closestToHalfIndex, stop, sym, depth+1, gt, usep, brace, line );
			if (brace)
				printf("%s}\n", TABS(depth));
			printf("%selse\n", TABS(depth));
			if (brace)
				printf("%s{\n", TABS(depth));
			if (!gt)
				output_btree( f, v, startp, start, closestToHalfIndex, sym, depth+1, gt, usep, brace, line );
			else
				output_btree( f, v, startp, closestToHalfIndex, stop, sym, depth+1, gt, usep, brace, line );
			if (brace)
				printf("%s}\n", TABS(depth));
		}
	}
}

void output( const Gaussian& f, const UVec& v, const UVec& startp, int name )
{
	printf("static int SlotForValue%d(int32 x)\n", name);
	printf("{\n");
	printf("\tuint32 slot;\n");
	output_btree( f, v, startp, 0, startp.size() - 1, "x", 1, false, true, false, output_slot );
	printf("\treturn slot;\n");
	printf("}\n");
	printf("static int SlotForEncoding%d(int32 x)\n", name);
	printf("{\n");
	printf("\tuint32 slot;\n");
	output_btree( f, v, startp, 0, startp.size() - 1, "x", 1, false, false, false, output_slot );
	printf("\treturn slot;\n");
	printf("}\n");

	printf("static SlotData%d[%d] = {\n", name, startp.size());
	printf("\t{%d,%d},\n", v.back(), v.size()-1);
	for (int i=0; i<startp.size()-1; i++)
		printf("\t{%d,%d},\n", v[startp[i+1]], startp[i+1]);
	printf("};\n", name);


}

int main(int argc, const char* argv[])
{
	for (int testbits = 3; testbits<=28; testbits++)
	{
		int test = 1<<testbits;
		UVec v;
		Gaussian g = dealias( mkgaussian(test, 0.99), test );
		fillvec(g , v, test);
		UVec startlines;
		fitlines(v, startlines);
		v.push_back(v.back() + 1);
		output(g, v, startlines, testbits);
	}

	return 0;
}

