////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2009.
// -------------------------------------------------------------------------
//  File name:   licenseconnection.h
//  Version:     v1.00
//  Created:     27/08/2009 by Younggi Lim
//  Compilers:   Visual Studio.NET
//  Description: Manage communication with License Server
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////
#include "stdafx.h"

#ifdef WIN32
#include <windows.h>
#endif
#include "LicenseConnection.h"
#include "TinyClientSocket.h"
#include "ProtocolBuilder.h"
#include "ProtocolAssembler.h"
#include "LicenseServerCommon.h"
#include <Drei/TimerUtil.h>

const uint32 RecviveBufferSize = 1024;
const uint32 ReceiveWaitTime = 1000;
const uint32 SendBufferSize = 1024;

CLicenseConnection::CLicenseConnection() : 
			m_clientSocket(new CTinyClientSocket()), 
			m_builder(new CProtocolBuilder()), 
			m_assembler(new CProtocolAssembler()),
			m_remainSecond(0),
			m_floatLicenses(0),
			m_progressStep(ePS_None)
{
	m_builder->EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
	m_clientSocket->InitNetwork();
	UpdateLastAuthenticatedTick();
}

CLicenseConnection::~CLicenseConnection()
{
	m_clientSocket->FiniNetwork();

	if (NULL != m_clientSocket)
	{
		delete m_clientSocket;
		m_clientSocket = NULL;
	}

	if (NULL != m_builder)
	{
		delete m_builder;
		m_builder = NULL;
	}

	if (NULL != m_assembler)
	{
		delete m_assembler;
		m_assembler = NULL;
	}
}

void CLicenseConnection::SetConnectionInfo( const char* ip, int port )
{
	m_serverIp = ip;
	m_serverPort = port;
}

void CLicenseConnection::SetDefaultConnectionInfo()
{
	m_serverIp = "mycryengine.com";
	m_serverPort = 25379;
}

void CLicenseConnection::GetLicenseKey( std::string& key )
{
	key = m_key;
}

void CLicenseConnection::GetEncryptKey( std::string& key )
{
	key = m_encryptKey;
}

EPMErrorCode CLicenseConnection::Authenticate( const char* key)
{
	if (false == VerifyLocalNetworkDeviceInfo())
		return PMEC_NETWORK_DEVICE;

	const int RetryCount = 2;
	EPMErrorCode retcode = PMEC_NO_ERROR;
	for (int i=0; i<RetryCount; ++i)
	{
		retcode = RemoteAuthenticate(key);
		if (
				PMEC_NO_ERROR == retcode || 
				PMEC_KEY_EXPIRED == retcode ||
				PMEC_KEY_NOT_FOUND == retcode ||
				PMEC_KEY_INVALID == retcode 
			)
			break;
	}
	return retcode;
}

bool CLicenseConnection::VerifyLocalNetworkDeviceInfo()
{
	if (false == m_localIp.empty() && false == m_localMac.empty())
		return true;

	if (false == m_clientSocket->GetLocalIPAddress(m_localIp))
		return false;
	if (false == m_clientSocket->GetLocalMacAddress(m_localMac))
		return false;
	return true;
}

EPMErrorCode CLicenseConnection::RemoteAuthenticate( const std::string& key )
{
	if (false == InitSocket())
		return PMEC_NETWORK_DEVICE;

	if (false == ConnectServer())
		return PMEC_CONNECT_FAILED;

	EPMErrorCode retcode = PMEC_NO_ERROR;
	retcode = LoginProcess(key);
	if (PMEC_NO_ERROR != retcode)
	{
		ResetConnection();
		return retcode;
	}

	retcode = AuthProcess(key);
	if (PMEC_NO_ERROR != retcode)
		return retcode;
	UpdateLastAuthenticatedTick();
	return PMEC_NO_ERROR;
}

bool CLicenseConnection::SendLogin( const std::string& key ) const
{
	if (m_progressStep >= ePS_LoginServer)
		return true;

	m_builder->Clear();
	m_builder->Add("type", "login");
	m_builder->Add("key", key);
	m_builder->Add("ip", m_localIp);
	m_builder->Add("mac", m_localMac);
	m_builder->Add("sessionkey", m_sessionKey);
	m_builder->Add("appname", m_appname);
	m_builder->Add("version", m_version);
	char sendBuffer[SendBufferSize] = {0,};
	uint32 sendSize = m_builder->MakeBuffer(sendBuffer, SendBufferSize);
	return m_clientSocket->Send(sendBuffer, sendSize);
}

bool CLicenseConnection::SendAuth( const std::string& key) const
{
	m_builder->Clear();
	m_builder->Add("type", "auth");
	m_builder->Add("key", key);
	m_builder->Add("ip", m_localIp);
	m_builder->Add("mac", m_localMac);
	m_builder->Add("sessionkey", m_sessionKey);
	m_builder->Add("sessionid", m_sessionId);
	m_builder->Add("appname", m_appname);
	char sendBuffer[SendBufferSize] = {0,};
	uint32 sendSize = m_builder->MakeBuffer(sendBuffer, SendBufferSize);
	return m_clientSocket->Send(sendBuffer, sendSize);
}

EPMErrorCode CLicenseConnection::ReceiveLoginResult()
{
	if (m_progressStep >= ePS_LoginServer)
		return PMEC_NO_ERROR;

	char receiveBuffer[RecviveBufferSize] = {0,};
	uint32 receivedSize = m_clientSocket->Receive(receiveBuffer, RecviveBufferSize, ReceiveWaitTime*2);
	if (0 == receivedSize)
		return PMEC_CANT_READ_FROM_SOCKET;
	m_builder->Clear();
	uint32 bufferOffset = 0;
	m_assembler->ApplyBuffer(receiveBuffer, bufferOffset, receivedSize);
	if (false == m_assembler->AssembleBuilder(*m_builder))
		return PMEC_CANT_READ_FROM_SOCKET;

	std::string received = m_builder->MakeString();
	//printf("received : %s\n", received.c_str());
	std::string strErrorCode = m_builder->GetValue<std::string>("errorcode");
	if (strErrorCode.empty())
		return PMEC_KEY_INVALID;
	EPMErrorCode resultCode = (EPMErrorCode)m_builder->GetValue<int32>("errorcode");
	if (resultCode != PMEC_NO_ERROR)
		return resultCode;

	m_key = m_builder->GetValue<std::string>("key");
	m_remainSecond = m_builder->GetValue<uint32>("remain");
	m_encryptKey = m_builder->GetValue<std::string>("encryptkey");
	m_floatLicenses = m_builder->GetValue<uint32>("floatlicenses");
	m_progressStep = ePS_LoginServer;
	m_sessionId = m_builder->GetValue<std::string>("sessionid");
	return resultCode;
}

EPMErrorCode CLicenseConnection::ReceiveAuthResult() const
{
	char receiveBuffer[RecviveBufferSize] = {0,};
	uint32 receivedSize = m_clientSocket->Receive(receiveBuffer, RecviveBufferSize, ReceiveWaitTime);
	if (0 == receivedSize)
		return PMEC_CANT_READ_FROM_SOCKET;
	m_builder->Clear();
	uint32 bufferOffset = 0;
	m_assembler->ApplyBuffer(receiveBuffer, bufferOffset, receivedSize);
	if (false == m_assembler->AssembleBuilder(*m_builder))
		return PMEC_CANT_READ_FROM_SOCKET;

	std::string received = m_builder->MakeString();
	std::string strErrorCode = m_builder->GetValue<std::string>("errorcode");
	if (strErrorCode.empty())
		return PMEC_KEY_INVALID;
	EPMErrorCode resultCode = (EPMErrorCode)m_builder->GetValue<int32>("errorcode");
	return resultCode;
}

void CLicenseConnection::ResetConnection()
{
	m_clientSocket->FiniSocket();
	m_progressStep = ePS_None;
}

EPMErrorCode CLicenseConnection::LoginProcess( const std::string& key )
{
	if (false == SendLogin(key))
		return PMEC_CANT_WRITE_TO_SOCKET;

	return ReceiveLoginResult();
}

EPMErrorCode CLicenseConnection::AuthProcess( const std::string& key )
{
	EPMErrorCode retcode = PMEC_NO_ERROR;
	do 
	{
		if (false == SendAuth(key))
		{
			retcode = PMEC_CANT_WRITE_TO_SOCKET;
			break;
		}

		retcode = ReceiveAuthResult();
		if (PMEC_NO_ERROR != retcode)
			break;
	} while(false);

	if (PMEC_NO_ERROR != retcode)
		ResetConnection();

	return retcode;
}

uint32 CLicenseConnection::GetRemainSecond()
{
	return m_remainSecond;
}

uint32 CLicenseConnection::GetDisconnectedPeriod()
{
	//return (GetTickCount() - m_lastAutheticatedTick);
	return (DreiNetwork::GetTick() - m_lastAutheticatedTick);
}
void CLicenseConnection::UpdateLastAuthenticatedTick()
{
	m_lastAutheticatedTick = DreiNetwork::GetTick();
}

void CLicenseConnection::SetProxy(std::string& ip, int port, std::string& authInfo)
{
	m_proxyIp = ip;
	m_proxyPort = port;
	m_proxyAuthInfo = authInfo;
}

bool CLicenseConnection::InitSocket()
{
	if (m_progressStep >= ePS_InitSocket)
		return true;
	bool result = m_clientSocket->InitSocket();
	if (result)
		m_progressStep = ePS_InitSocket;
	return result;
}

bool CLicenseConnection::ConnectServer()
{
	if (m_progressStep >= ePS_ConnectServer)
		return true;
	
	if (m_proxyIp.empty())
	{
		bool result = m_clientSocket->Connect(m_serverIp.c_str(), m_serverPort);
		if (result)
			m_progressStep = ePS_ConnectServer;
		return result;
	}

	// let proxy to connect
	std::string sProxyAuthorization;
	if (false == m_proxyAuthInfo.empty())
		sProxyAuthorization = "Proxy-authorization: Basic "+ m_proxyAuthInfo+ "\r\n";
	char sServerPort[32] = {0,};
	sprintf(sServerPort, "%d", m_serverPort);
	std::string sProxyConnect = std::string("CONNECT ")+ m_serverIp+ ":"+sServerPort+" HTTP/1.0\r\n";
	std::string sProxyPieces = sProxyConnect + sProxyAuthorization + "\r\n";

	if (false == m_clientSocket->Connect(m_proxyIp.c_str(), m_proxyPort))
		return false;

	m_clientSocket->Send(sProxyPieces.c_str(), (uint32)sProxyPieces.length());

	std::string received;
	char receiveBuffer[RecviveBufferSize] = {0,};
	uint32 recvBytes = m_clientSocket->Receive(receiveBuffer, RecviveBufferSize-1, ReceiveWaitTime);
	if (recvBytes == 0)
		return false;
	received = receiveBuffer;

	size_t iter = 0;
	while (received.c_str()[iter]!=32 && iter<recvBytes) 
		iter++;
	if (iter==recvBytes || received.find("200")!=iter+1)
		return false;

	m_progressStep = ePS_ConnectServer;
	return true;
}

uint32 CLicenseConnection::GetFloatLicenses()
{
	return m_floatLicenses;
}

void CLicenseConnection::ResetLogin()
{
	if (m_progressStep > ePS_ConnectServer)
		m_progressStep = ePS_ConnectServer;
}

void CLicenseConnection::GetLocalIp( std::string& ip )
{
	ip = m_localIp;
}

void CLicenseConnection::GetMacAddress(std::string& mac)
{
	mac = m_localMac;
}

void CLicenseConnection::SetAccountInfo( const std::string& account, const std::string& password )
{
	m_username = account;
	m_password = password;
}

bool CLicenseConnection::SendLoginAccount( const std::string& username, const std::string& password ) const
{
	//if (m_progressStep >= ePS_LoginServer)
	//	return true;

	m_builder->Clear();
	m_builder->Add("type", "loginaccount");
	m_builder->Add("username", username);
	m_builder->Add("password", password);
	m_builder->Add("mac", m_localMac);
	char sendBuffer[SendBufferSize] = {0,};
	uint32 sendSize = m_builder->MakeBuffer(sendBuffer, SendBufferSize);
	return m_clientSocket->Send(sendBuffer, sendSize);
}

bool CLicenseConnection::SendPasswordRemind(const std::string& username, const std::string& email) const
{
	m_builder->Clear();
	m_builder->Add("type", "passwordremind");
	m_builder->Add("username", username);
	m_builder->Add("email", email);
	char sendBuffer[SendBufferSize] = {0,};
	uint32 sendSize = m_builder->MakeBuffer(sendBuffer, SendBufferSize);
	return m_clientSocket->Send(sendBuffer, sendSize);
}

ELoginAccountResult CLicenseConnection::ReceiveLoginAccountResult(bool& agreeLicenseFlag)
{
	char receiveBuffer[RecviveBufferSize] = {0,};
	uint32 receivedSize = m_clientSocket->Receive(receiveBuffer, RecviveBufferSize, ReceiveWaitTime);
	if (0 == receivedSize)
		return eLA_Undefined;
	m_builder->Clear();
	uint32 bufferOffset = 0;
	m_assembler->ApplyBuffer(receiveBuffer, bufferOffset, receivedSize);
	if (false == m_assembler->AssembleBuilder(*m_builder))
		return eLA_Undefined;

	std::string received = m_builder->MakeString();
	//printf("received : %s\n", received.c_str());
	std::string packetType = m_builder->GetValue<std::string>("type");
	if ("loginaccount" != packetType)
		return eLA_Undefined;
	ELoginAccountResult resultCode = (ELoginAccountResult)m_builder->GetValue<uint8>("resultcode");
	agreeLicenseFlag = m_builder->GetValue<bool>("agreelicense");
	if (eLA_Success == resultCode)
		m_sessionKey = m_builder->GetValue<std::string>("sessionkey");
	return resultCode;
}

bool CLicenseConnection::ReceivePasswordRemindResult()
{
	char receiveBuffer[RecviveBufferSize] = {0,};
	uint32 receivedSize = m_clientSocket->Receive(receiveBuffer, RecviveBufferSize, ReceiveWaitTime);
	if (0 == receivedSize)
		return false;
	m_builder->Clear();
	uint32 bufferOffset = 0;
	m_assembler->ApplyBuffer(receiveBuffer, bufferOffset, receivedSize);
	if (false == m_assembler->AssembleBuilder(*m_builder))
		return false;

	std::string received = m_builder->MakeString();
	//printf("received : %s\n", received.c_str());
	std::string packetType = m_builder->GetValue<std::string>("type");
	if ("passwordremind" != packetType)
		return false;

	std::string result = m_builder->GetValue<std::string>("result");
	
	return (result == ResultTrue);
}

ECreateAccountResult CLicenseConnection::RequestCreateAccount( const SCreateAccountParam& param )
{
	if (false == InitSocket())
		return eCA_Undefined;
	if (false == ConnectServer())
		return eCA_Undefined;
	if (false == SendCreateAccount(param))
		return eCA_Undefined;
	return ReceiveCreateAccount();
}

bool CLicenseConnection::SendCreateAccount( const SCreateAccountParam& param ) const
{
	m_builder->Clear();
	m_builder->Add("type", "createaccount");
	m_builder->Add("username", param.username);
	m_builder->Add("password", param.password);
	m_builder->Add("firstname", param.firstname);
	m_builder->Add("lastname", param.lastname);
	m_builder->Add("email", param.email);
	m_builder->Add("address", param.address);
	m_builder->Add("city", param.city);
	m_builder->Add("country", param.country);
	m_builder->Add("state", param.state);
	m_builder->Add("zipcode", param.zipcode);
	m_builder->Add("mac", m_localMac);
	char sendBuffer[SendBufferSize] = {0,};
	uint32 sendSize = m_builder->MakeBuffer(sendBuffer, SendBufferSize);
	return m_clientSocket->Send(sendBuffer, sendSize);
}

ECreateAccountResult CLicenseConnection::ReceiveCreateAccount() const
{
	char receiveBuffer[RecviveBufferSize] = {0,};
	uint32 receivedSize = m_clientSocket->Receive(receiveBuffer, RecviveBufferSize, ReceiveWaitTime);
	if (0 == receivedSize)
		return eCA_Undefined;
	m_builder->Clear();
	uint32 bufferOffset = 0;
	m_assembler->ApplyBuffer(receiveBuffer, bufferOffset, receivedSize);
	if (false == m_assembler->AssembleBuilder(*m_builder))
		return eCA_Undefined;

	std::string received = m_builder->MakeString();
	//printf("received : %s\n", received.c_str());
	std::string packetType = m_builder->GetValue<std::string>("type");
	if ("createaccount" != packetType)
		return eCA_Undefined;
	ECreateAccountResult resultCode = (ECreateAccountResult)m_builder->GetValue<uint8>("resultcode");
	return resultCode;
}

ELoginAccountResult CLicenseConnection::RequestLoginAccount( const std::string& username, const std::string& password, bool& agreeLicenseFlag )
{
	if (false == InitSocket())
		return eLA_Undefined;
	if (false == ConnectServer())
		return eLA_Undefined;
	if (false == SendLoginAccount(username, password))
		return eLA_Undefined;

	return ReceiveLoginAccountResult(agreeLicenseFlag);
}

bool CLicenseConnection::RequestPasswordRemind( const std::string& username, const std::string& email )
{
	if (false == InitSocket())
		return false;
	if (false == ConnectServer())
		return false;
	if (false == SendPasswordRemind(username, email))
		return false;

	return ReceivePasswordRemindResult();
}

void CLicenseConnection::Logoff()
{
	m_clientSocket->Disconnect();
}

void CLicenseConnection::SetVersion( const std::string& version )
{
	m_version = version;
}

void CLicenseConnection::SetAppName( const std::string& appName )
{
	m_appname = appName;
}