////////////////////////////////////////////////////////////////////////////
//
//  Crytek Source File.
//  Copyright (C), Crytek Studios, 2009.
// -------------------------------------------------------------------------
//  File name:   ProtocolBuilder.h
//  Version:     v1.00
//  Created:     13/08/2009 by Younggi Lim
//  Compilers:   Visual Studio.NET
//  Description: make a simple protocol based on key/value string
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "ProtocolBuilder.h"

#ifndef NOT_USING_CRYPTOGRAPHY
#include "AesCryptography.h"
#endif // NOT_USING_CRYPTOGRAPHY

const uint8 ProtocolBuilderType = 0xD2;

CProtocolBuilder::CProtocolBuilder(void) : 
	m_encrypt(false)
#ifndef NOT_USING_CRYPTOGRAPHY
, m_cryptObj(new CAesCryptography())
#endif // NOT_USING_CRYPTOGRAPHY
{
}

CProtocolBuilder::CProtocolBuilder(const CProtocolBuilder& other)
{
	m_container.clear();
	StringMapConstIterator endIter = other.m_container.end();
	for(StringMapConstIterator iter = other.m_container.begin(); iter != endIter; ++iter)
		Add(iter->first, iter->second);
}

CProtocolBuilder::~CProtocolBuilder(void)
{
#ifndef NOT_USING_CRYPTOGRAPHY
	if (NULL != m_cryptObj)
	{
		delete m_cryptObj;
		m_cryptObj = NULL;
	}
#endif // NOT_USING_CRYPTOGRAPHY
}

bool CProtocolBuilder::AddImpl(const std::string& key, const std::string& value)
{
	std::pair<StringMapIterator, bool> result;
	result = m_container.insert(std::pair<std::string, std::string>(key, value));
	if (false == result.second)
		return false;
	return true;
}

bool CProtocolBuilder::Remove(const std::string& key)
{
	StringMapIterator iter = m_container.find(key);
	if (iter == m_container.end())
		return false;
	m_container.erase(iter);
	return true;
}

bool CProtocolBuilder::Parse(const std::string& source)
{
	if (-1 == source.find("=") || -1 == source.find(";"))
		return false;

	std::string subSource = source;
	std::string key, value;

	while(subSource.length() > 0)
	{
		int keyPos = (int)subSource.find("=");
		if (keyPos<0)
			break;	

		key = subSource.substr(0, keyPos);
		subSource = subSource.substr(keyPos+1);
		int valuePos = (int)subSource.find(";");
		if (valuePos<0)
			break;
		value = subSource.substr(0, valuePos);
		Add(key, value);
		subSource = subSource.substr(valuePos+1);
	}
	return true;
}

bool CProtocolBuilder::Parse( char* buffer, uint32 &offset, const uint32 bufferLength)
{
	uint32 protocolHeaderSize = sizeof(SProtocolHeader);
	if (bufferLength < protocolHeaderSize)
		return false;

	SProtocolHeader header;
	if (false == GetHeader(buffer, offset, bufferLength, header))
		return false;
	if (bufferLength < header.totalLength)
		return false;
	
	uint32 payloadSize = header.totalLength-protocolHeaderSize;
#ifndef NOT_USING_CRYPTOGRAPHY
	if (m_encrypt)
	{
		uint32 bufferLength = payloadSize;
		char* cryptBuffer = new char[bufferLength];
		m_cryptObj->DecryptBuffer((const unsigned char*)&buffer[offset+protocolHeaderSize], bufferLength, (unsigned char*)cryptBuffer);
		memcpy(&buffer[offset+protocolHeaderSize], cryptBuffer, bufferLength);
		delete [] cryptBuffer;
		cryptBuffer = NULL;
	}
#endif // NOT_USING_CRYPTOGRAPHY
	std::string payload(&buffer[offset+protocolHeaderSize], payloadSize);
	
	if (false == Parse(payload))
		return false;

	offset += header.totalLength;
	return true;
}

std::string CProtocolBuilder::GetValueImpl(const std::string& key) const
{
	StringMapConstIterator iter = m_container.find(key);
	if (iter != m_container.end())
		return iter->second;
	return "";
}

void CProtocolBuilder::Clear()
{
	m_container.clear();
}

std::string CProtocolBuilder::MakeString() const
{
	std::string result;
	StringMapReverseIterator endIter = m_container.rend();
	for(StringMapReverseIterator iter = m_container.rbegin(); iter != endIter; ++iter)
		result += iter->first + "=" + iter->second + ";";
	return result;
}

uint32 CProtocolBuilder::MakeBuffer(char* buffer, uint32 length) const
{
	std::string strResult = MakeString();
#ifndef NOT_USING_CRYPTOGRAPHY
	if (strResult.length()<CryptBlockSize)
		strResult += "forsmalldata=dummy;";
#endif // NOT_USING_CRYPTOGRAPHY
	uint32 checkSum = 0;
	uint32 headerLength = MakeHeader(buffer, (uint32)strResult.length(), checkSum);
	memcpy(&buffer[headerLength], strResult.c_str(), strResult.length());
#ifndef NOT_USING_CRYPTOGRAPHY
	if (m_encrypt)
	{
		uint32 bufferLength = (uint32)strResult.length();
		char* cryptBuffer = new char[bufferLength];
		m_cryptObj->EncryptBuffer((const unsigned char*)&buffer[headerLength], bufferLength, (unsigned char*)cryptBuffer);
		memcpy(&buffer[headerLength], cryptBuffer, bufferLength);
		delete [] cryptBuffer;
		cryptBuffer = NULL;
	}
#endif // NOT_USING_CRYPTOGRAPHY
	return headerLength+(uint32)strResult.length();
}

uint32 CProtocolBuilder::MakeHeader( char* buffer, uint32 dataLength, uint32 checkSum ) const
{
	uint32 protocolHeaderSize = sizeof(SProtocolHeader);
	SProtocolHeader header;
	header.type = ProtocolBuilderType;
	header.totalLength = dataLength+protocolHeaderSize;
	header.checkSum = checkSum;
	memcpy(buffer, &header, protocolHeaderSize);
	return protocolHeaderSize;
}

bool CProtocolBuilder::GetHeader( const char* buffer, uint32 offset, uint32 bufferLength, SProtocolHeader& header ) const
{
	const uint32 protocolHeaderSize = sizeof(SProtocolHeader);
	if (bufferLength < protocolHeaderSize)
		return false;

	memcpy(&header, &buffer[offset], protocolHeaderSize);
	if (ProtocolBuilderType != header.type)
		return false;

	if (protocolHeaderSize > header.totalLength)
		return false;

	// todo : checksum check

	return true;
}

void CProtocolBuilder::EncryptKey(unsigned char* key, unsigned char keyLen)
{
#ifndef NOT_USING_CRYPTOGRAPHY
	m_encrypt = true;
	m_cryptObj->SetKeyValue(key, keyLen);
#endif // NOT_USING_CRYPTOGRAPHY
}