////////////////////////////////////////////////////////////////////////////
//
//  Crytek Source File.
//  Copyright (C), Crytek Studios, 2009.
// -------------------------------------------------------------------------
//  File name:   ProtocolBuilder.h
//  Version:     v1.00
//  Created:     13/08/2009 by Younggi Lim
//  Compilers:   Visual Studio.NET
//  Description: make a simple protocol based on key/value string
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////
#ifndef __protocolbuilder_h__
#define __protocolbuilder_h__

#pragma once

#include <string>
#include <map>
#include "DataTypeConvertor.h"

class CAesCryptography;

class CProtocolBuilder
{
public:
	typedef std::map<std::string, std::string>				StringKeyValueMap;
	typedef StringKeyValueMap::iterator								StringMapIterator;
	typedef StringKeyValueMap::const_iterator					StringMapConstIterator;
	typedef StringKeyValueMap::const_reverse_iterator StringMapReverseIterator;

	struct SProtocolHeader
	{
		uint8		type;
		uint32	totalLength;
		uint32	checkSum;
	};

public:
	CProtocolBuilder();
	CProtocolBuilder(const CProtocolBuilder& other);
	virtual ~CProtocolBuilder();

	template<typename T>
	bool					Add(const std::string& key, const T& value)
	{
		const int MaxValueLength = 64;
		char strValue[MaxValueLength] = {0,};
		DataTypeConvertor::ToString(value, strValue, MaxValueLength);
		return AddImpl(key, strValue);
	}

	bool					Remove(const std::string& key);
	void					Clear();

	template<typename T>
	T		GetValue(const std::string& key) const
	{
		std::string result = GetValueImpl(key);
		T value;
		DataTypeConvertor::ToNumeric(result.c_str(), value);
		return value;
	}

	////////////////////////////////////////////////////////////////////////
	// Description:
	//   Parsing and Add to container 
	//   i.e.) name=alice;level=40; => ["name", "alice"], ["level", "40"]
	////////////////////////////////////////////////////////////////////////
	bool					Parse(const std::string& source);
	bool					Parse(char* buffer, uint32 &offset, const uint32 bufferLength);
	std::string		MakeString() const;
	uint32				MakeBuffer(char* buffer, uint32 length) const;
	void					EncryptKey(unsigned char* key, unsigned char keyLen);


protected:
	bool					AddImpl(const std::string& key, const std::string& value);
	////////////////////////////////////////////////////////////////////////
	// Description:
	//   If there is no value, return empty string("")
	////////////////////////////////////////////////////////////////////////
	std::string		GetValueImpl(const std::string& key) const;

	uint32				MakeHeader(char* buffer, uint32 dataLength, uint32 checkSum) const;
	bool					GetHeader(const char* buffer, uint32 offset, uint32 bufferLength, SProtocolHeader& header) const;

private:
	StringKeyValueMap m_container;
	bool							m_encrypt;
	CAesCryptography*	m_cryptObj;
};

#endif // __protocolbuilder_h__
