#include "stdafx.h"
#include "ClientModeServer.h"
#include "JobQueue.h"
#include "ClientModeManager.h"
#include "LicenseConnection.h"
#include <Drei/TimerUtil.h>
#include "ServerConfig.h"
#include <Drei/NetworkFacade.h>
#include "LicenseConnectionServer.h"

const uint32 LicenseConnectionCheckTerm = 60*1000;
//const uint32 LicenseConnectionCheckTerm = 10*1000;

static std::string Base64Encode(const char* pBytesToEncode) 
{
	static const std::string base64_chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
	unsigned int in_len = (uint32)strlen(pBytesToEncode);

	std::string ret;
	int i = 0;
	int j = 0;
	unsigned char char_array_3[3];
	unsigned char char_array_4[4];

	while (in_len--) 
	{
		char_array_3[i++] = (unsigned char)*(pBytesToEncode++);
		if (i == 3) {
			char_array_4[0] = (char_array_3[0] & 0xfc) >> 2;
			char_array_4[1] = ((char_array_3[0] & 0x03) << 4) + ((char_array_3[1] & 0xf0) >> 4);
			char_array_4[2] = ((char_array_3[1] & 0x0f) << 2) + ((char_array_3[2] & 0xc0) >> 6);
			char_array_4[3] = char_array_3[2] & 0x3f;

			for(i = 0; (i <4) ; i++)
				ret += base64_chars[char_array_4[i]];
			i = 0;
		}
	}

	if (i)
	{
		for(j = i; j < 3; j++)
			char_array_3[j] = '\0';

		char_array_4[0] = (char_array_3[0] & 0xfc) >> 2;
		char_array_4[1] = ((char_array_3[0] & 0x03) << 4) + ((char_array_3[1] & 0xf0) >> 4);
		char_array_4[2] = ((char_array_3[1] & 0x0f) << 2) + ((char_array_3[2] & 0xc0) >> 6);
		char_array_4[3] = char_array_3[2] & 0x3f;

		for (j = 0; (j < i + 1); j++)
			ret += base64_chars[char_array_4[j]];

		while((i++ < 3))
			ret += '=';
	}
	return ret;
}

//////////////////////////////////////////////////////////////////////////

CClientModeServer::CClientModeServer(void) : 
	m_init(false), 
	m_jobQueue(NULL), 
	m_clientmodeManager(new CClientModeManager()), 
	m_licenseCheckTimer(new DreiNetwork::CycleChecker(LicenseConnectionCheckTerm)),
	m_licenseConnectionServer(new CLicenseConnectionServer())
{
}

CClientModeServer::~CClientModeServer(void)
{
	SAFE_DELETE(m_clientmodeManager);
	SAFE_DELETE(m_licenseCheckTimer);
	SAFE_DELETE(m_licenseConnectionServer);
}

bool CClientModeServer::Init(CJobQueue* que)
{
	//SetupProxy();
	if (false == IsValidServerConfig())
		return false;

	CLicenseConnection licenseConnection;
	EPMErrorCode retCode = AuthLicenseConnectionSync(licenseConnection);
	std::string encryptKey;
	licenseConnection.GetEncryptKey(encryptKey);
	uint32 floatLicenses = licenseConnection.GetFloatLicenses();
	std::string localIp, localMac;
	licenseConnection.GetLocalIp(localIp);
	licenseConnection.GetMacAddress(localMac);
	ApplyLicenseData(retCode, encryptKey, floatLicenses, localIp, localMac);
	if (PMEC_NO_ERROR != retCode)
	{
		PrintErrorCode(retCode);
		return false;
	}

	std::string licenseKey;
	licenseConnection.GetLicenseKey(licenseKey);
	uint32 maxUserCount = licenseConnection.GetFloatLicenses();
	printf("License key is %s.\n", licenseKey.c_str());
	licenseConnection.Logoff();

	m_jobQueue = que;
	m_init = true;
	return true;
}

void CClientModeServer::Fini()
{
	if (false == m_init)
		return;
	m_licenseConnectionServer->Fini();
}

void CClientModeServer::Update()
{
	const uint32 jobFetchTimeout = 1000*5;
	while(true)
	{
		CAbstractJob* job = m_jobQueue->PopWork(jobFetchTimeout);
		if (NULL == job)
			break;

		job->ClientModeExecute(*m_clientmodeManager);
		job->Complete();
		delete job;
		job = NULL;
		if (m_jobQueue->Empty())
			break;
	}
	UpdateLicenseConnection();
}

bool CClientModeServer::GetProxyInfo( std::string& ip, int& port, std::string& authStr )
{
	ip = CServerConfig::GetValue<std::string>("net_proxy_ip");
	if (ip.empty())
		return false;

	std::string sAuthUser = CServerConfig::GetValue<std::string>("net_proxy_user");
	std::string sAuthPass = CServerConfig::GetValue<std::string>("net_proxy_pass");
	std::string sAuthUserPass = sAuthUser+":"+sAuthPass;
	if (false == sAuthUser.empty())
		authStr = Base64Encode(sAuthUserPass.c_str());
	std::string sProxyPort = CServerConfig::GetValue<std::string>("net_proxy_port");
	port = atoi(sProxyPort.c_str());
	return true;
}

EPMErrorCode CClientModeServer::AuthLicenseConnectionSync(CLicenseConnection& licenseConnection)
{
	licenseConnection.ResetLogin();
	licenseConnection.SetConnectionInfo(m_serverIp.c_str(), m_serverPort);
	std::string proxyServerIp;
	std::string authStr;
	int proxyPort = 0;
	if (GetProxyInfo(proxyServerIp, proxyPort, authStr))
		licenseConnection.SetProxy(proxyServerIp, proxyPort, authStr);
	licenseConnection.SetAppName(LicenseServerAppname);
	licenseConnection.SetVersion(LicenseServerVersion);
	return licenseConnection.Authenticate(m_licenseKey.c_str());
}

void CClientModeServer::AuthLicenseConnectionAsync()
{
	std::string localIp, localMac;
	m_clientmodeManager->GetLocalIp(localIp);
	m_clientmodeManager->GetMacAddress(localMac);
	uint32 streamId = m_clientmodeManager->GetGlobalLicenseServerStreamId();

	CProtocolBuilder builder;
	builder.EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
	builder.Add("type", "login");
	builder.Add("key", m_licenseKey);
	builder.Add("ip", localIp);
	builder.Add("mac", localMac);
	builder.Add("appname", LicenseServerAppname);
	builder.Add("version", LicenseServerVersion);
	const uint32 MaxBuffer = 1024;
	char sendBuffer[MaxBuffer] = {0,};
	uint32 sendSize = builder.MakeBuffer(sendBuffer, MaxBuffer);
	if (false == NetworkInstance->SendRequest(streamId, sendBuffer, sendSize))
		ResetConnection();
}

void CClientModeServer::ApplyLicenseData(EPMErrorCode retCode, const std::string& encryptKey, uint32 floatLicenses, const std::string localIp, const std::string localMac)
{
	if (m_clientmodeManager->GetErrorCode() != retCode)
	{
		printf("Authenticate result is %s.\n", (PMEC_NO_ERROR==retCode)?"true":"false");
		m_clientmodeManager->SetErrorCode(retCode);
	}

	if (m_clientmodeManager->GetMaxUserCount() != floatLicenses)
	{
		printf("Concurrent Floating License is %d.\n", floatLicenses);
		m_clientmodeManager->SetMaxUserCount(floatLicenses);
	}
	m_clientmodeManager->SetLicenseKey(m_licenseKey);
	m_clientmodeManager->SetEncryptKey(encryptKey);
	m_clientmodeManager->SetLocalIp(localIp);
	m_clientmodeManager->SetMacAddress(localMac);
}

void CClientModeServer::UpdateLicenseConnection()
{
	if (false == m_licenseCheckTimer->IsExpire())
		return;

	AuthLicenseConnectionAsync();
}

bool CClientModeServer::IsValidServerConfig()
{
	m_licenseKey = CServerConfig::GetValue<std::string>("ClientModeLicenseKey");
	if (m_licenseKey.empty())
		return false;
	m_serverIp = CServerConfig::GetValue<std::string>("ClientModeServerIp");
	if (m_serverIp.empty())
		m_serverIp = GlobalServerIp;
	m_serverPort = CServerConfig::GetValue<uint32>("ClientModeServerPort");
	if (0 == m_serverPort)
		m_serverPort = DefaultServicePort;
	return true;
}

void CClientModeServer::MakeConnectionWithGlobalServer()
{
	m_licenseConnectionServer->Init(m_jobQueue);
	std::string proxyServerIp;
	std::string authStr;
	int proxyPort = 0;
	if (GetProxyInfo(proxyServerIp, proxyPort, authStr))
		m_licenseConnectionServer->SetProxy(m_serverIp.c_str(), m_serverPort, authStr);
	ConnectGlobalServer();
}

void GetSpecificErrorMessage( EPMErrorCode errorCode, char* titleMsg, char* captionMsg )
{
	const char* ValidationErrorMsg = "Your License could not be validated.\nIn order to help you solving any connectivity issues, please provide the error code in the title of this dialog.";
	if (PMEC_CONNECT_FAILED == errorCode)
	{
		strcpy(captionMsg, "Could not connect to the License Server. Please establish an internet connection.\nIf you still cannot connect, check your firewall or contact Crytek to resolve this issue.");
		strcpy(titleMsg, "Connection Error 0x100");
	}
	else if (PMEC_PROXY_CONNECT_FAILED == errorCode)
	{
		strcpy(captionMsg, "Could not authenticate at proxy. Please verify your proxy settings.\nIf you still cannot connect, check your firewall or contact Crytek to resolve this issue.");
		strcpy(titleMsg, "Connection Error 0x110");
	}	
	else if (PMEC_CANT_WRITE_TO_SOCKET == errorCode)
	{
		strcpy(captionMsg, "Could not write to the License Server. Please check your firewall or contact Crytek to resolve this issue.");
		strcpy(titleMsg, "Connection Error 0x200");
	}
	else if (PMEC_CANT_READ_FROM_SOCKET == errorCode)
	{
		strcpy(captionMsg, "Could not read from the License Server. Please check your firewall or contact Crytek to resolve this issue.");
		strcpy(titleMsg, "Connection Error 0x300");
	}
	else if (PMEC_NETWORK_DEVICE == errorCode)
	{
		strcpy(captionMsg, "Could not find your network device.\nPlease check your device or contact Crytek to resolve this issue.");
		strcpy(titleMsg, "Connection Error 0x400");
	}	
	else if (PMEC_KEY_EXPIRED == errorCode)
	{
		strcpy(captionMsg, ValidationErrorMsg);
		strcpy(titleMsg, "Validation Error 0x100");
	}
	else if (PMEC_KEY_INVALID == errorCode)
	{
		strcpy(captionMsg, ValidationErrorMsg);
		strcpy(titleMsg, "Validation Error 0x200");
	}
	else if (PMEC_INVALID_TIMEFRAME == errorCode)
	{
		strcpy(captionMsg, ValidationErrorMsg);
		strcpy(titleMsg, "Validation Error 0x300");
	}
	else if (PMEC_KEY_NOT_FOUND == errorCode)
	{
		strcpy(captionMsg, ValidationErrorMsg);
		strcpy(titleMsg, "Validation Error 0x400");
	}
	else if (PMEC_OVER_CONNECTION == errorCode)
	{
		strcpy(captionMsg, ValidationErrorMsg);
		strcpy(titleMsg, "Validation Error 0x500");
	}
	else if (PMEC_NO_DONGLE == errorCode)
	{
		strcpy(captionMsg, ValidationErrorMsg);
		strcpy(titleMsg, "Validation Error 0x600");
	}
	else if (PMEC_INVALID_ADDRESS == errorCode)
	{
		strcpy(captionMsg, ValidationErrorMsg);
		strcpy(titleMsg, "Validation Error 0x700");
	}
	else if (PMEC_REGISTRY_NOT_FOUND == errorCode)
	{
		strcpy(captionMsg, "Could not find a valid License Information. Please run SettingsMgr or Sandbox to validate your CryENGINE installation.");
		strcpy(titleMsg, "Validation Error 0x800");
	}
	else if (PMEC_DOUBLE_LOGIN == errorCode)
	{
		strcpy(captionMsg, ValidationErrorMsg);
		strcpy(titleMsg, "Validation Error 0x900");
	}
	else if (PMEC_INVALID_CONNECT_TYPE == errorCode)
	{
		strcpy(captionMsg, "The License Key you entered is not meant for this type of application. License Keys can be of following types:\n");
		strcat(captionMsg, "- SDK activation (Sandbox & Tools)\n");
		strcat(captionMsg, "- Use of the Educational License Server only\n");
		strcat(captionMsg, "- Launcher application only\n");
		strcat(captionMsg, "Please double check the type of license you have and contact Crytek support if necessary.");
		strcpy(titleMsg, "Validation Error 0x110");
	}
	else if (PMEC_OLD_APP_VERSION == errorCode)
	{
		strcpy(captionMsg, "The version of this application is deprecated.\n");
		strcat(captionMsg, "Please download new version on mycryengine.com.\n");
		strcpy(titleMsg, "Validation Error 0x120");
	}
	else
	{
		char unknownErrorTitle[128] = {0,};
		sprintf(unknownErrorTitle, "Unknown Error 0x%d", (int)errorCode);
		strcpy(captionMsg, ValidationErrorMsg);
		strcpy(titleMsg, unknownErrorTitle);
	}
}


void CClientModeServer::PrintErrorCode( EPMErrorCode retCode )
{
	const int ErrorMessageSize = 1024;
	char titleErrorMessage[ErrorMessageSize] = {0,};
	char captionErrorMessage[ErrorMessageSize] = {0,};
	GetSpecificErrorMessage(retCode, titleErrorMessage, captionErrorMessage);
	printf("%s\n", titleErrorMessage);
	printf("%s\n", captionErrorMessage);
}

void CClientModeServer::ConnectGlobalServer()
{
	std::string proxyServerIp;
	std::string authStr;
	int proxyPort = 0;
	if (GetProxyInfo(proxyServerIp, proxyPort, authStr))
		NetworkInstance->Connect(proxyServerIp.c_str(), proxyPort, LicenseConnectionQueueId);
	else
		NetworkInstance->Connect(m_serverIp.c_str(), m_serverPort, LicenseConnectionQueueId);
}

void CClientModeServer::ResetConnection()
{
	uint32 streamId = m_clientmodeManager->GetGlobalLicenseServerStreamId();
	NetworkInstance->CloseStream(streamId);
	ConnectGlobalServer();
}