#include "stdafx.h"
#include "DatabaseModeManager.h"

#ifdef USING_DATABASE_MODE

#include "LicenseDBJob.h"
#include "LicensekeyPool.h"
#include "DBConnection.h"
#include "ServerConfig.h"
//#include "ServerLog.h"
#include "StringFilter.h"
#include <Drei/TimerUtil.h>

//////////////////////////////////////////////////////////////////////////
CDatabaseModeManager::CDatabaseModeManager() : m_dbConn(NULL), m_pool(new CLicensekeyPool())
{
}

CDatabaseModeManager::~CDatabaseModeManager()
{
	SAFE_DELETE(m_pool);
}

bool CDatabaseModeManager::Init()
{
	std::string dbIP = CServerConfig::GetValue<std::string>("DBIp");
	std::string dbUser = CServerConfig::GetValue<std::string>("DBUser");
	std::string dbPass = CServerConfig::GetValue<std::string>("DBPass");
	std::string defaultDB = CServerConfig::GetValue<std::string>("DefaultDB");
	uint32 dbPort = CServerConfig::GetValue<uint32>("DBPort");

	if (dbIP.empty())
		return false;
	if (dbUser.empty())
		return false;
	if (dbPass.empty())
		return false;
	if (defaultDB.empty())
		return false;
	if (0 == dbPort)
		return false;

	m_dbConn = new CDBConnection();

	printf("Connecting database.\n");
	bool result = false;
	result = m_dbConn->Init();
	if (false == result)
		return false;
	result = m_dbConn->Connect(
		dbIP.c_str(), 
		dbPort, 
		dbUser.c_str(),
		dbPass.c_str(),
		defaultDB.c_str());

	if (false == result)
		return false;

	return true;
}

void CDatabaseModeManager::Fini()
{
	m_dbConn->Disconnect();
	SAFE_DELETE(m_dbConn);
}

SLoginUserResult CDatabaseModeManager::LoginUser(uint32 userId, const SAuthenticateParam& authParam, const std::string& ip, const std::string& version)
{
	SLoginUserResult result;
	CAuthenticateLicenseKey job(authParam.licenseKey, ip, authParam.macAddress);
	job.SetMySQLConnection(*m_dbConn);
	bool fromGeneratedProcess  = false;
	result.errorCode = job.AssignedKeyProcess();
	if (PMEC_NO_ERROR != result.errorCode)
	{
		result.errorCode = job.GeneratedKeyProcess();
		if (PMEC_NO_ERROR == result.errorCode)
			fromGeneratedProcess = true;
	}

	if (PMEC_NO_ERROR != result.errorCode)
		return result;

	SSelectAssignedKeyParam& resultSet = job.ResultSet();
	// remove ip check
	//if (resultSet.ipAddress != m_ip)
	//	m_result = PMEC_INVALID_ADDRESS;
	// remove MacAddress restriction
	//if (resultSet.macAddress != authParam.macAddress)
	//	result.errorCode = PMEC_INVALID_ADDRESS;
	if (false == resultSet.isValid)
		result.errorCode = PMEC_KEY_EXPIRED;
	if (PMEC_NO_ERROR == result.errorCode)
	{
		result.remainSecond = resultSet.remainSecond;
		if (0 == resultSet.remainSecond)
			result.errorCode = PMEC_KEY_EXPIRED;
		result.companyID = resultSet.companyID;
		result.connectType = resultSet.connectType;
		result.floatLicenses = resultSet.floatLicenses;
		if (eLT_LicenseServer == result.connectType)
		{
			if (false == resultSet.hasFloatConnectRight)
			{
				result.allowFloatConnection = false;
				result.floatLicenses = 0;
				result.errorCode = PMEC_INVALID_CONNECT_TYPE;
			}
		}	
		else
		{
			result.floatLicenses = 0;
		}
	}

	if (PMEC_NO_ERROR != result.errorCode)
		return result;
	
	SLicenseKeyPoolElement element;
	element.licenseKey = authParam.licenseKey;
	element.macAddress = authParam.macAddress;
	element.application = authParam.application;
	element.version = version;
	element.loginTick = DreiNetwork::GetTick();
	element.licenseType = (ELicenseType)resultSet.connectType;
	GenerateEncryptKey generator;
	element.sessionId = generator.GenerateGuid();
	if (false == m_pool->IsValidConnectType(element))
	{
		result.errorCode = PMEC_INVALID_CONNECT_TYPE;
		return result;
	}

	if (false == m_pool->Login(userId, element))
	{
		result.errorCode = PMEC_DOUBLE_LOGIN;
		return result;
	}

	result.sessionId = element.sessionId;

	if (fromGeneratedProcess)
		job.MoveKeyFromGeneratedToAssigned();

	return result;
}

void CDatabaseModeManager::LogoffUser(uint32 userId)
{
	SLicenseKeyPoolElement element;
	if (false == m_pool->GetElement(userId, element))
		return;

	m_pool->Logoff(userId);
	uint32 currentTick = DreiNetwork::GetTick();
	uint32 usageTimeByMin = 0;
	if (currentTick>element.loginTick)
		usageTimeByMin = (currentTick - element.loginTick)/(1000*60);
	CUpdateUsageTimeAndVersion job(element.licenseKey, element.application, element.version, usageTimeByMin);
	job.SetMySQLConnection(*m_dbConn);
	job.Execute();
}

EPMErrorCode CDatabaseModeManager::AuthenticateUser(uint32 userId, const SAuthenticateParam& authParam, const std::string& ip)
{
	if (false == m_pool->Authenticate(userId, authParam))
	{
		// Todo
		return PMEC_DOUBLE_LOGIN;
	}

	CAuthenticateLicenseKey job(authParam.licenseKey, ip, authParam.macAddress);
	job.SetMySQLConnection(*m_dbConn);
	EPMErrorCode errorCode = job.AssignedKeyProcess();
	if (PMEC_NO_ERROR != errorCode)
		return errorCode;

	SSelectAssignedKeyParam& resultSet = job.ResultSet();
	// remove MacAddress restriction
	//if (resultSet.macAddress != authParam.macAddress)
	//	errorCode = PMEC_INVALID_ADDRESS;
	if (false == resultSet.isValid)
		errorCode = PMEC_KEY_EXPIRED;
	if (PMEC_NO_ERROR == errorCode)
	{
		if (0 == resultSet.remainSecond)
			errorCode = PMEC_KEY_EXPIRED;
	}

	return errorCode;
}

ECreateAccountResult CDatabaseModeManager::CreateAccount( const SCreateAccountParam& param, const std::string& licenseKey, uint32& userId )
{
	ECreateAccountResult resultCode = eCA_Success;
	if (false == IsValidUsernameString(param.username))
		return eCA_BanCharacter;
	CSandboxStudent job;
	job.SetMySQLConnection(*m_dbConn);

	resultCode = job.CreateAccount(param, licenseKey);
	if (eCA_Success == resultCode)
		job.SelectUserId(param.username, userId);
	return resultCode;
}

ELoginAccountResult CDatabaseModeManager::AuthenticateAccount( const std::string& username, const std::string& password, const std::string& licenseKey, bool& agreelicenseFlag )
{
	CSandboxStudent job;
	job.SetMySQLConnection(*m_dbConn);
	ELoginAccountResult resultCode = job.Authenticate(username, password, licenseKey, agreelicenseFlag);
	return resultCode;
}

void CDatabaseModeManager::UpdateAgreeLicense( const std::string& username )
{
	CSandboxStudent job;
	job.SetMySQLConnection(*m_dbConn);
	job.UpdateAgreeLicense(username);
}

bool CDatabaseModeManager::IsValidUsernameString( const std::string& username )
{
	CEngLowerString engLowerFilter(NULL);
	CEngUpperString engUpperFilter(&engLowerFilter);
	CNumericString completeFilter(&engUpperFilter);
	return completeFilter.IsValid(username);
}

bool CDatabaseModeManager::GetStudentPassword( const std::string& username, const std::string& email, std::string& password, uint32& userId )
{
	if (false == IsValidUsernameString(username))
		return false;

	CSandboxStudent job;
	job.SetMySQLConnection(*m_dbConn);
	if (email.empty())
		return job.SelectPassword(username, userId, password);
	else
		return job.SelectPasswordByEmail(email, userId, password);
}

#endif // USING_DATABASE_MODE