#include "stdafx.h"
#include "LicensekeyPool.h"
#include "ServerLog.h"

enum EApplicationType
{
	eAT_Editor = 0,
	eAT_Launcher,
	eAT_DedicatedServer,
	eAT_LicenseServer,
	eAT_SettingsManager,
	eAT_Unknown,
	eAT_Max,
};

const char* ApplicationName[eAT_Max] = 
{
	"Editor",
	"Launcher",
	"DedicatedServer",
	"LicenseServer",
	"SettingsManager",
	"Unknown"
};

CLicensekeyPool::CLicensekeyPool(void)
{
}

CLicensekeyPool::~CLicensekeyPool(void)
{
}

bool CLicensekeyPool::Login( uint32 userId, const SLicenseKeyPoolElement& newElement)
{
	if (false == IsValidConnectType(newElement))
		return false;
	IndexPoolConstIterator findIndex = m_indexContainer.find(userId);
	if (m_indexContainer.end() != findIndex)
	{
		bool isValidWithExistingElement = true;
		const SLicenseKeyPoolElement& existingElement = findIndex->second;
		isValidWithExistingElement &= (existingElement.licenseKey == newElement.licenseKey);
		isValidWithExistingElement &= (existingElement.macAddress == newElement.macAddress);
		isValidWithExistingElement &= (existingElement.application == newElement.application);
		if (false == isValidWithExistingElement)
		{
			DREI_LOG(	ErrorLogger, 
				"LicensekeyPool Login failed. userId(%d), existing(%s, %s, %s), new(%s, %s, %s)",
				userId,
				existingElement.licenseKey.c_str(), existingElement.macAddress.c_str(), existingElement.application.c_str(),
				newElement.licenseKey.c_str(), newElement.macAddress.c_str(), newElement.application.c_str());
		}
		return isValidWithExistingElement;
	}

	if (false == IsValidWithExistingElemnt(newElement))
		return false;

	return AddElement(userId, newElement);
}

void CLicensekeyPool::Logoff(uint32 userId)
{
	std::string key;
	std::string sessionId;
	RemoveIndexElement(userId, key, sessionId);
	if (key.empty())
		return;
	RemoveLicenseKeyElement(key, userId);
	RemoveSessionIdElement(sessionId);
	//printf("Logoff id(%d), %d, %d\n",
	//	userId, m_indexContainer.size(), m_keyContainer.size());
}

bool CLicensekeyPool::Authenticate(uint32 userId, const SAuthenticateParam& authParam) const
{
	SLicenseKeyPoolElement existingElement;
	if (false == GetElement(userId, existingElement))
	{
		DREI_LOG( ErrorLogger, 
			"LicensekeyPool Authenticate ErrorCode(Can't find userId) userId(%d), key(%s), mac(%s)", 
			userId,	authParam.licenseKey.c_str(), authParam.macAddress.c_str());
		return false;
	}

	bool isValidWithExistingElement = true;
	isValidWithExistingElement &= (existingElement.licenseKey == authParam.licenseKey);
	isValidWithExistingElement &= (existingElement.macAddress == authParam.macAddress);
	isValidWithExistingElement &= (existingElement.application == authParam.application);
	if (false == isValidWithExistingElement)
	{
		DREI_LOG( ErrorLogger, 
			"LicensekeyPool Authenticate failed. userId(%d), existing(%s, %s, %s), new(%s, %s, %s)",
			userId,
			existingElement.licenseKey.c_str(), existingElement.macAddress.c_str(), existingElement.application.c_str(),
			authParam.licenseKey.c_str(), authParam.macAddress.c_str(), authParam.application.c_str());
		return false;
	}

	SLicenseKeyPoolElement tempElem;
	if (false == GetElement(authParam.licenseKey, userId, tempElem))		// key pool check
		return false;

	return true;
}

size_t CLicensekeyPool::GetElementCount() const
{
	return m_indexContainer.size();
}

void CLicensekeyPool::RemoveIndexElement( uint32 userId, std::string& licenseKey, std::string& sessionId )
{
	if (m_indexContainer.size() <= 0)
		return;
	IndexPoolIterator removeIndex = m_indexContainer.find(userId);
	if (m_indexContainer.end() == removeIndex)
		return;
	licenseKey = (removeIndex->second).licenseKey;
	sessionId = (removeIndex->second).sessionId;
	m_indexContainer.erase(removeIndex);
}

void CLicensekeyPool::RemoveLicenseKeyElement( const std::string& key, uint32 userId )
{
	if (key.empty())
		return;
	if (m_keyContainer.size() <= 0)
		return;

	std::pair<KeyPoolIterator,KeyPoolIterator> result;
	result = m_keyContainer.equal_range(key);
	if (result.first == result.second)	// no matched element
		return;

	KeyPoolIterator curIter = result.first;
	for (;curIter!=result.second; ++curIter)
	{
		if (curIter->second != userId)
			continue;

		m_keyContainer.erase(curIter);
		break;
	}
}

void CLicensekeyPool::RemoveSessionIdElement( const std::string& sessionId)
{
	if (m_sessionIdContainer.size() <= 0)
		return;
	SessionIdPoolIterator removeIndex = m_sessionIdContainer.find(sessionId);
	if (m_sessionIdContainer.end() == removeIndex)
		return;
	m_sessionIdContainer.erase(removeIndex);
}

bool CLicensekeyPool::AddElement( uint32 userId, const SLicenseKeyPoolElement& element )
{
	std::pair<IndexPoolIterator,bool> indexResult;
	indexResult = m_indexContainer.insert(std::make_pair(userId, element));
	if (false == indexResult.second)
		return false;

	KeyPoolIterator insertedIter = m_keyContainer.insert(std::make_pair(element.licenseKey, userId));
	if ( insertedIter->first != element.licenseKey || insertedIter->second != userId)
	{
		std::string dummyKey, dummySessionId;
		RemoveIndexElement(userId, dummyKey, dummySessionId);
		return false;
	}
	
	SessionIdPoolIterator sessionIdInsertedIter = m_sessionIdContainer.insert(std::make_pair(element.sessionId, userId));
	return true;
}

bool CLicensekeyPool::GetElement( uint32 userId, SLicenseKeyPoolElement& outputElement ) const
{
	IndexPoolConstIterator findIndex = m_indexContainer.find(userId);
	if (m_indexContainer.end() == findIndex)
		return false;

	outputElement = (findIndex->second);
	return true;
}

bool CLicensekeyPool::GetElement( const std::string& licenseKey, uint32 userId, SLicenseKeyPoolElement& outputElement ) const
{
	std::pair<KeyPoolConstIterator,KeyPoolConstIterator> result;
	result = m_keyContainer.equal_range(licenseKey);
	if (result.first == result.second)	// no matched element
		return false;

	KeyPoolConstIterator curIter = result.first;
	for (;curIter!=result.second; ++curIter)
	{
		if (curIter->second != userId)
			continue;
		if (false == GetElement(curIter->second, outputElement))
			continue;

		return true;
	}
	return false;
}
#if 0
bool CLicensekeyPool::ChangeElementIf( uint32 oldUserId, uint32 newUserId, const SLicenseKeyPoolElement& newElement )
{
	SLicenseKeyPoolElement oldElement;
	if (false == GetElement(oldUserId, oldElement))
	{
		DREI_LOG(
			ErrorLogger, 
			"LicensekeyPool ChangeElementIf ErrorCode(Can't find oldElement) oldId(%d), newId(%d)", 
			oldUserId, newUserId);
		return false;
	}

	if (oldElement.licenseKey != newElement.licenseKey)
	{
		DREI_LOG(
			ErrorLogger, 
			"LicensekeyPool ChangeElementIf ErrorCode(Different key) oldId(%d), newId(%d), oldKey(%s), newKey(%s), oldMac(%s), newMac(%s)", 
			oldUserId, newUserId,
			oldElement.licenseKey.c_str(), newElement.licenseKey.c_str(), 
			oldElement.macAddress.c_str(), newElement.macAddress.c_str());
		return false;
	}

	if (oldElement.macAddress != newElement.macAddress)
	{
		DREI_LOG(
			ErrorLogger, 
			"LicensekeyPool ChangeElementIf ErrorCode(Different Mac) oldId(%d), newId(%d), oldKey(%s), newKey(%s), oldMac(%s), newMac(%s)", 
			oldUserId, newUserId,
			oldElement.licenseKey.c_str(), newElement.licenseKey.c_str(), 
			oldElement.macAddress.c_str(), newElement.macAddress.c_str());
		return false;
	}

	DREI_LOG(
		ErrorLogger, 
		"LicensekeyPool ChangeElementIf oldId(%d), newId(%d), changeCount(%d), key(%s), mac(%s)", 
		oldUserId, newUserId, oldElement.changeCount,
		oldElement.licenseKey.c_str(), 
		oldElement.macAddress.c_str());

	//const uint32 AllowChangeCount = 20;
	//if (AllowChangeCount < oldElement.changeCount)
	//	return false;

	Logoff(oldUserId);
	++oldElement.changeCount;
	return AddElement(newUserId, oldElement);
}

#endif

bool CLicensekeyPool::IsValidWithExistingElemnt( const SLicenseKeyPoolElement& newElement ) const
{
	ElementVector existingElemList;
	GetElementListWith(newElement.licenseKey, existingElemList);
	ElementConstIterator curIter = existingElemList.begin();
	for (;curIter!=existingElemList.end(); ++curIter)
	{
		const SLicenseKeyPoolElement& existingElement = *curIter;
		bool compareResult = true;
		//compareResult &= (newElement.application != existingElement.application);
		compareResult &= (newElement.macAddress == existingElement.macAddress);
		compareResult &= (eLT_Full == existingElement.licenseType);
		if (false == compareResult)
		{
			DREI_LOG(	ErrorLogger, 
				"LicensekeyPool IsValidWithExistingElemnt failed. existing(%s, %s, %s), new(%s, %s, %s)",
				existingElement.licenseKey.c_str(), existingElement.macAddress.c_str(), existingElement.application.c_str(),
				newElement.licenseKey.c_str(), newElement.macAddress.c_str(), newElement.application.c_str());
			return false;
		}
	}

	return true;
}

bool CLicensekeyPool::GetIndexListWith( const std::string& key, IndexVector& list ) const
{
	std::pair<KeyPoolConstIterator,KeyPoolConstIterator> result;
	result = m_keyContainer.equal_range(key);
	KeyPoolConstIterator curIter = result.first;
	for (;curIter!=result.second; ++curIter)
		list.push_back(curIter->second);
	return (false == list.empty());
}

bool CLicensekeyPool::GetElementListWith( const std::string& key, ElementVector& outputList ) const
{
	std::pair<KeyPoolConstIterator,KeyPoolConstIterator> result;
	result = m_keyContainer.equal_range(key);
	if (result.first == result.second)	// no matched element
		return false;
	KeyPoolConstIterator curIter = result.first;
	for (;curIter!=result.second; ++curIter)
	{
		SLicenseKeyPoolElement existingElement;
		if (false == GetElement(curIter->second, existingElement))
			continue;
		outputList.push_back(existingElement);
	}
	return true;
}

bool CLicensekeyPool::IsValidConnectType( const SLicenseKeyPoolElement& element ) const
{
	bool result = false;
	if (eLT_Full == element.licenseType)
	{
		result |= (ApplicationName[eAT_Editor] == element.application);
		result |= (ApplicationName[eAT_Launcher] == element.application);
		result |= (ApplicationName[eAT_DedicatedServer] == element.application);
		result |= (ApplicationName[eAT_LicenseServer] == element.application);
		result |= (ApplicationName[eAT_SettingsManager] == element.application);
		result |= (ApplicationName[eAT_Unknown] == element.application);
	}
	else if (eLT_Launcher == element.licenseType)
	{
		result = (ApplicationName[eAT_Launcher] == element.application);
	}
	else if (eLT_LicenseServer == element.licenseType)
	{
		result = (ApplicationName[eAT_LicenseServer] == element.application);
	}

	if (false == result)
	{
		DREI_LOG( ErrorLogger,
			"LicensekeyPool IsValidConnectType ErrorCode(Invalid ConnectType) type(%d), appname(%s)", 
			element.licenseType, 
			element.application.c_str());
	}
	return result;
}