#ifndef __licensekeypool_h__
#define __licensekeypool_h__

#pragma once

#include <map>
#include <vector>
#include "LicenseServerCommon.h"

enum ELicenseType
{
	eLT_Full,
	eLT_LicenseServer,
	eLT_Launcher,
};

struct SLicenseKeyPoolElement
{
	std::string licenseKey;
	std::string version;
	std::string macAddress;
	std::string application;
	std::string sessionId;
	uint32 loginTick;
	ELicenseType licenseType;
};

class CLicensekeyPool
{
public:
	CLicensekeyPool(void);
	~CLicensekeyPool(void);

public:
	bool Login(uint32 userId, const SLicenseKeyPoolElement& newElement);
	void Logoff(uint32 userId);
	bool Authenticate(uint32 userId, const SAuthenticateParam& authParam) const;
	size_t GetElementCount() const;
	bool GetElement(uint32 userId, SLicenseKeyPoolElement& outputElement) const;
	bool IsValidConnectType(const SLicenseKeyPoolElement& element) const;

protected:
	void RemoveIndexElement(uint32 userId, std::string& licenseKey, std::string& sessionKey);
	void RemoveLicenseKeyElement(const std::string& key, uint32 userId);
	void RemoveSessionIdElement(const std::string& sessionKey);

	bool AddElement(uint32 userId, const SLicenseKeyPoolElement& element);
	bool ChangeElementIf(uint32 oldUserId, uint32 newUserId, const SLicenseKeyPoolElement& element);
	bool GetElement(const std::string& licenseKey, uint32 userId, SLicenseKeyPoolElement& outputElement) const;
	bool IsValidWithExistingElemnt(const SLicenseKeyPoolElement& newElement) const;

	typedef std::vector<uint32> IndexVector;
	typedef IndexVector::iterator				IndexIterator;
	typedef IndexVector::const_iterator	IndexConstIterator;

	typedef std::vector<SLicenseKeyPoolElement> ElementVector;
	typedef ElementVector::iterator				ElementIterator;
	typedef ElementVector::const_iterator	ElementConstIterator;

	bool GetIndexListWith(const std::string& key, IndexVector& list) const;
	bool GetElementListWith(const std::string& key, ElementVector& list) const;

private:
	// (UserId, SLicenseKeyPoolElement)
	typedef std::map<uint32, SLicenseKeyPoolElement> IndexPoolType;
	typedef IndexPoolType::iterator				IndexPoolIterator;
	typedef IndexPoolType::const_iterator	IndexPoolConstIterator;

	// (LicenseKey, UserId)
	typedef std::multimap<std::string, uint32> KeyPoolType;
	typedef KeyPoolType::iterator				KeyPoolIterator;
	typedef KeyPoolType::const_iterator	KeyPoolConstIterator;

	// (SessionKey, UserId)
	typedef std::multimap<std::string, uint32> SessionIdPoolType;
	typedef SessionIdPoolType::iterator				SessionIdPoolIterator;
	typedef SessionIdPoolType::const_iterator	SessionIdPoolConstIterator;

	IndexPoolType	m_indexContainer;
	KeyPoolType m_keyContainer;
	SessionIdPoolType m_sessionIdContainer;
};

#endif // __licensekeypool_h__