#include "stdafx.h"
#include "NetworkClient.h"
#include "ProtocolBuilder.h"
#include "ProtocolAssembler.h"
#include "LicenseServerCommon.h"
#include "JobQueue.h"
#include "ServerLog.h"

CNetworkClient::CNetworkClient(uint32 streamId, CJobQueue* que) : 
			m_streamId(streamId), m_jobQueue(que), m_assembler(new CProtocolAssembler())
{
}

CNetworkClient::~CNetworkClient(void)
{
	if (NULL != m_assembler)
	{
		delete m_assembler;
		m_assembler = NULL;
	}
}

void CNetworkClient::OnAccept(const char* buffer, uint32 len)
{
#ifdef DEBUG_MESSAGE
	printf("[%02d]Accept\n", m_streamId);
#endif
	m_remoteIp = buffer;
	DREI_LOG(PacketLogger, ACE_TEXT("Accept Stream(%d): %s"), m_streamId, m_remoteIp.c_str());
}

void CNetworkClient::OnReceive(const char* buffer, uint32 len )
{
#ifdef DEBUG_MESSAGE
	printf("[%02d]Receive\n", m_streamId);
#endif
	uint32 bufferOffset = 0;
	m_assembler->ApplyBuffer(buffer, bufferOffset, len);

	CProtocolBuilder builder;
	builder.EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
	while(m_assembler->AssembleBuilder(builder))
	{
		std::string packetType = builder.GetValue<std::string>("type");
		if ("auth" == packetType)
			PacketHandle_Auth(builder);
		else if ("login" == packetType)
			PacketHandle_Login(builder);
		else if ("createaccount" == packetType)
			PacketHandle_CreateAccount(builder);
		else if ("loginaccount" == packetType)
			PacketHandle_LoginAccount(builder);
		else if ("passwordremind" == packetType)
			PacketHandle_PasswordRemind(builder);
		else if ("requestcreateaccount" == packetType)
			PacketHandle_RequestCreaterAccount(builder);
		else if ("requestloginaccount" == packetType)
			PacketHandle_RequestLoginAccount(builder);
		else if ("licenseagree" == packetType)
			PacketHandle_LicenseAgree(builder);
		else if ("requestlicenseagree" == packetType)
			PacketHandle_RequestLicenseAgree(builder);
		else if ("requestpasswordremind" == packetType)
			PacketHandle_RequestPasswordRemind(builder);

		std::string receivedString = builder.MakeString();
		DREI_LOG(PacketLogger, "Recv Stream(%d) Len(%d) : %s", m_streamId, len, receivedString.c_str());
		builder.Clear();

#ifdef DEBUG_MESSAGE
		printf("[%02d]Receive(%s)\n", m_streamId, receivedString.c_str());
#endif
	}
}

void CNetworkClient::OnClose()
{
#ifdef DEBUG_MESSAGE
	printf("[%02d]Close\n", m_streamId);
#endif
	DREI_LOG(PacketLogger, ACE_TEXT("Close Stream(%d): %s"), m_streamId, m_remoteIp.c_str());

	std::string key;
	std::string mac;
	uint32 tick = 0;
	CLogoffJob* job = new CLogoffJob(key, mac, m_streamId, tick);
	m_jobQueue->PushWork(job);
}

void CNetworkClient::PacketHandle_Login( CProtocolBuilder& packet )
{
	uint32 tick = packet.GetValue<uint32>("tick");

	CLoginJob* job = new CLoginJob(packet.MakeString(), m_streamId, tick);
	m_jobQueue->PushWork(job);
}

void CNetworkClient::PacketHandle_Auth( CProtocolBuilder& packet )
{
	uint32 tick = packet.GetValue<uint32>("tick");
	CAuthJob* job = new CAuthJob(packet.MakeString(), m_streamId, tick);
	m_jobQueue->PushWork(job);
}

void CNetworkClient::PacketHandle_CreateAccount( CProtocolBuilder& packet )
{
	CCreateAccountJob* job = new CCreateAccountJob(packet.MakeString(), m_streamId);
	m_jobQueue->PushWork(job);
}

void CNetworkClient::PacketHandle_LoginAccount( CProtocolBuilder& packet )
{
	CLoginAccountJob* job = new CLoginAccountJob(packet.MakeString(), m_streamId);
	m_jobQueue->PushWork(job);
}

void CNetworkClient::PacketHandle_PasswordRemind( CProtocolBuilder& packet )
{
	CPasswordRemindJob* job = new CPasswordRemindJob(packet.MakeString(), m_streamId);
	m_jobQueue->PushWork(job);
}

void CNetworkClient::PacketHandle_RequestCreaterAccount( CProtocolBuilder& packet )
{
	CCreateAccountRequest* job = new CCreateAccountRequest(packet.MakeString(), m_streamId);
	m_jobQueue->PushWork(job);
}

void CNetworkClient::PacketHandle_RequestLoginAccount( CProtocolBuilder& packet )
{
	CLoginAccountRequest* job = new CLoginAccountRequest(packet.MakeString(), m_streamId);
	m_jobQueue->PushWork(job);
}

void CNetworkClient::PacketHandle_LicenseAgree( CProtocolBuilder& packet )
{
	CLicenseAgreeJob* job = new CLicenseAgreeJob(packet.MakeString(), m_streamId);
	m_jobQueue->PushWork(job);
}

void CNetworkClient::PacketHandle_RequestLicenseAgree( CProtocolBuilder& packet )
{
	CLicenseAgreeRequestJob* job = new CLicenseAgreeRequestJob(packet.MakeString(), m_streamId);
	m_jobQueue->PushWork(job);
}

void CNetworkClient::PacketHandle_RequestPasswordRemind( CProtocolBuilder& packet )
{
	CPasswordRemindRequestJob* job = new CPasswordRemindRequestJob(packet.MakeString(), m_streamId);
	m_jobQueue->PushWork(job);
}