#include "stdafx.h"
#include "NetworkServer.h"
#include <Drei/NetworkFacade.h>
#ifdef WIN32
#include <Drei/ProactorFactory.h>
#else
#include <Drei/ReactorFactory.h>
#endif

#include <Drei/MessageHeader.h>
#include "ProtocolBuilder.h"
#include "ProtocolAssembler.h"
#include "JobQueue.h"
#include "NetworkClient.h"
#include "ServerConfig.h"
#include "ServerLog.h"


CNetworkServer::CNetworkServer() : 
		m_init(false), m_jobQueue(NULL), m_clientMap(new ClientMapType())
{
	
}

CNetworkServer::~CNetworkServer(void)
{
	if (NULL != m_clientMap)
	{
		delete m_clientMap;
		m_clientMap = NULL;
	}
}

bool CNetworkServer::Init(CJobQueue* que)
{
	m_jobQueue = que;
	ACE::init();

	DreiNetwork::SystemConfig config;
	config.Worker_Thread_Count = CServerConfig::GetValue<uint32>("WorkerThreadCount");
	config.Receive_Buffer_Len = 1024*4;
	config.Interval_Send_Term = 30;
	config.Send_Mode = DreiNetwork::eSM_Direct;
	config.ZombieConnectionTerm = 1000*60*2 + 1000*20;	// 2 min 20 sec
#ifdef WIN32
	if (false == NetworkInstance->Create<DreiNetwork::ProactorFactory>())
#else
	if (false == NetworkInstance->Create<DreiNetwork::ReactorFactory>())
#endif
		return false;

	int servicePort = CServerConfig::GetValue<uint32>("ServicePort");
	if (false == NetworkInstance->AddAcceptor(NULL, servicePort, ReceiveQueueId))
		return false;
	if (false == NetworkInstance->Open(&config))
		return false;

	// Create network thread
	if (false == this->Open(1))
		return false;

	printf("Start accept service port is %d.\n", servicePort);

	m_init = true;
	return true;
}

void CNetworkServer::Fini()
{
	if (false == m_init)
		return;

	NetworkInstance->Close();
	this->Close();
	NetworkInstance->Destroy();
	ACE::fini();
}

void CNetworkServer::Update()
{
	ACE_Message_Block* headBlock = NULL;
	ACE_Message_Block* commandBlock = NULL;
	using namespace DreiNetwork;
	bool receiveData = NetworkInstance->PopAllMessage(ReceiveQueueId, &headBlock, -1);
	if (false == receiveData)
		return;
	commandBlock = headBlock;

	while(NULL != commandBlock)
	{
		MessagePostee postee;
		memcpy(&postee, commandBlock->rd_ptr(), sizeof(MessagePostee));
		commandBlock->rd_ptr(sizeof(MessagePostee));
		ACE_Message_Block* payloadBlock = commandBlock->cont();

		switch(postee.command)
		{
		case eMH_Establish:
			OnAccept(postee, payloadBlock->rd_ptr(), (::uint32)payloadBlock->length());
			break;
		case eMH_Read:
			OnReceive(postee, payloadBlock->rd_ptr(), (::uint32)payloadBlock->length());
			break;
		case eMH_Close:
			OnClose(postee);
			break;
		}
		commandBlock = payloadBlock->cont();
	}
	headBlock->release();
}

void CNetworkServer::OnAccept( DreiNetwork::MessagePostee& postee, const char* buffer, uint32 len )
{
	CNetworkClient* client = new CNetworkClient(postee.stream_id, m_jobQueue);
	m_clientMap->Register(client, postee.stream_id);
	client->OnAccept(buffer, len);
}

void CNetworkServer::OnReceive( DreiNetwork::MessagePostee& postee, const char* buffer, uint32 len )
{
	CNetworkClient* client = m_clientMap->Get(postee.stream_id);
	if (NULL == client)
		return;
	client->OnReceive(buffer, len);
}

void CNetworkServer::OnClose( DreiNetwork::MessagePostee& postee )
{
	CNetworkClient* client = m_clientMap->Get(postee.stream_id);
	if (NULL == client)
		return;
	client->OnClose();
	m_clientMap->UnRegister(postee.stream_id);
	delete client;
	client = NULL;
}