#include "stdafx.h"
#include "ProcessJobs.h"

#ifdef USING_DATABASE_MODE
#include "DatabaseModeManager.h"
#endif
#include <Drei/NetworkFacade.h>
#include "ProtocolBuilder.h"
#include "UniKeyManager.h"
#include "ServerConfig.h"
#include "ServerLog.h"
#include "AesCryptography.h"
#include "ClientModeManager.h"
#include "TinyClientSocket.h"
#include "StringFilter.h"

#ifdef SEND_AUTH_EMAIL
const char* WebServerHost = "mycryengine.com";
#else
//const char* WebServerHost = "test.mycryengine.com";
const char* WebServerHost = "mycryengine-test.com";
#endif

//////////////////////////////////////////////////////////////////////////
CLoginJob::CLoginJob( std::string received, uint32 streamId, uint32 tick ) :
		m_receivedString(received),
		m_streamId(streamId), 
		m_tick(tick),
		m_remainSecond(0),
		m_allowFloatConnection(false),
		m_floatLicenses(0)
{
}

void CLoginJob::DBExecute( CDatabaseModeManager& mgr )
{
#ifdef USING_DATABASE_MODE	
	CProtocolBuilder builder;
	builder.Parse(m_receivedString);
	std::string ip = builder.GetValue<std::string>("ip");
	std::string version = builder.GetValue<std::string>("version");
	SAuthenticateParam authParam;
	authParam.licenseKey = builder.GetValue<std::string>("key");
	authParam.macAddress = builder.GetValue<std::string>("mac");
	authParam.application = builder.GetValue<std::string>("appname");
	// For the old version user
	if (authParam.application.empty())
		authParam.application = "Unknown";
		//authParam.application = "Editor";
	SLoginUserResult resultParam = mgr.LoginUser(m_streamId, authParam, ip, version);
	m_result = resultParam.errorCode;
	if (PMEC_NO_ERROR != m_result)
		return;

	MakeEncryptionKey(resultParam.companyID);
	m_remainSecond = resultParam.remainSecond;
	m_allowFloatConnection = resultParam.allowFloatConnection;
	m_floatLicenses = resultParam.floatLicenses;
	m_sessionId = resultParam.sessionId;
#endif
}

void CLoginJob::UniKeyExecute( CUniKeyManager& unikey )
{
	CProtocolBuilder builder;
	builder.Parse(m_receivedString);
	std::string licenseKey = builder.GetValue<std::string>("key");
	std::string ip = builder.GetValue<std::string>("ip");
	std::string mac = builder.GetValue<std::string>("mac");
	m_result = unikey.LoginUser(m_streamId, licenseKey, mac);
	TimeType unikeyRemainTerm = unikey.GetRemainSecond();
	int32 cachedLicenseLifeSec = CServerConfig::GetValue<uint32>("CachedLicenseHour")*60*60;
	TimeType remainSecond = (cachedLicenseLifeSec < unikeyRemainTerm)?cachedLicenseLifeSec:unikeyRemainTerm;
	m_remainSecond = (uint32)remainSecond;
	if (PMEC_NO_ERROR == m_result)
	{
		unikey.GetLicenseKey(licenseKey);
		m_encryptionKey = licenseKey;
	}
}

void CLoginJob::ClientModeExecute(CClientModeManager& mgr)
{
	CProtocolBuilder builder;
	builder.Parse(m_receivedString);
	std::string licenseKey = builder.GetValue<std::string>("key");
	std::string ip = builder.GetValue<std::string>("ip");
	std::string mac = builder.GetValue<std::string>("mac");
	std::string sessionKey = builder.GetValue<std::string>("sessionkey");
	if (sessionKey.empty())
	{
		m_result = PMEC_REQUEST_ACCOUNTLOGIN;
		return;
	}

	m_result = mgr.LoginUser(m_streamId, licenseKey, mac, sessionKey);
	mgr.GetLicenseKey(licenseKey);
	mgr.GetEncryptKey(m_encryptionKey);
}

void CLoginJob::Complete()
{
	CProtocolBuilder receivedBuilder;
	receivedBuilder.Parse(m_receivedString);
	std::string licenseKey = receivedBuilder.GetValue<std::string>("key");
	CProtocolBuilder builder;
	builder.EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
	builder.Add("type", "login");
	if (PMEC_NO_ERROR== m_result)
		builder.Add("result", ResultTrue);
	else
		builder.Add("result", ResultFalse);
	builder.Add("tick", m_tick);
	builder.Add("key", licenseKey);
	builder.Add("errorcode", (int32)m_result);
	builder.Add("encryptkey", m_encryptionKey);
	builder.Add("sessionid", m_sessionId);
	if (m_allowFloatConnection)
		builder.Add("floatlicenses", m_floatLicenses);
	// Get an exact time from UniKeyManager 
	builder.Add("remain", m_remainSecond);
	std::string sendString = builder.MakeString();
#ifdef DEBUG_MESSAGE
	printf("Send(%s)\n", sendString.c_str());
#endif
	DREI_LOG(PacketLogger, "Send Stream(%d): %s", m_streamId, sendString.c_str());
		
	const uint32 MaxBuffer = 1024;
	char sendBuffer[MaxBuffer] = {0,};
	uint32 sendSize = builder.MakeBuffer(sendBuffer, MaxBuffer);
	NetworkInstance->SendRequest(m_streamId, sendBuffer, sendSize);
}

void CLoginJob::MakeEncryptionKey(uint32 seed)
{
	const int KeySize = 20;
	char buffer[KeySize] = {0,};
	GenerateEncryptKey generator;
	generator.Generate(seed, buffer, KeySize);
	m_encryptionKey.assign((char*)buffer, KeySize-1);
}

//////////////////////////////////////////////////////////////////////////
CAuthJob::CAuthJob(std::string received, uint32 streamId, uint32 tick) :
	m_receivedString(received),
	m_streamId(streamId), 
	m_tick(tick)
{
}

void CAuthJob::DBExecute(CDatabaseModeManager& mgr)
{
#ifdef USING_DATABASE_MODE
	CProtocolBuilder builder;
	builder.Parse(m_receivedString);
	std::string ip = builder.GetValue<std::string>("ip");
	SAuthenticateParam authParam;
	authParam.licenseKey = builder.GetValue<std::string>("key");
	authParam.macAddress = builder.GetValue<std::string>("mac");
	authParam.application = builder.GetValue<std::string>("appname");
	authParam.sessionId = builder.GetValue<std::string>("sessionid");
	// For the old version user
	if (authParam.application.empty())
		authParam.application = "Unknown";
		//authParam.application = "Editor";
	m_result = mgr.AuthenticateUser(m_streamId, authParam, ip);
#else
	m_result = PMEC_NO_ERROR;
#endif
}

void CAuthJob::UniKeyExecute(CUniKeyManager& unikey)
{
	CProtocolBuilder builder;
	builder.Parse(m_receivedString);
	std::string licenseKey = builder.GetValue<std::string>("key");
	std::string ip = builder.GetValue<std::string>("ip");
	std::string mac = builder.GetValue<std::string>("mac");

	m_result = unikey.AuthenticateUser(m_streamId, licenseKey, mac);
}

void CAuthJob::ClientModeExecute(CClientModeManager& mgr)
{
	CProtocolBuilder builder;
	builder.Parse(m_receivedString);
	std::string sessionKey = builder.GetValue<std::string>("sessionkey");
	if (sessionKey.empty())
	{
		m_result = PMEC_REQUEST_ACCOUNTLOGIN;
		return;
	}
	std::string licenseKey = builder.GetValue<std::string>("key");
	std::string ip = builder.GetValue<std::string>("ip");
	std::string mac = builder.GetValue<std::string>("mac");
	m_result = mgr.AuthenticateUser(m_streamId, licenseKey, mac, sessionKey);
}

void CAuthJob::Complete()
{
	CProtocolBuilder builder;
	builder.EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
	builder.Add("type", "auth");
	if (PMEC_NO_ERROR== m_result)
		builder.Add("result", ResultTrue);
	else
		builder.Add("result", ResultFalse);
	builder.Add("tick", m_tick);
	builder.Add("errorcode", (int32)m_result);
	std::string sendString = builder.MakeString();
#ifdef DEBUG_MESSAGE
	printf("Send(%s)\n", sendString.c_str());
#endif
	DREI_LOG(PacketLogger, "Send Stream(%d): %s", m_streamId, sendString.c_str());
	
	const uint32 MaxBuffer = 1024;
	char sendBuffer[MaxBuffer] = {0,};
	uint32 sendSize = builder.MakeBuffer(sendBuffer, MaxBuffer);
	NetworkInstance->SendRequest(m_streamId, sendBuffer, sendSize);
}

//////////////////////////////////////////////////////////////////////////
CLogoffJob::CLogoffJob( std::string& key, std::string& mac, uint32 streamId, uint32 tick ) :
	m_key(key), m_mac(mac), m_result(false), m_streamId(streamId), m_tick(tick)
{

}

void CLogoffJob::DBExecute(CDatabaseModeManager& mgr)
{
#ifdef USING_DATABASE_MODE
	mgr.LogoffUser(m_streamId);
#endif 
}

void CLogoffJob::UniKeyExecute( CUniKeyManager& unikey )
{
	unikey.LogoffUser(m_streamId);
}

void CLogoffJob::ClientModeExecute(CClientModeManager& mgr)
{
	mgr.LogoffUser(m_streamId);
}

void CLogoffJob::Complete()
{
}

//////////////////////////////////////////////////////////////////////////

CCreateAccountJob::CCreateAccountJob(std::string received, uint32 streamId) : 
	m_receivedString(received), m_streamId(streamId)
{
}

void CCreateAccountJob::ClientModeExecute( CClientModeManager& mgr)
{
	CProtocolBuilder receivedBuilder;
	receivedBuilder.Parse(m_receivedString);
	std::string username = receivedBuilder.GetValue<std::string>("username");
	std::string password = receivedBuilder.GetValue<std::string>("password");
	std::string email = receivedBuilder.GetValue<std::string>("email");
	std::string firstname = receivedBuilder.GetValue<std::string>("firstname");
	std::string lastname = receivedBuilder.GetValue<std::string>("lastname");
	std::string address = receivedBuilder.GetValue<std::string>("address");
	std::string city = receivedBuilder.GetValue<std::string>("city");
	std::string country = receivedBuilder.GetValue<std::string>("country");
	std::string state = receivedBuilder.GetValue<std::string>("state");
	std::string zipcode = receivedBuilder.GetValue<std::string>("zipcode");
	std::string licenseKey;
	mgr.GetLicenseKey(licenseKey);

	CProtocolBuilder builder;
	builder.EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
	builder.Add("type", "requestcreateaccount");
	builder.Add("username", username);
	builder.Add("password", password);
	builder.Add("email", email);
	builder.Add("firstname", firstname);
	builder.Add("lastname", lastname);
	builder.Add("address", address);
	builder.Add("city", city);
	builder.Add("country", country);
	builder.Add("state", state);
	builder.Add("zipcode", zipcode);
	builder.Add("clientstreamid", m_streamId);
	builder.Add("licensekey", licenseKey);

	std::string sendString = builder.MakeString();
	DREI_LOG(PacketLogger, "Send LicenseConn : %s", sendString.c_str());

	const uint32 MaxBuffer = 1024;
	char sendBuffer[MaxBuffer] = {0,};
	uint32 sendSize = builder.MakeBuffer(sendBuffer, MaxBuffer);
	NetworkInstance->SendRequest(mgr.GetGlobalLicenseServerStreamId(), sendBuffer, sendSize);
}

//////////////////////////////////////////////////////////////////////////

CLoginAccountJob::CLoginAccountJob(std::string received, uint32 streamId) : 
	m_receivedString(received), m_streamId(streamId)
{
}

void CLoginAccountJob::ClientModeExecute( CClientModeManager& mgr)
{
	CProtocolBuilder receivedBuilder;
	receivedBuilder.Parse(m_receivedString);
	std::string username = receivedBuilder.GetValue<std::string>("username");
	std::string password = receivedBuilder.GetValue<std::string>("password");

	if (username.empty() || password.empty())
	{
		CProtocolBuilder builder;
		builder.EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
		builder.Add("type", "loginaccount");
		builder.Add("result", ResultFalse);
		const uint32 MaxBuffer = 1024;
		char sendBuffer[MaxBuffer] = {0,};
		uint32 sendSize = builder.MakeBuffer(sendBuffer, MaxBuffer);
		NetworkInstance->SendRequest(m_streamId, sendBuffer, sendSize);
		return;
	}

	CProtocolBuilder builder;
	builder.EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
	builder.Add("type", "requestloginaccount");
	builder.Add("username", username);
	builder.Add("password", password);
	builder.Add("clientstreamid", m_streamId);

	std::string sendString = builder.MakeString();
	DREI_LOG(PacketLogger, "Send LicenseConn : %s", sendString.c_str());

	const uint32 MaxBuffer = 1024;
	char sendBuffer[MaxBuffer] = {0,};
	uint32 sendSize = builder.MakeBuffer(sendBuffer, MaxBuffer);
	NetworkInstance->SendRequest(mgr.GetGlobalLicenseServerStreamId(), sendBuffer, sendSize);
}

//////////////////////////////////////////////////////////////////////////


CPasswordRemindJob::CPasswordRemindJob(std::string received, uint32 streamId) : 
m_receivedString(received), m_streamId(streamId)
{
}

void CPasswordRemindJob::ClientModeExecute( CClientModeManager& mgr)
{
	CProtocolBuilder receivedBuilder;
	receivedBuilder.Parse(m_receivedString);
	std::string username = receivedBuilder.GetValue<std::string>("username");
	std::string email = receivedBuilder.GetValue<std::string>("email");

	if (username.empty() && email.empty())
	{
		CProtocolBuilder builder;
		builder.EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
		builder.Add("type", "passwordremind");
		builder.Add("result", ResultFalse);
		const uint32 MaxBuffer = 1024;
		char sendBuffer[MaxBuffer] = {0,};
		uint32 sendSize = builder.MakeBuffer(sendBuffer, MaxBuffer);
		NetworkInstance->SendRequest(m_streamId, sendBuffer, sendSize);
		return;
	}

	CProtocolBuilder builder;
	builder.EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
	builder.Add("type", "requestpasswordremind");
	builder.Add("username", username);
	builder.Add("email", email);
	builder.Add("clientstreamid", m_streamId);

	std::string sendString = builder.MakeString();
	DREI_LOG(PacketLogger, "Send LicenseConn : %s", sendString.c_str());

	const uint32 MaxBuffer = 1024;
	char sendBuffer[MaxBuffer] = {0,};
	uint32 sendSize = builder.MakeBuffer(sendBuffer, MaxBuffer);
	NetworkInstance->SendRequest(mgr.GetGlobalLicenseServerStreamId(), sendBuffer, sendSize);
}

//////////////////////////////////////////////////////////////////////////

CCreateAccount4LicenseConnection::CCreateAccount4LicenseConnection(std::string received, uint32 streamId) : 
	m_receivedString(received), m_streamId(streamId)
{
}

void CCreateAccount4LicenseConnection::ClientModeExecute( CClientModeManager& mgr)
{
	CProtocolBuilder receivedBuilder;
	receivedBuilder.Parse(m_receivedString);
	uint32 clientStreamId = receivedBuilder.GetValue<uint32>("clientstreamid");
	uint8 resultCode = receivedBuilder.GetValue<uint8>("resultcode");
	
	CProtocolBuilder builder;
	builder.EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
	builder.Add("type", "createaccount");
	builder.Add("resultcode", resultCode);
	std::string sendString = builder.MakeString();
#ifdef DEBUG_MESSAGE
	printf("Send(%s)\n", sendString.c_str());
#endif
	DREI_LOG(PacketLogger, "Send Stream(%d) : %s", clientStreamId,  sendString.c_str());

	const uint32 MaxBuffer = 1024;
	char sendBuffer[MaxBuffer] = {0,};
	uint32 sendSize = builder.MakeBuffer(sendBuffer, MaxBuffer);
	NetworkInstance->SendRequest(clientStreamId, sendBuffer, sendSize);
}

void CCreateAccount4LicenseConnection::Complete()
{
}

//////////////////////////////////////////////////////////////////////////

CLoginAccount4LicenseConnection::CLoginAccount4LicenseConnection(std::string received, uint32 streamId) : 
	m_receivedString(received), m_streamId(streamId)
{
}

void CLoginAccount4LicenseConnection::ClientModeExecute( CClientModeManager& mgr)
{
	CProtocolBuilder receivedBuilder;
	receivedBuilder.Parse(m_receivedString);
	uint32 clientStreamId = receivedBuilder.GetValue<uint32>("clientstreamid");
	std::string username = receivedBuilder.GetValue<std::string>("username");
	uint8 resultCode = receivedBuilder.GetValue<uint8>("resultcode");
	bool agreelicense = receivedBuilder.GetValue<bool>("agreelicense");

	CProtocolBuilder builder;
	builder.EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
	builder.Add("type", "loginaccount");
	builder.Add("resultcode", resultCode);
	builder.Add("agreelicense", agreelicense);

	// Create sessionkey
	if (eLA_Success == resultCode)
	{
		//std::string sessionKey = GenerateSessionKey(username);
		GenerateEncryptKey generator;
		std::string sessionKey = generator.GenerateGuid();
		builder.Add("sessionkey", sessionKey);
	}

	std::string sendString = builder.MakeString();
#ifdef DEBUG_MESSAGE
	printf("Send(%s)\n", sendString.c_str());
#endif
	DREI_LOG(PacketLogger, "Send Stream(%d) : %s", clientStreamId, sendString.c_str());

	const uint32 MaxBuffer = 1024;
	char sendBuffer[MaxBuffer] = {0,};
	uint32 sendSize = builder.MakeBuffer(sendBuffer, MaxBuffer);
	NetworkInstance->SendRequest(clientStreamId, sendBuffer, sendSize);
}

void CLoginAccount4LicenseConnection::Complete()
{
}

//////////////////////////////////////////////////////////////////////////

CConnectionEstblished4LicenseConnection::CConnectionEstblished4LicenseConnection(uint32 streamId) : m_streamId(streamId)
{
}

void CConnectionEstblished4LicenseConnection::ClientModeExecute( CClientModeManager& mgr)
{
	mgr.SetGlobalLicenseServerStreamId(m_streamId);

	std::string licenseKey, localIp, localMac;
	mgr.GetLicenseKey(licenseKey);
	mgr.GetLocalIp(localIp);
	mgr.GetMacAddress(localMac);
	CProtocolBuilder builder;
	builder.EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
	builder.Add("type", "login");
	builder.Add("key", licenseKey);
	builder.Add("ip", localIp);
	builder.Add("mac", localMac);
	builder.Add("appname", LicenseServerAppname);
	builder.Add("version", LicenseServerVersion);
	const uint32 MaxBuffer = 1024;
	char sendBuffer[MaxBuffer] = {0,};
	uint32 sendSize = builder.MakeBuffer(sendBuffer, MaxBuffer);
	NetworkInstance->SendRequest(m_streamId, sendBuffer, sendSize);
}

void CConnectionEstblished4LicenseConnection::Complete()
{

}

//////////////////////////////////////////////////////////////////////////

CConnectionClose4LicenseConnection::CConnectionClose4LicenseConnection(uint32 streamId) : m_streamId(streamId)
{
}

void CConnectionClose4LicenseConnection::ClientModeExecute( CClientModeManager& mgr)
{
	mgr.SetErrorCode(PMEC_CONNECT_FAILED);
}

void CConnectionClose4LicenseConnection::Complete()
{
}

//////////////////////////////////////////////////////////////////////////

CLoginResult4LicenseConnection::CLoginResult4LicenseConnection( std::string received, uint32 streamId, uint32 tick ) :
	m_receivedString(received), m_streamId(streamId)
{

}

void CLoginResult4LicenseConnection::ClientModeExecute( CClientModeManager& mgr )
{
	CProtocolBuilder builder;
	builder.Parse(m_receivedString);
	EPMErrorCode resultCode = (EPMErrorCode)builder.GetValue<int32>("errorcode");
	std::string licenseKey = builder.GetValue<std::string>("key");
	uint32 remainSecond = builder.GetValue<uint32>("remain");
	std::string encryptKey = builder.GetValue<std::string>("encryptkey");
	uint32 floatLicenses = builder.GetValue<uint32>("floatlicenses");

	if (mgr.GetErrorCode() != resultCode)
	{
		printf("Authenticate result is %s.\n", (PMEC_NO_ERROR==resultCode)?"true":"false");
		mgr.SetErrorCode(resultCode);
	}

	if (mgr.GetMaxUserCount() != floatLicenses)
	{
		if (0 < floatLicenses)
			printf("Concurrent Floating License is %d.\n", floatLicenses);
		mgr.SetMaxUserCount(floatLicenses);
	}
	mgr.SetEncryptKey(encryptKey);
}

//////////////////////////////////////////////////////////////////////////

CPasswordRemind4LicenseConnection::CPasswordRemind4LicenseConnection( std::string received, uint32 streamId ) :
m_receivedString(received), m_streamId(streamId)
{

}

void CPasswordRemind4LicenseConnection::ClientModeExecute( CClientModeManager& mgr )
{
	CProtocolBuilder receivedBuilder;
	receivedBuilder.Parse(m_receivedString);
	std::string result = receivedBuilder.GetValue<std::string>("result");
	uint32 clientStreamId = receivedBuilder.GetValue<uint32>("clientstreamid");

	CProtocolBuilder builder;
	builder.EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
	builder.Add("type", "passwordremind");
	builder.Add("result", result);
	const uint32 MaxBuffer = 1024;
	char sendBuffer[MaxBuffer] = {0,};
	uint32 sendSize = builder.MakeBuffer(sendBuffer, MaxBuffer);
	NetworkInstance->SendRequest(clientStreamId, sendBuffer, sendSize);
}

//////////////////////////////////////////////////////////////////////////

CCreateAccountRequest::CCreateAccountRequest(std::string received, uint32 streamId) : 
	m_receivedString(received), m_streamId(streamId)
{
}

void CCreateAccountRequest::DBExecute( CDatabaseModeManager& mgr )
{
#ifdef USING_DATABASE_MODE
	CProtocolBuilder receivedBuilder;
	receivedBuilder.Parse(m_receivedString);
	std::string username = receivedBuilder.GetValue<std::string>("username");
	std::string password = receivedBuilder.GetValue<std::string>("password");
	std::string email = receivedBuilder.GetValue<std::string>("email");
	std::string firstname = receivedBuilder.GetValue<std::string>("firstname");
	std::string lastname = receivedBuilder.GetValue<std::string>("lastname");
	std::string address = receivedBuilder.GetValue<std::string>("address");
	std::string city = receivedBuilder.GetValue<std::string>("city");
	std::string country = receivedBuilder.GetValue<std::string>("country");
	std::string state = receivedBuilder.GetValue<std::string>("state");
	std::string zipcode = receivedBuilder.GetValue<std::string>("zipcode");
	std::string licenseKey = receivedBuilder.GetValue<std::string>("licensekey");

	CStringFilter filter;
	filter.Convert2SmallLetter(username);
	filter.Convert2SmallLetter(email);

	SCreateAccountParam param;
	param.username = username;
	param.password = password;
	param.email = email;
	param.firstname = firstname;
	param.lastname = lastname;
	param.address = address;
	param.city = city;
	param.country = country;
	param.state = state;
	param.zipcode = zipcode;

	ECreateAccountResult resultCode = eCA_Success;
	uint32 userId = 0;
	resultCode = mgr.CreateAccount(param, licenseKey, userId);

	CProtocolBuilder builder;
	builder.EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
	builder.Add("type", "requestcreateaccount");
	builder.Add("resultcode", (uint8)resultCode);
	uint32 clientStreamId = receivedBuilder.GetValue<uint32>("clientstreamid");
	builder.Add("clientstreamid", clientStreamId);
	std::string sendString = builder.MakeString();
#ifdef DEBUG_MESSAGE
	printf("Send(%s)\n", sendString.c_str());
#endif
	DREI_LOG(PacketLogger, "Send Stream(%d) : %s", m_streamId, sendString.c_str());

	const uint32 MaxBuffer = 1024;
	char sendBuffer[MaxBuffer] = {0,};
	uint32 sendSize = builder.MakeBuffer(sendBuffer, MaxBuffer);
	NetworkInstance->SendRequest(m_streamId, sendBuffer, sendSize);

	if (eCA_Success == resultCode)
	{
		// Call send email url
		const int TryCount = 2;
		for(int i=0; i<TryCount; ++i)
			if (SendEmailToUser(userId))
				break;
	}

#endif
}

bool CallWebURL(const std::string& host, const std::string& url)
{
	CTinyClientSocket webSocket;
	webSocket.InitNetwork();
	webSocket.InitSocket();
	bool functionResult = false;
	do 
	{
		const uint32 MaxUrlLength = 1024;
		char requestURL[MaxUrlLength] = {0,};
		sprintf(requestURL, 
			"GET %s HTTP/1.1\nHost:%s\n\n", url.c_str(), host.c_str());

#ifndef SEND_AUTH_EMAIL
		// test //////////////////////////////////////////////////////////////////
		//printf("%s\n", requestURL);
		functionResult = true;
		break;
		//////////////////////////////////////////////////////////////////////////
#endif

		const int WebServicePort = 80;
		bool socketResult = webSocket.Connect(host.c_str(), WebServicePort);
		if (false == socketResult)
			break;

		socketResult = webSocket.Send(requestURL, (const uint32)strlen(requestURL));
		if (false == socketResult)
			break;

		const int MaxBuffer = 1024;
		char buffer[MaxBuffer] = {0,};
		const uint32 WaitTimeFromWebServer = 1000;
		uint32 recvSize = webSocket.Receive(buffer, MaxBuffer, WaitTimeFromWebServer);
		if (0 >= recvSize)
			break;
		std::string resultString = std::string(buffer, recvSize);

		if (std::string::npos != resultString.find("true"))
			functionResult = true;
	} while (false);

	webSocket.FiniSocket();
	webSocket.FiniNetwork();
	return functionResult;
}

bool CCreateAccountRequest::SendEmailToUser(uint32 userId)
{
	const uint32 MaxUrlLength = 1024;
	char requestURL[MaxUrlLength] = {0,};
	sprintf(requestURL, 
		"/core/UserAction.php?action=sendauthmailtostudents&id=%d", userId);

	return CallWebURL(WebServerHost, requestURL);
}

//////////////////////////////////////////////////////////////////////////

CLoginAccountRequest::CLoginAccountRequest(std::string received, uint32 streamId) : 
	m_receivedString(received), m_streamId(streamId)
{
}

void CLoginAccountRequest::DBExecute( CDatabaseModeManager& mgr )
{
#ifdef USING_DATABASE_MODE
	CProtocolBuilder receivedBuilder;
	receivedBuilder.Parse(m_receivedString);
	std::string username = receivedBuilder.GetValue<std::string>("username");
	std::string password = receivedBuilder.GetValue<std::string>("password");
	std::string licenseKey = receivedBuilder.GetValue<std::string>("licensekey");

	CStringFilter filter;
	filter.Convert2SmallLetter(username);

	bool agreeLicenseFlag = false;
	ELoginAccountResult resultCode = mgr.AuthenticateAccount(username, password, licenseKey, agreeLicenseFlag);

	CProtocolBuilder builder;
	builder.EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
	builder.Add("type", "requestloginaccount");
	builder.Add("resultcode", (uint8)resultCode);
	builder.Add("agreelicense", agreeLicenseFlag);

	uint32 clientStreamId = receivedBuilder.GetValue<uint32>("clientstreamid");
	builder.Add("clientstreamid", clientStreamId);
	std::string sendString = builder.MakeString();
#ifdef DEBUG_MESSAGE
	printf("Send(%s)\n", sendString.c_str());
#endif
	DREI_LOG(PacketLogger, "Send Stream(%d) : %s", m_streamId, sendString.c_str());

	const uint32 MaxBuffer = 1024;
	char sendBuffer[MaxBuffer] = {0,};
	uint32 sendSize = builder.MakeBuffer(sendBuffer, MaxBuffer);
	NetworkInstance->SendRequest(m_streamId, sendBuffer, sendSize);
#endif
}

//////////////////////////////////////////////////////////////////////////

CLicenseAgreeJob::CLicenseAgreeJob(std::string received, uint32 streamId) : 
	m_receivedString(received), m_streamId(streamId)
{
}

void CLicenseAgreeJob::ClientModeExecute( CClientModeManager& mgr)
{
	CProtocolBuilder receivedBuilder;
	receivedBuilder.Parse(m_receivedString);
	std::string username = receivedBuilder.GetValue<std::string>("username");

	if (username.empty())
		return;
	
	CProtocolBuilder builder;
	builder.EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
	builder.Add("type", "requestlicenseagree");
	builder.Add("username", username);
	builder.Add("clientstreamid", m_streamId);

	std::string sendString = builder.MakeString();
	DREI_LOG(PacketLogger, "Send LicenseConn : %s", sendString.c_str());

	const uint32 MaxBuffer = 1024;
	char sendBuffer[MaxBuffer] = {0,};
	uint32 sendSize = builder.MakeBuffer(sendBuffer, MaxBuffer);
	NetworkInstance->SendRequest(mgr.GetGlobalLicenseServerStreamId(), sendBuffer, sendSize);
}

//////////////////////////////////////////////////////////////////////////

CLicenseAgreeRequestJob::CLicenseAgreeRequestJob(std::string received, uint32 streamId) : 
	m_receivedString(received), m_streamId(streamId)
{
}

void CLicenseAgreeRequestJob::DBExecute( CDatabaseModeManager& mgr )
{
#ifdef USING_DATABASE_MODE
	CProtocolBuilder receivedBuilder;
	receivedBuilder.Parse(m_receivedString);
	std::string username = receivedBuilder.GetValue<std::string>("username");
	CStringFilter filter;
	filter.Convert2SmallLetter(username);
	mgr.UpdateAgreeLicense(username);
#endif
}


//////////////////////////////////////////////////////////////////////////

std::string Dec2Hex(int dec)
{
	const int bufferSize = 12;
	char temp[bufferSize] = {0,};
	sprintf(temp, "%x", dec);
	return std::string(temp);
}

void PhpEncryptString(std::string& dst, const std::string& src)
{
	srand((unsigned )time(NULL));
	for(size_t i=0; i<src.length();++i)
	{
		int seed = rand()%255;
		dst += ((seed<16)?"0":"") + Dec2Hex(seed);
		int dec = (src.at(i) + seed)%256;
		dst += ((dec<16)?"0":"") + Dec2Hex(dec);
	}
}

bool SendPasswordRemindEmail(uint32 userId, const std::string& password)
{
	CStringFilter filter;
	std::string base64DecodedPassword = filter.Base64Decode(password);
	std::string encryptedPassword;
	PhpEncryptString(encryptedPassword, base64DecodedPassword);

	const uint32 MaxUrlLength = 1024;
	char requestURL[MaxUrlLength] = {0,};
	sprintf(requestURL, 
		"/core/UserAction.php?action=sendpasswordmailtostudents&id=%d&password=%s", userId, encryptedPassword.c_str());

	return CallWebURL(WebServerHost, requestURL);
}

CPasswordRemindRequestJob::CPasswordRemindRequestJob(std::string received, uint32 streamId) : 
m_receivedString(received), m_streamId(streamId)
{
}

void CPasswordRemindRequestJob::DBExecute( CDatabaseModeManager& mgr )
{
#ifdef USING_DATABASE_MODE
	CProtocolBuilder receivedBuilder;
	receivedBuilder.Parse(m_receivedString);
	std::string username = receivedBuilder.GetValue<std::string>("username");
	std::string email = receivedBuilder.GetValue<std::string>("email");
	std::string password;
	uint32 userId = 0;

	bool result = mgr.GetStudentPassword(username, email, password, userId);
	if (result)
		result = SendPasswordRemindEmail(userId, password);

	CProtocolBuilder builder;
	builder.EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
	builder.Add("type", "requestpasswordremind");
	if (result)
		builder.Add("result", ResultTrue);
	else
		builder.Add("result", ResultFalse);
	uint32 clientStreamId = receivedBuilder.GetValue<uint32>("clientstreamid");
	builder.Add("clientstreamid", clientStreamId);
	std::string sendString = builder.MakeString();
#ifdef DEBUG_MESSAGE
	printf("Send(%s)\n", sendString.c_str());
#endif
	DREI_LOG(PacketLogger, "Send Stream(%d) : %s", m_streamId, sendString.c_str());

	const uint32 MaxBuffer = 1024;
	char sendBuffer[MaxBuffer] = {0,};
	uint32 sendSize = builder.MakeBuffer(sendBuffer, MaxBuffer);
	NetworkInstance->SendRequest(m_streamId, sendBuffer, sendSize);
#endif
}

//////////////////////////////////////////////////////////////////////////