#include "stdafx.h"
#include "ServerLog.h"
#include <ace/Time_Value.h>
#include <ace/OS_NS_time.h>
#include <ace/OS_NS_sys_time.h>
#include <ace/OS_NS_stdio.h>
#include <Drei/TimerUtil.h>

const char* LogDirectory = "Log";
const char* DBLogPrefix = "LicenseServerDatabaseLog";
const char* PacketLogPrefix = "LicenseServerPacketLog";
const char* ErrorLogPrefix = "LicenseServerErrorLog";
const uint32 LogFileSizeCheckTerm = 5*60*1000; // 5 min

size_t GetFileSizeFromHandle(FILE* handle)
{
	if (NULL == handle)
		return 0;
	ACE_OS::fseek(handle, 0, SEEK_END);
	size_t filesize = ACE_OS::ftell(handle);
	ACE_OS::fseek(handle, 0, SEEK_SET);
	return filesize;
}

size_t GetFileSizeFromPath(const char* filePath)
{
	FILE* fileHandle = ACE_OS::fopen(filePath, "rb");
	if (NULL == fileHandle)
		return 0;
	size_t filesize = GetFileSizeFromHandle(fileHandle);
	ACE_OS::fclose(fileHandle);
	return filesize;
}

void GenerateLogFilename(const char* prefix, char* filename)
{
	time_t currentTime = ACE_OS::gettimeofday().sec();
	struct tm* today = ACE_OS::localtime(&currentTime);
	char strDate[64] = {0,};
	ACE_OS::sprintf( strDate, "%4d%02d%02d", 
		today->tm_year+1900, today->tm_mon+1, today->tm_mday);
	//ACE_OS::sprintf( strDate, "%4d%02d%02d%02d%02d%02d", 
	//	today->tm_year+1900, today->tm_mon+1, today->tm_mday, today->tm_hour, today->tm_min, today->tm_sec);
	ACE_OS::sprintf(filename, "%s/%s%s.txt", 
		LogDirectory, prefix, strDate);
}

void SwitchLogfileIfEnoughBig(char* filename, const char* prefix, DreiNetwork::Logger* logger)
{
	const int LimitLogFileSize = 200*1024*1024;	// 200 MB
	size_t logFileSize = GetFileSizeFromPath(filename);
	if (logFileSize < LimitLogFileSize)
		return;

	GenerateLogFilename(prefix, filename);
	logger->SetFileLog(filename);
}

void CreateLogDirectory( const char* directory )
{
#ifdef WIN32
	CreateDirectory(directory, NULL);
#else
	mkdir(directory, 509);
#endif
}

//////////////////////////////////////////////////////////////////////////

CServerLog::CServerLog(void) : m_dbLogger(NULL), m_packetLogger(NULL), m_errorLogger(NULL)
{
}

CServerLog::~CServerLog(void)
{
}

bool CServerLog::Init()
{
#ifndef NO_DREI_LOG
	CreateLogDirectory(LogDirectory);
#endif

	m_dbLogger = new DreiNetwork::Logger();
	m_packetLogger = new DreiNetwork::Logger();
	m_errorLogger =  new DreiNetwork::Logger();

	GenerateLogFilename(DBLogPrefix, m_dbLogFilename);
	m_dbLogger->SetFileLog(m_dbLogFilename);
	m_dbLogger->AddDate();
	m_dbLogger->AddThreadId();

	GenerateLogFilename(PacketLogPrefix, m_packetLogFilename);
	m_packetLogger->SetFileLog(m_packetLogFilename);
	m_packetLogger->AddDate();
	m_packetLogger->AddThreadId();

	GenerateLogFilename(ErrorLogPrefix, m_errorLogFilename);
	m_errorLogger->SetFileLog(m_errorLogFilename);
	m_errorLogger->AddDate();

	//NetworkInstance->CreateNetworkLog();

#ifdef NO_DREI_LOG
	m_dbLogger->SetDisable(true);
	m_packetLogger->SetDisable(true);
	m_errorLogger->SetDisable(true);
#endif

	m_logFileSizeCheckTimer = new DreiNetwork::CycleChecker(LogFileSizeCheckTerm);

	return true;
}

void CServerLog::Fini()
{
	SAFE_DELETE(m_dbLogger);
	SAFE_DELETE(m_packetLogger);
	SAFE_DELETE(m_errorLogger);
	SAFE_DELETE(m_logFileSizeCheckTimer);
}

void CServerLog::Update()
{
#ifdef NO_DREI_LOG
	return;
#endif
	LogFileSizeCheck();
}

DreiNetwork::Logger* CServerLog::GetDBLogger()
{
	return m_dbLogger;
}

DreiNetwork::Logger* CServerLog::GetPacketLogger()
{
	return m_packetLogger;
}

DreiNetwork::Logger* CServerLog::GetErrorLogger()
{
	return m_errorLogger;
}

void CServerLog::LogFileSizeCheck()
{
	if (false == m_logFileSizeCheckTimer->IsExpire())
		return;

	SwitchLogfileIfEnoughBig(m_dbLogFilename, DBLogPrefix, m_dbLogger);
	SwitchLogfileIfEnoughBig(m_packetLogFilename, PacketLogPrefix, m_packetLogger);
	SwitchLogfileIfEnoughBig(m_errorLogFilename, ErrorLogPrefix, m_errorLogger);
}