#include "stdafx.h"
#ifdef UNIT_TEST
#include "Testcase.h"
#include "ProcessJobs.h"
#include <set>
#include "ProtocolBuilder.h"
#include "ProtocolAssembler.h"
#include "EncryptedLicenseData.h"
#include "UserPool.h"
#include "AesCryptography.h"
#include "TinyClientSocket.h"
#include "StringFilter.h"
#include "LicensekeyPool.h"

class CTestFunctionPrint
{
public:
	CTestFunctionPrint(const char* functionname) : m_functionname(functionname)
	{
		printf("Begin %s\n", m_functionname);
	}

	~CTestFunctionPrint()
	{
		printf("End.. %s\n", m_functionname);
	}

private:
	const char* m_functionname;
};

CLicenseServerTestcase::CLicenseServerTestcase( void )
{
}

CLicenseServerTestcase::~CLicenseServerTestcase( void )
{
}

bool CLicenseServerTestcase::Run()
{
	TestMakeEncryptKey();
	TestEncryptedData();
	TestUserPool();
	TestCrypto();
	TestStringValidCheck();
	TestProtocolBuilder();
	TestStringFilter();
	TestLicensekeyPool();
	TestLicensekeyPoolFullLicense();
	TestLicensekeyPoolLauncherLicense();
	TestLicensekeyPoolLicenseServerLicense();
	TestBase64Encode();
	return true;
}

void CLicenseServerTestcase::TestMakeEncryptKey()
{
	CTestFunctionPrint autoPrint(__FUNCTION__);
	std::set<std::string> encryptKeySet;
	GenerateEncryptKey generator;
	const int KeySize = 20;
	char buffer[KeySize] = {0,};
	const int TestCount = 5000;
	for (int i=0; i<TestCount; ++i)
	{
		generator.Generate(i, buffer, KeySize);
		ACE_TEST_ASSERT(CheckNotAllowedCharacter(buffer, KeySize-1));
		ACE_TEST_ASSERT(encryptKeySet.end() == encryptKeySet.find(std::string(buffer, KeySize)));
		std::string keyStr(buffer, KeySize-1);
		encryptKeySet.insert(keyStr);
		//printf("%04d = %s\n", i, buffer);

		CProtocolBuilder builder;
		builder.Add("EncryptKey", keyStr);
		std::string s = builder.GetValue<std::string>("EncryptKey");
		ACE_TEST_ASSERT(keyStr == s);
	}

	std::set<std::string> encryptKeySet2;
	for (int i=0; i<TestCount; ++i)
	{
		generator.Generate(i, buffer, KeySize);
		ACE_TEST_ASSERT(CheckNotAllowedCharacter(buffer, KeySize-1));
		ACE_TEST_ASSERT(encryptKeySet2.end() == encryptKeySet2.find(std::string(buffer, KeySize)));
		encryptKeySet2.insert(std::string(buffer, KeySize-1));
	}

	std::set<std::string>::iterator firstIter = encryptKeySet.begin();
	std::set<std::string>::iterator secondIter = encryptKeySet2.begin();
	for (int i=0; i<TestCount; ++i)
	{
		std::string a = firstIter->c_str();
		std::string b = secondIter->c_str();
		ACE_TEST_ASSERT(a == b);
		++firstIter; ++secondIter;
	}
}

bool CLicenseServerTestcase::CheckNotAllowedCharacter( char* buffer, uint32 len )
{
	for (uint32 i=0; i<len; ++i)
	{
		char c = buffer[i];
		ACE_TEST_ASSERT(c>=48);
		ACE_TEST_ASSERT(c<=125);
		ACE_TEST_ASSERT(c!=';');
	}
	return true;
}

void CLicenseServerTestcase::TestEncryptedData()
{
	CTestFunctionPrint autoPrint(__FUNCTION__);
	CEncryptedLicenseData licenseData;
	std::string licenseKey = "A4EC-5087-63FC-13CD";
	std::string encryptKey = "lTx?<y<ikGRCO=i?@b|";
	uint32 maxUserCount = 3244;
	uint32 validityTerm = 365;
	TimeType startTime = 13232324;
	TimeType expireTime = startTime+12346;
	TimeType lastLoginTime = 121213232;
	EPMErrorCode errorCode = PMEC_KEY_EXPIRED;
	licenseData.SetLicenseKey(licenseKey.c_str());
	licenseData.SetEncryptKey(encryptKey.c_str());
	licenseData.SetUserCount(maxUserCount);
	licenseData.SetStartTime(startTime);
	licenseData.SetExpireTime(expireTime);
	licenseData.SetLastLoginTime(lastLoginTime);
	licenseData.SetValidityTerm(validityTerm);
	licenseData.SetErrorCode(errorCode);

	std::string licenseKeyFromData;
	licenseData.GetLicenseKey(licenseKeyFromData);
	ACE_TEST_ASSERT(licenseKey == licenseKeyFromData);
	std::string encryptKeyFromData;
	licenseData.GetEncryptKey(encryptKeyFromData);
	ACE_TEST_ASSERT(encryptKey == encryptKeyFromData);
	ACE_TEST_ASSERT(maxUserCount == licenseData.GetUserCount());
	ACE_TEST_ASSERT(startTime == licenseData.GetStartTime());
	ACE_TEST_ASSERT(expireTime == licenseData.GetExpireTime());
	ACE_TEST_ASSERT(lastLoginTime == licenseData.GetLastLoginTime());
	ACE_TEST_ASSERT(validityTerm == licenseData.GetValidityTerm());
	ACE_TEST_ASSERT(errorCode == licenseData.GetErrorCode());
}

void CLicenseServerTestcase::TestUserPool()
{
	CTestFunctionPrint autoPrint(__FUNCTION__);
	const uint32 maxUserCount = 489;
	CUserPool pool;
	pool.SetMaxUser(maxUserCount);

	std::string testMacAddress0 = "00:14:85:3a:4e:f4";
	std::string testMacAddress1 = "00:15:85:3a:4e:f4";
	ACE_TEST_ASSERT(pool.Login(0, testMacAddress0));
	ACE_TEST_ASSERT(pool.Login(1, testMacAddress1));
	ACE_TEST_ASSERT(pool.Authenticate(1, testMacAddress1));
	ACE_TEST_ASSERT(false == pool.Authenticate(0, testMacAddress1));
	ACE_TEST_ASSERT(false == pool.Authenticate(1, testMacAddress0));
	pool.Logoff(1);
	ACE_TEST_ASSERT(false == pool.Authenticate(1, testMacAddress1));
	ACE_TEST_ASSERT(pool.Authenticate(0, testMacAddress0));
	pool.Logoff(0);

	for(uint32 i=0;i<maxUserCount; ++i)
		ACE_TEST_ASSERT(pool.Login(i, testMacAddress0));
	for(uint32 i=0;i<maxUserCount; ++i)
		ACE_TEST_ASSERT(pool.Authenticate(i, testMacAddress0));
	ACE_TEST_ASSERT(false == pool.Login(99999, testMacAddress0));
	pool.Logoff(maxUserCount-100);
	ACE_TEST_ASSERT(pool.Login(99999, testMacAddress0));
	for(uint32 i=0;i<maxUserCount; ++i)
		pool.Logoff(i);
}

void CLicenseServerTestcase::TestCrypto()
{
	CTestFunctionPrint autoPrint(__FUNCTION__);
	const uint8 TestcaseCryptKey[] = {0x93, 0xE2, 0x34, 0x3B, 0xF2, 0x04, 0x10, 0x4D};
	const int TestcaseBufferSize = 523;
	char srcBuffer[TestcaseBufferSize] = {0,};
	char encryptBuffer[TestcaseBufferSize] = {0,};
	char decryptBuffer[TestcaseBufferSize] = {0,};

	for(int i=0;i<TestcaseBufferSize; ++i)
		srcBuffer[i] = i%128;

	CAesCryptography cryptObj;
	cryptObj.SetKeyValue(TestcaseCryptKey, sizeof(TestcaseCryptKey));
	cryptObj.EncryptBuffer((uint8*)srcBuffer, TestcaseBufferSize, (uint8*)encryptBuffer);
	ACE_TEST_ASSERT(0 != memcmp(srcBuffer, encryptBuffer, TestcaseBufferSize));
	cryptObj.DecryptBuffer((uint8*)encryptBuffer, TestcaseBufferSize, (uint8*)decryptBuffer);
	ACE_TEST_ASSERT(0 != memcmp(decryptBuffer, encryptBuffer, TestcaseBufferSize));
	ACE_TEST_ASSERT(0 == memcmp(decryptBuffer, srcBuffer, TestcaseBufferSize));
}

void CLicenseServerTestcase::TestStringValidCheck()
{
	CTestFunctionPrint autoPrint(__FUNCTION__);
	CEngLowerString engLowerString(NULL);
	ACE_TEST_ASSERT(engLowerString.IsValid("abcdeioertuieo"));
	ACE_TEST_ASSERT(false == engLowerString.IsValid("abcdei4oer8tuieo"));
	ACE_TEST_ASSERT(false == engLowerString.IsValid("abcdeiAoertuieo"));
	CEngUpperString engUpperString(&engLowerString);
	ACE_TEST_ASSERT(engUpperString.IsValid("AbCdEfG"));
	ACE_TEST_ASSERT(false == engUpperString.IsValid("AbCdEf7G"));

	CNumericString numericString(&engUpperString);
	ACE_TEST_ASSERT(numericString.IsValid("AbCdEf7G"));
	ACE_TEST_ASSERT(false == numericString.IsValid("AbCdEf7+-/G"));

	CSymbolStringGroup0 symbol0(&numericString);
	ACE_TEST_ASSERT(symbol0.IsValid("AbCdEf7+-/G"));
	ACE_TEST_ASSERT(false == symbol0.IsValid("AbCdEf7+-/G;<"));

	CSymbolStringGroup1 symbol1(&symbol0);
	ACE_TEST_ASSERT(symbol1.IsValid("AbCdEf7+-/G;<"));
	ACE_TEST_ASSERT(false == symbol1.IsValid("AbCdEf7+-/G;<[]"));

	CSymbolStringGroup2 symbol2(&symbol1);
	ACE_TEST_ASSERT(symbol2.IsValid("AbCdEf7+-/G;<[]"));
	ACE_TEST_ASSERT(false == symbol2.IsValid("AbCdEf7+-/G;<[]{}"));

	CSymbolStringGroup3 symbol3(&symbol2);
	ACE_TEST_ASSERT(symbol3.IsValid("AbCdEf7+-/G;<[]{}"));
	ACE_TEST_ASSERT(symbol3.IsValid("!~"));
	ACE_TEST_ASSERT(false == symbol3.IsValid("AbCdEf7+-/G;<[]{} 3"));
}

bool IsEqualProtocol(std::string& str0, std::string& str1)
{
	CProtocolBuilder builder0, builder1;
	builder0.Parse(str0);
	builder1.Parse(str1);
	return true;
}

void CLicenseServerTestcase::TestProtocolBuilder()
{
	CTestFunctionPrint autoPrint(__FUNCTION__);
	CProtocolBuilder inputBuilder;
	//type=loginaccount;account=knight;password=1414;mac=00:21:85:19:94:45;
	std::string inputString[] = 
	{
		"type=createaccount;account=knight;password=1414;email=javawork@test.com;mac=00:21:85:19:94:45;",
		"type=loginaccount;account=alice;password=9898;mac=00:21:85:19:94:45;"
	};
	inputBuilder.EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
	inputBuilder.Parse(inputString[0]);
	const uint32 MaxBuffer = 1024;
	char sendBuffer[MaxBuffer] = {0,};
	uint32 sendSize = inputBuilder.MakeBuffer(sendBuffer, MaxBuffer);

	uint32 bufferOffset = 0;
	CProtocolAssembler assembler;
	assembler.ApplyBuffer(sendBuffer, bufferOffset, sendSize);

	inputBuilder.Clear();
	inputBuilder.Parse(inputString[1]);
	memset(sendBuffer, 0, MaxBuffer);
	sendSize = inputBuilder.MakeBuffer(sendBuffer, MaxBuffer);
	bufferOffset = 0;
	assembler.ApplyBuffer(sendBuffer, bufferOffset, sendSize);

	int i = 0;
	CProtocolBuilder outputBuilder;
	outputBuilder.EncryptKey((unsigned char*)NetworkCryptKey, sizeof(NetworkCryptKey));
	while(assembler.AssembleBuilder(outputBuilder))
	{
		std::string parsedMessage =  outputBuilder.MakeString();
		//ACE_TEST_ASSERT(parsedMessage == inputString[i++]);
		IsEqualProtocol(parsedMessage, inputString[i]);
		ACE_TEST_ASSERT(parsedMessage.length() == inputString[i].length());
		outputBuilder.Clear();
		i++;
	}

	ACE_TEST_ASSERT(i == 2);
}

void CLicenseServerTestcase::TestStringFilter()
{
	CTestFunctionPrint autoPrint(__FUNCTION__);
	std::string targetString = "abcd32423--4fsfsd'fds";
	CStringFilter filter;
	filter.RemoveSpecificString(targetString, "--");
	filter.RemoveSpecificString(targetString, "'");
	ACE_TEST_ASSERT(targetString == "abcd324234fsfsdfds");

	filter.RemoveSpecificString(targetString, "--");
	filter.RemoveSpecificString(targetString, "'");
	ACE_TEST_ASSERT(targetString == "abcd324234fsfsdfds");

	std::string sourceStr = "aMaziNg234*%ZmW";
	filter.Convert2SmallLetter(sourceStr);
	ACE_TEST_ASSERT(sourceStr == "amazing234*%zmw");

	std::string seedStr = "1111";
	const int KeySize = 20;
	char buffer[KeySize] = {0,};
	GenerateEncryptKey generator;
	generator.Generate(seedStr, buffer, KeySize);
	std::string key0((char*)buffer, KeySize-3);

	generator.Generate(seedStr, buffer, KeySize);
	std::string key1((char*)buffer, KeySize-3);
	
	ACE_TEST_ASSERT(key0 == key1);
}

void CLicenseServerTestcase::TestLicensekeyPool()
{
	CTestFunctionPrint autoPrint(__FUNCTION__);

	uint32 userId0 = 8;
	std::string key0 = "B8F3-C75A-EB62-9E0A";
	std::string mac0 = "00:14:85:3a:4e:f3";
	const char* EditerAppName = "Editor";

	CLicensekeyPool pool;
	SLicenseKeyPoolElement element;
	SAuthenticateParam authParam;
	element.licenseKey = key0;
	element.macAddress = mac0;
	element.application = EditerAppName;
	element.loginTick = 0;
	element.licenseType = eLT_Full;
	bool result = pool.Login(userId0, element);
	ACE_TEST_ASSERT(result);
	ACE_TEST_ASSERT(1 == pool.GetElementCount());
	authParam.licenseKey = key0;
	authParam.macAddress = mac0;
	authParam.application = EditerAppName;
	result = pool.Authenticate(userId0, authParam);
	ACE_TEST_ASSERT(result);
	SLicenseKeyPoolElement elementResult;
	result = pool.GetElement(userId0, elementResult);
	ACE_TEST_ASSERT(result);
	ACE_TEST_ASSERT(element.licenseKey == elementResult.licenseKey);
	ACE_TEST_ASSERT(element.application == elementResult.application);
	pool.Logoff(userId0);
	ACE_TEST_ASSERT(0 == pool.GetElementCount());

	uint32 userId1 = 12;
	std::string key1 = "28F4-F75A-EB62-9E0A";
	std::string mac1 = "00:15:85:3a:41:f3";

	element.licenseKey = key0;
	element.macAddress = mac0;
	result = pool.Login(userId0, element);
	ACE_TEST_ASSERT(true == result);
	result = pool.Login(userId0, element);
	ACE_TEST_ASSERT(true == result);

	element.macAddress = mac1;
	result = pool.Login(userId1, element);
	ACE_TEST_ASSERT(false == result);
	element.licenseKey = key1;
	element.macAddress = mac1;
	result = pool.Login(userId1, element);
	ACE_TEST_ASSERT(true == result);
	result = pool.Login(userId1, element);
	ACE_TEST_ASSERT(true == result);
	result = pool.Login(userId0, element);
	ACE_TEST_ASSERT(false == result);
	
	result = pool.GetElement(userId0, elementResult);
	ACE_TEST_ASSERT(result);
	ACE_TEST_ASSERT(key0 == elementResult.licenseKey);
	result = pool.GetElement(userId1, elementResult);
	ACE_TEST_ASSERT(result);
	ACE_TEST_ASSERT(key1 == elementResult.licenseKey);
	result = pool.GetElement(userId1+10, elementResult);
	ACE_TEST_ASSERT(false == result);

	element.licenseKey = key0;
	element.macAddress = mac0;
	authParam.licenseKey = key0;
	authParam.macAddress = mac0;
	authParam.application = EditerAppName;
	result = pool.Authenticate(userId0, authParam);
	ACE_TEST_ASSERT(result);
	element.licenseKey = key1;
	element.macAddress = mac1;
	authParam.licenseKey = key1;
	authParam.macAddress = mac1;
	authParam.application = EditerAppName;
	result = pool.Authenticate(userId1, authParam);
	ACE_TEST_ASSERT(result);
	result = pool.Authenticate(userId0, authParam);
	ACE_TEST_ASSERT(false == result);
	result = pool.Authenticate(userId1+10, authParam);
	ACE_TEST_ASSERT(false == result);
	ACE_TEST_ASSERT(2 == pool.GetElementCount());

	pool.Logoff(userId0);
	ACE_TEST_ASSERT(1 == pool.GetElementCount());
	pool.Logoff(userId1);
	ACE_TEST_ASSERT(0 == pool.GetElementCount());

	element.licenseKey = key0;
	element.macAddress = mac0;
	authParam.licenseKey = key0;
	authParam.macAddress = mac0;
	authParam.application = EditerAppName;
	result = pool.Authenticate(userId0, authParam);
	ACE_TEST_ASSERT(false == result);

	// test again for sure
	element.licenseKey = key0;
	element.macAddress = mac0;
	result = pool.Login(userId0, element);
	ACE_TEST_ASSERT(true == result);
	result = pool.Login(userId1, element);
	ACE_TEST_ASSERT(true == result);

	result = pool.Authenticate(userId0, authParam);
	ACE_TEST_ASSERT(true == result);
	result = pool.Authenticate(userId1, authParam);
	ACE_TEST_ASSERT(true == result);
}

void CLicenseServerTestcase::TestLicensekeyPoolFullLicense()
{
	CTestFunctionPrint autoPrint(__FUNCTION__);

	uint32 userId0 = 8;
	std::string key0 = "B8F3-C75A-EB62-9E0A";
	std::string mac0 = "00:14:85:3a:4e:f3";

	CLicensekeyPool pool;
	SLicenseKeyPoolElement element;
	element.licenseKey = key0;
	element.macAddress = mac0;
	element.application = "Editor";
	element.loginTick = 0;
	element.licenseType = eLT_Full;
	bool result = pool.Login(userId0, element);
	ACE_TEST_ASSERT(true == result);
	ACE_TEST_ASSERT(1 == pool.GetElementCount());
	SAuthenticateParam authParam;
	authParam.licenseKey = key0;
	authParam.macAddress = mac0;
	authParam.application = "Editor";
	result = pool.Authenticate(userId0, authParam);
	ACE_TEST_ASSERT(result);
	SLicenseKeyPoolElement elementResult;
	result = pool.GetElement(userId0, elementResult);
	ACE_TEST_ASSERT(result);
	ACE_TEST_ASSERT(element.licenseKey == elementResult.licenseKey);
	ACE_TEST_ASSERT(element.application == elementResult.application);

	element.application = "Launcher";
	result = pool.Login(userId0, element);
	ACE_TEST_ASSERT(false == result);

	element.application = "Editor";
	element.macAddress = "00:14:85:3a:4e:f4";
	result = pool.Login(userId0, element);
	ACE_TEST_ASSERT(false == result);

	element.macAddress = mac0;
	uint32 userId1 = 21;

	element.application = "Launch";
	result = pool.Login(userId1, element);
	ACE_TEST_ASSERT(false == result);

	element.application = "Launcher";
	result = pool.Login(userId1, element);
	ACE_TEST_ASSERT(true == result);
	ACE_TEST_ASSERT(2 == pool.GetElementCount());

	uint32 userId2 = 27;
	element.application = "DedicatedServer";
	result = pool.Login(userId2, element);
	ACE_TEST_ASSERT(true == result);
	ACE_TEST_ASSERT(3 == pool.GetElementCount());

	uint32 userId3 = 37;
	element.licenseType = eLT_Launcher;
	result = pool.Login(userId3, element);
	ACE_TEST_ASSERT(false == result);

	element.application = "LicenseServer";
	result = pool.Login(userId3, element);
	ACE_TEST_ASSERT(false == result);

	element.application = "Launcher_";
	result = pool.Login(userId3, element);
	ACE_TEST_ASSERT(false == result);

	authParam.licenseKey = key0;
	authParam.macAddress = mac0;
	authParam.application = "Launcher_";
	result = pool.Authenticate(userId3, authParam);
	ACE_TEST_ASSERT(false == result);

	pool.Logoff(userId1);
	ACE_TEST_ASSERT(2 == pool.GetElementCount());

	authParam.application = "DedicatedServer";
	result = pool.Authenticate(userId2, authParam);
	ACE_TEST_ASSERT(true == result);

	element.application = "SettingsManager";
	element.licenseType = eLT_Full;
	result = pool.Login(userId3, element);
	ACE_TEST_ASSERT(true == result);

	pool.Logoff(userId2);
	pool.Logoff(userId3);
	ACE_TEST_ASSERT(1 == pool.GetElementCount());
	authParam.application = "Editor";
	result = pool.Authenticate(userId0, authParam);
	ACE_TEST_ASSERT(true == result);
	authParam.application = "DedicatedServer";
	result = pool.Authenticate(userId2, authParam);
	ACE_TEST_ASSERT(false == result);
}

void CLicenseServerTestcase::TestLicensekeyPoolLauncherLicense()
{
	CTestFunctionPrint autoPrint(__FUNCTION__);

	uint32 userId0 = 8;
	std::string key0 = "B8F3-C75A-EB62-9E0A";
	std::string mac0 = "00:14:85:3a:4e:f3";

	CLicensekeyPool pool;
	SLicenseKeyPoolElement element;
	element.licenseKey = key0;
	element.macAddress = mac0;
	element.application = "Editor";
	element.loginTick = 0;
	element.licenseType = eLT_Launcher;
	bool result = pool.Login(userId0, element);
	ACE_TEST_ASSERT(false == result);

	element.application = "Launcher";
	result = pool.Login(userId0, element);
	ACE_TEST_ASSERT(true == result);
	ACE_TEST_ASSERT(1 == pool.GetElementCount());
	SAuthenticateParam authParam;
	authParam.licenseKey = key0;
	authParam.macAddress = mac0;
	authParam.application = "Launcher";
	result = pool.Authenticate(userId0, authParam);
	ACE_TEST_ASSERT(result);

	authParam.application = "Editor";
	result = pool.Authenticate(userId0, authParam);
	ACE_TEST_ASSERT(false == result);
	SLicenseKeyPoolElement elementResult;
	result = pool.GetElement(userId0, elementResult);
	ACE_TEST_ASSERT(result);
	ACE_TEST_ASSERT(element.licenseKey == elementResult.licenseKey);
	ACE_TEST_ASSERT(element.application == elementResult.application);

	uint32 userId1 = 21;
	result = pool.Login(userId1, element);
	ACE_TEST_ASSERT(false == result);

	element.application = "Editor";
	result = pool.Login(userId1, element);
	ACE_TEST_ASSERT(false == result);

	uint32 userId2 = 27;
	element.application = "DedicatedServer";
	result = pool.Login(userId2, element);
	ACE_TEST_ASSERT(false == result);

	element.application = "DedicatedSer";
	result = pool.Login(userId2, element);
	ACE_TEST_ASSERT(false == result);

	uint32 userId3 = 37;
	element.licenseType = eLT_Full;
	result = pool.Login(userId3, element);
	ACE_TEST_ASSERT(false == result);

	authParam.application = "Editor";
	result = pool.Authenticate(userId2, authParam);
	ACE_TEST_ASSERT(false == result);
}

void CLicenseServerTestcase::TestLicensekeyPoolLicenseServerLicense()
{
	CTestFunctionPrint autoPrint(__FUNCTION__);

	uint32 userId0 = 8;
	std::string key0 = "B8F3-C75A-EB62-9E0A";
	std::string mac0 = "00:14:85:3a:4e:f3";

	CLicensekeyPool pool;
	SLicenseKeyPoolElement element;
	element.licenseKey = key0;
	element.macAddress = mac0;
	element.application = "Editor";
	element.loginTick = 0;
	element.licenseType = eLT_LicenseServer;
	bool result = pool.Login(userId0, element);
	ACE_TEST_ASSERT(false == result);

	element.application = "LicenseServer";
	result = pool.Login(userId0, element);
	ACE_TEST_ASSERT(true == result);
	ACE_TEST_ASSERT(1 == pool.GetElementCount());
	SAuthenticateParam authParam;
	authParam.licenseKey = key0;
	authParam.macAddress = mac0;
	authParam.application = "LicenseServer";
	result = pool.Authenticate(userId0, authParam);
	ACE_TEST_ASSERT(true == result);
	authParam.application = "Editor";
	result = pool.Authenticate(userId0, authParam);
	ACE_TEST_ASSERT(false == result);
	authParam.application = "Launcher";
	result = pool.Authenticate(userId0, authParam);
	ACE_TEST_ASSERT(false == result);
	SLicenseKeyPoolElement elementResult;
	result = pool.GetElement(userId0, elementResult);
	ACE_TEST_ASSERT(result);
	ACE_TEST_ASSERT(element.licenseKey == elementResult.licenseKey);
	ACE_TEST_ASSERT(element.application == elementResult.application);

	uint32 userId1 = 21;
	result = pool.Login(userId1, element);
	ACE_TEST_ASSERT(false == result);

	element.application = "Editor";
	result = pool.Login(userId1, element);
	ACE_TEST_ASSERT(false == result);

	uint32 userId2 = 27;
	element.application = "DedicatedServer";
	result = pool.Login(userId2, element);
	ACE_TEST_ASSERT(false == result);

	uint32 userId3 = 37;
	element.licenseType = eLT_Full;
	result = pool.Login(userId3, element);
	ACE_TEST_ASSERT(false == result);

	authParam.application = "Editor";
	result = pool.Authenticate(userId2, authParam);
	ACE_TEST_ASSERT(false == result);
}

void CLicenseServerTestcase::TestBase64Encode()
{
	CTestFunctionPrint autoPrint(__FUNCTION__);
	CStringFilter filter;

	const int TestArraySize = 42;
	struct TestStruct
	{
		int serial;
		float posX;
		float posY;
		char testArray[TestArraySize];
	};

	TestStruct test0;
	test0.serial = 735;
	test0.posX = 333453.440f;
	test0.posY = 3639453.201f;
	for(int i=0;i<TestArraySize;++i)
		test0.testArray[i] = i;
	std::string encodedStr = filter.Base64Encode((const uint8*)&test0, sizeof(TestStruct));
	std::string decodedStr = filter.Base64Decode(encodedStr);

	TestStruct test1;
	memcpy(&test1, decodedStr.c_str(), decodedStr.length());
	ACE_TEST_ASSERT(test0.serial == test1.serial);
	ACE_TEST_ASSERT(test0.posX == test1.posX);
	ACE_TEST_ASSERT(test0.posY == test1.posY);
	ACE_TEST_ASSERT(0 == memcmp(test0.testArray, test1.testArray, TestArraySize));
}

#endif // UNIT_TEST