////////////////////////////////////////////////////////////////////////////
//
//  Crytek Source File.
//  Copyright (C), Crytek Studios, 2009.
// -------------------------------------------------------------------------
//  File name:   UniKeyManager.h
//  Version:     v1.00
//  Created:     13/08/2009 by Younggi Lim
//  Compilers:   Visual Studio.NET
//  Description: make easy to use UniKey Api
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////
#include "stdafx.h"
#include "UniKeyManager.h"

#include "UniKeyFR.h"
#include <time.h>
#include "UserPool.h"
#include "EncryptedLicenseData.h"

const uint32 LicenseDataSyncTerm = 5000;
const uint32 LoginWriteTerm = 1000*60*10;		// every 10 min
const uint32 UniKeyIsOnCheckTerm = 5000;		// 5 sec

//////////////////////////////////////////////////////////////////////////
CUniKeyManager::CUniKeyManager() : 
		m_unikeyOn(false), 
		m_unikeyLastCheckTick(0), 
		m_unikeyLoginWriteTick(0),
		m_userPool(new CUserPool()),
		m_licenseData(new CEncryptedLicenseData())
{
}


//////////////////////////////////////////////////////////////////////////
CUniKeyManager::~CUniKeyManager()
{
	SAFE_DELETE(m_userPool);
	SAFE_DELETE(m_licenseData);
}


bool CUniKeyManager::Find()
{
	unsigned long retcode, lp1, lp2;
	lp1 = 0, lp2 = 1;
	retcode = UniKey_Find(&m_handle, &lp1, &lp2);
	if (0 != retcode)
		return false;

	return true;
}

bool CUniKeyManager::LoginUniKey()
{
	printf("Find Dongle...\n");
	uint16 p1, p2, p3, p4;
	p1 = 19067;	// passwords
	p2 = 43378;
	p3 = 64671;
	p4 = 8007;
	unsigned long retcode, lp1, lp2;
	retcode = UniKey_Find(&m_handle, &lp1, &lp2);
	if (SUCCESS != retcode)
	{
		printf("Cann't find dongle.\n");
		return false;
	}
	printf("Dongle is on.\n");
	retcode = UniKey_Vender_Logon(&m_handle, &p1, &p2, &p3, &p4);
	if (SUCCESS != retcode)
	{
		printf("Dongle Login fail.\n");
		return false;
	}
	printf("Dongle Login success.\n");

	uint16 offset = 0;
	uint16 dataSize = LicenseDataSize;
	char buffer[256] = {0,};
	retcode = UniKey_Read_Memory(&m_handle, &offset, &dataSize, (uint8*)buffer);
	if (SUCCESS != retcode)
		return false;

	m_licenseData->ApplyEncryptedBuffer(buffer);
	printf("Dongle data version is %d.\n", m_licenseData->GetDataVersion()); 
	if (UniKeyDataVersion != m_licenseData->GetDataVersion())
	{
		printf("Dongle data version error.\n"); 
		printf("DongleVersion(%d), ServerVersion(%d)\n", 
			m_licenseData->GetDataVersion(), UniKeyDataVersion);
		return false;
	}
	printf("Concurrent Floating Licenses is %d.\n", m_licenseData->GetUserCount());

	WriteValidityTime();
	if (false == ValidDate(m_licenseData->GetStartTime(), m_licenseData->GetExpireTime(), m_licenseData->GetLastLoginTime()))
	{
		printf("Validity term is expired.\n");
		return false;
	}

	TimeType remainSecond = GetRemainSecond();
	uint32 remainMonth = 0;
	uint32 remainDay = 0;
	uint32 remainHour = 0;
	ConvertToDate(remainSecond, remainMonth, remainDay, remainHour);
	printf("%d month(s) %d day(s) %d hour(s) left.\n", remainMonth, remainDay, remainHour);
	m_userPool->SetMaxUser(m_licenseData->GetUserCount());
	m_unikeyOn = true;
	WriteLoginTime();
	return true;
}

void CUniKeyManager::LogOffUniKey()
{
	UniKey_Logoff(&m_handle);
}

EPMErrorCode CUniKeyManager::LoginUser(uint32 userId, std::string key, std::string mac)
{
	if (false == CheckUnikeyIsOn())
		return PMEC_NO_DONGLE;

	//if (false == ValidKey(key))
	//	return false;

	if (false == ValidDate(m_licenseData->GetStartTime(), m_licenseData->GetExpireTime(), m_licenseData->GetLastLoginTime()))
		return PMEC_KEY_EXPIRED;

	WriteLoginTime();
	bool poolResult = m_userPool->Login(userId, mac);
	if (false == poolResult)
		return PMEC_OVER_CONNECTION;

	return PMEC_NO_ERROR;

}

void CUniKeyManager::LogoffUser(uint32 userId)
{
	m_userPool->Logoff(userId);
}

EPMErrorCode CUniKeyManager::AuthenticateUser( uint32 userId, std::string key, std::string mac )
{
	if (false == CheckUnikeyIsOn())
		return PMEC_NO_DONGLE;

	//if (false == ValidKey(key))
	//	return false;

	if (false == ValidDate(m_licenseData->GetStartTime(), m_licenseData->GetExpireTime(), m_licenseData->GetLastLoginTime()))
		return PMEC_KEY_EXPIRED;

	bool poolResult = m_userPool->Authenticate(userId, mac);
	if (false == poolResult)
		return PMEC_OVER_CONNECTION;

	return PMEC_NO_ERROR;
}

TimeType CUniKeyManager::GetRemainSecond()
{
	time_t currentTime;
	time((time_t*)&currentTime);

	if (m_licenseData->GetExpireTime() <= (TimeType)currentTime)
		return 0;
	return (m_licenseData->GetExpireTime() - (TimeType)currentTime);
}


bool CUniKeyManager::ValidKey( std::string& key ) 
{
	std::string keyFromUniKey;
	m_licenseData->GetLicenseKey(keyFromUniKey);
	return (keyFromUniKey == key);
}

bool CUniKeyManager::ValidDate( const TimeType& start, const TimeType& expire, const TimeType& last ) const
{
	time_t currentTime;
	time((time_t*)&currentTime);
	if (expire <= currentTime)
		return false;
	if (start > currentTime)
		return false;
	if (last > currentTime)
		return false;
	
	return true;
}

bool CUniKeyManager::CheckUnikeyIsOn()
{
	uint32 currentTick = this->GetTick(); 

	if (0 == m_unikeyLastCheckTick)
		m_unikeyLastCheckTick = currentTick;

	if ((currentTick-m_unikeyLastCheckTick) < UniKeyIsOnCheckTerm)
		return m_unikeyOn;

	m_unikeyLastCheckTick = currentTick;
	
	bool result = Find();
	//printf("Find() elasped time %d\n", this->GetTick() - currentTick);
	m_unikeyOn = result;
	return result;
}

void CUniKeyManager::WriteLoginTime()
{
	uint32 currentTick = this->GetTick(); 
	if (0 == m_unikeyLoginWriteTick)
		m_unikeyLoginWriteTick = currentTick;

	if ((currentTick-m_unikeyLoginWriteTick) < LoginWriteTerm)
		return;

	m_unikeyLoginWriteTick = currentTick;

	uint16 offset = 0;
	uint16 dataSize = LicenseDataSize;
	char buffer[256] = {0,};
	unsigned long retcode = UniKey_Read_Memory(&m_handle, &offset, &dataSize, (uint8*)buffer);
	if (SUCCESS != retcode)
		return;

	m_licenseData->ApplyEncryptedBuffer(buffer);

	time_t t = 0;
	time(&t);
	m_licenseData->SetLastLoginTime((TimeType)t);
	m_licenseData->GetEncryptedBuffer(buffer);
	UniKey_Write_Memory(&m_handle, &offset, &dataSize, (uint8*)buffer);
}

void CUniKeyManager::WriteValidityTime()
{
	if ( 0 != m_licenseData->GetStartTime())
		return;
	if ( 0 != m_licenseData->GetExpireTime())
		return;

	time_t startTime = 0;
	time(&startTime);
	m_licenseData->SetStartTime((TimeType)startTime);
	m_licenseData->SetExpireTime(m_licenseData->GetStartTime() + m_licenseData->GetValidityTerm());

	uint16 offset = 0;
	uint16 dataSize = LicenseDataSize;
	char buffer[256] = {0,};

	time_t t = 0;
	time(&t);
	m_licenseData->SetLastLoginTime((TimeType)t);
	m_licenseData->GetEncryptedBuffer(buffer);
	UniKey_Write_Memory(&m_handle, &offset, &dataSize, (uint8*)buffer);
}

uint32 CUniKeyManager::GetTick()
{
#ifdef WIN32
	return GetTickCount();
#else 
	timeval tick;
	gettimeofday (&tick, 0);
	return (tick.tv_sec*1000 + tick.tv_usec/1000);
#endif
}

void CUniKeyManager::GetLicenseKey(std::string& key)
{
	m_licenseData->GetLicenseKey(key);
}

void CUniKeyManager::ConvertToDate(const TimeType paramRemainSecond, uint32& month, uint32& day, uint32& hour)
{
	const uint32 OneMonthSecond = 30*24*60*60;
	const uint32 OneDaySecond = 24*60*60;
	const uint32 OneHourSecond = 60*60;
	TimeType remainSecond = paramRemainSecond;
	month = remainSecond/OneMonthSecond;
	remainSecond -= month*OneMonthSecond;
	day = remainSecond/OneDaySecond;
	remainSecond -= day*OneDaySecond;
	hour = remainSecond/OneHourSecond;
}