#ifndef HEADER_Math
#define HEADER_Math

#include <math.h>

//

#define PIf 3.141592653589793f

//

class Vec3f32;
class Vec4f32;

class Mat33f32;
class Mat44f32;

class Quatf32;

//

class Vec3f32
{
public:
	float x, y, z;

public:
	inline Vec3f32()
	{
	}

	inline Vec3f32(float x, float y, float z) :
		x(x), y(y), z(z)
	{
	}

public:
	inline float& operator [](unsigned int index)
	{
		return (&x)[index];
	}

	inline float operator [](unsigned int index) const
	{
		return (&x)[index];
	}

	inline Vec3f32 operator -() const
	{
		return Vec3f32(-x, -y, -z);
	}

	inline Vec3f32 operator +(const Vec3f32& v) const
	{
		return Vec3f32(x + v.x, y + v.y, z + v.z);
	}

	inline Vec3f32 operator -(const Vec3f32& v) const
	{
		return Vec3f32(x - v.x, y - v.y, z - v.z);
	}

	inline Vec3f32 operator *(float v) const
	{
		return Vec3f32(x * v, y * v, z * v);
	}

	inline Vec3f32 operator /(float v) const
	{
		return Vec3f32(x / v, y / v, z / v);
	}

	inline Vec3f32& operator +=(const Vec3f32& v)
	{
		x += v.x; y += v.y; z += v.z;
		return *this;
	}

	inline Vec3f32& operator -=(const Vec3f32& v)
	{
		x -= v.x; y -= v.y; z -= v.z;
		return *this;
	}

	inline Vec3f32& operator *=(float v)
	{
		x *= v; y *= v; z *= v;
		return *this;
	}

	inline Vec3f32 operator *(const Mat33f32& matrix) const
	{
		Vec3f32 result;
		const float* pV = (const float*)&x;
		const float* pM = (const float*)&matrix;
		float* pR = (float*)&result;
		pR[0] = pV[0]*pM[0*3+0] + pV[1]*pM[1*3+0] + pV[2]*pM[2*3+0];
		pR[1] = pV[0]*pM[0*3+1] + pV[1]*pM[1*3+1] + pV[2]*pM[2*3+1];
		pR[2] = pV[0]*pM[0*3+2] + pV[1]*pM[1*3+2] + pV[2]*pM[2*3+2];
		return result;
	}

public:
	inline float Length() const
	{
		return ::sqrt(x*x + y*y + z*z);
	}

	inline Vec3f32& Normalize()
	{
		const float s = 1.0f / Length();
		x *= s; y *= s; z *= s;
		return *this;
	}

	inline Vec3f32 Normalized() const
	{
		const float s = 1.0f / Length();
		return Vec3f32(x * s, y * s, z * s);
	}

	inline Vec3f32 Cross(const Vec3f32& vector) const
	{
		return Vec3f32(
			y * vector.z - z * vector.y,
			z * vector.x - x * vector.z,
			x * vector.y - y * vector.x);
	}
};

class Vec4f32
{
public:
	float x, y, z, w;

public:
	inline float& operator [](unsigned int index)
	{
		return (&x)[index];
	}

	inline float operator [](unsigned int index) const
	{
		return (&x)[index];
	}
};

class Mat33f32
{
public:
	static inline Mat33f32 RotationX(float angle)
	{
		float c = ::cosf(angle);
		float s = ::sinf(angle);
		return Mat33f32(
			1.0f,	0.0f,	0.0f,
			0.0f,	c,		s,
			0.0f,	-s,		c);
	}

	static inline Mat33f32 RotationY(float angle)
	{
		float c = ::cosf(angle);
		float s = ::sinf(angle);
		return Mat33f32(
			c,		0.0f,	-s,
			0.0f,	1.0f,	0.0f,
			s,		0.0f,	c);
	}

	static inline Mat33f32 RotationZ(float angle)
	{
		float c = ::cosf(angle);
		float s = ::sinf(angle);
		return Mat33f32(
			c,		s,		0.0f,
			-s,		c,		0.0f,
			0.0f,	0.0f,	1.0f);
	}

	static inline Mat33f32 LookAt(const Vec3f32& direction)
	{
		Vec3f32 zAxis(direction);
		zAxis.Normalize();
		Vec3f32 yAxis(zAxis.y, zAxis.z, zAxis.x);
		Vec3f32 xAxis = yAxis.Cross(zAxis).Normalize();
		yAxis = zAxis.Cross(xAxis);
		return Mat33f32(
			xAxis.x, xAxis.y, xAxis.z,
			yAxis.x, yAxis.y, yAxis.z,
			zAxis.x, zAxis.y, zAxis.z);
	}

	static inline Mat33f32 Scaling(float x, float y, float z)
	{
		return Mat33f32(
			x, 0.0f, 0.0f,
			0.0f, y, 0.0f,
			0.0f, 0.0f, z);
	}

	static inline Mat33f32 Scaling(float v)
	{
		return Scaling(v, v, v);
	}

public:
	Vec3f32 _[3];

public:
	inline Mat33f32()
	{
	}

	inline Mat33f32(
		float _00, float _01, float _02,
		float _10, float _11, float _12,
		float _20, float _21, float _22)
	{
		_[0][0] = _00; _[0][1] = _01; _[0][2] = _02;
		_[1][0] = _10; _[1][1] = _11; _[1][2] = _12;
		_[2][0] = _20; _[2][1] = _21; _[2][2] = _22;
	}

public:
	inline Vec3f32& operator [](unsigned int index)
	{
		return _[index];
	}

	inline const Vec3f32& operator [](unsigned int index) const
	{
		return _[index];
	}

	inline Mat33f32 operator *(const Mat33f32& matrix) const
	{
		Mat33f32 result;
		const float* pM0 = (const float*)_;
		const float* pM1 = (const float*)&matrix;
		float* pR = (float*)&result;

		pR[0*3+0] = pM0[0*3+0] * pM1[0*3+0] + pM0[0*3+1] * pM1[1*3+0] + pM0[0*3+2] * pM1[2*3+0];
		pR[0*3+1] = pM0[0*3+0] * pM1[0*3+1] + pM0[0*3+1] * pM1[1*3+1] + pM0[0*3+2] * pM1[2*3+1];
		pR[0*3+2] = pM0[0*3+0] * pM1[0*3+2] + pM0[0*3+1] * pM1[1*3+2] + pM0[0*3+2] * pM1[2*3+2];

		pR[1*3+0] = pM0[1*3+0] * pM1[0*3+0] + pM0[1*3+1] * pM1[1*3+0] + pM0[1*3+2] * pM1[2*3+0];
		pR[1*3+1] = pM0[1*3+0] * pM1[0*3+1] + pM0[1*3+1] * pM1[1*3+1] + pM0[1*3+2] * pM1[2*3+1];
		pR[1*3+2] = pM0[1*3+0] * pM1[0*3+2] + pM0[1*3+1] * pM1[1*3+2] + pM0[1*3+2] * pM1[2*3+2];

		pR[2*3+0] = pM0[2*3+0] * pM1[0*3+0] + pM0[2*3+1] * pM1[1*3+0] + pM0[2*3+2] * pM1[2*3+0];
		pR[2*3+1] = pM0[2*3+0] * pM1[0*3+1] + pM0[2*3+1] * pM1[1*3+1] + pM0[2*3+2] * pM1[2*3+1];
		pR[2*3+2] = pM0[2*3+0] * pM1[0*3+2] + pM0[2*3+1] * pM1[1*3+2] + pM0[2*3+2] * pM1[2*3+2];

		return result;
	}

public:
	inline void Identity()
	{
		_[0] = Vec3f32(1.0f, 0.0f, 0.0f);
		_[1] = Vec3f32(0.0f, 1.0f, 0.0f);
		_[2] = Vec3f32(0.0f, 0.0f, 1.0f);
	}

	inline Mat33f32 Transposed() const
	{
		return Mat33f32(
			_[0][0], _[1][0], _[2][0],
			_[0][1], _[1][1], _[2][1],
			_[0][2], _[1][2], _[2][2]);
	}
};

class Mat44f32
{
public:
	inline static Mat44f32 Translation(float x, float y, float z)
	{
		return Mat44f32(
			1.0f, 0.0f, 0.0f, 0.0f,
			0.0f, 1.0f, 0.0f, 0.0f,
			0.0f, 0.0f, 1.0f, 0.0f,
			x, y, z, 1.0f);
	}

public:
	Vec4f32 _[4];

public:
	inline Mat44f32()
	{
	}

	inline Mat44f32(
		float _00, float _01, float _02, float _03,
		float _10, float _11, float _12, float _13,
		float _20, float _21, float _22, float _23,
		float _30, float _31, float _32, float _33)
	{
		_[0].x = _00; _[0].y = _01; _[0].z = _02; _[0].w = _03;
		_[1].x = _10; _[1].y = _11; _[1].z = _12; _[1].w = _13;
		_[2].x = _20; _[2].y = _21; _[2].z = _22; _[2].w = _23;
		_[3].x = _30; _[3].y = _31; _[3].z = _32; _[3].w = _33;
	}

	inline explicit Mat44f32(const Mat33f32& matrix)
	{
		_[0][0] = matrix._[0][0]; _[0][1] = matrix._[0][1]; _[0][2] = matrix._[0][2]; _[0][3] = 0.0f;
		_[1][0] = matrix._[1][0]; _[1][1] = matrix._[1][1]; _[1][2] = matrix._[1][2]; _[1][3] = 0.0f;
		_[2][0] = matrix._[2][0]; _[2][1] = matrix._[2][1]; _[2][2] = matrix._[2][2]; _[2][3] = 0.0f;
		_[3][0] = 0.0f; _[3][1] = 0.0f; _[3][2] = 0.0f; _[3][3] = 1.0f;
	}

public:
	inline Vec4f32& operator [](unsigned int index)
	{
		return _[index];
	}

	inline const Vec4f32& operator [](unsigned int index) const
	{
		return _[index];
	}

	inline Mat44f32& operator =(const Mat33f32& matrix)
	{
		_[0][0] = matrix._[0][0]; _[0][1] = matrix._[0][1]; _[0][2] = matrix._[0][2]; _[0][3] = 0.0f;
		_[1][0] = matrix._[1][0]; _[1][1] = matrix._[1][1]; _[1][2] = matrix._[1][2]; _[1][3] = 0.0f;
		_[2][0] = matrix._[2][0]; _[2][1] = matrix._[2][1]; _[2][2] = matrix._[2][2]; _[2][3] = 0.0f;
		_[3][0] = 0.0f; _[3][1] = 0.0f; _[3][2] = 0.0f; _[3][3] = 1.0f;

		return *this;
	}

	inline Mat44f32 operator *(const Mat44f32& matrix) const
	{
		Mat44f32 result;
		const float* pM0 = (const float*)_;
		const float* pM1 = (const float*)&matrix;
		float* pR = (float*)&result;

		pR[0*4+0] = pM0[0*4+0] * pM1[0*4+0] + pM0[0*4+1] * pM1[1*4+0] + pM0[0*4+2] * pM1[2*4+0] + pM0[0*4+3] * pM1[3*4+0];
		pR[0*4+1] = pM0[0*4+0] * pM1[0*4+1] + pM0[0*4+1] * pM1[1*4+1] + pM0[0*4+2] * pM1[2*4+1] + pM0[0*4+3] * pM1[3*4+1];
		pR[0*4+2] = pM0[0*4+0] * pM1[0*4+2] + pM0[0*4+1] * pM1[1*4+2] + pM0[0*4+2] * pM1[2*4+2] + pM0[0*4+3] * pM1[3*4+2];
		pR[0*4+3] = pM0[0*4+0] * pM1[0*4+3] + pM0[0*4+1] * pM1[1*4+3] + pM0[0*4+2] * pM1[2*4+3] + pM0[0*4+3] * pM1[3*4+3];

		pR[1*4+0] = pM0[1*4+0] * pM1[0*4+0] + pM0[1*4+1] * pM1[1*4+0] + pM0[1*4+2] * pM1[2*4+0] + pM0[1*4+3] * pM1[3*4+0];
		pR[1*4+1] = pM0[1*4+0] * pM1[0*4+1] + pM0[1*4+1] * pM1[1*4+1] + pM0[1*4+2] * pM1[2*4+1] + pM0[1*4+3] * pM1[3*4+1];
		pR[1*4+2] = pM0[1*4+0] * pM1[0*4+2] + pM0[1*4+1] * pM1[1*4+2] + pM0[1*4+2] * pM1[2*4+2] + pM0[1*4+3] * pM1[3*4+2];
		pR[1*4+3] = pM0[1*4+0] * pM1[0*4+3] + pM0[1*4+1] * pM1[1*4+3] + pM0[1*4+2] * pM1[2*4+3] + pM0[1*4+3] * pM1[3*4+3];

		pR[2*4+0] = pM0[2*4+0] * pM1[0*4+0] + pM0[2*4+1] * pM1[1*4+0] + pM0[2*4+2] * pM1[2*4+0] + pM0[2*4+3] * pM1[3*4+0];
		pR[2*4+1] = pM0[2*4+0] * pM1[0*4+1] + pM0[2*4+1] * pM1[1*4+1] + pM0[2*4+2] * pM1[2*4+1] + pM0[2*4+3] * pM1[3*4+1];
		pR[2*4+2] = pM0[2*4+0] * pM1[0*4+2] + pM0[2*4+1] * pM1[1*4+2] + pM0[2*4+2] * pM1[2*4+2] + pM0[2*4+3] * pM1[3*4+2];
		pR[2*4+3] = pM0[2*4+0] * pM1[0*4+3] + pM0[2*4+1] * pM1[1*4+3] + pM0[2*4+2] * pM1[2*4+3] + pM0[2*4+3] * pM1[3*4+3];

		pR[3*4+0] = pM0[3*4+0] * pM1[0*4+0] + pM0[3*4+1] * pM1[1*4+0] + pM0[3*4+2] * pM1[2*4+0] + pM0[3*4+3] * pM1[3*4+0];
		pR[3*4+1] = pM0[3*4+0] * pM1[0*4+1] + pM0[3*4+1] * pM1[1*4+1] + pM0[3*4+2] * pM1[2*4+1] + pM0[3*4+3] * pM1[3*4+1];
		pR[3*4+2] = pM0[3*4+0] * pM1[0*4+2] + pM0[3*4+1] * pM1[1*4+2] + pM0[3*4+2] * pM1[2*4+2] + pM0[3*4+3] * pM1[3*4+2];
		pR[3*4+3] = pM0[3*4+0] * pM1[0*4+3] + pM0[3*4+1] * pM1[1*4+3] + pM0[3*4+2] * pM1[2*4+3] + pM0[3*4+3] * pM1[3*4+3];

		return result;
	}

public:
	inline Mat44f32 Transposed() const
	{
		return Mat44f32(
			_[0][0], _[1][0], _[2][0], _[3][0],
			_[0][1], _[1][1], _[2][1], _[3][1],
			_[0][2], _[1][2], _[2][2], _[3][2],
			_[0][3], _[1][3], _[2][3], _[3][3]);
	}
};

class Quatf32
{
public:
	float x, y, z, w;

public:
	inline Quatf32()
	{
	}

	inline Quatf32(float x, float y, float z, float w) :
		x(x), y(y), z(z), w(w)
	{
	}

public:
	inline Vec3f32 GetColumn0() const { return Vec3f32(2.0f * (x*x+w*w) - 1.0f, 2.0f * (y*x+z*w), 2.0f * (z*x-y*w)); }
	inline Vec3f32 GetColumn1() const { return Vec3f32(2.0f * (x*y-z*w), 2.0f * (y*y+w*w) - 1.0f, 2.0f * (z*y+x*w)); }
	inline Vec3f32 GetColumn2() const { return Vec3f32(2.0f * (x*z+y*w), 2.0f * (y*z-x*w), 2.0f * (z*z+w*w) - 1.0f); }
	inline Vec3f32 GetColumn(unsigned int index) const
	{
		switch (index)
		{
		case 0: return GetColumn0();
		case 1: return GetColumn0();
		case 2: return GetColumn0();
		}

		return Vec3f32(0.0f, 0.0f, 0.0f);
	}

	inline Vec3f32 GetRow0() const { return Vec3f32(2.0f * (x*x+w*w) - 1.0f, 2.0f * (x*y-z*w), 2.0f * (x*z+y*w)); }
	inline Vec3f32 GetRow1() const { return Vec3f32(2.0f * (y*x+z*w), 2.0f * (y*y+w*w) - 1.0f, 2.0f * (y*z-x*w)); }
	inline Vec3f32 GetRow2() const { return Vec3f32(2.0f * (z*x-y*w), 2.0f * (z*y+x*w), 2.0f * (z*z+w*w) - 1.0f); }

	Mat33f32 ToMatrix()
	{
		Mat33f32 matrix;
		matrix[0] = GetColumn0();
		matrix[1] = GetColumn1();
		matrix[2] = GetColumn2();
		return matrix;
	}

	void FromMatrix(const Mat33f32& m)
	{ 
		float tr = m[0][0] + m[1][1] + m[2][2];
		if (tr > 0.0f)
		{
			float s = ::sqrt(tr+1.0f);
//			assert(s);
			float p = 0.5f / s;
				x = (m[1][2] - m[2][1]) * p;
				y = (m[2][0] - m[0][2]) * p;
				z = (m[0][1] - m[1][0]) * p;
				w = s*0.5f;

		}
		else if ((m[0][0] >= m[1][1]) && (m[0][0] >= m[2][2]))
		{
			float s = ::sqrt(m[0][0]-m[1][1]-m[2][2]+1.0f);
//			assert(s);
			float p = 0.5f / s;
			x = s * 0.5f;
			y = (m[0][1] + m[1][0]) * p;
			z = (m[0][2] + m[2][0]) * p;
			w = (m[1][2] - m[2][1]) * p;
		}
		else if ((m[1][1] >= m[0][0]) && (m[1][1] >= m[2][2]))
		{
			float s = ::sqrt(m[1][1]-m[2][2]-m[0][0]+1.0f);
//			assert(s);
			float p = 0.5f / s;
			x = (m[1][0] + m[0][1]) * p;
			y = s * 0.5f;
			z = (m[1][2] + m[2][1]) * p;
			w = (m[2][0] - m[0][2]) * p;
		}
		else if ((m[2][2] >= m[0][0]) && (m[2][2] >= m[1][1]))
		{
			float s = ::sqrt(m[2][2]-m[0][0]-m[1][1]+1.0f);
//			assert(s);
			float p = 0.5f / s;
			x = (m[2][0] + m[0][2]) * p;
			y = (m[2][1] + m[1][2]) * p;
			z = s * 0.5f;
			w = (m[0][1] - m[1][0]) * p;
		}
		else
		{
			x = 0.0f;
			y = 0.0f;
			z = 0.0f;
			w = 1.0f;
		}
	}
};

// TEMP

class Quatf32T
{
public:
	Quatf32 q;
	Vec3f32 t;

public:
	void SetIdentity()
	{
		// TODO
	}
};

//

inline Quatf32 exp(const Vec3f32& v) 
{
	float length = ::sqrt(v.x*v.x + v.y*v.y + v.z*v.z);
	float s = ::sin(length);
	float c = ::cos(length);
	if (length > 0.0f)
		return Quatf32(c, v.x*s/length, v.y*s/length, v.z*s/length);
	return Quatf32(0.0f, 0.0f, 0.0f, 1.0f);
}

inline int sgn(int x) { return (x>>31)+((x-1)>>31)+1; }

#endif // HEADER_Math
