#include <stdio.h>
#include <fcntl.h>
#include <io.h>

#include <Windows.h>
#include <gl/GL.h>
#include <LiveCreate.h>

#include "../Library/Library.h"

#include "Scene.h"

#include "Camera.h"

CCamera camera;

//

Vec3f32 move(0.0f, 0.0f, 0.0f);

bool bRotate = false;
POINT anchor;

void OnKeyDown(char key)
{
	switch (key)
	{
	case 'W': move.y = +1.0f; break;
	case 'S': move.y = -1.0f; break;
	case 'A': move.x = -1.0f; break;
	case 'D': move.x = +1.0f; break;
	}
}

void OnKeyUp(char key)
{
	switch (key)
	{
	case 'W': if (move.y > 0.0f) move.y = 0.0f; break;
	case 'S': if (move.y < 0.0f) move.y = 0.0f; break;
	case 'A': if (move.x < 0.0f) move.x = 0.0f; break;
	case 'D': if (move.x > 0.0f) move.x = 0.0f; break;
	}
}

void OnMouseDown(char key)
{
	switch (key)
	{
	case 1: bRotate = true; break;
	}
}

void OnMouseUp(char key)
{
	switch (key)
	{
	case 1: bRotate = false; break;
	}
}

void ProcessInput()
{
	POINT cursor;
	::GetCursorPos(&cursor);

	if (bRotate)
	{
		::camera.angles.x -= float(cursor.y - ::anchor.y) * 0.01f;
		::camera.angles.y -= float(cursor.x - ::anchor.x) * 0.01f;
		::camera.Update();

		::SetCursorPos(::anchor.x, ::anchor.y);
	}
	else
	{
		::anchor = cursor;
	}

	Vec3f32 move = ::move * ::camera.orientation;
	camera.position += move * 0.1f;
}

//

static const Mat33f32 zForwardToYForward(
   1.0f, 0.0f, 0.0f,
   0.0f, 0.0f, 1.0f,
   0.0f, 1.0f, 0.0f);


inline Mat44f32 ProjectionPerspective(float left, float top, float right, float bottom, float zNear, float zFar)
{
	return Mat44f32(::zForwardToYForward.Transposed()) * Mat44f32(
		2.0f*zNear / (right - left), 0.0f, 0.0f, 0.0f,
		0.0f, 2.0f*zNear / (top-bottom), 0.0f, 0.0f,
		0.0f, 0.0f, zFar / (zFar-zNear), 1.0f,
		0.0f, 0.0f, - zFar*zNear / (zFar-zNear), 0.0f);
}

inline Mat44f32 ProjectionPerspectiveVerticalFOV(float fov, float aspect, float zNear, float zFar)
{
	
	float y = ::tanf(fov * 0.5f) * zNear;
	float x = y * aspect;
	return ::ProjectionPerspective(-x, +y, +x, -y, zNear, zFar);
}

//

class CSystem :
	public ILCSystem
{
	// ILCSystem
public:
	void Log(LCString message, ...)
	{
		char buffer[2048];

		va_list arguments;
		va_start(arguments, message);
		::vsnprintf(buffer, sizeof(buffer), message, arguments);
		va_end(arguments);

		::printf(buffer);
	}
} _system;

//

CScene scene;
LCLiveCreate* pLiveCreate = NULL;

//

bool glInitialize(HWND hWindow);
void glShutdown();
void glPresent();

HWND AppInitialize(HINSTANCE hInstance);
void AppShutdown();
bool AppRun();

//

HMODULE liveCreateModule = NULL;

LCLiveCreate* LiveCreateLoad(const char* name, ILCScene& scene)
{
	::liveCreateModule = ::LoadLibrary(name);
	if (!::liveCreateModule)
	{
		::printf("Unable to load \"%s\"\n", name);
		return NULL;
	}

	FLiveCreate LiveCreate = (FLiveCreate)
		::GetProcAddress(::liveCreateModule, "LiveCreate");
	if (!LiveCreate)
	{
		::printf("Unable to retrieve LiveCreate from \"%s\"\n", name);
		::FreeLibrary(::liveCreateModule);
		return NULL;
	}

	LCLiveCreate* pLiveCreate = LiveCreate(&::_system);
	if (!pLiveCreate)
	{
		::printf("Unable to instance LiveCreate from \"%s\"", name);
		::FreeLibrary(::liveCreateModule);
		return NULL;
	}

	::printf("Loaded LiveCreate: \"%s\"\n", name);

	if (!pLiveCreate->Connect("192.168.6.81", scene))
		return false;

	return pLiveCreate;
}

void LiveCreateClose(LCLiveCreate* pLiveCreate)
{
	if (pLiveCreate)
		pLiveCreate->Release();

	if (::liveCreateModule)
		::FreeLibrary(::liveCreateModule);
	::liveCreateModule = NULL;
}

//

void glSphereLine(const Vec3f32& position, const Vec3f32& scale, LCu32 sections = 16)
{
	LCf32 step = PIf * 2.0f / LCf32(sections);
	LCf32 angle = step;

	LCf32 x = 1.0f, y = 0.0f;
	for (LCu32 i=0; i<sections; ++i)
	{
		LCf32 c = ::cos(angle);
		LCf32 s = ::sin(angle);

		::glVertex3f(position.x, position.y + y*scale.y, position.z + x*scale.x);
		::glVertex3f(position.x, position.y + s*scale.y, position.z + c*scale.x);
		::glVertex3f(position.x + x*scale.x, position.y, position.z + y*scale.y);
		::glVertex3f(position.x + c*scale.x, position.y, position.z + s*scale.y);
		::glVertex3f(position.x + x*scale.x, position.y + y*scale.y, position.z);
		::glVertex3f(position.x + c*scale.x, position.y + s*scale.y, position.z);

		x = c;
		y = s;

		angle += step;
	}
}

LCi32 GetIndexByName(const LCSet& set, LCString name)
{
	LCu32 nodeCount = set.GetNodeCount();
	for (LCu32 i=0; i<nodeCount; ++i)
	{
		if (stricmp(set.GetNodeName(i), name))
			continue;

		return i;
	}

	return -1;
}

void glDrawGrid(float width, float height, int sections = 16)
{
	float xStep = width / float(sections);
	float yStep = height / float(sections);

	float xPosition = width * -0.5f;
	float yPosition = height * -0.5f;

	float xOffset = xPosition;
	float yOffset = yPosition;

	::glBegin(GL_LINES);
	::glColor4f(0.0f, 0.0f, 0.0f, 1.0f);
	for (int i=0; i<sections+1; ++i)
	{
		::glVertex3f(xPosition, yOffset, 0.0f);
		::glVertex3f(xPosition+width, yOffset, 0.0f);

		::glVertex3f(xOffset, yPosition, 0.0f);
		::glVertex3f(xOffset, yPosition+height, 0.0f);

		xOffset += xStep;
		yOffset += yStep;
	}
	::glBegin(GL_LINES);
}

void GeneratePrismBoneVertices(const Vec3f32& origin, const Mat33f32& orientationScale, float length, Vec3f32 vertices[6])
{
	vertices[0] = origin;
	vertices[1] = origin + Vec3f32(-0.5f, +0.5f, +1.0f) * orientationScale;
	vertices[2] = origin + Vec3f32(+0.5f, +0.5f, +1.0f) * orientationScale;
	vertices[3] = origin + Vec3f32(+0.5f, -0.5f, +1.0f) * orientationScale;
	vertices[4] = origin + Vec3f32(-0.5f, -0.5f, +1.0f) * orientationScale;
	vertices[5] = origin + orientationScale[2].Normalized() * length;
}

Vec3f32 ComputeTriangleNormal(const Vec3f32& v0, const Vec3f32& v1, const Vec3f32& v2)
{
	return (v0 - v1).Cross(v2 - v1).Normalize() * ::zForwardToYForward;
}

void glDrawPrismBone(const Vec3f32& origin, const Mat33f32& orientationScale, float length)
{
	Vec3f32 vertices[6];
	::GeneratePrismBoneVertices(origin, orientationScale, length, vertices);

	::glBegin(GL_TRIANGLES);

	::glColor4f(0.75f, 1.0f, 0.75f, 1.0f);
	::glNormal3fv((GLfloat*)&ComputeTriangleNormal(vertices[0], vertices[1], vertices[2]));
	::glVertex3fv((GLfloat*)&vertices[0]); ::glVertex3fv((GLfloat*)&vertices[1]); ::glVertex3fv((GLfloat*)&vertices[2]);
	::glNormal3fv((GLfloat*)&ComputeTriangleNormal(vertices[2], vertices[1], vertices[5]));
	::glVertex3fv((GLfloat*)&vertices[2]); ::glVertex3fv((GLfloat*)&vertices[1]); ::glVertex3fv((GLfloat*)&vertices[5]);

	::glNormal3fv((GLfloat*)&ComputeTriangleNormal(vertices[0], vertices[3], vertices[4]));
	::glVertex3fv((GLfloat*)&vertices[0]); ::glVertex3fv((GLfloat*)&vertices[3]); ::glVertex3fv((GLfloat*)&vertices[4]);
	::glNormal3fv((GLfloat*)&ComputeTriangleNormal(vertices[4], vertices[3], vertices[5]));
	::glVertex3fv((GLfloat*)&vertices[4]); ::glVertex3fv((GLfloat*)&vertices[3]); ::glVertex3fv((GLfloat*)&vertices[5]);

	::glColor4f(1.0f, 0.75f, 0.75f, 1.0f);
	::glNormal3fv((GLfloat*)&ComputeTriangleNormal(vertices[0], vertices[2], vertices[3]));
	::glVertex3fv((GLfloat*)&vertices[0]); ::glVertex3fv((GLfloat*)&vertices[2]); ::glVertex3fv((GLfloat*)&vertices[3]);
	::glNormal3fv((GLfloat*)&ComputeTriangleNormal(vertices[3], vertices[2], vertices[5]));
	::glVertex3fv((GLfloat*)&vertices[3]); ::glVertex3fv((GLfloat*)&vertices[2]); ::glVertex3fv((GLfloat*)&vertices[5]);

	::glNormal3fv((GLfloat*)&ComputeTriangleNormal(vertices[0], vertices[4], vertices[1]));
	::glVertex3fv((GLfloat*)&vertices[0]); ::glVertex3fv((GLfloat*)&vertices[4]); ::glVertex3fv((GLfloat*)&vertices[1]);
	::glNormal3fv((GLfloat*)&ComputeTriangleNormal(vertices[1], vertices[4], vertices[5]));
	::glVertex3fv((GLfloat*)&vertices[1]); ::glVertex3fv((GLfloat*)&vertices[4]); ::glVertex3fv((GLfloat*)&vertices[5]);
	::glEnd();
}

void glDrawFrame(const Vec3f32& origin, const Mat33f32& orientation, float length)
{
	::glBegin(GL_LINES);
	::glColor3f(1.0f, 0.0f, 0.0f);
	::glVertex3fv((GLfloat*)&origin);
	::glVertex3fv((GLfloat*)&(origin + orientation[0] * length));
	::glColor3f(0.0f, 1.0f, 0.0f);
	::glVertex3fv((GLfloat*)&origin);
	::glVertex3fv((GLfloat*)&(origin + orientation[1] * length));
	::glColor3f(0.0f, 0.0f, 1.0f);
	::glVertex3fv((GLfloat*)&origin);
	::glVertex3fv((GLfloat*)&(origin + orientation[2] * length));
	::glEnd();
}

void Draw()
{
	::glClearColor(0.25f, 0.25f, 0.25f, 1.0f);
	::glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	::glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);

	::glEnable(GL_CULL_FACE);
	::glCullFace(GL_BACK);
	::glEnable(GL_DEPTH_TEST);
	::glDepthFunc(GL_LEQUAL);

	::glShadeModel(GL_SMOOTH);

	::glMatrixMode(GL_PROJECTION);
	::glLoadIdentity();
	LCf32 size = 3.0f;
	LCf32 aspect = 640.0f / 480.0f;
	::glLoadMatrixf((GLfloat*)&::ProjectionPerspectiveVerticalFOV(PIf*0.4f, aspect, 0.1f, 100.0f));

	Mat44f32 matrix = ::camera.GetViewMatrix();
	::glMatrixMode(GL_MODELVIEW);
	::glLoadIdentity();
	::glLoadMatrixf((GLfloat*)&matrix);

	::glLightfv(GL_LIGHT0, GL_SPOT_DIRECTION, (GLfloat*)&::camera.orientation[1]);

	//

//	::glEnable(GL_LIGHTING);
//	::glEnable(GL_LIGHT0);

//	::glEnable(GL_COLOR_MATERIAL);
//	::glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);

	size_t actorCount = scene.GetActorCount();
	for (size_t i=0; i<actorCount; ++i)
	{
		CActor& actor = scene.GetActor(i);
		const LCSet* pSet = actor.GetSet();
		if (!pSet)
			continue;

		LCu32 nodeCount = LCu32(pSet->GetNodeCount());
		for (LCu32 j=0; j<nodeCount; ++j)
		{
			CActor::SNode* pNode = actor.GetNodeByName(pSet->GetNodeName(j));
			if (!pNode)
				continue;

//			::glDrawFrame(*(Vec3f32*)&pNode->position, ((Quatf32*)&pNode->orientation)->ToMatrix(), 0.1f);

			LCi32 parent = pSet->GetNodeParent(j);
			if (parent < 0)
				continue;

			CActor::SNode* pParent = actor.GetNodeByName(pSet->GetNodeName(parent));
			if (!pParent)
				continue;

			Vec3f32 n = *(Vec3f32*)&pNode->position;
			Vec3f32 p = *(Vec3f32*)&pParent->position;

			Vec3f32 direction = n - p;
			float length = direction.Length();
			if (length > 0.0f)
				direction *= 1.0f / length; // Normalize.

			Mat33f32 orientation = Mat33f32::LookAt(direction);
//			orientation = ((Quatf32*)&pParent->orientation)->ToMatrix();
			glDrawPrismBone(p, orientation * Mat33f32::Scaling(0.05f), length);

			n = p + orientation[0] * length;

//			::glDrawFrame(p, orientation, 0.1f);

/*
			::glBegin(GL_LINES);
			::glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
			::glVertex3fv((GLfloat*)&p);
			::glColor4f(0.5f, 0.5f, 0.5f, 1.0f);
			::glVertex3fv((GLfloat*)&n);
			::glEnd();*/
		}
	}

	::glDisable(GL_LIGHTING);
	::glDisable(GL_LIGHT0);

	//


	::glDrawGrid(10.0f, 10.0f);

	::glBegin(GL_LINES);
	::glColor4f(1.0f, 0.0f, 0.0f, 1.0f);
	::glVertex3f(0.0f, 0.0f, 0.0f);
	::glVertex3f(1.0f, 0.0f, 0.0f);
	::glColor4f(0.0f, 1.0f, 0.0f, 1.0f);
	::glVertex3f(0.0f, 0.0f, 0.0f);
	::glVertex3f(0.0f, 1.0f, 0.0f);
	::glColor4f(0.0f, 0.0f, 1.0f, 1.0f);
	::glVertex3f(0.0f, 0.0f, 0.0f);
	::glVertex3f(0.0f, 0.0f, 1.0f);
	::glEnd();

	::glBegin(GL_LINES);
	for (size_t i=0; i<actorCount; ++i)
	{
		CActor& actor = scene.GetActor(i);
		const LCSet* pSet = actor.GetSet();

		LCu32 nodeCount = LCu32(actor.GetNodeCount());
		for (LCu32 j=0; j<nodeCount; ++j)
		{
			CActor::SNode* pNode = actor.GetNodeByIndex(j);
			if (!pNode)
				continue;

			if (pSet)
				::glColor4f(1.0f, 0.5f, 0.5f, 1.0f);
			else
				::glColor4f(0.5f, 0.5f, 1.0f, 1.0f);
		
			::glSphereLine(*(const Vec3f32*)&pNode->position, Vec3f32(0.0125f, 0.0125f, 0.0125f));
		}
/*
		nodeCount = pSet ? pSet->GetNodeCount() : 0;
		for (size_t j=0; j<nodeCount; ++j)
		{
			CActor::SNode* pNode = actor.GetNodeByName(pSet->GetNodeName(j));
			if (!pNode)
				continue;

			LCi32 parent = pSet->GetNodeParent(j);
			if (parent < 0)
				continue;

			CActor::SNode* pParent = actor.GetNodeByName(pSet->GetNodeName(parent));
			if (!pParent)
				continue;

			::glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
			::glVertex3fv((GLfloat*)&pParent->position);
			::glColor4f(0.5f, 0.5f, 0.5f, 1.0f);
			::glVertex3fv((GLfloat*)&pNode->position);
		}*/
	}
	::glEnd();

	::glPresent();
}

bool Initialize(HINSTANCE hInstance)
{
	camera.position = Vec3f32(0.0f, -5.0f, +1.0f);

	HWND hWindow = ::AppInitialize(hInstance);
	if (!hWindow)
		return false;

	if (!::glInitialize(hWindow))
		return false;

	::pLiveCreate = ::LiveCreateLoad("VICON", ::scene);
//	::pLiveCreate = ::LiveCreateLoad("MAYA", ::scene);
	if (!::pLiveCreate)
		return false;

	return true;
}

void Shutdown()
{
	::LiveCreateClose(::pLiveCreate);

	::glShutdown();
	::AppShutdown();
}

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE, LPSTR pParameters, int)
{
	if (!::Initialize(hInstance))
	{
		::Shutdown();
		return NULL;
	}

	while (::AppRun())
	{
		if (::pLiveCreate)
			::pLiveCreate->Update();

		::ProcessInput();
		::Draw();
	}

	::Shutdown();
	return NULL;
}
