#include "Client.h"
#include "Biped.h"

#include <algorithm>

//

extern ILCSystem* pSystem;

//

bool ParseInformationString(const std::string& source, std::string& type, std::string& name)
{
	size_t start = source.find('<');
	if (start == std::string::npos)
		return false;

	size_t end = source.find('>');
	if (end == std::string::npos)
		return false;

	type = source.substr(start, (end+1)-start);
	name = source.substr(0, start);
	return true;
}

/*

  CClient

*/

CClient::CClient()
{
}

CClient::~CClient()
{
	Disconnect();	
}

//

bool CClient::Connect(const char* address)
{
	Disconnect();

	if (!m_connection.Connect(address, 800))
		return false;

	//

	if (!Request(eRequest_Info))
		return false;

	std::vector<std::string> information;
	if (!ReceiveInformation(information))
		return false;

	if (!ParseInformation(information))
		return false;

	//

	if (!m_connection.MakeNonBlocking())
		return false;

	return true;
}

bool CClient::IsConnected()
{
	return m_connection.IsConnected();
}

void CClient::Disconnect()
{
	m_connection.Disconnect();

	m_actors.clear();

	m_joints.clear();
	m_markers.clear();	

	m_dataCount = NULL;
	m_data.clear();
}

void CClient::Update(ILCScene& scene)
{
	if (!m_connection.IsConnected())
		return;

	if (!Request(eRequest_Data))
		return;

	if (!ReceiveData())
		return;

	if (!ParseData(scene))
		return;
}

//

bool CClient::SendPacketHeader(ERequest eRequest)
{
	static const LCu32 request = 0;
	const LCu32 header[2] = { eRequest, request };
	if (!m_connection.Send(header, sizeof(header)))
		return false;

	return true;
}

bool CClient::ReceivePacketHeader(ERequest eFromRequest)
{
	LCu32 header[2];
	if (!m_connection.Receive(header, sizeof(header)))
		return false;

	static const LCu32 reply = 1;
	if (header[0] != eFromRequest ||
		header[1] != reply)
	{
		::pSystem->Log("ViconClient: Received invalid packet header.\n");
		Disconnect();
		return false;
	}

	return true;
}

bool CClient::Request(ERequest eRequest)
{
	if (!SendPacketHeader(eRequest))
		return false;
	if (!ReceivePacketHeader(eRequest))
		return false;
	return true;
}

//

bool CClient::ReceiveInformation(std::vector<std::string>& information)
{
	LCu32 count;
	if (!m_connection.Receive(&count, sizeof(LCu32)))
		return false;

	information.clear();
	information.resize(count);
	for (LCu32 i=0; i<count; ++i)
	{
		LCu32 length;
		if (!m_connection.Receive(&length, sizeof(LCu32)))
			return false;

		char c[255];
		if (!m_connection.Receive(c, length))
			return false;

		information[i].append(c, length);
	}

	return true;
}

CClient::SJoint& CClient::AddJoint(const char* name)
{
	size_t count = m_joints.size();
	for (size_t i=0; i<count; ++i)
	{
		if (::stricmp(m_joints[i].name.c_str(), name))
			continue;

		return m_joints[i];
	}

	m_joints.push_back(SJoint(name));
	return m_joints.back();
}

CClient::SMarker& CClient::AddMarker(const char* name)
{
	size_t count = m_markers.size();
	for (size_t i=0; i<count; ++i)
	{
		if (::stricmp(m_markers[i].name.c_str(), name))
			continue;

		return m_markers[i];
	}

	m_markers.push_back(SMarker(name));
	return m_markers.back();
}

bool CClient::AddChannel(const char* type, const char* name, LCu32 index)
{
	if		(::stricmp(type, "<A-X>") == 0)	AddJoint(name).ax = index;
	else if	(::stricmp(type, "<A-Y>") == 0)	AddJoint(name).ay = index;
	else if	(::stricmp(type, "<A-Z>") == 0)	AddJoint(name).az = index;
	else if	(::stricmp(type, "<T-X>") == 0)	AddJoint(name).tx = index;
	else if	(::stricmp(type, "<T-Y>") == 0)	AddJoint(name).ty = index;
	else if	(::stricmp(type, "<T-Z>") == 0)	AddJoint(name).tz = index;
	else if	(::stricmp(type, "<P-X>") == 0)	AddMarker(name).x = index;
	else if	(::stricmp(type, "<P-Y>") == 0)	AddMarker(name).y = index;
	else if	(::stricmp(type, "<P-Z>") == 0)	AddMarker(name).z = index;
	else return false;
	return true;
}

bool CClient::ParseInformation(const std::vector<std::string>& info)
{
	m_dataCount = LCu32(info.size());
	for (std::vector<std::string>::const_iterator p=info.begin(); p!=info.end(); ++p)
	{
		std::string type, name;
		if (!::ParseInformationString(*p, type, name))
		{
			::pSystem->Log("ViconClient: Received invalid information.\n");
			Disconnect();
			return false;
		}

		ClampToLastSpace(name);

		LCu32 offset = LCu32(p - info.begin());
		AddChannel(type.c_str(), name.c_str(), offset);
	}

	m_data.resize(info.size());
	return true;
}

//

bool CClient::ReceiveData()
{
	LCu32 dataCount;
	if (!m_connection.Receive(&dataCount, sizeof(LCu32)))
		return false;

	if (dataCount != m_dataCount)
	{
		::pSystem->Log("ViconClient: Received invalid data.\n");
		return false;
	}

	for (std::vector<LCf64>::iterator i=m_data.begin(); i!=m_data.end(); i++)
	{
		if (!m_connection.Receive(&*i, sizeof(LCf64)))
			return false;
	}

	return true;
}

bool CClient::ParseData(ILCScene& scene)
{
	for (std::vector<SJoint>::iterator i=m_joints.begin(); i!=m_joints.end(); i++)
	{
		const char* name = i->name.c_str();
		size_t offset = ::strstr(name, ":") - name;

		std::string actorName;
		actorName.append(name, offset);

		name += offset+1;

		ILCActor* pActor = GetActor(scene, actorName.c_str(), &::viconBipedSet);
		if (!pActor)
			continue;

		// TODO: Pass correct time.
		pActor->SetOrientation(name, 0, ::QuaternionExponent(
			LCf32(m_data[i->ax]) * 0.5f, LCf32(m_data[i->ay]) * 0.5f, LCf32(m_data[i->az]) * 0.5f));
		pActor->SetPosition(name, 0, *(const LCVec3f32*)&Vec3f32(
			LCf32(m_data[i->tx]), LCf32(m_data[i->ty]), LCf32(m_data[i->tz])));
	}

	for (std::vector<SMarker>::iterator i=m_markers.begin(); i!=m_markers.end(); i++)
	{
		const char* name = i->name.c_str();
		size_t offset = ::strstr(name, ":") - name;

		std::string actorName;
		actorName.append(name, offset);

		name += offset+1;

		ILCActor* pActor = GetActor(scene, actorName.c_str(),& ::viconBipedSet);
		if (!pActor)
			continue;

		// TODO: Pass correct time.
		pActor->SetPosition(name, 0, *(const LCVec3f32*)&Vec3f32(
			Vec3f32(LCf32(m_data[i->x]), LCf32(m_data[i->y]), LCf32(m_data[i->z]))));
	}

	return true;
}

//

ILCActor* CClient::GetActor(ILCScene& scene, const char* name, LCSet* pSet)
{
	ILCActor* pActor = m_actors[name];
	if (pActor)
		return pActor;

	pActor = scene.CreateActor(name, pSet);
	m_actors[name] = pActor;
	return pActor;
}
