#include "stdafx.h"
#include "ResourceCompilerHelper.h"
#include "Export/ExportStatusWindow.h"
#include "Export/IExportContext.h"
#include "LogFile.h"
#include "PakSystem.h"
#include "PathHelpers.h"
#include "CryExportTool.h"
#include "ISettings.h"
#include "EngineSettingsManager.h"
#include "DebugCallStack.h"

struct RCPathMissingErrorTag {};
typedef Exception<RCPathMissingErrorTag> RCPathMissingError;

namespace
{
	void CheckResourceCompiler(ILogFile* log)
	{
		log->Log(ILogFile::MessageSeverity_Debug, "Checking resource compiler.");
		CResourceCompilerHelper rcHelper;
		switch (rcHelper.CallResourceCompiler(0, 0, 0, false, false, CResourceCompilerHelper::eRcExePath_registry, true, true))
		{
		case CResourceCompilerHelper::eRcCallResult_notFound:
			log->Log(ILogFile::MessageSeverity_Error, "Resource compiler not found.");
			throw RCPathMissingError("RC cannot be executed.");
		case CResourceCompilerHelper::eRcCallResult_error:
			log->Log(ILogFile::MessageSeverity_Error, "Resource compiler error.");
			throw RCPathMissingError("RC cannot be executed.");
		case CResourceCompilerHelper::eRcCallResult_success:
			log->Log(ILogFile::MessageSeverity_Debug, "Resource compiler check passed.");
			break;
		default:
			log->Log(ILogFile::MessageSeverity_Error, "Resource compiler unknown failure.");
			throw RCPathMissingError("RC cannot be executed.");
		}
	}
}

class MBCryExportContext : public IExportContext
{
public:
	MBCryExportContext(ExportStatusWindow& statusWindow, ILogFile* logFile, IPakSystem* pakSystem, ISettings* settings)
		: m_statusWindow(statusWindow), m_logFile(logFile), m_pakSystem(pakSystem), m_settings(settings) {}

	// IExportContext
	virtual void SetProgress(float progress);
	virtual void SetCurrentTask(const std::string& id);
	virtual void Log(MessageSeverity severity, const char* message);
	virtual IPakSystem* GetPakSystem();
	virtual ISettings* GetSettings();
	virtual void GetRootPath(char* buffer, int bufferSize);

private:
	ExportStatusWindow& m_statusWindow;
	ILogFile* m_logFile;
	IPakSystem* m_pakSystem;
	ISettings* m_settings;
};

class MBCryExportSettings : public ISettings
{
public:
	MBCryExportSettings();

	// ISettings
	virtual bool GetSettingString(char* buffer, int bufferSize, const char* key);
	virtual bool GetSettingInt(int& value, const char* key);

private:
	std::string GetModuleName() const;

	CEngineSettingsManager m_engineSettings;
};

FBToolImplementation(CryExportTool);
FBRegisterTool(
	CryExportTool,
	"CryEngine 2 Exporter",
	"Exports animations to CryEngine 2",
	FB_DEFAULT_SDK_ICON);	// Icon filename (default=Open Reality icon)

bool CryExportTool::FBCreate()
{
	CreateUI();
	return true;
}

void CryExportTool::FBDestroy()
{
}

void CryExportTool::CreateUI()
{
	int lW = 200;
	int lS = 5;
	int lH = 25;

	// Add regions.
	AddRegion("LabelExport", "LabelExport",
		lS,   kFBAttachLeft,    "",                 1.0,
		lS,   kFBAttachTop,     "",                 1.0,
		lW,   kFBAttachNone,		NULL,               1.0,
		lH,   kFBAttachNone,		NULL,               1.0);
	AddRegion("ButtonExport", "ButtonExport",
		0,    kFBAttachLeft,    "LabelExport",      1.0,
		2*lS, kFBAttachBottom,  "LabelExport",      1.0,
		0,    kFBAttachWidth,   "LabelExport",      1.0,
		0,    kFBAttachHeight,  "LabelExport",      1.0 );
	AddRegion("ButtonConfigRC", "ButtonConfigRC",
		0,    kFBAttachLeft,    "ButtonExport",     1.0,
		lS,   kFBAttachBottom,  "ButtonExport",     1.0,
		0,    kFBAttachWidth,   "ButtonExport",     1.0,
		0,    kFBAttachHeight,  "ButtonExport",     1.0 );

	// Assign regions.
	SetControl("LabelExport", m_LabelExport);
	SetControl("ButtonExport", m_ButtonExport);
	SetControl("ButtonConfigRC", m_ButtonConfigRC);

	int windowWidth = 226, windowHeight = 140;
	StartSize[0] = windowWidth; StartSize[1] = windowHeight;
	MinSize[0] = windowWidth; MinSize[1] = windowHeight;
	MaxSize[0] = windowWidth; MaxSize[1] = windowHeight;

	// Configure UI elements.
	m_LabelExport.Caption = "Exporting";
	m_LabelExport.Justify = kFBTextJustifyCenter;
	m_ButtonExport.Caption = "Export Animations";
	m_ButtonConfigRC.Caption = "Configure Resource Compiler";

	// Add callbacks.
	m_ButtonExport.OnClick.Add(this, (FBCallback)&CryExportTool::OnExportClicked);
	m_ButtonConfigRC.OnClick.Add(this, (FBCallback)&CryExportTool::OnConfigRCClicked);
}

namespace
{
	void ConfigureRC()
	{
		CResourceCompilerHelper rcHelper;
		rcHelper.ResourceCompilerUI(0);
	}
}

void CryExportTool::OnExportClicked(HISender pSender, HKEvent pEvent)
{
	bool retry = true;
	bool rcNeedsConfig = false;

	// In some cases we might need to try exporting several times, so loop until we decide to stop.
	for (int attemptCount = 2; attemptCount > 0; --attemptCount)
	{
		rcNeedsConfig = false;

		try
		{
			// Create the log file.
			std::string logPath;
			{
				std::string const filename = m_source.GetDCCFileName();
				std::string const dir = m_source.GetExportDirectory();
				if (filename.empty() || dir.empty())
				{
					throw IExportContext::NeedSaveError("Scene must be saved before exporting.");
				}
				logPath = PathHelpers::Join(dir, PathHelpers::RemoveExtension(PathHelpers::GetFilename(filename)) + ".exportlog");
			}
			LogFile logFile(logPath);

			// Check whether the RC exists and can be found.
			CheckResourceCompiler(&logFile);

			std::vector<std::pair<std::string, std::string> > tasks;
			tasks.push_back(std::make_pair("dae", "Generating intermediate file (*.dae)."));
			tasks.push_back(std::make_pair("rc", "Compiling intermediate file to engine format."));
			tasks.push_back(std::make_pair("compress", "Compressing animations in engine format."));
			ExportStatusWindow window(400, 600, tasks);

			PakSystem pakSystem;
			MBCryExportSettings settings;
			MBCryExportContext context(window, &logFile, &pakSystem, &settings);
			if (!logFile.IsOpen())
			{
				char buffer[2048];
				sprintf(buffer, "Unable to open log file for writing: \"%s\"", logPath.c_str());
				context.Log(IExportContext::MessageSeverity_Warning, buffer);
			}

			// Handle any crashes.
			DebugCallStack::ErrorHandlerScope errorHandlerScope(&logFile);

			// Perform the export.
			m_writer.Export(&m_source, &context);

			attemptCount = 0; // If we got here, make no more attempts to export.
		}
		catch (IExportContext::NeedSaveError e)
		{
			FBMessageBox(
				"Exporting Problem",
				"This scene has never been saved. The exporter needs the scene to be saved to disk so that\n"
				"it can figure out where to export files to.\n"
				"\n"
				"Please save the scene and try exporting again.",
				"OK");

			attemptCount = 0; // No point retrying, will fail again.
		}
		catch (RCPathMissingError e)
		{
			FBMessageBox(
				"Exporting Problem",
				"The Exporter does not know where to find the Resource Compiler. This is stored in the bin32\n"
				"directory under the main build path.\n"
				"\n",
				"OK");
			if (attemptCount > 1)
				rcNeedsConfig = true;
		}

		// If exporting failed because the RC path was not configured, let the user do so now.
		if (rcNeedsConfig)
			ConfigureRC();
	}
}

void CryExportTool::OnConfigRCClicked(HISender pSender, HKEvent pEvent)
{
	ConfigureRC();
}

void MBCryExportContext::SetProgress(float progress)
{
	m_statusWindow.SetProgress(progress);
}

void MBCryExportContext::SetCurrentTask(const std::string& id)
{
	m_statusWindow.SetCurrentTask(id);
}

void MBCryExportContext::Log(MessageSeverity severity, const char* message)
{
	m_statusWindow.Log((ExportStatusWindow::MessageSeverity) severity, message);
	if (m_logFile)
		m_logFile->Log((ILogFile::MessageSeverity)severity, message);
}

IPakSystem* MBCryExportContext::GetPakSystem()
{
	return m_pakSystem;
}

ISettings* MBCryExportContext::GetSettings()
{
	return m_settings;
}

void MBCryExportContext::GetRootPath(char* buffer, int bufferSize)
{
	CEngineSettingsManager settingsManager;

	std::string rootPath = settingsManager.GetRootPath();

	if (rootPath.empty())
		settingsManager.CallSettingsDialog(0); // TODO: Get top-level window.

	strncpy(buffer, rootPath.c_str(), bufferSize);
}

MBCryExportSettings::MBCryExportSettings()
: m_engineSettings(GetModuleName().c_str())
{
}

bool MBCryExportSettings::GetSettingString(char* buffer, int bufferSize, const char* key)
{
	string value;
	if (m_engineSettings.GetModuleSpecificEntry(key, value))
	{
		strncpy(buffer, value.c_str(), bufferSize);
		return true;
	}

	return false;
}

bool MBCryExportSettings::GetSettingInt(int& value, const char* key)
{
	return (m_engineSettings.GetModuleSpecificEntry(key, value));
}

std::string MBCryExportSettings::GetModuleName() const
{
	return "MBCryExport";
}
