#include "StdAfx.h"
#include "ExportSource.h"
#include "Export/ISkeletonData.h"
#include "Export/IAnimationData.h"
#include "Export/IExportContext.h"
#include "Export/IModelData.h"
#include "Export/IGeometryData.h"
#include "Export/IGeometryFileData.h"
#include "Export/AnimationData.h"
#include "PathHelpers.h"
#include <vector>
#include <cmath>
#include <algorithm>

#ifdef new
#undef new
#endif

#define NOT_USE_CRY_MEMORY_MANAGER
#include <Cry_Math.h>

ExportSource::ExportSource()
{
}

void ExportSource::GetMetaData(SExportMetaData& metaData) const
{
	strcpy( metaData.authoring_tool,"CryENGINE Motion Builder COLLADA Exporter" );

	metaData.source_data[0] = 0;
	{
		const char* const name = (const char*)m_application.FBXFileName;
		if (name && name[0])
		{
			_snprintf_s(metaData.source_data, sizeof(metaData.source_data), sizeof(metaData.source_data) - 1, "file://%s", name);
		}
	}

	{
		char szNameBuffer[sizeof(metaData.author)];
		memset(szNameBuffer, 0, sizeof(szNameBuffer));
		DWORD dwSize = sizeof(szNameBuffer);
		if (!::GetUserNameA(szNameBuffer, &dwSize))
		{
			szNameBuffer[0] = 0;
		}
		strcpy(metaData.author, szNameBuffer);
	}

	// Implementation note: Looks like there is no any legal way to get MotionBuilder's
	// unit and axis settings by using Motion Builder's SDK, so we're just 
	// using MotionBuilder's default settings below:
	metaData.up_axis = SExportMetaData::Y_UP;
	metaData.fMeterUnit = 0.01f;  
}

std::string ExportSource::GetDCCFileName() const
{
	return std::string((const char*)m_application.FBXFileName);
}

std::string ExportSource::GetExportDirectory() const
{
	return PathHelpers::GetDirectory(GetDCCFileName());
}

namespace
{
	void GetPreRotation(Matrix33& preRotation, HFBModel pModel, HFBModel pParent)
	{
		Matrix44 worldRotation(IDENTITY), worldParentRotationInverse(IDENTITY), localRotationInverse(IDENTITY);
		{
			FBMatrix m;
			pModel->GetMatrix(m, kModelRotation, true);
			worldRotation = Matrix44(*reinterpret_cast<Matrix44_f64*>(&m));
		}
		{
			FBMatrix m;
			pModel->GetMatrix(m, kModelRotation, false);
			localRotationInverse = Matrix44(*reinterpret_cast<Matrix44_f64*>(&m));
			localRotationInverse.Invert();
		}
		if (pParent)
		{
			FBMatrix m;
			pParent->GetMatrix(m, kModelRotation, true);
			worldParentRotationInverse = Matrix44(*reinterpret_cast<Matrix44_f64*>(&m));
			worldParentRotationInverse.Invert();
		}

		preRotation = Matrix33((worldParentRotationInverse * localRotationInverse) * worldRotation);
	}

	std::string GetModelName(const std::string& modelName)
	{
		// We use the namespace of the MB model as the name of the exported model. The namespace
		// is encoded in the Name property using : to separate namespaces in the path.
		int namespaceEndPos = int(modelName.find_last_of(":"));
		int namespaceLength = (namespaceEndPos != std::string::npos ? namespaceEndPos : 0);
		std::string nameSpace = modelName.substr(0, namespaceLength);
		std::replace(nameSpace.begin(), nameSpace.end(), ':', '_');
		return (nameSpace.empty() ? std::string("default") : nameSpace);
	}

	std::pair<std::string, std::string> GetPropertyFromModel(HFBModel model)
	{
		// Look for a name of the form '<key>=<value>'.
		std::string name = model->ShortName;
		std::string::size_type equalPos = name.find('=');
		std::string::size_type keyStart = name.find_first_not_of(" \t");
		std::string::size_type keyEnd = (equalPos != std::string::npos && equalPos > 0 ? name.find_last_not_of(" \t", equalPos - 1) : std::string::npos);
		std::string::size_type valueStart = (equalPos != std::string::npos && equalPos < name.size() ? name.find_first_not_of(" \t", equalPos + 1) : std::string::npos);
		std::string::size_type valueEnd = (name.find_last_not_of(" \t"));

		std::pair<std::string, std::string> result;
		if (keyStart != std::string::npos && keyEnd != std::string::npos && valueStart != std::string::npos && valueEnd != std::string::npos)
		{
			result.first = name.substr(keyStart, keyEnd - keyStart + 1);
			result.second = name.substr(valueStart, valueEnd - valueStart + 1);
		}

		return result;
	}

	void RecurseHierarchyAndListBones(ISkeletonData* skeletonData, int parentIndex, HFBModel pModel, HFBModel pParent)
	{
		// Add the model to the model data.
		FBString name = pModel->ShortName;
		int modelIndex = skeletonData->AddBone(pModel, (const char*)name, parentIndex);

		const double* translationValues = FBVector3d(pModel->Translation).mValue;
		const double* rotationValues = FBVector3d(pModel->Rotation).mValue;
		const double* scalingValues = FBVector3d(pModel->Scaling).mValue;
		float translationArray[3] = {float(translationValues[0]), float(translationValues[1]), float(translationValues[2])};
		float rotationArray[3] = {float(rotationValues[0]), float(rotationValues[1]), float(rotationValues[2])};
		float scalingArray[3] = {float(scalingValues[0]), float(scalingValues[1]), float(scalingValues[2])};

		// Apply the prerotation value to the rotation.
		Matrix33 preRotation;
		GetPreRotation(preRotation, pModel, pParent);
		Matrix33 rotation = Matrix33(Ang3(rotationArray[0], rotationArray[1], rotationArray[2]));
		rotation = preRotation.GetInverted() * rotation;
		Ang3 rotationAngles(rotation);
		rotationArray[0] = rotationAngles.x;
		rotationArray[1] = rotationAngles.y;
		rotationArray[2] = rotationAngles.z;

		skeletonData->SetTranslation(modelIndex, translationArray);
		skeletonData->SetRotation(modelIndex, rotationArray);
		skeletonData->SetScale(modelIndex, scalingArray);

		// Recurse through children
		for (int i=0;i<pModel->Children.GetCount();i++)
			RecurseHierarchyAndListBones(skeletonData, modelIndex, pModel->Children[i], pModel);
	}

	void RecurseHierarchyFindModelsAndListTheirBones(ISkeletonData* skeletonData, const std::string& modelName, HFBModel pModel)
	{
		// Check whether this node represents a property.
		std::pair<std::string, std::string> property = GetPropertyFromModel(pModel);
		if (!property.first.empty())
			return;

		// Check whether this model is a skeleton or a skeleton root object.
		bool isBone = (strcmp(pModel->ClassName(), "FBModelSkeleton") == 0 || strcmp(pModel->ClassName(), "FBModelRoot") == 0);
		bool isCorrectModel = (_stricmp(GetModelName((const char*)pModel->Name).c_str(), modelName.c_str()) == 0);
		if (isBone && isCorrectModel)
		{
			RecurseHierarchyAndListBones(skeletonData, -1, pModel, 0);

			// Don't recurse to children, since we are only interested in the roots of the skeletons.
		}
		else
		{
			// Recurse through children
			for (int i=0;i<pModel->Children.GetCount();i++)
				RecurseHierarchyFindModelsAndListTheirBones(skeletonData, modelName, pModel->Children[i]);
		}
	}

	/*
	// Sokov: I've commented out collecting properties below, because I think that 
	// we need no any properties for GeometryFileData node except filetype==CRY_FILE_TYPE_CAF,
	// and DoNotMerge==true (these properties were already remembered by us).

	void RecurseHierarchyAndExtractProperties(IGeometryFileData* geometryFileData, int const geometryFileIndex, HFBModel model)
	{
		// Check whether this node represents a property.
		std::pair<std::string, std::string> property = GetPropertyFromModel(model);
		std::string key = property.first;
		std::string value = property.second;
		if (!key.empty())
		{
			geometryFileData->AddProperties(geometryFileIndex, (key + "=" + value));
		}

		// Recurse through children
		for (int i=0;i<model->Children.GetCount();i++)
		{
			RecurseHierarchyAndExtractProperties(geometryFileData, geometryFileIndex, model->Children[i]);
		}
	}
	*/

	void RecurseHierarchyAndFindGeometryFiles(IGeometryFileData* geometryFileData, HFBModel pModel, HFBModel pParent)
	{
		// Check whether this model is a skeleton or a skeleton root object.
		if (strcmp(pModel->ClassName(), "FBModelSkeleton") == 0 || strcmp(pModel->ClassName(), "FBModelRoot") == 0)
		{
			// Add the model, using the namespace as the name.
			int geometryFileIndex;
			{
				const std::string name = GetModelName((const char*)pModel->Name).c_str();
				const int count = geometryFileData->GetGeometryFileCount();
				for (geometryFileIndex = 0; geometryFileIndex < count; ++geometryFileIndex)
				{
					if (_stricmp(name.c_str(), geometryFileData->GetGeometryFileName(geometryFileIndex)) == 0)
					{
						break;
					}
				}
				if (geometryFileIndex >= count)
				{
					IGeometryFileData::SProperties props;
					props.filetypeInt = CRY_FILE_TYPE_CAF;
					props.bDoNotMerge = true;
					geometryFileIndex = geometryFileData->AddGeometryFile(pModel, name.c_str(), props);
				}
			}

			// Sokov: I've commented out collecting properties below, because I think that 
			// we need no any properties for GeometryFileData node except filetype==CRY_FILE_TYPE_CAF,
			// and DoNotMerge==true (these properties were already remembered by us).
			/*
			// Extract any properties from child nodes in the hierarchy.
			RecurseHierarchyAndExtractProperties(geometryFileData, geometryFileIndex, pModel);
			*/

			// Don't recurse to children, since we are only interested in the roots of the skeletons.
		}
		else
		{
			// Recurse through children
			for (int i=0;i<pModel->Children.GetCount();i++)
			{
				RecurseHierarchyAndFindGeometryFiles(geometryFileData, pModel->Children[i], pModel);
			}
		}
	}
}

void ExportSource::ReadGeometryFiles(IExportContext* context, IGeometryFileData* geometryFileData)
{
	RecurseHierarchyAndFindGeometryFiles(geometryFileData, m_system.SceneRootModel, 0);
}

bool ExportSource::ReadMaterials(IExportContext* context, const IGeometryFileData* const geometryFileData, IMaterialData* materialData)
{
	// Materials are irrelevant for MB.
	return true;
}

void ExportSource::ReadModels(const IGeometryFileData* geometryFileData, int geometryFileIndex, IModelData* modelData)
{
	HFBModel model = static_cast<HFBModel>(geometryFileData->GetGeometryFileHandle(geometryFileIndex));

	std::string modelName = geometryFileData->GetGeometryFileName(geometryFileIndex);
	modelData->AddModel(model, modelName.c_str(), -1, false, SHelperData(), "");
}

void ExportSource::ReadSkinning(IExportContext* context, ISkinningData* skinningData, const IModelData* const modelData, int modelIndex, ISkeletonData* skeletonData)
{
	// No skinning info required for MB.
}

bool ExportSource::ReadSkeleton(const IGeometryFileData* const geometryFileData, int geometryFileIndex, const IModelData* const modelData, int modelIndex, const IMaterialData* materialData, ISkeletonData* skeletonData)
{
	RecurseHierarchyFindModelsAndListTheirBones(skeletonData, geometryFileData->GetGeometryFileName(geometryFileIndex), m_system.SceneRootModel);
	return true;
}

int ExportSource::GetAnimationCount() const
{
	return m_system.Takes.GetCount();
}

std::string ExportSource::GetAnimationName(const IGeometryFileData* geometryFileData, int geometryFileIndex, int animationIndex) const
{
	HFBModel model = static_cast<HFBModel>(geometryFileData->GetGeometryFileHandle(geometryFileIndex));
	std::string modelName = geometryFileData->GetGeometryFileName(geometryFileIndex);
	std::string modelPrefix = (modelName.empty() ? "" : modelName + "_");
	const char* name = (const char*)m_system.Takes[animationIndex]->Name;
	std::string safeAnimationName = modelPrefix + (name ? name : "UNNAMED");
	std::replace(safeAnimationName.begin(), safeAnimationName.end(), ' ', '_');
	return safeAnimationName;
}

void ExportSource::GetAnimationTimeSpan(float& start, float& stop, int animationIndex) const
{
	FBTimeSpan timespan = FBTimeSpan(m_system.Takes[animationIndex]->LocalTimeSpan);
	start = float(timespan.GetStart().GetSecondDouble());
	stop = float(timespan.GetStop().GetSecondDouble());
}

namespace
{
	struct BoneEntry
	{
		BoneEntry(HFBModel model, int parentIndex, bool isRoot): model(model), parentIndex(parentIndex), isRoot(isRoot) {}

		HFBModel model;
		int parentIndex;
		bool isRoot;
	};
	void RecurseAndGetBoneList(std::vector<BoneEntry>& bones, HFBModel model, int parentIndex, bool isRoot)
	{
		int index = int(bones.size());
		bones.push_back(BoneEntry(model, parentIndex, isRoot));

		// Recurse through children
		for (int i=0;i<model->Children.GetCount();i++)
			RecurseAndGetBoneList(bones, model->Children[i], index, false);
	}

	void RecurseFindModelsAndGetBoneList(std::vector<BoneEntry>& bones, HFBModel model, const std::string& modelName)
	{
		// Check whether this model is a skeleton or a skeleton root object.
		bool isBone = (strcmp(model->ClassName(), "FBModelSkeleton") == 0 || strcmp(model->ClassName(), "FBModelRoot") == 0);
		bool isCorrectModel = (_stricmp(GetModelName((const char*)model->Name).c_str(), modelName.c_str()) == 0);
		if (isBone && isCorrectModel)
		{
			RecurseAndGetBoneList(bones, model, -1, true);

			// Don't recurse to children, since we are only interested in the roots of the skeletons.
		}
		else
		{
			// Recurse through children
			for (int i=0;i<model->Children.GetCount();i++)
				RecurseFindModelsAndGetBoneList(bones, model->Children[i], modelName);
		}
	}

	void GetBoneList(std::vector<BoneEntry>& bones, HFBModel model, const std::string& modelName)
	{
		RecurseFindModelsAndGetBoneList(bones, model, modelName);
	}

	enum AnimationNodeCategory
	{
		AnimationNodeCategory_None = -1,
		AnimationNodeCategory_Translation = 0,
		AnimationNodeCategory_Rotation,
		AnimationNodeCategory_Scaling
	};
	struct AnimationUserNameInfo
	{
	public:
		AnimationUserNameInfo(std::string name, AnimationNodeCategory category, int parameterIndex)
			: name(name), category(category), parameterIndex(parameterIndex) {}
		std::string name;
		AnimationNodeCategory category;
		int parameterIndex;
	};
	AnimationUserNameInfo animationUserNameInfos[] = {
		AnimationUserNameInfo("Lcl Translation", AnimationNodeCategory_Translation, -1),
		AnimationUserNameInfo("Lcl Rotation", AnimationNodeCategory_Rotation, -1),
		AnimationUserNameInfo("Lcl Scaling", AnimationNodeCategory_Scaling, -1),
		AnimationUserNameInfo("X", AnimationNodeCategory_None, 0),
		AnimationUserNameInfo("Y", AnimationNodeCategory_None, 1),
		AnimationUserNameInfo("Z", AnimationNodeCategory_None, 2)};
	enum {NUM_USER_NAMES = sizeof(animationUserNameInfos) / sizeof(animationUserNameInfos[0])};
	void SampleCurvesRecurse(HFBAnimationNode animationNode, int startFrame, int frameCount, float fps, AnimationNodeCategory parentCategory, float* translation, int translationStride, float* rotation, int rotationStride, float* scaling, int scalingStride)
	{
		FBFCurve* curve = animationNode->FCurve;

		float* arrays[] = {translation, rotation, scaling};
		float strides[] = {translationStride, rotationStride, scalingStride};

		// Read the name of the node to figure out what it controls.
		string name = animationNode->Name;
		AnimationNodeCategory category = AnimationNodeCategory_None;
		int parameterIndex = -1;
		for (int i = 0; i < NUM_USER_NAMES; ++i)
		{
			if (animationUserNameInfos[i].name == name)
			{
				category = animationUserNameInfos[i].category;
				parameterIndex = animationUserNameInfos[i].parameterIndex;
			}
		}

		// If the name told us that this node controls a valid value, evaluate the curve at the current time and
		// store the value.
		if (curve && parentCategory != AnimationNodeCategory_None && parameterIndex >= 0)
		{
			for (int frameIndex = startFrame; frameIndex < startFrame + frameCount; ++frameIndex)
			{
				float frameTime = float(frameIndex) / fps;
				FBTime fbTime;
				fbTime.SetSecondDouble(frameTime);
				float value = curve->Evaluate(fbTime);
				int frameSlotStart = strides[parentCategory] * (frameIndex - startFrame);
				arrays[parentCategory][frameSlotStart + parameterIndex] = value;
			}
		}

		// Recurse to the children.
		for (int i = 0, count = animationNode->Nodes.GetCount(); i < count; ++i)
			SampleCurvesRecurse(animationNode->Nodes[i], startFrame, frameCount, fps, category, translation, translationStride, rotation, rotationStride, scaling, scalingStride);
	}

	void CheckModelForExportKeyRecurse(IExportContext* context, HFBModel model, HFBAnimationNode animationNode, bool& key)
	{
		// Check whether the property is an "export" custom property, and if it is,
		// whether it has the value true.
		string name = animationNode->Name;
		if (_stricmp(name.c_str(), "export") == 0)
		{
			FBFCurve* curve = animationNode->FCurve;
			FBTime fbTime;
			fbTime.SetSecondDouble(0.0f);
			float const value = curve->Evaluate(fbTime);
			if (value > 0.1f)
				key = true;
			Log(context, IExportContext::MessageSeverity_Debug, "'export' string found for model \"%s\", in animation node \"%s\". Value is %f", (const char*)model->Name, name.c_str(), value);
		}

		// Recurse to the children.
		for (int i = 0, count = animationNode->Nodes.GetCount(); i < count; ++i)
			CheckModelForExportKeyRecurse(context, model, animationNode->Nodes[i], key);
	}

	bool CheckModelForExportKey(IExportContext* context, HFBModel model)
	{
		bool key = false;
		CheckModelForExportKeyRecurse(context, model, model->AnimationNode, key);

		if (key)
			Log(context, IExportContext::MessageSeverity_Debug, "'Export' key found in node \"%s\"", (const char*)model->Name);

		return key;
	}
}

void ExportSource::ReadAnimationFlags(IExportContext* context, IAnimationData* animationData, const IGeometryFileData* const geometryFileData, const IModelData* modelData, int modelIndex, const ISkeletonData* skeletonData, int animationIndex) const
{
	// Get a list of all the bones we want to export.
	std::vector<BoneEntry> bones;
	GetBoneList(bones, m_system.SceneRootModel, modelData->GetModelName(modelIndex));

	// Look up the bones in the skeleton data.
	std::vector<int> boneIndices(bones.size());
	for (int modelIndex = 0, modelCount = int(bones.size()); modelIndex < modelCount; ++modelIndex)
		boneIndices[modelIndex] = skeletonData->FindBone(bones[modelIndex].model->ShortName);

	// Set this take to be the current one.
	m_system.CurrentTake = m_system.Takes[animationIndex];

	// Look for bones that are marked as the root of an export hierarchy.
	Log(context, IExportContext::MessageSeverity_Debug, "Looking for bones that are marked as the root of an export hierarchy (scanning %i nodes)", int(bones.size()));
	std::vector<bool> exportRootFlags(bones.size());
	int rootCount = 0;
	for (int modelIndex = 0, modelCount = int(bones.size()); modelIndex < modelCount; ++modelIndex)
	{
		HFBModel model = bones[modelIndex].model;
		exportRootFlags[modelIndex] = CheckModelForExportKey(context, model);
		if (exportRootFlags[modelIndex])
			++rootCount;
	}
	Log(context, IExportContext::MessageSeverity_Debug, "%i of %i nodes are marked for export", rootCount, int(bones.size()));

	// If there are no roots, default to exporting the entire hierarchy.
	if (rootCount == 0)
	{
		for (int modelIndex = 0, modelCount = int(bones.size()); modelIndex < modelCount; ++modelIndex)
			exportRootFlags[modelIndex] = true;
	}

	for (int modelIndex = 0, modelCount = int(bones.size()); modelIndex < modelCount; ++modelIndex)
	{
		// Check whether this bone is one of the ones we are meant to export.
		int modelDataIndex = boneIndices[modelIndex];

		if (modelDataIndex >= 0)
		{
			// Check whether this bone is part of the hierarchies that we are supposed to export.
			// We do this by checking whether the node was marked for export, or any of its ancestors.
			bool shouldExport = false;
			for (int index = modelIndex; index >= 0; index = bones[index].parentIndex)
				shouldExport = shouldExport || exportRootFlags[index];

			// Store the information in the animation data.
			unsigned modelFlags = 0;
			modelFlags |= (!shouldExport ? IAnimationData::ModelFlags_NoExport : 0);
			animationData->SetModelFlags(modelDataIndex, modelFlags);
		}
	}
}

IAnimationData * ExportSource::ReadAnimation(IExportContext* context, const IGeometryFileData* const geometryFileData, const IModelData* modelData, int modelIndex, const ISkeletonData* skeletonData, int animationIndex, float fps) const
{
	IAnimationData *animationData = new AnimationData(skeletonData->GetBoneCount(), fps, 
		float(FBTimeSpan(m_system.Takes[animationIndex]->LocalTimeSpan).GetStart().GetSecondDouble()));

	// Get a list of all the bones we want to export.
	std::vector<BoneEntry> bones;
	GetBoneList(bones, m_system.SceneRootModel, modelData->GetModelName(modelIndex));

	// Look up the bones in the skeleton data.
	std::vector<int> boneIndices(bones.size());
	for (int boneIndex = 0, boneCount = int(bones.size()); boneIndex < boneCount; ++boneIndex)
		boneIndices[boneIndex] = skeletonData->FindBone(bones[boneIndex].model->ShortName);

	// Set this take to be the current one.
	m_system.CurrentTake = m_system.Takes[animationIndex];

	// Sample the take at 30 fps.
	int firstFrame = int(FBTimeSpan(m_system.CurrentTake->LocalTimeSpan).GetStart().GetMilliSeconds() * fps / 1000.0f + 0.5f);
	int lastFrame = int(FBTimeSpan(m_system.CurrentTake->LocalTimeSpan).GetStop().GetMilliSeconds() * fps / 1000.0f + 0.5f);
	int frameCount = lastFrame - firstFrame + 1;
	animationData->SetFrameCount(frameCount);

	// For each bone sample the transform at each frame.
	std::vector<float> translationArray(frameCount * 3);
	std::fill_n(translationArray.begin(), frameCount * 3, 0.0f);
	std::vector<float> rotationArray(frameCount * 3);
	std::fill_n(rotationArray.begin(), frameCount * 3, 0.0f);
	std::vector<float> scalingArray(frameCount * 3);
	std::fill_n(scalingArray.begin(), frameCount * 3, 1.0f);
	for (int boneIndex = 0, boneCount = int(bones.size()); boneIndex < boneCount; ++boneIndex)
	{
		HFBModel boneModel = bones[boneIndex].model;
		int parentIndex = bones[boneIndex].parentIndex;
		HFBModel parent = (parentIndex != -1 ? bones[parentIndex].model : 0);

		// Check whether this bone is one of the ones we are meant to export.
		int skeletonDataIndex = boneIndices[boneIndex];

		if (skeletonDataIndex >= 0)
		{
			// Sample the transform for this bone for each frame.
			SampleCurvesRecurse(boneModel->AnimationNode, firstFrame, frameCount, fps, AnimationNodeCategory_None, &translationArray[0], 3, &rotationArray[0], 3, &scalingArray[0], 3);

			// Apply the prerotation for this node.
			Matrix33 preRotation;
			GetPreRotation(preRotation, boneModel, parent);
			for (int frameIndex = firstFrame; frameIndex <= lastFrame; ++frameIndex)
			{
				float* rotationValues = &rotationArray[(frameIndex - firstFrame) * 3];
				Matrix33 rotation = Matrix33(Ang3(DEG2RAD(rotationValues[0]), DEG2RAD(rotationValues[1]), DEG2RAD(rotationValues[2])));
				rotation = preRotation.GetInverted() * rotation;
				Ang3 rotationAngles(rotation);
				rotationValues[0] = RAD2DEG(rotationAngles.x);
				rotationValues[1] = RAD2DEG(rotationAngles.y);
				rotationValues[2] = RAD2DEG(rotationAngles.z);
			}

			// HACK: If the node is the root of the skeleton (ie not the root of the scene) rotate about X
			// to convert to CryEngine coordinate system.
			// TODO: Do this in RC, and specify input coordinate system in COLLADA.

			// It turns out that there are two nodes that have Parent==0 - the master root node (not the scene root)
			// and the children of the scene root (as weird as that sounds). As the master root node will not appear
			// in the list of nodes we have assembled (since we started at the scene root), Parent==0 must mean
			// that the node is a child of the scene root, and should have the values rotated.

			if (!boneModel->Parent)
			{
				//// Rotate 90 degrees to handle coordinate system.
				//Matrix33 coordConv(Ang3(DEG2RAD(90.0f), 0.0f, 0.0f));
				//for (int frameIndex = firstFrame; frameIndex <= lastFrame; ++frameIndex)
				//{
				//	std::swap(translationArray[(frameIndex - firstFrame) * 3 + 1], translationArray[(frameIndex - firstFrame) * 3 + 2]);
				//	translationArray[(frameIndex - firstFrame) * 3 + 1] *= -1;

				//	float* rotationValues = &rotationArray[(frameIndex - firstFrame) * 3];
				//	Matrix33 rotation = Matrix33(Ang3(DEG2RAD(rotationValues[0]), DEG2RAD(rotationValues[1]), DEG2RAD(rotationValues[2])));
				//	rotation = coordConv * rotation;
				//	Ang3 rotationAngles(rotation);
				//	rotationValues[0] = RAD2DEG(rotationAngles.x);
				//	rotationValues[1] = RAD2DEG(rotationAngles.y);
				//	rotationValues[2] = RAD2DEG(rotationAngles.z);
				//}

				//// Force animation to begin at 0,0,0.
				//float animOrigin[3];
				//for (int axis = 0; axis < 3; ++axis)
				//	animOrigin[axis] = translationArray[axis];
				//for (int frameIndex = firstFrame; frameIndex <= lastFrame; ++frameIndex)
				//{
				//	for (int axis = 0; axis < 3; ++axis)
				//		translationArray[(frameIndex - firstFrame) * 3 + axis] -= animOrigin[axis];
				//}
			}

			// Round scale values to nearest 0.001.
			for (int frameIndex = firstFrame; frameIndex <= lastFrame; ++frameIndex)
			{
				for (int axis = 0; axis < 3; ++axis)
				{
					float& value = scalingArray[(frameIndex - firstFrame) * 3 + axis];
					value = std::floor(value * 1000.0f + 0.5f) / 1000.0f;
				}
			}

			// Store the information in the animation data.
			for (int frameIndex = firstFrame; frameIndex <= lastFrame; ++frameIndex)
			{
				animationData->SetFrameData(skeletonDataIndex, frameIndex - firstFrame,
					&translationArray[3 * (frameIndex - firstFrame)],
					&rotationArray[3 * (frameIndex - firstFrame)],
					&scalingArray[3 * (frameIndex - firstFrame)]);
			}
		}
	}

	return animationData;
}

bool ExportSource::ReadGeometry(IExportContext* context, IGeometryData* geometry, const IModelData* const modelData, const IMaterialData* const materialData, int modelIndex)
{
	return false;
}

bool ExportSource::ReadGeometryMaterialData(IExportContext* context, IGeometryMaterialData* geometryMaterialData, const IModelData* const modelData, const IMaterialData* const materialData, int modelIndex) const
{
	return false;
}

bool ExportSource::ReadBoneGeometry(IExportContext* context, IGeometryData* geometry, ISkeletonData* skeletonData, int boneIndex, const IMaterialData* const materialData)
{
	return false;
}

bool ExportSource::ReadBoneGeometryMaterialData(IExportContext* context, IGeometryMaterialData* geometryMaterialData, ISkeletonData* skeletonData, int boneIndex, const IMaterialData* const materialData) const
{
	return false;
}

void ExportSource::ReadMorphs(IExportContext* context, IMorphData* morphData, const IModelData* const modelData, int modelIndex)
{
}

bool ExportSource::ReadMorphGeometry(IExportContext* context, IGeometryData* geometry, const IModelData* const modelData, int modelIndex, const IMorphData* const morphData, int morphIndex, const IMaterialData* materialData)
{
	assert(0);
	return false;
}

bool ExportSource::HasValidPosController(const IModelData* modelData, int modelIndex) const
{
	assert(0);
	return false;
}

bool ExportSource::HasValidRotController(const IModelData* modelData, int modelIndex) const
{
	assert(0);
	return false;
}

bool ExportSource::HasValidSclController(const IModelData* modelData, int modelIndex) const
{
	assert(0);
	return false;
}