#############################################################################
## Crytek Source File
## Copyright (C) 2006, Crytek Studios
##
## Creator: Sascha Demetrio
## Date: Jul 31, 2006
## Description: GNU-make based build system
#############################################################################

# Common include of the GNU make-based build system.
#
# This file should be included by the top level makefile of the project.
# Common.mk expects the following make variables to be set:
#
# CODE_ROOT
#	The root directory of the source code.  This is the directory
#	containing the CryEngine sub-directory.
#
# BUILD_ROOT
#	The output root directory of the build process.  All files are
#	generated within the build root directory.
#
#	Note: This makefile will override the BUILD_ROOT variable in case of a
#	mixed build of a module (see below).
#
# REPO_ROOT
#	The root directory of the repository.
#
# PS3_SDK_ROOT
#	The root folder where the PS3-SDK will be stored/read from
#
# CYGWIN_BASEDIR (optional, only when building on Cygwin)
#	The Windows path to the '/base' directory mounted in the Cygwin
#	environment.  Typically this is the common base directory of
#	$CODE_ROOT, $BUILD_ROOT, and $REPO_ROOT (in Windows DRIVE:PATH
#	notation).
#
# STAGES
#	A whitespace separated list of build stages to be performed by the
#	build system.  The make system will process all listed stages in order
#	- the current stage is exposed through the exported variable STAGE.
#
#	If STAGES is undefined but STAGE is defined, then STAGES is set to
#	contain only the named STAGE.
#
# ARCH
#	The target architecure.  This is a combination of an OS- or system
#	name and a processor type.  System names are Win32, Win64, Linux, PS3.
#	Processor type names are x86 (Intel 586 compatbible), amd64, cell.
#	The list of supported architecures can be obtained by listing the *.mk
#	files in the Arch.mk subdirectory.
#
# SOLUTION
#	The base name of the solution.  The build system expects a 'solution
#	makefile' in the Code/Solutions directory.  The solution makefile
#	defines the valid targets of the solution as well as build-order and
#	link dependencies between them.  (Note: These are not code level
#	dependencies.)
#
# TARGETS
#	The (symbolic) names of the build targets of the entire build process
#	(i.e. the combination of all stages).  Valid targets are defined
#	through the solution makefile.  The build system executes the named
#	targets in the specified order.  The current target is exposed through
#	the exported variable TARGET.
#
#	If TARGETS is undefined but TARGET is defined, then TARGETS is set to
#	contain only the named TARGET.
#
# This file provides the following make targets (we'll call them
# build-commands):
#
# default
#	The default build target for target and stage.  Typically this implies
#	the generation of all result files of the target and stage.
#
# clean
#	Remove all output files of the target and stage.  This includes all
#	files pulled out of the repository.
#
# post
#	Perform only the post action of the selected stages (use only if you
#	know what you are doing).  The post action does _not_ depend on the
#	default action of the stage!
#
# store (depends on 'default')
#	Store the results of the build process into the repository.
#
# The build system output may be controlled through some MKOPTION_* variables:
#
# MKOPTION_VERBOSE
#	Verbose operation of the make system.  This causes all build process
#	commands executed to be echoed.  If this is 0 or undefined, then a
#	less verbose output format is used.
#
# MKOPTION_MSVC
#	Convert output messages to MSVC format (required for intergration into
#	Visual Studio).
#
# MKOPTION_COMPACT
#	Compact output messages (similar to MSVC style).
#
# MKOPTION_NOWRAPPER
#	If this is 1, then the build wrapper is skipped when executing build
#	process commands.

# Archtecture specific makefiles ############################################
#
# All architecure specific makefiles live in '$(MAKE_ROOT)/Arch'.  The
# base-name if these makefile matches the '$(ARCH)' variable.  Architecture
# specific makefiles are included after the configuration makefiles
# (Config.mk) have been included and before any other makefile is included.

# Solution makefiles ########################################################
#
# All solution makefiles live in '$(SOLUTION_DIR)' (which may or may not
# reside withing '$(MAKE_ROOT)').  A solution makefile defines the following:
# - A list of projects.
# - (Optional) A default project.
# - Project dependencies.  These are link order dependencies, not code
#   dependencies.
#
# PROJECTS
#	A whitespace separated list of project names.  If some projects are
#	not relevant for all architecures, then the solution makefile must set
#	the PROJECTS variable depending on the selected architecure.  For
#	that, it may use the following variables: ARCH, ARCH_OS, ARCH_CPU.
#
# (ProjectName)_DEPS
#	A whitespace separated list of dependencies for the project
#	(ProjectName).
#
# (ProjectName)_CODE
#	The code directory for the project.  If this is undefined, then the
#	project is a 'phony' project, i.e. project that just causes the listed
#	dependencies to be built.
#
#	The specifed path _must_ be relative to the $(CODE_ROOT) directory.
#	The build system expects a project specific makefile with the name
#	Project.mk there.
#
# (ProjectName)_XFLAGS
#	Project specific extended flags (typically optimization flags).  These
#	settings override the architecure specific $(XFLAGS).
#
# (ProjectName)_XFLAGS_(SourceName)
#	Per-file override for the extended flags.  (SourceName) is the full
#	name of the source file, relative to the (ProjectName)_CODE, with all
#	dashes, slashes, and dots converted to underscores.  Example: The
#	XFLAGS for the source file Foo/Bar.cpp are specified in the variable
#	(ProjectName)_XFLAGS_Foo_Bar_cpp.  It is possible to specify extended
#	flags for a precompiled header file.
#
# DEFAULT_PROJECT
#	Optional.  The name of the default project.
#
# The projects specified in a solution makefile correspond to the valid
# targets for the TARGETS variable.  If both TARGET and TARGETS are undefined
# and a default project is defined for the selected solution, then the default
# project is chosen as the target.
#
# Note: It is valid to specify (ProjectName)_DEPS and/or (ProjectName)_CODE
# for a project and then skip that project in the PROJECTS variable.

# Stage Makefiles ###########################################################
#
# Stage specific makefiles live in the Stage subdirectory.  A stage specific
# makefile defines the following targets:
#
# _prepare
#	Phony target for setting up the compilation stage.
#
# _command_$(COMMAND)
#	Phony target executing the requested build command.

#############################################################################

# Check the directory layout variables.
ifeq ($(strip $(CODE_ROOT)),)
 $(error No CODE_ROOT specified)
endif
ifndef _CODE_CONFIG_INCLUDED
 -include $(CODE_ROOT)/Config.mk
_CODE_CONFIG_INCLUDED := 1
endif
ifeq ($(strip $(BUILD_ROOT)),)
 $(error No BUILD_ROOT specified)
endif
ifndef _BUILD_CONFIG_INCLUDED
 -include $(BUILD_ROOT)/Config.mk
 _BUILD_CONFIG_INCLUDED := 1
endif
ifeq ($(strip $(REPO_ROOT)),)
 $(error No REPO_ROOT specified)
endif
PS3_SDK_ROOT := $(CODE_ROOT)/$(PS3-SDK-FOLDER)

# to support build with the eval sdk and not having the generated spu sources
# use an external folder for the PS3-SDK in case of building with the PS3-eval-SDK
ifeq ($(BUILD_WITH_PS3_SDK),1)
	PS3_SPU_ROOT := $(PS3_SDK_ROOT)/Lib
else
	PS3_SPU_ROOT := $(BUILD_ROOT)/Lib
endif
export PS3_SDK_ROOT
export PS3_SPU_ROOT
export CODE_ROOT
export BUILD_ROOT
export REPO_ROOT
ifdef CYGWIN_BASEDIR
 override CYGWIN_BASEDIR := $(subst \,/,$(CYGWIN_BASEDIR))
 export CYGWIN_BASEDIR
endif

# Define the MAKE_ROOT variable.
#
# This is the root directory for all shared makefiles.  This variable is
# exported.  All shared makefiles should be accessed relative to this
# variable.
ifeq ($(origin MAKE_ROOT),undefined)
 MAKE_ROOT := $(CODE_ROOT)/Tools/Make
endif
export MAKE_ROOT

# Why, Why, Why, Why is the below block also defined verbatim in
# Default.mk?  To prevent the reader from being the same poor schmuck
# that the author of this comment is, here the nice hint that if you
# change the below block, please also change the values in Default.mk
# accordingly.
ifeq ($(HOST_SYSTEM),Windows)
 # For now HOST_SYSTEM=Windows maps to HOST_SYSTEM=MingW
 override HOST_SYSTEM := MingW
endif
ifeq ($(HOST_SYSTEM),Linux)
 HOST_BINDIR := host-linux
endif
ifeq ($(HOST_SYSTEM),Cygwin)
 HOST_BINDIR := host-win32
endif
ifeq ($(HOST_SYSTEM),MingW)
 HOST_BINDIR := host-win32
endif
ifndef HOST_BINDIR
 $(error set HOST_SYSTEM to 'Linux', 'Windows', or 'Cygwin' or 'MingW')
endif
export HOST_BINDIR
MAKE_TOOLS_BINDIR = $(MAKE_ROOT)/Tools/$(HOST_BINDIR)
export MAKE_TOOLS_BINDIR

# Mixed build overrides.
ifndef TARGET_BUILD_ROOT
 export TARGET_BUILD_ROOT := $(BUILD_ROOT)
endif
ifeq ($(MIXED_BUILD),1)
 EFFECTIVE_CONFIG_NAME := $(VS_CONFIG_$(VS_CONFIG))
 ifeq ($(EFFECTIVE_CONFIG_NAME),)
  $(error No mixed configuration mapping for build configuration $(VS_CONFIG))
 endif
 EFFECTIVE_BUILD_NAME := $(ARCH)-$(EFFECTIVE_CONFIG_NAME)
 EFFECTIVE_BUILD_ROOT := \
   $(patsubst %/$(BUILD_NAME),%/$(EFFECTIVE_BUILD_NAME),$(BUILD_ROOT))
 #
 # IMPLEMENTATION NOTE:
 #
 # The BUILD_ROOT variable will be changed to EFFECTIVE_BUILD_ROOT after the
 # Project.mk file has been included.  The original BUILD_ROOT is available
 # through the (exported) variable TARGET_BUILD_ROOT.
 #
 # It is critical, that no references to $(BUILD_ROOT) between here and the
 # point where BUILD_ROOT is set to $(EFFECTIVE_BUILD_ROOT) are expanded when
 # the makefile is read - i.e. BUILD_ROOT may _not_ be used in so called
 # "simply expanded variables" (see GNU make manual for details).
else
 EFFECTIVE_BUILD_NAME := $(BUILD_NAME)
 EFFECTIVE_BUILD_ROOT := $(BUILD_ROOT)
endif

ifeq ($(MKOPTION_PS3_MIXED_BUILD),1)
 MIXED_BUILD_NAME := $(ARCH)-mixed
 MIXED_BUILD_ROOT := \
   $(patsubst %/$(BUILD_NAME),%/$(MIXED_BUILD_NAME),$(BUILD_ROOT))
endif

ifndef BUILD_TEMP
 BUILD_TEMP = $(BUILD_ROOT)/Temp
endif

# Directory containing the solution makefiles.
ifeq ($(origin SOLUTION_DIR),undefined)
 SOLUTION_DIR := $(CODE_ROOT)/Solutions
endif
export SOLUTION_DIR

# Check if the architecure is supported.  An architecure is supported if and
# only if the associated architecure include file is present and can be
# included without indicating an error.  The include statement for the
# architecure file is below the solution file include clause.
ifeq ($(strip $(ARCH)),)
 $(error No target architecure specified)
endif # !ARCH
_arch := $(subst -, ,$(strip $(ARCH)))
ifneq ($(words $(_arch)),2)
 $(error Bad format in architecure string $(ARCH))
endif
ARCH_OS := $(word 1, $(_arch))
ARCH_CPU := $(word 2, $(_arch))
export ARCH_OS
export ARCH_CPU

#ifndef MKOPTION_DISTCC
# ifndef MKOPTION_INCREDIBUILD
#  MKOPTION_DISTCC = 1
#  export MKOPTION_DISTCC
# endif
#endif

ifndef OPTION_DEVIRTUALIZE
 OPTION_DEVIRTUALIZE = 1
 export OPTION_DEVIRTUALIZE
endif

# default settings for some options needed for PS3-BuildEvaluation SDK
ifndef BUILD_WITH_PS3_SDK
	BUILD_WITH_PS3_SDK = 0	
	export BUILD_WITH_PS3_SDK
endif

ifndef GENERATE_PS3_SDK
	GENERATE_PS3_SDK = 0
	export GENERATE_PS3_SDK
endif

ifndef MKOPTION_OPTIMIZE_SHELL
 MKOPTION_OPTIMIZE_SHELL = 1
 ifeq ($(HOST_SYSTEM),Linux)
  MKOPTION_OPTIMIZE_SHELL = 0
 endif
endif
ifeq ($(MKOPTION_VERBOSE),1)
 MKOPTION_OPTIMIZE_SHELL = 0
endif

ifeq ($(MKOPTION_CRYCGPROFILE),1)
 MKOPTION_OPTIMIZE_SHELL = 0
endif

# MKOPTION_OPTIMIZE_SHELL is currently broken due to shell quoting
# issues. It will be fixed as soons as the underlying cygwin shell has
# been completely removed. 
ifeq ($(MKOPTION_OPTIMIZE_SHELL),1)
 MKOPTION_OPTIMIZE_SHELL = 0
endif 

# Check if the requested solution is supported (i.e. if a matching solution
# makefile exists).
ifeq ($(strip $(SOLUTION)),)
 $(error No solution specified)
endif # !SOLUTION
include $(SOLUTION_DIR)/$(SOLUTION).mk
ifndef TARGET
 ifndef TARGETS
  ifdef DEFAULT_PROJECT
   TARGET := $(DEFAULT_PROJECT)
  endif # DEFAULT_PROJECT
 endif # !TARGETS
endif # !TARGET
export SOLUTION

# Command targets recognized by the build system.
COMMANDS := default
ifneq ($(MKOPTION_FASTCLEAN),1)
 COMMANDS += clean
 option_fastclean := 0
else
 # The fast-clean option is available only for the default project of the
 # selected solution.
 ifneq ($(TARGET),$(DEFAULT_PROJECT))
  COMMANDS += clean
  option_fastclean := 0
 else
  option_fastclean := 1
 endif
endif

# The Windows platform needs an explicit .exe extension or all
# executable calls will fail with PS3 SDK version 2.4
ifeq ($(HOST_SYSTEM), Cygwin)
  export BUILD_EXE_EXT := .exe
else 
  export BUILD_EXE_EXT :=
endif 

ifeq ($(MAKE_SYSTEM),MinGW)
MAKE_SYSTEM_MAKEFLAGS='MAKE=$(MAKE) $(filter -j%,$(MKOPTION_MAKEFLAGS_$(1)))'
else
MAKE_SYSTEM_MAKEFLAGS=
endif

include $(MAKE_ROOT)/Arch/$(ARCH).mk
ifdef ERROR
 $(error $(ERROR))
endif # ERROR
export ARCH

default_STAGES ?= compile link
STAGES ?= $(default_STAGES)

# Common variable setup #####################################################

# Location of this Common.mk makefile.
#
COMMON_MK := $(MAKE_ROOT)/Common.mk

# Common arguments passed to all sub-makes.
#
COMMON_ARGS = ARCH=$(ARCH) SOLUTION=$(SOLUTION)

# Default options ###########################################################

MKOPTION_DISTRIBUTED := 0
ifeq ($(MKOPTION_INCREDIBUILD),1)
 MKOPTION_DISTRIBUTED := 1
endif
ifeq ($(MKOPTION_DISTCC),1)
 MKOPTION_DISTRIBUTED := 1
endif
export MKOPTION_DISTRIBUTED

ifeq ($(MKOPTION_INCREDIBUILD_SHOWCONSOLE),1)
	SHOW_INCREDIBUILD_CONSOLE := --openmonitor --UseIDEMonitor
else
	SHOW_INCREDIBUILD_CONSOLE := 
endif

# For the stages 'link' and 'scan', default to -j2.
ifeq ($(strip $(MKOPTION_MAKEFLAGS_link)),)
 ifeq ($(MAKE_SYSTEM),MinGW)
  MKOPTION_MAKEFLAGS_link := -j5 TARGET_ARGS=-j4
 else
  MKOPTION_MAKEFLAGS_link := -j5
 endif
endif
ifeq ($(strip $(MKOPTION_MAKEFLAGS_scan)),)
 MKOPTION_MAKEFLAGS_scan := -j4
endif

# For Incredibuild use some arbitrary huge parallel number as the compilation is not started before 
#  all compilation commands are generated.
ifeq ($(MKOPTION_INCREDIBUILD),1)
 MKOPTION_MAKEFLAGS_compile := -j50 TARGET_ARGS=-j50
 MKOPTION_MAKEFLAGS_link := -j50 TARGET_ARGS=-j50
else
 # For DISTCC builds, we'll use -j10 as a default for the MAKEOPTIONS of the
 # 'compile' stage, for local build we'll use -j2.
 ifeq ($(strip $(MKOPTION_MAKEFLAGS_compile)),)
  ifeq ($(MKOPTION_DISTCC),1)
   ifeq ($(HOST_SYSTEM),Linux) 
    MKOPTION_MAKEFLAGS_compile := -j20
   else
    ifeq ($(MAKE_SYSTEM),MinGW)
      ifeq ($(MKOPTION_UNITYBUILD), 1)
        MKOPTION_MAKEFLAGS_compile := -j10 TARGET_ARGS=-j30
      else
        MKOPTION_MAKEFLAGS_compile := -j4 TARGET_ARGS=-j4
      endif
    else
      MKOPTION_MAKEFLAGS_compile := -j13
    endif
   endif
  else
    ifeq ($(MAKE_SYSTEM),MinGW)
      MKOPTION_MAKEFLAGS_compile := -j2 TARGET_ARGS=-j2
    else
      MKOPTION_MAKEFLAGS_compile := -j3
    endif
  endif
 endif
endif

# The linker map file works only if the translation unit object files are
# linked individually.
# XXX disabled temporarily
#ifeq ($(OPTION_LD_MAPFILE),1)
# MKOPTION_LDFILES_ALL := 1
#endif

ifeq ($(MKOPTION_LDFILES_ALL),1)
  MKOPTION_LDFILES := 1
endif

# Build wrapper setup #######################################################
#
# Most commands executed in the build process are filtered through a build
# wrapper script 'build.pl' (part of the MakeTools project).  The build
# wrapper is invoked by prepending the value of one of the $(BUILD*) variables
# to the build process command.
#
# $(BUILD)
#	Compile a C or C++ file.
# $(BUILD_PCH)
#	Compile a C++ header file to a precompiled header.
# $(BUILD_LINK)
#	Link to an executable, shared object, DLL, relocatable object, or
#	static library.
# $(BUILD_ECHO)
#	When building non-verbose, then the specified argument(s) are echoed.
#	May perform a simple path-normalization on the specified argument(s).
# $(BUILD_SILENT)
#	Silence command execution if $(MKOPTION_VERBOSE) is not set.
#
# The build commands come in a variant with an underscore prefix (e.g.
# $(_BUILD).  These variants contain the 'raw' command without any make
# specific prefixes.  Within a shell command (e.g. a shell-level if
# statement), the raw variant should be used instead of the default variant.

ifeq ($(MKOPTION_TIME_COMMANDS),1)
 TIME = time
else
 TIME =
endif

#ifeq ($(SILENT),)
# SHELL_NOP := 
#else
 SHELL_NOP := :
#endif

SILENT_NOP := $(SILENT) $(SHELL_NOP)

ifeq ($(MKOPTION_NOWRAPPER),1)
# No build wrapper.
 ifneq ($(MKOPTION_VERBOSE),1)
  _BUILD = $(TIME)
  _BUILD_PCH = $(TIME)
  _BUILD_LINK = $(TIME)
  _BUILD_ECHO = echo
  BUILD = $(SILENT) $(_BUILD)
  BUILD_PCH = $(SILENT) $(_BUILD_PCH)
  BUILD_LINK = $(SILENT) $(_BUILD_LINK)
  BUILD_ECHO = $(SILENT) $(_BUILD_ECHO)
 else # MKOPTION_VERBOSE
  _BUILD = $(TIME)
  _BUILD_PCH = $(TIME)
  _BUILD_LINK = $(TIME)
  _BUILD_ECHO = $(SILENT_NOP)
  BUILD = $(_BUILD)
  BUILD_PCH = $(_BUILD_PCH)
  BUILD_LINK = $(_BUILD_LINK)
  BUILD_ECHO = $(SILENT) $(_BUILD_ECHO)
 endif # MKOPTION_VERBOSE else
else # MKOPTION_NOWRAPPER
 BUILDTOOL := $(PERL) $(MAKE_ROOT)/Tools/build.pl
 BUILDECHOTOOL := $(PERL) $(MAKE_ROOT)/Tools/echo.pl
 ifeq ($(MKOPTION_MSVC),1)
  BUILDTOOL += -x
 endif
BUILDTOOL += -H $(HOST_SYSTEM)
BUILDECHOTOOL += -H $(HOST_SYSTEM)
 ifeq ($(HOST_SYSTEM),Cygwin)
  # Cygwin mode.
  ifdef CYGWIN_BASEDIR
   BUILDTOOL += -b '$(CYGWIN_BASEDIR)'
   BUILDECHOTOOL += -b '$(CYGWIN_BASEDIR)'
  endif
 endif
 ifeq ($(ARCH_OS),PS3)
  # PS3 mode (move stub libs to the end).
  BUILDTOOL += --ps3
 endif
 ifneq ($(MKOPTION_VERBOSE),1)
  BUILDTOOL += -s
  ifeq ($(MKOPTION_MSVC),1)
   _BUILD_ECHO = echo
   BUILD_ECHO = $(SILENT) $(_BUILD_ECHO)
  else
   _BUILD_ECHO = $(BUILDECHOTOOL) -t "$(TARGET)" --
   BUILD_ECHO = $(SILENT) $(BUILDECHOTOOL) -t "$(TARGET)" --
  endif
 else
  ifeq ($(MKOPTION_MSVC),1)
   _BUILD_ECHO = echo
   BUILD_ECHO = $(SILENT) $(_BUILD_ECHO)
  else
   _BUILD_ECHO = $(SILENT_NOP)
   BUILD_ECHO = $(SILENT) $(_BUILD_ECHO)
  endif
 endif
 _BUILD = $(BUILDTOOL) -m compile -- $(TIME)
 _BUILD_PCH = $(BUILDTOOL) -m compile_pch -- $(TIME)
 _BUILD_LINK = $(BUILDTOOL) -m link -- $(TIME)
 BUILD = $(SILENT) $(_BUILD)
 BUILD_PCH = $(SILENT) $(_BUILD_PCH)
 BUILD_LINK = $(SILENT) $(_BUILD_LINK)
endif # MKOPTION_NOWRAPPER else
ifeq ($(MKOPTION_VERBOSE),1)
 _BUILD_SILENT =
 BUILD_SILENT = $(_BUILD_SILENT)
else
 _BUILD_SILENT =
 BUILD_SILENT = $(SILENT) $(_BUILD_SILENT)
endif

# The log-rotate command.  Log files in the per-stage subdirectories of
# $(LOGDIR) are rotated at the beginning of the stage.
LOGDIR = $(BUILD_ROOT)/Log
LOGROTATE = $(PERL) "$(MAKE_ROOT)/Tools/logrotate.pl"
ifeq ($(MKOPTION_VERBOSE),1)
 LOGROTATE += --verbose
endif
ifneq ($(MKOPTION_LOGROTATE),1)
 LOGROTATE += -m 0
endif

# Devirtualization cache directory.
DEVIRT_CACHE_DIR = $(BUILD_ROOT)/../DeVirtCache
# The devirtualization virtual mapping file 
DEVIRT_VIRTUAL_MAPFILE = crycg_virtual_functions_list

# The special functions $(BUILD_MAKE_LDFILES) and $(BUILD_LDFILES) handle the
# generation and referencing of files passed to a linker.
#
# $(call BUILD_MAKE_LDFILES,$(OBJECTS),$(TARGET))
#   If $(MKOPTION_LDFILES) is set, generate the list file from the specified
#   objects.  On Windows systems, the file names are transformed into Windows
#   notation.  If $(MKOPTION_LDFILES) is not set, then the functions expands
#   to nothing.
#   Note: The expanded function already contains the appropriate silence
#   prefixes and it should be called without any prefixes.
# $(call BUILD_LDFILES,$(OBJECTS),$(TARGET))
#   If $(MKOPTION_LDFILES) is set, then the function expands to the list file
#   with a '@' prefix.  If $(MKOPTION_LDFILES) is not set, the function
#   expands to the specified objects list.
# $(call BUILD_RM_LDFILES,$(TARGET))
#   Remove the list file.

# HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK
# Response file prefix hack for msys sh.exe doing some funky expansion
# when '@$PATH' is encountered Look at build.pl in function
# processCmd() to see what happens exactly.
# HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK
ifeq ($(MKOPTION_MSVC),1)
 export RESPONSE_FILE_PREFIX := --!--
else
 export RESPONSE_FILE_PREFIX := @
endif


EXT_LDFILES := files
ifeq ($(MKOPTION_LDFILES),1)
 build_make_ldfiles_echo := $(PERL) $(MAKE_ROOT)/Tools/echo.pl
 ifeq ($(HOST_SYSTEM),Cygwin)
  build_make_ldfiles_echo += -c
  ifdef CYGWIN_BASEDIR
   build_make_ldfiles_echo += -b '$(CYGWIN_BASEDIR)'
  endif
 endif
 ifeq ($(MKOPTION_LDFILES_ALL),1)
  build_make_ldfiles_echo += -a
 endif
 BUILD_MAKE_LDFILES = cd $(BUILD_STRIP_PREFIX) && $(build_make_ldfiles_echo) -- $(1) > '$(2).$(EXT_LDFILES)'
 BUILD_MAKE_LDFILES_APPEND = cd $(BUILD_STRIP_PREFIX) && $(build_make_ldfiles_echo) -- $(1) >> '$(2).$(EXT_LDFILES)'
 BUILD_LDFILES = '$(RESPONSE_FILE_PREFIX)$(2).$(EXT_LDFILES)'
else # MKOPTION_LDFILES == 1
 BUILD_MAKE_LDFILES =
 BUILD_MAKE_LDFILES_APPEND =
 BUILD_LDFILES = $(1)
endif # else MKOPTION_LDFILES == 1

# Make target definitions. ##################################################

# Pseudo stages.  If $(STAGE) or $(STAGES) contain one of the pseudo stages,
# then the pseudo stage is expanded to the list of stages defined as
# STAGES_(pseudoStage).
STAGES_all := compile link

# Flags to be passed to all sub-makes.
# Note that the name is MAKE_FLAGS with an underscore in the middle!  GNU make
# uses a magic variable named $(MAKEFLAGS) for communication with sub-make
# processes.
MAKE_FLAGS := --no-print-directory
ifneq ($(MKOPTION_VERBOSE),1)
 MAKE_FLAGS += --quiet
endif

# Build progress messages.
#
# MSG_STAGE_BEGIN (stage, command)
#	Echo a message at the beginning of a stage.
#
# MSG_STAGE_END (stage, command)
#	Echo a message at the (successful) end of a stage.
#
# MSG_STAGE_FAILED (stage, command)
#	Echo a message indicating the failure of a stage.
#
# MSG_TARGET_BEGIN (target, stage, command)
#	Echo a message at the beginning of a target.
#
# MSG_TARGET_END (target, stage, command)
#	Echo a message at the (successful) end of a target.
#
# MSG_TARGET_FAILED (target, stage, command)
#	Echo a message indicating the failure of a target.
#
# MSG_READING_VCPROJ
#	Echo a message indicating that a VCPROJ file is being read.
include $(MAKE_ROOT)/Messages.mk

ifeq ($(MKOPTION_INCREDIBUILD),1)
 START_INCREDIBUILD := \
  ($(PERL) $(MAKE_ROOT)/Tools/invoke_incredibuild.pl \
	--AvoidLocal=on --MaxCPUS=128 \
	$(SHOW_INCREDIBUILD_CONSOLE) --NoLogo --out='$(INCREDIBAT_LOG)' \
	--command='$(INCREDIBAT)' &)
 SYNC_INCREDIBUILD := \
  ($(PERL) $(MAKE_ROOT)/Tools/invoke_incredibuild.pl --SILENT --NoLogo \
	--command='$(MAKE_ROOT)/Tools/incredibuild_sync.bat' 2>&1 || : )\
  && $(PERL) $(MAKE_ROOT)/Tools/compiler_output.pl \
	-f -x -d -s -e $(MAKE_ROOT)/Stage/compile.error '$(INCREDIBAT_LOG)'
else
  START_INCREDIBUILD := :
  SYNC_INCREDIBUILD := :  
endif

# Template for a sub-make invocation executing a stage.
# $(1) is the make command passed through the $(COMMAND) variable
#   (i.e.  'default', 'clean', ...);
# $(2) is the stage name.
define execstage_template
	if [ '$(1)' = default ]; then \
	  if [ '$(2)' = scan ]; then \
	    $$(LOGROTATE) -l "$$(LOGDIR)/$(2)"; \
	  fi \
	fi &&\
  $$(if $$(subst link,,$(2)),,$$(if $$(subst default,,$(1)),, \
    $(SYNC_INCREDIBUILD) &&))\
	$$(MAKE) -r \
	  -f "$(COMMON_MK)" \
	  $$(MAKE_FLAGS) \
	  $$(MKOPTION_MAKEFLAGS) \
	  $$(MKOPTION_MAKEFLAGS_$(2)) \
	  $$(call MAKE_SYSTEM_MAKEFLAGS,$(2)) \
	  COMMAND="$(1)" \
	  STAGE="$(2)" \
	  STAGES= \
	  $(COMMON_ARGS) \
	  _exec_stage && \
	  $$(if $$(subst compile,,$(2)),,$$(if $$(subst default,,$(1)),, \
		(test -r '$(INCREDIBAT)' && $(START_INCREDIBUILD) || :) && ))
endef

# If incredibuild support is active we unconditionally execute fixdeps
# after every stage just to be on the safe side in case there are any
# dependencies to be fixed
ifeq ($(MKOPTION_INCREDIBUILD),1)
 export fixdeps := $(PERL) "$(MAKE_ROOT)/Tools/fixdeps.pl" --host-system $(HOST_SYSTEM)
 define incredibuild_fixdeps
	$(SILENT) find $(BUILD_ROOT) -iname "*.d_" -print0 | xargs -r -iXXX -0 -n 1 \
	sh -c '$(fixdeps) < XXX > `dirname XXX`/`basename XXX | sed "s/\.d_/.d/g"` && rm -f XXX' 
 endef
endif


# Define the make command targets for iterating over the specifed stages.  If
# STAGE is defined instead of STAGES, then we'll call the command target to
# execute only that stage.
ifneq ($(strip $(STAGES)),)
export FIRST_STAGE = $(word 1, $(STAGES))
define command_template
.PHONY: $(1)
$(1):
	$(SILENT) mkdir -p "$(BUILD_ROOT)" "$(BUILD_TEMP)" && \
 	  $(foreach stage,\
	  $(STAGES),\
	  $(call execstage_template,$(1),$(stage))) :
	$(incredibuild_fixdeps)
endef
else # STAGES
ifndef STAGE
 $(error No STAGES specified)
endif # !STAGE
# Check if this is a pseudo stage.  We don't want pseudo stages to be
# introduces through environment variables or overrides, so we insist on the
# origin 'file' here.
ifeq ($(origin STAGES_$(STAGE)),file)
define command_template
.PHONY: $(1)
$(1):
	$(SILENT) mkdir -p "$(BUILD_ROOT)" "$(BUILD_TEMP)" && \
	  $(foreach stage,\
	  $(STAGES_$(STAGE)),\
	  $(call execstage_template,$(1),$(stage))) :
endef
else # if pseudo stage
define command_template
.PHONY: $(1)
$(1):
	$(SILENT) mkdir -p "$(BUILD_ROOT)" "$(BUILD_TEMP)"
	$(SILENT) $(call MSG_STAGE_BEGIN,$(STAGE),$(1))
	$(SILENT) $(call execstage_template,$(1),$(STAGE)) : || { \
	  $(call MSG_STAGE_FAILED,$(STAGE),$(1)); \
	  false; \
	}
	$(SILENT) $(call MSG_STAGE_END,$(STAGE),$(1))
endef
endif # if pseudo stage; else
endif # STAGES else
$(foreach command,$(COMMANDS),$(eval $(call command_template,$(command))))

.PHONY: _exec_stage
ifdef TARGETS
_exec_stage:
	$(SILENT) for TARGET in $(TARGETS); do \
	  $(MAKE) -r -f "$(COMMON_MK)" \
	    COMMAND=$(COMMAND) \
	    STAGE=$(STAGE) \
	    TARGET=$$TARGET \
	    $(COMMON_ARGS) \
	    _exec_target \
	  || exit $$?; \
	done
else # TARGETS
ifndef TARGET
 $(error No TARGETS specified)
endif # !TARGET
_exec_stage:
	$(SILENT) $(MAKE) -r -f "$(COMMON_MK)" \
	  COMMAND=$(COMMAND) \
	  STAGE=$(STAGE) \
	  TARGET=$(TARGET) \
	  $(COMMON_ARGS) \
	  _exec_target_$(TARGET)
endif # TARGETS else

ifdef TARGET

# The MakeTools project is always present.  All other (non-phony) projects
# depend on it.  Note that we're specifying an absolute path, because the make
# system might be called from a project outside of our code tree.
MakeTools_CODE := $(MAKE_ROOT)/Tools

ifdef $(TARGET)_CODE
define target_template
.PHONY: _exec_target_$(1)
ifeq ($(MIXED_BUILD),1)
_exec_target_$(1):
else
_exec_target_$(1): $$(addprefix _exec_target_,$$($(1)_DEPS)) $(2)
endif
	$(SILENT) $(call MSG_TARGET_BEGIN,$(1),$(STAGE),$(COMMAND))
	$(SILENT) cd "$(BUILD_ROOT)" && $$(MAKE) -r -f "$(COMMON_MK)" \
	  $(TARGET_ARGS) \
	  COMMAND=$(COMMAND) \
	  STAGE=$(STAGE) \
	  TARGET=$(1) \
	  BUILD_STRIP_PREFIX=$(CURDIR) \
	  $(COMMON_ARGS) \
	  EXEC_TARGET=1 \
	  _exec_target || { \
	    $(call MSG_TARGET_FAILED,$(1),$(STAGE),$(COMMAND)); \
	    false; \
	  }
	$(SILENT) $(call MSG_TARGET_END,$(1),$(STAGE),$(COMMAND))
endef
else # $(TARGET)_CODE
# If  $(TARGET)_CODE is not defined, then the project is a phony project and
# no execute steps are preformed.
define target_template
.PHONY: _exec_target_$(1)
ifeq ($(MIXED_BUILD),1)
_exec_target_$(1):
else
_exec_target_$(1): $$(addprefix _exec_target_,$$($(1)_DEPS))
endif
	$(SILENT_NOP)
endef
endif # $(TARGET)_CODE else
endif # TARGET
$(foreach project,\
	$(PROJECTS),\
	$(eval $(call target_template,$(project),_exec_target_MakeTools)))
$(eval $(call target_template,MakeTools,))

# The $(EXEC_TARGET) variable is defined only when we're called from an
# _exec_target_* command.  We'll include the stage specific code in this case.
#
ifdef EXEC_TARGET

# Define $(PROJECT_CODE) to point to the code directory of the target project.
# This is an absolute path.  Note that $($(TARGET)_CODE) may be an absolute
# directory.
ifeq ($(patsubst /%,,$($(TARGET)_CODE)),)
 PROJECT_CODE := $($(TARGET)_CODE)
else
 ifneq ($(shell X='$($(TARGET)_CODE)'; echo $${X\#?:}),$($(TARGET)_CODE))
  PROJECT_CODE := $($(TARGET)_CODE)
 else
  PROJECT_CODE := $(CODE_ROOT)/$($(TARGET)_CODE)
 endif
endif
export PROJECT_CODE

# $(PROJECT_BUILD) points to the build directory for the project.  We'll
# remove the strip prefix from $($(TARGET)_CODE).
_target_code := $($(TARGET)_CODE)
ifdef DEPOT_MAIN
 _target_code := $(patsubst $(DEPOT_MAIN)%,%,$($(TARGET)_CODE))
else
 _target_code := $(patsubst $(BUILD_STRIP_PREFIX)%,%,$(_target_code))
endif
_target_code := $(patsubst /%,%,$(_target_code))
PROJECT_BUILD = $(BUILD_ROOT)/$(_target_code)
export PROJECT_BUILD

# The cleanup file lists for the supported stages.  All potentially generated
# files for a stage should be added to the corresponding clean lists for that
# stage.  The only exceptions are the 'Files.mk' files generated from the
# VCPROJ files - these are created on demand (in any stage) and are collected
# in the final cleanup list $(FINAL_CLEAN_$(TARGET)).
ifneq ($(TARGET),)
 SCAN_CLEAN_$(TARGET) :=
 COMPILE_CLEAN_$(TARGET) :=
 LINK_CLEAN_$(TARGET) :=
 FINAL_CLEAN_$(TARGET) :=
endif

include $(PROJECT_CODE)/Project.mk

ifeq ($(MIXED_BUILD),1)
 ifneq ($(PROJECT_TYPE),program)
  BUILD_NAME := $(EFFECTIVE_BUILD_NAME)
  BUILD_ROOT := $(EFFECTIVE_BUILD_ROOT)
 endif
endif

# Load the architecure specific setup file, if present.
-include $(MAKE_ROOT)/Arch/$(ARCH).setup.mk

# Set default values for various PROJECT_* variables.
PROJECT_TYPE ?= module

include $(MAKE_ROOT)/Stage/$(STAGE).mk

.PHONY: _exec_target
_exec_target: _prepare _command_$(COMMAND)
ifeq ($(COMMAND),clean)
	$(BUILD_SILENT) rm -f $(FINAL_CLEAN_$(TARGET)) 2>&1 || : 
else
	$(SILENT_NOP)
endif

endif # EXEC_TARGET

ifeq ($(option_fastclean),1)
clean:
	$(SILENT) echo 'Removing build directory (fast clean)...'
	$(SILENT) rm -rf '$(BUILD_ROOT)' 2>&1 || : 
endif

# Common.mk
# vim:ts=8:sw=2

