#!/usr/bin/perl
#############################################################################
## Crytek Source File
## Copyright (C) 2006, Crytek Studios
##
## Creator: Sascha Demetrio
## Date: Jul 13, 2007
## Description: GNU-make based build system
#############################################################################

# This script reads an element file written by the CryCG utility and
# extracts/converts information for the make environment.

# Description of the element file format read by the script:
#
# All lines consisting only of whitespace and lines where the first
# non-whitespace character is a hash ('#', comment lines) are ignored.
#
# A line containing an element definition starts with a single alphanumeric
# character indicating the type of element.  The following element types are
# recognized:
#
# V  External variable.
# v  Static variable.
# I  Instance variable.
# i  Class variable.
# F  External function.
# f  Static function.
# M  Instance method.
# m  Class method.
# C  Compound type (class, struct, or union).
# E  Enum type.
# T  Typedef type.
# U  A translation unit.  Translation units are a special case with a unique
#    parameter format, see below.
#
# Following the element type, 9 positional parameters follow.  These
# positional parameters are either scalar or lists of scalars.  A list of
# scalars is enclosed in curly braces.  Scalars are either integers,
# strings, or enumerated identifiers.  All integers are signed.
#
# Element parameters:
# - The element ID.  This is a unique integer value (within the associated
#   index file) identifying the element.
# - The element name.
# - The element mangled name (may be empty if not available).
# - The element's unique id 
# - A list of element dependencies.  Every dependency is represented as a
#   dependency type enumeration constant followed by the element ID of the
#   element.  The dependency type is one of the following:
#   LL - DECL/DECL dependency.
#   LF - DECL/DEF dependency.
#   FL - DEF/DECL dependency.
#   FF - DEF/DEF dependency.
# - The ID of the associated translation unit.
# - The ID of the tree node within the associated translation unit.
# - The element ID of the context of the element.  -1 if the element does not
#   have a context element.
# - A list of children of the element.  The list contains the element IDs of
#   the children.
# - A flag string.  The presence of a flag character in the string indicating
#   that the associated flag is set.  The following flag character may appear
#   in the string:
#   D - A definition of the element is available.
#   C - A CFG (control flow graph) of the element is available (relevant only
#       for functions and methods).
#   P - A function or method is considered primitive.
#   T - The element is being traced for debugging.
# - A list of attribute strings.  Attribute strings are formed as
#   "KEY=VALUE", where KEY is a valid C/C++ identitier and VALUE is arbitrary
#   (typically a comma separated string).  Note: The only KEY supported at
#   present is 'entry' (indicating an entry point of an SPU job).
#
# Translation unit parameters:
# - The ID of the translation unit.  This ID is unique within the generated
#   element file and is used only for the purpose of referring to translation
#   units.  It is otherwise meaningless.
# - The name of the C/C++ source file associated with the translation unit.
# - The name of the CryCG translation unit file.
#
# Integer scalars are encoded in plain decimal, string are encoded using the
# notation 'sLENGTH(STRING)' where LENGTH is the length of the string (in
# decimal) and STRING is the raw (unquoted) string.
#
# In an element file, all translation units are listed before any other
# elements.

use warnings;
use strict;
use sort 'stable';

use Getopt::Std;

##############################################################################
# Module boilerplate.

BEGIN
{
  use Exporter ();
  our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

  $VERSION = 1.00;
  @ISA = qw(Exporter);
  @EXPORT = qw(&Element::new);
  %EXPORT_TAGS = ( );
  @EXPORT_OK = qw();
}
our @EXPORT_OK;

END { }

##############################################################################
# Parser utility functions.

# Get a decoded string.
#
# Parameters:
# - The string to be parsed.
#
# Returns:
# - The decoded string, or 'undef' in case of an error.
# - The suffix of the parsed string.
sub getString ($)
{
  my $string = shift;

  if ($string =~ /^\s*s([0-9]+)\(/)
  {
    my $length = $1;
    my $prefix = $&;
    my $value = substr($string, length $prefix, $length);
    $string = substr($string, length($prefix) + $length + 1);
    return $value, $string;
  }
  else
  {
    return undef, $string;
  }
}

# Get a token.
#
# Parameters:
# - The string to be parsed.
#
# Returns:
# - The token string, or 'undef' in case of an error.
# - The suffix of the parsed string.
sub getToken ($)
{
  my $string = shift;

  if ($string =~ /^\s*(\w+)/)
  {
    my $match = $&;
    my $value = $1;
    $string = substr($string, length($match));
    return $value, $string;
  }
  else
  {
    return undef, $string;
  }
}

# Get a number.
#
# Parameters:
# - The string to be parsed.
#
# Returns:
# - The number, or 'undef' in case of an error.
# - The suffix of the parsed string.
sub getNumber ($)
{
  my $string = shift;

  if ($string =~ /^\s*(-?\d+)/)
  {
    my $match = $&;
    my $value = $1 + 0;
    $string = substr($string, length($match));
    return $value, $string;
  }
  else
  {
    return undef, $string;
  }
}

# Get a sub-attribute from an attribute string.
#
# Parameters:
# - The attribute string.
# - The name of the requested sub-attribute.
#
# Returns:
# The value of the requested sub-attribute or 'undef' if the attribute is not
# found.
sub getSubAttr ($$)
{
  my $attrString = shift;
  my $attrName = shift;
  my @attrList = split /\s*(?![^)]*\)),\s*/, $attrString;
  my $attr;

  foreach $attr (@attrList)
  {
    $attr =~ /^\s*(\w+)\s*=\s*(.+)$/;
    my ($name, $value) = ($1, $2);
    if ($name eq $attrName) { return $value; }
  }
  return undef;
}

##############################################################################
# Class 'Unit' represents a translation unit.

sub Unit::new ($$)
{
  my $class = shift;
  my $spec = shift;
  my $self = { };

  $self->{ID} = undef;
  $self->{SOURCE_FILE} = undef;
  $self->{UNIT_FILE} = undef;
  bless($self, $class);
  $self->init($spec);
  return $self;
}

sub Unit::init ($$)
{
  my $self = shift;
  my $spec = shift;
  my ($token, $id, $sourceFileName, $unitFileName);

  ($token, $spec) = getToken $spec;
  defined $token or die "invalid unit spec line '$spec' (token)";
  $token eq 'U' or die "spec line '$spec' not a unit spec";

  ($id, $spec) = getNumber $spec;
  defined $id or die "invalid unit spec line '$spec' (id)";

  ($sourceFileName, $spec) = getString $spec;
  defined $sourceFileName
    or die "invalid unit spec line '$spec' (source file)";

  ($unitFileName, $spec) = getString $spec;
  defined $unitFileName
    or die "invalid unit spec line '$spec' (unit file)";

  $self->{ID} = $id;
  $self->{SOURCE_FILE} = $sourceFileName;
  $self->{UNIT_FILE} = $unitFileName;
  return $self;
}

sub Unit::id ($) { return shift->{ID}; }
sub Unit::sourceFile ($) { return shift->{SOURCE_FILE}; }
sub Unit::unitFile ($) { return shift->{UNIT_FILE}; }

##############################################################################
# Class 'Element' represents an element within the translation unit.
#
# Instance of the 'Element' class only contain a subset of all fields provided
# in the element file.  More fields will be added as needed.

# Enumeration of element types.
my (
  $ET_NONE,
  $ET_EXTERNAL_VARIABLE,
  $ET_STATIC_VARIABLE,
  $ET_INSTANCE_VARIABLE,
  $ET_CLASS_VARIABLE,
  $ET_EXTERNAL_FUNCTION,
  $ET_STATIC_FUNCTION,
  $ET_INSTANCE_METHOD,
  $ET_CLASS_METHOD,
  $ET_COMPOUND_TYPE,
  $ET_ENUM_TYPE,
  $ET_TYPEDEF_TYPE
) = ( 0 .. 11 );

my %_Element_typeMap = (
  'V' => $ET_EXTERNAL_VARIABLE,
  'v' => $ET_STATIC_VARIABLE,
  'I' => $ET_INSTANCE_VARIABLE,
  'i' => $ET_CLASS_VARIABLE,
  'F' => $ET_EXTERNAL_FUNCTION,
  'f' => $ET_STATIC_FUNCTION,
  'M' => $ET_INSTANCE_METHOD,
  'm' => $ET_CLASS_METHOD,
  'C' => $ET_COMPOUND_TYPE,
  'E' => $ET_ENUM_TYPE,
  'T' => $ET_TYPEDEF_TYPE
);

sub Element::new ($$$)
{
  my $class = shift;
  my $spec = shift;
  my $unitSet = shift;
  my $self = { };

  $self->{TYPE} = undef;
  $self->{ID} = undef;
  $self->{NAME} = undef;
  $self->{MANGLED} = undef;
  $self->{UUID} = undef; 
  $self->{ATTRS} = ( );
  $self->{UNIT} = undef;
  $self->{ET_MAP} = \%_Element_typeMap;
  $self->{JOB_MAP} = undef;
  $self->{ENTRY_POINTS} = undef;
  bless($self, $class);
  $self->init($spec, $unitSet);
  return $self;
}

sub Element::init ($$$)
{
  my $self = shift;
  my $spec = shift;
  my $unitSet = shift;
  my ($token, $type, $id, $name, $mangledName, $uuid);
  my ($unitId, $unit, %attributes, $value);
  my $typeMap = $self->{ET_MAP};

  ($token, $spec) = getToken $spec;
  defined $token or die "invalid element spec line '$spec' (token)";
  defined $typeMap->{$token}
    or die "unrecognized token '$token' in element spec line '$spec'";
  $type = $typeMap->{$token};

  ($id, $spec) = getNumber $spec;
  defined $id or die "invalid element spec line '$spec' (id)";

  ($name, $spec) = getString $spec;
  defined $name or die "invalid element spec line '$spec' (name)";

  ($mangledName, $spec) = getString $spec;
  defined $mangledName
    or die "invalid element spec line '$mangledName' (mangled mame)";

  ($uuid, $spec) = getString $spec; 
  defined $uuid or die "invalid element spec line '$spec' (uuid)";

  $spec =~ s/^\s*\{[^}]*\}//
    or die "invalid element spec line '$spec' (deps)";

  ($unitId, $spec) = getNumber $spec;
  defined $unitId or die "invalid element spec line '$spec' (unit)";
  $unit = $unitSet->getUnit($unitId);
  defined $unit or die "invalid unit ID in spec line '$spec'";

  ($value, $spec) = getNumber $spec;
  defined $value or die "invalid element spec line '$spec' (node)";

  ($value, $spec) = getNumber $spec;
  defined $value or die "invalid element spec line '$spec' (context)";

  $spec =~ s/^\s*\{[^}]*\}//
    or die "invalid element spec line '$spec' (children)";

  ($value, $spec) = getString $spec;
  defined $value or die "invalid element spec line '$spec' (flags)";

  $spec =~ s/^\s*\{//
    or die "invalid element spec line '$spec' (attrs)";
  %attributes = ( );
  while (not $spec =~ s/^\s*\}//)
  {
    my $attr;
    ($attr, $spec) = getString $spec;
    defined $attr or die "invalid element spec line '%spec' (attrs)";
    $attr =~ /(\w+)=(.*)$/ or die "malformed element attribute '$attr'";
    my ($key, $value) = ($1, $2);
    $attributes{$key} = $value;
  }

  $self->{TYPE} = $type;
  $self->{ID} = $id;
  $self->{NAME} = $name;
  $self->{MANGLED} = $mangledName;
  $self->{UUID} = $uuid; 
  $self->{UNIT} = $unit;
  $self->{ATTRS} = \%attributes;

  return $self;
}

sub Element::type ($) { return shift->{TYPE}; }
sub Element::id ($) { return shift->{ID}; }
sub Element::name ($) { return shift->{NAME}; }
sub Element::mangledName ($) { return shift->{MANGLED}; }
sub Element::uuid ($) { return shift->{UUID}; }
sub Element::unit ($) { return shift->{UNIT}; }
sub Element::attrs ($) { return shift->{ATTRS}; }

##############################################################################
# Class 'UnitSet' represents a set of translation units.

sub UnitSet::new ($)
{
  my $class = shift;
  my $self = { };

  $self->{INDEX_FILE} = undef;
  $self->{UNIT_MAP} = { };
  $self->{ELEMENT_MAP} = { };
  $self->{ELEMENT_UUID_MAP} = { };
  bless($self, $class);
  return $self;
}

sub UnitSet::load ($$)
{
  my $self = shift;
  my $elementFileName = shift;
  local *IN;

  open(IN, '<', $elementFileName)
    or die "Can not open element file '$elementFileName': $!";
  while (<IN>)
  {
    chomp;
    next if (/^\s*$/ or /^\s*#/);
    my $initial = substr($_, 0, 1);
    if ($initial eq 'U')
    {
      my $unit = Unit->new($_);
      $self->{UNIT_MAP}{$unit->id} = $unit;
    }
    else
    {
      my $element = Element->new($_, $self);
      $self->{ELEMENT_MAP}{$element->id} = $element;
      $self->{ELEMENT_UUID_MAP}{$element->uuid} = $element;
    }
  }
}

# Get a list of function/method elements.
sub UnitSet::getFunctionElements ($)
{
  my $self = shift;
  my $element;
  my @list = ( );
  my $elementMap = $self->{ELEMENT_MAP};

  foreach $element (values %$elementMap)
  {
    my $type = $element->type;
    if ($type == $ET_EXTERNAL_FUNCTION
      or $type == $ET_STATIC_FUNCTION
      or $type == $ET_INSTANCE_METHOD
      or $type == $ET_CLASS_METHOD)
    {
      push @list, $element;
    }
  }
  return @list;
}

# Get a translation unit by ID.
sub UnitSet::getUnit ($$)
{
  my $self = shift;
  my $unitId = shift;
  my $unitMap = $self->{UNIT_MAP};

  if (not defined $unitMap->{$unitId}) { return undef; }
  return $unitMap->{$unitId};
}

# Get a mapping of all jobs.
#
# The UnitSet keeps a cached copy of the job map in the attribute JOB_MAP.
sub UnitSet::getJobMap ($)
{
  my $self = shift;

  if (defined $self->{JOB_MAP})
  {
    return %{$self->{JOB_MAP}};
  }

  my %jobMap = ( );
  my @functionElements = $self->getFunctionElements();
  my $error = 0;

  foreach my $element (@functionElements)
  {
    my $attrs = $element->attrs();
    next unless defined $attrs->{entry};
    my $jobName = getSubAttr $attrs->{entry}, 'job';
    if (defined $jobName)
    {
      my $elementName = $element->name();
      $elementName =~ s/\(.*//;
      if (defined $jobMap{$jobName})
      {
        print STDERR "$0: ERROR: multiple definitions for job '$jobName'!\n";
        $error = 1;
        next;
      }
      $jobMap{$jobName} = Job->new($jobName, $element);
    }
  }
  die if $error;

  foreach my $element (@functionElements)
  {
    my $attrs = $element->attrs();
    next unless defined $attrs->{entry};
    my $indirectSpec = getSubAttr $attrs->{entry}, 'indirect';
    if (defined $indirectSpec)
    {
      my @indirectList = ( );
      while (1)
      {
        if ($indirectSpec =~ /^\s*([a-zA-Z0-9_]+)\s*($|,)/)
        {
          push @indirectList, $1;
          $indirectSpec =~ s/^[^,]+,?\s*//;
        }
        elsif ($indirectSpec =~ /^\s*([a-zA-Z0-9_]+)\s*(\([^\)]*\))\s*/)
        {
          push @indirectList, $1.$2;
          $indirectSpec =~ s/^[^\(]+\([^\)]*\)\s*,?\s*//;
        }
        else
        {
          print STDERR
            "$0: WARNING: invalid SPU indirect specification ",
            "'$indirectSpec'\n";
          last;
        }
        last if $indirectSpec =~ /^\s*$/;
      }
      foreach my $indirect (@indirectList)
      {
        my $jobName = $indirect;
        $jobName =~ s/\s*\(.*//;
        if (not defined $jobMap{$jobName})
        {
          my $elementName = $element->name();
#          print STDERR
#            "$0: WARNING: unknown job '$jobName' referenced ",
#            "for indirect element '$elementName'\n";
          next;
        }
        my $job = $jobMap{$jobName};
        my @variantList = ( );
        if ($indirect =~ /\(([^\)]*)\)/)
        {
          my $variantSpec = $1;
          $variantSpec =~ s/^\s+//;
          $variantSpec =~ s/\s+$//;
          foreach my $variant (split /\s*,\s*/, $variantSpec)
          {
            $variant = uc $variant;
            $variant =~ s/^V//;
            $variant =~ s/=V/=/;
            $variant =~ s/\s+=\s+/=/;
            if ($variant =~ /^([ML]+)=([ML]+)$/)
            {
              # Mapped variant of the form (FROM)=(TO).  Check if the
              # specified variants are compatible.
              my ($from, $to) = ($1, $2);
              my $notCompatible = 0;
              if (length($from) != length($to))
              {
                $notCompatible = 1;
              }
              else
              {
                my $length = length $from;
                for (my $i = 0; $i < $length; ++$i)
                {
                  my $toDomain = substr $to, $i, 1;
                  if ($toDomain eq 'L')
                  {
                    my $fromDomain = substr $from, $i, 1;
                    if ($fromDomain ne 'L')
                    {
                      print STDERR
                        "$0: WARNING: ",
                        "can not map L-domain to M-domain ",
                        "in variant mapping '$variant'\n";
                      $notCompatible = 1;
                      last;
                    }
                  }
                }
              }
              if ($notCompatible)
              {
                print STDERR
                  "$0: WARNING: ",
                  "invalid mapped SPU indirect variant specification ",
                  "'$variant': mapped variants are not compatible\n";
                next;
              }
            }
            elsif (not $variant =~ /^[ML]+$/)
            {
              print STDERR
                "$0: WARNING: ",
                "invalid SPU indirect variant specification '$variant'\n";
              next;
            }
            push @variantList, $variant;
          }
        }
        else
        {
          push @variantList, undef;
        }
        foreach my $variant (@variantList)
        {
          my $foundElement = 0;
          foreach my $entryPoint (@{$job->indirect()})
          {
            next unless $element == $entryPoint->element();
            my $entryPointVariant = $entryPoint->variant();
            if (not defined($variant) and not defined($entryPointVariant))
            {
              $foundElement = 1;
              last;
            }
            next if not defined($variant) or not defined($entryPointVariant);
            if ($variant eq $entryPointVariant)
            {
              $foundElement = 1;
              last;
            }
          }
          if (not $foundElement) { $job->addIndirect($element, $variant); }
        }
      }
    }
  }

  $self->{JOB_MAP} = \%jobMap;

  return %jobMap;
}


##############################################################################
# Class 'Job' represents an SPU job.

# Create a new SPU job instance.
#
# Parameters:
# - The job name.
# - The entry element of the job.
sub Job::new ($$$)
{
  my $class = shift;
  my $name = shift;
  my $entryElement = shift;
  my $self = { };

  $self->{NAME} = $name;
  $self->{ENTRY_POINT} = EntryPoint->new($entryElement);
  $self->{INDIRECT} = [ ];
  bless($self, $class);
  return $self;
}

sub Job::name ($) { return shift->{NAME}; }
sub Job::entryPoint ($) { return shift->{ENTRY_POINT}; }
sub Job::entryElement ($) { return shift->{ENTRY_POINT}->element(); }

# Get the indirect call targets of the job.
sub Job::indirect ($)
{
  my $self = shift;
  my @indirect = sort 
  { 
      $a->element()->uuid() cmp $b->element()->uuid() 
  } @{$self->{INDIRECT}};
  return @indirect if wantarray;
  return \@indirect;
}

# Add an indirect call target.
#
# Parameters:
# - The element.
# - The variant identifier.
sub Job::addIndirect ($$$)
{
  my $self = shift;
  my $element = shift;
  my $variant = shift;

  push @{$self->{INDIRECT}}, EntryPoint->new($element, $variant);
}

# Get the list of all entry points, including the indirect call targets.
sub Job::getEntryPointList ($)
{
  my $self = shift;
  my @entryPointList = @{$self->indirect()};
  unshift @entryPointList, $self->{ENTRY_POINT};
  return @entryPointList if wantarray;
  return \@entryPointList;
}

# Set the entry point symbols for all EntryPoint instances of the job.
#
# Parameters:
# - The PPU symbol table.  If this is undefined, then the PPU symbol
#   attributes will not be set.
# - The SPU symbol table.  If this is undefined, then the SPU symbol
#   attributes will not be set.
sub Job::setEntryPointSymbols ($$$)
{
  my $self = shift;
  my $ppuSymTab = shift;
  my $spuSymTab = shift;

  foreach my $entryPoint ($self->getEntryPointList())
  {
    my $element = $entryPoint->element();
    my $elementName = $element->name();

    if (defined $ppuSymTab)
    {
      my $ppuMangledName = $element->mangledName();
      my $ppuSymbol = $ppuSymTab->get($ppuMangledName);
      if (not defined $ppuSymbol)
      {
        # It is not a problem if the PPU symbol can not be found if the
        # element is not an indirect jump target.
        my $isIndirect = 0;
        foreach my $indirectEntryPoint ($self->indirect())
        {
          if ($indirectEntryPoint == $entryPoint)
          {
            $isIndirect = 1;
            last;
          }
        }
        if ($isIndirect)
        {
          print STDERR "$0: in Job::setEntryPointSymbols: ",
            "can not find PPU symbol '$ppuMangledName' ",
            "for element '$elementName'\n";
          exit 1;
        }
      }
      $entryPoint->setPPUSymbol($ppuSymbol);
    }
    else
    {
      $entryPoint->setPPUSymbol(undef);
    }

    if (defined $spuSymTab)
    {
      my $elementId = $element->id();
      my $variant = $entryPoint->variant();
      if (not $entryPoint->isMapped())
      {
        my $spuMangledName = $spuSymTab->findSymbol($elementId, $variant);
        if (not defined $spuMangledName)
        {
          if (not defined $variant)
          {
            print STDERR "$0: in Job::setEntryPointSymbols: ",
              "can not find SPU symbol for default variant of ",
              "element '$elementName' (ID $elementId)\n";
            exit 1;
          }
          else
          {
            print STDERR "$0: in Job::setEntryPointSymbols: ",
              "can not find SPU symbol for variant '$variant' of ",
              "element '$elementName' (ID $elementId)\n";
            exit 1;
          }
        }
        my $spuSymbol = $spuSymTab->get($spuMangledName);
        die if not defined $spuSymbol;
        $entryPoint->setSPUSymbol($spuSymbol);
      }
      else
      {
        # For mapped variants we'll check if an SPU symbol is available, because
        # this is an indication of a performance problem.
        my $fromVariant = $variant;
        $fromVariant =~ s/=[ML]+$//;
        my $spuMangledName = $spuSymTab->findSymbol($elementId, $fromVariant);
        if (defined $spuMangledName)
        {
          print STDERR "$0: in Job::setEntryPointSymbols: ",
            "PERFORMANCE WARNING: ",
            "entry point '$elementName' ($variant) is mapped ",
            "but an SPU symbol for the from-variant $fromVariant ",
            "is available ('$spuMangledName')\n";
        }
      }
    }
    else
    {
      $entryPoint->setSPUSymbol(undef);
    }
  }
}

##############################################################################
# Class 'EntryPoint' representing an entry point of a job.
#
# Fields:
# - ELEMENT: The entry point element.
# - VARIANT: The variant of the element, represented as a variant string (a
#   sequence of 'L' and 'M' characters).  The default (main) variant is
#   assumed if undefined.
# - FNID: The function ID.
# - PPU_SYMBOL: The PPU symbol of the entry point.
# - SPU_SYMBOL: The SPU symbol of the entry point.
#
# Conventional fields:
# - SPU_SYMBOL_NAME: Code wishing to cache the SPU symbol name of the entry
#   point should use this field.
#
# Note that instances of EntryPoint are _not_ shared among jobs, so the same
# logical entry point may have different function IDs in different jobs.

# Create a new entry point.
#
# Parameters:
# - The element.
# - The variant (optional).  If no variant is specified, then the default
#   variant is assumed.
sub EntryPoint::new ($$;$)
{
  my $class = shift;
  my $element = shift;
  my $variant = shift;
  my $self = { };

  $self->{ELEMENT} = $element;
  $self->{VARIANT} = $variant;
  $self->{FNID} = undef;
  $self->{PPU_SYMBOL} = undef;
  $self->{SPU_SYMBOL} = undef;
  bless($self, $class);
  return $self;
}

sub EntryPoint::element ($) { return shift->{ELEMENT}; }
sub EntryPoint::variant ($) { return shift->{VARIANT}; }
sub EntryPoint::fnid ($) { return shift->{FNID}; }

sub EntryPoint::setFnid ($$)
{
  my $self = shift;
  my $fnid = shift;

  die "multiple FNIDs in entry point" if defined $self->{FNID};
  $self->{FNID} = $fnid;
}

sub EntryPoint::ppuSymbol ($) { return shift->{PPU_SYMBOL}; }
sub EntryPoint::spuSymbol ($) { return shift->{SPU_SYMBOL}; }

sub EntryPoint::setPPUSymbol ($$)
{
  my $self = shift;
  my $ppuSymbol = shift;

  $self->{PPU_SYMBOL} = $ppuSymbol;
}

sub EntryPoint::setSPUSymbol ($$)
{
  my $self = shift;
  my $spuSymbol = shift;

  $self->{SPU_SYMBOL} = $spuSymbol;
}

# Check if the entry point is 'mapped'.
#
# A mapped entry point maps to another variant of the same element.  Mapped
# entry points are not generated but are only handled in the PPU function
# address resolver function.
sub EntryPoint::isMapped ($)
{
  my $self = shift;
  my $variant = $self->{VARIANT};

  if (defined($variant) and $variant =~ /=/) { return 1; }
  return 0;
}

# Find the mapped entry point in the specified list of entry points.
#
# If the entry point is not mapped, it will always return itself (without
# consulting the specified list).
#
# Parameters:
# - The list of entry points to be searched.
#
# The method returns the mapped entry point or undef if no matching entry
# point was found in the specified list.
sub EntryPoint::findMapped ($\@)
{
  my $self = shift;
  my $list = shift;

  return $self unless $self->isMapped();
  my $variant = $self->{VARIANT};
  die unless defined $variant;
  die unless $variant =~ /=([ML]+)$/;
  my $toVariant = $1;
  my $element = $self->{ELEMENT};
  foreach my $entryPoint (@$list)
  {
    next unless $entryPoint->{ELEMENT} == $element;
    my $entryPointVariant = $entryPoint->{VARIANT};
    next unless defined $entryPointVariant;
    next unless $entryPointVariant eq $toVariant;
    return $entryPoint;
  }
  return undef;
}

# Check if the entry point is weak.
#
# A weak entry point indicates that the called entry function or method will
# not call out of it's containing page.  For single page jobs, all entry
# points are weak.
sub EntryPoint::isWeak ($)
{
  my $self = shift;

  return 1;
}

##############################################################################
# Class 'PageItem' represents an element on a code page 
#
# Fields:
# - NAME:  The name of the element 
# - MANGLED:  The mangled name
# - ID: The UUID of the element 
# - VARIANTKEY: The variant index of the element
# - VARIANTNAME: The variant's suffix 
# - WEAK:  Boolean value describing if the element is weak 
# - ENTRY: Boolean value describing if the element is a page relative
#          entry point

# PageItem constructor.
sub PageItem::new ($$$$$$$$)
{
	my $class = shift; 
	my $self = { };

	$self->{NAME} = shift;
	$self->{MANGLED} = shift;
	$self->{UUID} = shift;
	$self->{VARIANTNAME} = shift;
	$self->{VARIANTKEY} = shift;
	$self->{WEAK} = shift;
	$self->{ENTRY} = shift; 
	if ($self->{VARIANTNAME} eq "undef") 
	{ 
			$self->{VARIANTNAME} = ""; 
	} 
	bless($self, $class);

	return $self;
}

##############################################################################
# Class 'CodePage' represents code page information
#
# Fields:
# - METHODS: An array of element names of the methods distributed on
#            the code page 
# - CROSSCALLS: The names of the elements which can be reached via a
#               cross page call from this page 

# CodePage constructor.
#
sub CodePage::new ($)
{
	my $class = shift;
	my $self = { };
	
	$self->{METHODS} = [];
	$self->{CROSSCALLS} = [];
	bless($self, $class);
	
	return $self;
}

##############################################################################
# Class 'PageTable' a table of elements distributed onto code pages 
#
# Fields:
# - CODEPAGES: An array of CodePage instances 

# PageTable constructor.
#
sub PageTable::new ($)
{
	my $class = shift;
	my $self = { };

	$self->{CODEPAGES} = [];
	bless($self, $class);

	return $self;
}

# Load a page table from a file 
# 
# Parameters: 
# - The page table filename 
sub PageTable::load($$)
{
  my $self = shift;
  my $pageTableFileName = shift;
  local *IN;
  my $lineNumber = 0; 
  my $codePage = CodePage->new();

  if (not open(IN, '<', $pageTableFileName))
  {
    print STDERR
      "$0: can not open page table file '$pageTableFileName': $!\n";
    exit 1;
  }

  while (<IN>)
  {
	  ++$lineNumber; 
	  # Skip lines consisting of comments or lines that are simply
	  # empty 
	  if (/^#.*$/)
	  {
		  next;
	  }
	  if (/^\s*$/)
	  {
		  next;
	  }

	  # Parse a page header. Create a new code page instance and
	  # append the previous the array of code pages 
	  if (/^\[([0-9]+)\].*$/)
	  {
		  if ($1 != "0")
		  {
			  push(@{$self->{CODEPAGES}}, $codePage);
			  $codePage = CodePage->new();
		  }
		  next;
	  }

	  # Parse a page item line. 
	  if(/^\s+item:\s+([a-z,0-9]+)\s+'(.*)'\s+(.*)\s+(.*)\s+([0-9]+)\s+([0,1])\s+([0,1]).*$/)
	  {
		  # extract the information about the page item 
		  my $elementUUID = $1; 
		  my $elementName = $2; 
		  my $mangledName = $3; 
		  my $variantKey  = $4;
		  my $variantName = $5;
		  my $isWeak      = $6;
		  my $isEntry     = $7; 
		  
		  # create a new page item and append it to the code page
		  my $pageItem = PageItem->new(
                 $elementName,
                 $mangledName,
					       $elementUUID,
					       $variantKey,
					       $variantName,
					       $isWeak,
					       $isEntry);
		  push(@{$codePage->{METHODS}}, $pageItem);
		  
		  next;
	  }

	  # Parse a cross call line.
	  if (/^\s+crosscall:\s+([a-z,0-9]+)\s+'(.*)'\s+.*\s+(.*)\s+([0-9]+).*$/)
	  {
		  # extract the information about the page item 
		  my $crossCall = {
			  uuid => $1,
			  name => $2, 
			  key => $4,
			  variant => $3,
		  };
		  # create a new page item and append it to the code page
		  push(@{$codePage->{CROSSCALLS}}, $crossCall);
		  
		  next;
	  }
 
	  print STDERR "Error: unparseable line '$lineNumber' in pagetable file",
	  " '$pageTableFileName'\n";
  }

  push(@{$self->{CODEPAGES}}, $codePage);

}

##############################################################################
# Class 'PPUSymbol' represents a global symbol of the PPU.
#
# Fields:
# - NAME: The symbol name
# - TYPE: The symbol type (this is the type character from the NM output).
# - ADDR: The symbol address, may be undefined.

# Helper function checking if a type character represents a global symbol.
#
sub isGlobal ($)
{
  my $type = shift;

  return 1 if ord($type) >= ord('A') and ord($type) <= ord('Z');
  return 0;
}

# PPUSymbol constructor.
#
# Parameters:
# - The symbol name.
# - The symbol type character (from the NM output).
# - (Optional.) The address of the symbol.
sub PPUSymbol::new ($$$;$)
{
  my $class = shift;
  my $name = shift;
  my $type = shift;
  my $addr = shift;
  my $self = { };

  $self->{NAME} = $name;
  $self->{TYPE} = $type;
  $self->{ADDR} = $addr;
  bless($self, $class);
  return $self;
}

sub PPUSymbol::name ($) { return shift->{NAME}; }
sub PPUSymbol::type ($) { return shift->{TYPE}; }
sub PPUSymbol::addr ($) { return shift->{ADDR}; }
sub PPUSymbol::isGlobal ($) { return isGlobal(shift->{TYPE}); }

##############################################################################
# Class 'PPUSymTab' represents a PPU symbol table.
#
# Fields:
# - TABLE: The symbol name to PPUSymbol instance mapping.

sub PPUSymTab::new ($)
{
  my $class = shift;
  my $self = { };

  $self->{TABLE} = { };
  bless($self, $class);
  return $self;
}

# Load the PPU symbol table from an NM file.
#
# Parameters:
# - The PPU NM file name.
sub PPUSymTab::loadNM ($$)
{
  my $self = shift;
  my $ppuNmFileName = shift;
  local *IN;
  my %table = ( );

  if (not open(IN, '<', $ppuNmFileName))
  {
    print STDERR
      "$0: can not open PPU NM file '$ppuNmFileName': $!\n";
    exit 1;
  }
  my $line = 0;
  while (<IN>)
  {
    ++$line;
    if (not /^([0-9a-f]+)\s+(\w)\s+([^\s]+)\s*$/)
    {
      print STDERR
        "$0: malformed entry in PPU NM file '$ppuNmFileName', line $line\n";
      exit 1;
    }
    my $addr = $1;
    my $type = $2;
    my $name = $3;
    next if not isGlobal($type);
    my $symbol = PPUSymbol->new($name, $type, $addr);
    if (defined $table{$name})
    {
      print STDERR "$0: in PPU NM file '$ppuNmFileName': ",
        "multiple definitions of PPU symbol '$name'\n";
      exit 1;
    }
    $table{$name} = $symbol;
  }
  close(IN);

  $self->{TABLE} = \%table;
  return \%table;
}

# Load the PPU symbol table from a map file.
#
# IMPORTANT NOTE:
# This method works only if the specified linker map file containes the
# _mangled_ names of the PPU symbols.  Unfortunately the linker map file
# generated by the Windows toolchain of the Cell SDK generats a map file with
# demangled names (and this can not be switched off using the --no-demangle
# linker option).  Linker map files generated on Windows can _not_ be used to
# populate a PPUSymTab!
#
# Parameters:
# - The PPU linker map file name.
sub PPUSymTab::loadMap ($$)
{
  my $self = shift;
  my $ppuMapFileName = shift;
  local *IN;
  my %table = ( );

  if (not open(IN, '<', $ppuMapFileName))
  {
    print STDERR
      "$0: can not open PPU map file '$ppuMapFileName': $!\n";
    exit 1;
  }
  my $line = 0;
  # Skip everything up to the beginning of the text segment definition.
  # (For now we're only interested in the text segment, support for reading
  # data and bss will be added on demand.)
  my $textStart;
  my $textSize;
  while (<IN>)
  {
    ++$line;
    if (/^\.text\s+0x([0-9a-f]+)\s+0x([0-9a-f]+)\s*$/)
    {
      $textStart = $1;
      $textSize = $2;
      last;
    }
  }
  if (not defined $textStart)
  {
    print STDERR
      "$0: no text segment definition in PPU map '$ppuMapFileName'\n";
    exit 1;
  }
  while (<IN>)
  {
    ++$line;
    last if /^\s*$/ or not /^\s/;
    if (/^\s+0x([0-9a-f]+)\s+\.([^\s]+)\s*$/)
    {
      my $addr = $1;
      my $name = $2;
      my $symbol = PPUSymbol->new($name, 'T', $addr);
      if (defined $table{$name})
      {
        print STDERR "$0: in PPU map file '$ppuMapFileName': ",
          "multiple definitions of PPU symbol '$name'\n";
        exit 1;
      }
      $table{$name} = $symbol;
    }
  }
  close(IN);

  $self->{TABLE} = \%table;
  return \%table;
}

# Load the symbol table from a symbol file.
#
# A symbol file (*.sym) is a file generated by the 'GetSym' tool.  It contains
# a mapping from external PPU symbols to PPU addresses (similar to an NM
# file).  All symbols are of type 'T'.
#
# Parameters:
# - The symbol file.
sub PPUSymTab::loadSym ($$)
{
  my $self = shift;
  my $ppuSymFileName = shift;
  local *IN;
  my %table = ( );

  if (not open(IN, '<', $ppuSymFileName))
  {
    print STDERR
      "$0: can not open PPU symbol file '$ppuSymFileName': $!\n";
    exit 1;
  }
  my $line = 0;
  while (<IN>)
  {
    ++$line;
    if (not /^([0-9a-f]+)\s+([^\s]+)\s*$/)
    {
      print STDERR
        "$0: malformed entry in PPU symbol file '$ppuSymFileName', line $line\n";
      exit 1;
    }
    my $addr = $1;
    my $name = $2;
    my $symbol = PPUSymbol->new($name, 'T', $addr);
    if (defined $table{$name})
    {
      print STDERR "$0: in PPU symbol file '$ppuSymFileName': ",
        "multiple definitions of PPU symbol '$name'\n";
      exit 1;
    }
    $table{$name} = $symbol;
  }
  close(IN);

  $self->{TABLE} = \%table;
  return \%table;
}

# Get the named PPU symbol.
#
# Parameters:
# - The PPU symbol name.
#
# The method returns the PPUSymbol instance or 'undef' if the symbol is not
# found.
sub PPUSymTab::get ($$)
{
  my $self = shift;
  my $name = shift;

  if (defined $self->{TABLE}->{$name}) { return $self->{TABLE}->{$name}; }
  return undef;
}

##############################################################################
# Class 'SPUSymbol'.
#
# Fields:
# - NAME: The symbol name
# - TYPE: The symbol type (this is the type character from the NM output).
# - PAGE: The page number.

# SPUSymbol constructor.
#
# Parameters:
# - The symbol name.
# - The page number.
sub SPUSymbol::new ($$$;$)
{
  my $class = shift;
  my $name = shift;
  my $type = shift;
  my $page = shift;

  my $isGlobal = 0;
  if (ord($type) >= ord('A') and ord($type) <= ord('Z')) { $isGlobal = 1; }

  my $self = { };
  $self->{NAME} = $name;
  $self->{TYPE} = $type;
  $self->{PAGE} = $page;
  bless($self, $class);
  return $self;
}

sub SPUSymbol::name ($) { return shift->{NAME}; }
sub SPUSymbol::type ($) { return shift->{TYPE}; }
sub SPUSymbol::page ($) { return shift->{PAGE}; }
sub SPUSymbol::isGlobal ($) { return isGlobal(shift->{TYPE}); }

sub SPUSymbol::variant ($)
{
  my $name = shift->{NAME};

  if ($name =~ /\$V([ML]+)_/) { return $1; }
  return undef;
}

##############################################################################
# Class 'SPUSymTab'.
#
# The SPU symbol table contains a symbol name to instance mapping and an
# element ID mapping.
#
# Fields:
# - TABLE: The SPU symbol name to SPUSymbol instance mapping.
# - ID_MAP: The element ID to SPU symbol names mapping.  Every ID maps to an
#   array of symbol names.

sub SPUSymTab::new ($)
{
  my $class = shift;
  my $self = { };

  $self->{TABLE} = { };
  $self->{ID_MAP} = { };
  bless($self, $class);
  return $self;
}

# Load the SPU symbol table.
#
# The caller species the directory containing the job description and SPU NM
# files.  By default (unless a page NM file is specified), the method will
# load _all_ pages.
#
# Parameters:
# - The job name.
# - The directory containing the description and SPU NM files.
# - (Optional.)  The name of the page NM files.  If this is undefined or
#   unspecified, then the method will load all pages.
#
# The method returns a reference to the name to symbol map.
sub SPUSymTab::load ($$$;$)
{
  my $self = shift;
  my $jobName = shift;
  my $jobDir = shift;
  my $pageNmFileName = shift;
  local *IN;

  my $pageTable = PageTable->new();
  $pageTable->load("$jobDir/${jobName}.pagetable");

  my %table = ( );
  my %idMap = ( );

  my $pageNum = 0;
  foreach my $codePage (@{$pageTable->{CODEPAGES}})
  {
      foreach my $pageItem (@{$codePage->{METHODS}})
      {
	  my $symbolId = $pageItem->{UUID};
	  my $symbolVariant = $pageItem->{VARIANTNAME};
	  my $symbolName = sprintf("\$E%s_%s",  $symbolId, $symbolVariant);
	  my $symbol = SPUSymbol->new($symbolName, "T", $pageNum);
	  my $symbolList;
	  if (defined $idMap{$symbolId})
	  {
	      $symbolList = $idMap{$symbolId};
	  }
	  else
	  {
	      $symbolList = [ ];
	      $idMap{$symbolId} = $symbolList;
	  }
	  push @$symbolList, $symbol;
	  $table{$pageItem->{NAME}} = $symbol;
      }
      ++$pageNum;
  }

  $self->{TABLE} = \%table;
  $self->{ID_MAP} = \%idMap;
  return \%table;
}

# Get the SPU symbol for the specified mangled name.
#
# Parameters:
# - The SPU mangled name.
#
# The method returns the SPUSymbol instance or undef if the symbol is not
# found.
sub SPUSymTab::get ($$)
{
  my $self = shift;
  my $name = shift;

  if (not defined $self->{TABLE}->{$name}) { return undef; }
  return $self->{TABLE}->{$name};
}

# Find an SPU symbol name for a given element ID and variant.
#
# Parameters:
# - The element's unique ID.
# - (Optional.) The variant.  If omitted or undefined, the default variant is
#   assumed.
#
# The method returns the requested SPU symbol name or 'undef' if the requested
# symbol is not found.
sub SPUSymTab::findSymbol ($$;$)
{
  my $self = shift;
  my $elementUUID = shift;
  my $variant = shift;
  my $symbolName = undef;
  my $idMap = $self->{ID_MAP};
  return undef if not defined $idMap->{$elementUUID};

  my $symbolList = $idMap->{$elementUUID};
  if (not defined $variant)
  {
    # Get the default variant.
    foreach my $symbol (@$symbolList)
    {
      my $symbolVariant = $symbol->variant();
      return $symbol->name() 
		  if not defined $symbolVariant or $symbolVariant =~ /^M+$/;
    }
  }
  else
  {
    foreach my $symbol (@$symbolList)
    {
      my $symbolVariant = $symbol->variant();
      return $symbol->name() if $variant eq $symbolVariant;
    }
  }
  return undef;
}

# Tools/Elements.pm
# vim:ts=2:sw=2:expandtab

