#!/usr/bin/perl
#############################################################################
## Crytek Source File
## Copyright (C) 2006, Crytek Studios
##
## Creator: Sascha Demetrio
## Date: Jul 31, 2006
## Description: GNU-make based build system
#############################################################################

# Build wrapper script for compiler and linker invocations.
# Call with option -h for a description.
#
# Note that the script currently assumes that the compiler accepts the a file
# list through the -Wl,@$file syntax.  @$file is translated to -Wl,@$file in
# the compiler invocation (because some versions of GCC don't understand
# @$file).  This feature is controlled through the --ldlist option, which is
# enabled by default.

use warnings;
use strict;

use Getopt::Long qw(:config no_auto_abbrev bundling);
use String::ShellQuote;

our $outputFile = '';
$outputFile = ''; # stupid perl waring 

# Parse the command line.
my $optMode = 'compile';
my $optConvertMSVC = 0;
my $optHostSystem;
my $optCygwinBaseDir;
my $optPrefix = '';
my $optIgnoreFile;
my $optErrorFile;
my $optSilentMode = 0;
my $optPS3Mode = 0;
my $optLdList = 1;
my $optCryCGMode = 0;
my $optCD = 0;
my $optHelp = 0;
GetOptions(
    'm|mode=s' => \$optMode,
    'x|msvc' => \$optConvertMSVC,
    'H|host-system=s' => \$optHostSystem,
    'b|basedir|base-dir=s' => \$optCygwinBaseDir,
    'p|prefix=s' => \$optPrefix,
    'i|ignore=s' => \$optIgnoreFile,
    'e|error=s' => \$optErrorFile,
    's|silent' => \$optSilentMode,
    'cd!' => \$optCD,
    'ps3!' => \$optPS3Mode,
    'ldlist!' => \$optLdList,
    'cgmode!' => \$optCryCGMode,
    'h|help' => \$optHelp);
if ($optHelp)
{
  print <<EOF;
build.pl: Build wrapper script.
Synopsis:
  \$PERL build.pl (Options) -- (Command)
Options:
-m|--mode MODE
  Build mode.  One of 'compile', 'compile_pch', 'link'.  Default is 'compile'.
-x|--msvc
  Convert the build output messages to MSVC format.
-H|--host-system HOST_SYSTEM
  The name of the host system.  One of 'Linux', 'Windows', or 'Cygwin'.
-b|--basedir BASEDIR
	Used for Cygwin only.  The base directory in Windows notation (drive:path).
	If this option is specified, then the path prefix '/base' is converted to
	the specified base directory.
-p|--prefix PREFIX
  Strip the specified prefix from filenames in the output (works only for
  recognized output lines, unrecognized lines are echoed as-is).
-i|--ignore IGNOREFILE
  Specify a file containing ignore patterns.  All non-empty lines of the
  specified file are added to the list of ignore patterns.
-e|--error ERRORFILE
  Specify a file containing error patterns.  All non-empty lines of the
	specified file are added to the list of error patterns.  Whenever one of the
	error patterns is recognized in the compiler or linker output, then the
	build script will report an error to the caller (error code 7).
-s|--silent
  Silent operation.  No output except for the command output.  (Normal
  operation will echo the command executed.)
--ps3
	PS3 mode.  In PS3 mode and link mode, the link library order is patched and
	stub libraries are moved to the end of the list.
--ldlist
  LD file list prefix mode.  If this option is sepcified, then file list
	arguments are marked as linker arguments.  This means that argument of the
	form \@\$file are transformed into -Wl,\@\$file.  If the called executable
	is a linker (executable names 'ld' or 'PREFIX-ld'), then the LD file list
	mode is turned off unconditionally.  LD file list mode is enabled by default
	and can be turned off by passing the '--no-ldlist' option.
--cgmode
  Code generator mode.  This option enables some special argument
	transformations for the CryCG code generator.  This mode is enabled
	automatically if the executable name is 'crycg'.
--cd
  Change the current working directory to output directory.  The output
	directory is determined from the path specified with the -o command option.
	For the 'compile' and 'compile_pch' modes, this option is enabled
	automatically if the '-save-temps' command option is present.
-h|--help
  Display this help screen and exit.
Note:  Separating the command from the options with a double hyphen is not
strictly required, but strongly recommended.
EOF
  exit 0;
}

# Update the operation mode based on the command executable name.
if ($#ARGV > 0)
{
	my $cmd = $ARGV[0];

	if ($cmd =~ /\bld(\.exe)?$/
			or $cmd =~ /\/ld(\.exe)?$/
			or $cmd =~ /-ld(\.exe)?$/)
	{ $optLdList = 0; }

	if ($cmd =~ /\bcrycg(\.exe)?$/
			or $cmd =~ /\/crycg(\.exe)?$/)
	{ $optCryCGMode = 1; }

	# same rules for crycg apply for pagedist
	if ($cmd =~ /\bpagedist(\.exe)?$/
			or $cmd =~ /\/pagedist(\.exe)?$/)
	{ $optCryCGMode = 1; }
}

# Check the host system.
if (not defined $optHostSystem)
{
	print STDERR "$0: no host system specified (option -H)\n";
	exit 1;
}
if ($optHostSystem ne 'Windows'
    and $optHostSystem ne 'Cygwin'
    and $optHostSystem ne 'Linux'
    and $optHostSystem ne 'MingW')
{
	print STDERR "$0: unrecognized host system '$optHostSystem'\n";
	exit 1;
}
my $cygwinMode = 0;
my $mingwMode = 0; 
if ($optHostSystem eq 'Cygwin') { $cygwinMode = 1; }
if ($optHostSystem eq 'MingW') { $mingwMode = 1; }

if ($cygwinMode or $mingwMode)
{
    my $cmd = $ARGV[0];

    if ($optCryCGMode)
    {
        if ($cmd =~ /\/crycg$/)
        {
            $cmd =~ s/crycg$/crycg.exe/g;
        }
    }
    $ARGV[0] = $cmd;
}

# List of message blocks.
my @blockList = ( );

# List of skip patterns.
my @skipPatterns = ( );

# List of error patterns.
my @errorPatterns = ( );

# Read the list of skip patterns.
sub readSkipPatterns
{
  local *IN;
  if ($optIgnoreFile)
  {
    open(IN, $optIgnoreFile)
			or die "Can't open skip pattern file '$optIgnoreFile': $!";
    while (<IN>)
    {
      if (/^\s*$/ or /^#/) { next; }
		  s/^\s+|\s+$//g;
      push @skipPatterns, $_;
    }
    close IN;
  }
}

# Read the list of error patterns.
sub readErrorPatterns
{
    local *IN;
    if ($optErrorFile)
    {
	open(IN, $optErrorFile)
	    or die "Can't open error pattern file '$optErrorFile': $!";
	while (<IN>)
	{
	    if (/^\s*$/ or /^#/) 
		{ 
		    next; 
		}
		s/^\s+|\s+$//g;
		push @errorPatterns, $_;
		}
	    close IN;
	}
}

# This flag is set whenever an error pattern is found in the tool output.
my $errorPatternFound = 0;

# Convert a filename for the target environment.
sub convertFilename
{
  my $filename = shift;
  if ($optPrefix and $filename =~ /^${optPrefix}(.*$)/i)
  {
      $filename = $1;
  }
  # Remove /X/../ and /./ sequences from the filenames.
  # Note: This will not work correctly if the source tree is fucked up with
  # directory symlinks - don't do that!
  while ($filename =~ /\/\.\//)
  {
      $filename =~ s/\/\.\//\//;
  }
  while ($filename =~ /\/[^\/]+\/\.\.\//)
  {
      $filename =~ s/\/[^\/]+\/\.\.\//\//;
  }
  return $filename;
}

# Convert the messages for the target environment.
sub convert
{
  my $line = shift;
  if ($line =~ /^In file included from ([^:]+):([0-9]+)([,:])$/)
  {
    my $lineno = $2; my $sep = $3;
    my $filename = convertFilename $1;
    if ($optConvertMSVC) {
      $line = "$filename($lineno): Included from here...\n";
    } else {
      $line = "In file included from $filename:${lineno}$sep\n";
    }
  }
  elsif ($line =~ /^\s*from ([^:]+):([0-9]+)([,:])$/)
  {
    my $lineno = $2; my $sep = $3;
    my $filename = convertFilename $1;
    if ($optConvertMSVC) {
      $line = "$filename($lineno): from here...\n";
    } else {
      $line = "                 from $filename:${lineno}$sep\n";
    }
  }
  elsif ($line =~ /^(\S..[^:]*):([0-9]+):([0-9]+):(.*)$/)
  {
    my $lineno = $2; my $column = $3; my $message = $4;
    my $filename = convertFilename $1;
    if ($optConvertMSVC) {
      $line = "$filename($lineno):$message\n";
    } else {
      $line = "$filename:$lineno:$column:$message\n";
    }
  }
  elsif ($line =~ /^(\S..[^:]*):([0-9]+):(.*)$/)
  {
    my $lineno = $2; my $message = $3;
    my $filename = convertFilename $1;
    if ($optConvertMSVC) {
      $line = "$filename($lineno):$message\n";
    } else {
      $line = "$filename:$lineno:$message\n";
    }
  }
  elsif ($line =~ /^(\S..[^:]*):(.*)$/)
  {
    my $message = $2;
    my $filename = convertFilename $1;
    $line = "$filename:$message\n";
  }
  return $line;
}

# Filter input lines.  The function returns 1 if the line contains one of the
# filter patterns specified by the script caller.  If an error pattern is
# found, then the $errorPatternFound flag is set and an error message is
# written to the standard error.
sub filter
{
  my $line = shift;
  my $pattern;
  foreach $pattern (@skipPatterns)
  {
    if ($line =~ /$pattern/) { return 1; }
  }
  foreach $pattern (@errorPatterns)
  {
      if ($line =~ /$pattern/)
      {
	  $line =~ s/warning:/error:/g;
	  $line = convert $line;
	  $line =~ s/\s+$//;
	  print STDERR "$line [error pattern]\n";
	  $errorPatternFound = 1;
      }
  }
  return 0;
}

# Process a block of command output.
#
# Parameters:
# - blockList (reference) - the list of processed blocks.
# - block (reference) - the block to be processed and appended to the block
#   ist.
sub processBlock (\@\@)
{
    my $blockList = shift;
    my $block = shift;
    my @blockProcessed = ( );

  foreach (@$block)
  {
    if (filter $_) { return; }
    # push @blockProcessed, convert $_;
    print convert $_;
  }
  push @$blockList, join '', @blockProcessed;
}

# Check if the message string indicates a PCH compilation error.
sub checkErrorPCH (@)
{
    while (my $line = shift)
    {
	return 1 if $line =~ /cc1plus[^:]*: .*\.h\.gch: not used/;
    }
    return 0;
}

# Strip the PCH include option from the specified argument list.
sub stripCommandPCH (\@)
{
    my $args = shift;
    my $nArgs = $#$args + 1;
    my $pch;

    for (my $i = 0; $i < $nArgs; ++$i)
    {
	my $arg = $args->[$i];
	if ($arg eq '-include' and $i < $nArgs - 1)
	{
	    $pch = $args->[$i + 1];
	    if ($pch =~ /\.h$/)
	    {
		splice @$args, $i, 2;
		$nArgs -= 2;
		last;
	    }
	}
    }
    if (defined $pch)
    {
	my $pchdir = $pch;
	$pchdir =~ s/\/[^\/]*$//;
	for (my $i = 0; $i < $nArgs; ++$i)
	{
	    my $arg = $args->[$i];
	    if ($arg eq "-I$pchdir" or $arg eq "-I$pchdir/")
	    {
		splice @$args, $i, 1;
		$nArgs -= 1;
		last;
	    }
	    elsif ($arg eq '-I' and $i < $nArgs - 1)
	    {
		my $dirarg = $args->[$i + 1];
		if ($dirarg eq $pchdir or $dirarg eq "$pchdir/")
		{
		    splice @$args, $i, 2;
		    $nArgs -= 2;
		    last;
		}
	    }
	}
    }
}

sub runCompilerCommand (\@\@$\@;$); # Recursive -> need prototype.

# Run the specified compiler command and process the command output.
#
# Parameters:
# - args (by reference) - The command to be executed.
# - echoArgs (by reference) - The command to be echoed in verbose mode.
# - dir - If this is defined, then the command will be executed in the
#   specified directory.
# - blockList (by reference) - List filtered output blocks.
#
# The last parameter is a recursion indicator and should be omitted by the
# top-level caller.
#
# Return value:
# The function returns the exit code of the compiler command.
sub runCompilerCommand (\@\@$\@;$)
{
    my $args = shift;
    my $echoArgs = shift;
    my $dir = shift;
    my $blockList = shift;
    my $recursion = shift;
    $recursion = 0 unless defined $recursion;
    my $exitCode = 0;

    my $distccMode = 0;
    if ($#$args > -1)
    {
	my $commandName = $args->[0];
	if ($commandName =~ /^distcc/ or $commandName =~ /[\\\/]distcc/)
	{
	    $distccMode = 1;
	}
    }
    my $cmd = shell_quote @$args;
    if (defined $dir) { $cmd = shell_quote('cd', $dir) . '&&' . $cmd; }
    if (not $optSilentMode)
    {
	my $echoCmd = shell_quote @$echoArgs;
	print STDERR "$echoCmd\n";
    }

    local *CMDOUT;
    open CMDOUT, '-|', $cmd.' 2>&1 || echo @FAILED-$?@'
	or die "can not execute command: $!";
    $_ = <CMDOUT>;
    my @cmdOutput = ( );
    my @block = ( );
    my $lineCount = 0;
    my $maxLineCount = 1000000;
    while ($_)
    {
	my $pushedBlocks = 0;
	if (/\@FAILED-([0-9]+)\@/)
	{
	    $exitCode = $1;
	    if (not <CMDOUT>) { last; }
	}
	if (/^distcc/)
	{
	    # DistCC messages are displayed immediately.
	    print STDERR $_;
	    $_ = <CMDOUT>;
	    next;
	}
	push @cmdOutput, $_;
	$lineCount += 1;
	if ($lineCount > $maxLineCount)
	{
	    print STDERR
		"$0: more than $maxLineCount lines of command output, terminating\n";
	    print STDERR
		"$0: command: '$cmd'\n";
	    close CMDOUT;
	    exit 1;
	}
	if (/^In /)
	{
	    push @block, $_;
	    while (<CMDOUT>)
	    {
		if (/^\s*from /) { push @block, $_; } else { last; }
	    }
	    if (not $_) { last; }
	}
	if (/^[^:]*: In /)
	{
	    push @block, $_;
	    while (<CMDOUT>)
	    {
		if (/^[^:]*: In /) { push @block, $_; } else { last; }
	    }
	    if (not $_) { last; }
	}
	if ($_) 
	{ 
	    push @block, $_; 
	}
	while (<CMDOUT>)
	{
	    if (/^[^:]*:   /) { push @block, $_; } else { last; }
	}
	processBlock(@$blockList, @block);
	@block = ( );
	if (not $_) { last; }
    }
    close CMDOUT;

    if ($exitCode and not $recursion)
    {
	if (checkErrorPCH @cmdOutput)
	{
	    # If the DistCC command failed because of a PCH mismatch, then we'll
	    # rerun the command without the PCH.  It is assumed that the PCH is
	    # specified with the first -include statement on the command line -
	    # skipping this statement effectively disables the PCH.
	    stripCommandPCH(@$args);
	    stripCommandPCH(@$echoArgs);
	    if (not $optSilentMode)
	    {
		print STDERR "build: PCH mismatch, compiling without PCH ...\n";
	    }
	    @$blockList = ( );
	    $exitCode = runCompilerCommand(@$args, @$echoArgs, $dir, @$blockList, 1);
	}
    }
    return $exitCode;
}

# Execute the specified command and process the output
sub processCmd
{
    my $exitCode = 0;
    my $arg;
    my @args = ( );
    my @echoArgs = ( );
    my @cdArgs = ( );
    my @ps3StubArgs = ( );
    my $outputFile;
    my $outputDir;
    my $cd = $optCD;
    my $saveTemps = 0;
    my $oOption = 0;

    foreach $arg (@ARGV)
    {
	if ($mingwMode) 
	{
	    $arg =~ s/--!--/@/;
	    if ($arg =~ m/[a-zA-Z]:/)
	    {
		$arg =~ s/\//\\/g;
	    }
	}
	my $echoSuffix = '';
	if ($oOption)
	{
	    $outputFile = $arg;
	    $oOption = 0;
	}
	elsif ($arg =~ /^-o(.*$)/)
	{
	    if ($1) { $outputFile = $1; } else { $oOption = 1; }
	}
	if (not $optSilentMode and $arg =~ /^@(.*$)/)
	{
	    my $listFile = $1;
	    $echoSuffix = '(';
	    my $space = '';
	    local *LIST;
	    open(LIST, $listFile)
		or die "Can't read command input file '$listFile': $!";
	    while (<LIST>)
	    {
		s/^\s+|\s+$//g;
		$echoSuffix .= $_.$space;
		$space = ' ';
	    }
	    $echoSuffix .= ')';
	    close LIST;
	}
	if ($cygwinMode)
	{
	    if ($arg =~ /^(-[IL]|)\/base\/(.*$)/)
	    {
		$arg = "$1$optCygwinBaseDir/$2";
	    }
	    if ($arg =~ /^(-[IL]|)\/cygdrive\/([a-z])\/(.*$)/)
	    {
		$arg = "$1$2:/$3";
	    }
	    elsif ($arg =~ /^(-[IL]|)\/([a-z])\/(.*$)/)
	    {
		$arg = "$1$2:/$3";
	    }
	    if ($arg =~ /^@\/base\/(.*$)/)
	    {
		$arg = "\@$optCygwinBaseDir/$1";
	    }
	    if ($arg =~ /^@\/cygdrive\/([a-z])\/(.*$)/)
	    {
		$arg = "\@$1:/$2";
	    }
	    elsif ($arg =~ /^@\/([a-z])\/(.*$)/)
	    {
		$arg = "\@$1:/$2";
	    }
	}
	if ($optLdList)
	{
	    $arg =~ s/^@/-Wl,@/;
	}
	if ($optCryCGMode and $cygwinMode)
	{
	    if ($arg =~ /^(-[CD][^:=]+[:=])(.*)/)
	    {
		my $prefix = $1;
		my @values = split(/\s*,\s*/, $2);
		my @patchedValues = ( );
		foreach my $value (@values)
		{
		    if ($value =~ /\/base\/(.*)/)
		    {
			$value = "$optCygwinBaseDir/$1";
		    }
		    elsif ($value =~ /\/cygdrive\/([a-zA-Z])\/(.*)/)
		    {
			$value = "$1:/$2";
		    }
		    elsif ($value =~ /\/([a-zA-Z])\/(.*)/)
		    {
			$value = "$1:/$2";
		    }
		    push @patchedValues, $value;
		}
		$arg = "$prefix" . join(', ', @patchedValues);
	    }
	}
	if ($cygwinMode and $optMode eq 'link')
	{
	    if ($arg =~ /^-Wl,-Map,\/base\/(.*$)/)
	    {
		$arg = "-Wl,-Map,$optCygwinBaseDir/$1";
	    }
	    if ($arg =~ /^-Wl,-Map,\/cygdrive\/([a-z])\/(.*$)/)
	    {
		$arg = "-Wl,-Map,$1:/$2";
	    }
	    elsif ($arg =~ /^-Wl,-Map,\/([a-z])\/(.*$)/)
	    {
		$arg = "-Wl,-Map,$1:/$2";
	    }
	}
	if ($optMode eq 'compile' or $optMode eq 'compile_pch')
	{
	    if ($arg =~ /^-save-temps$/) { $saveTemps = 1; }
	}
	if (not $mingwMode)
	{
	    $arg =~ s/\\/\//g;
	    $arg =~ s/\/\/*/\//g;
	}
	if ($optPS3Mode
	    and $optMode eq 'link'
	    and ($arg =~ /^-l.*_stub$/ or $arg =~ /^-Wl,--(no-)?whole-archive$/))
	{
	    push @ps3StubArgs, $arg;
	}
	else
	{
	    push @args, $arg;
	    push @echoArgs, $arg . $echoSuffix;
	}
    }
    if ($outputFile)
    {
	$outputDir = $outputFile;
	$outputDir =~ s/[\/\\][^\/\\]+$//;
	if ($outputFile eq $outputDir) { $outputDir = '.'; }
	# If the output file is a precompiled header, then we'll remove
	# 'localhost' from DistCC's host list.
	if ($outputFile =~ /\.h\.gch$/ and $args[0] =~ /distcc/)
	{
	    my $hostList = $ENV{DISTCC_HOSTS};
	    if (defined $hostList)
	    {
		my @hostList = split /\s+/, $hostList;
		for (my $i = $#hostList; $i >= 0; --$i)
		{
		    my $host = $hostList[$i];
		    if ($host =~ /^localhost/) { splice @hostList, $i, 1; }
		}
		my $patchedHostList = join ' ', @hostList;
		if ($patchedHostList ne $hostList)
		{
		    print STDERR "build: removed localhost from DISTCC_HOSTS for PCH\n";
		    $ENV{DISTCC_HOSTS} = $hostList;
		}
	    }
	}
    }
    foreach $arg (@ps3StubArgs)
    {
	push @args, $arg;
	push @echoArgs, $arg;
    }
    if ($saveTemps and $outputDir) { $cd = 1; }
    $exitCode = runCompilerCommand(
	@args,
	@echoArgs,
	$cd ? $outputDir : undef,
	@blockList);
    return $exitCode;
}

readSkipPatterns;
readErrorPatterns;
my $exitCode = processCmd;
foreach (@blockList) { print; }
if ($errorPatternFound and $exitCode == 0) { $exitCode = 7; }
exit $exitCode;

# Tools/build.pl
# vim:ts=2:sw=2

