#!/usr/bin/perl
#############################################################################
## Crytek Source File
## Copyright (C) 2006, Crytek Studios
##
## Creator: Sascha Demetrio
## Date: Jul 31, 2006
## Description: GNU-make based build system
#############################################################################

use XML::Parser;



# Description:
# Extract the list of source files from a Visual Studio project file.
#
# Invocation:
# extract.pl \
#   (ProjectFile.vcproj) (ProjectCodeDir) (MakeVarPrefix) (OutputFile)

my $filename = @ARGV[0];
my $basedir = @ARGV[1];
my $prefix = @ARGV[2];
my $output = @ARGV[3];

# Flag indicating if we're within a <Files> element.
my $inFiles = 0;

# The list of all input files referenced through <File> elements.
my @files = ( );

# Cache of case translated path names, relative to the specified base
# directory.
my %pathmap;

# Set of directories referenced in the path map.  This set is generated as a
# side-effect of setting up the path cache.
my %dirset;

# Lists of C++ files, C files, and H files.
my @cpp_files = ( );
my @c_files = ( );
my @h_files = ( );

# The name of the PCH cpp file (if any).
my $pch_file = 0;

# Flag indicating that the tool should operate in Cygwin mode.
#
# In Cygwin mode, we don't rely on any 'exists' test to check if the character
# case of a file is correct.  Instead, we'll _always_ scan the containing
# directory to locate the file.
my $cygwin_mode = 1;

# Flag indicating strict path mapping.
#
# If this flag is set, then all paths from the project file which can not be
# resolved are treated as errors.
my $strict_path_mapping = 1;

# Generate the XML parser and parse the input file.  The result will be a list
# of input files and input directories.
sub parseProject
{
	my $parser = new XML::Parser();
	$parser->setHandlers(
			Start => \&handleStart,
			End => \&handleEnd);
	$parser->parsefile($filename);
	@dirs = sort(keys(%dirset));
	@_ = sort(@files);
	@files = ( );
	my $prev = 0;
	foreach (@_)
	{
		if ($_ eq $prev) { next; }
		$prev = $_;
		push(@files, $_);
	}
	foreach (@files)
	{
		if (/^StdAfx\.h$/i or /^stdafx\.h$/i or /\/stdafx\.h$/i) {
			$pch_file = $_;
		}
		if (/\.cpp$/) { push @cpp_files, $_; }
		elsif (/\.c$/) { push @c_files, $_; }
		elsif (/\.h$/) { push @h_files, $_; }
	}
}

parseProject;

# Output make code defining the variables SOURCES_CPP, SOURCES_C, SOURCES_H,
# and SOURCE_DIRS.
open OUT, '>', $output || die "Can't open output file '$output': $!";
print OUT "${prefix}SOURCES_CPP =";
for (@cpp_files) { print OUT " $_"; }
print OUT "\n${prefix}SOURCES_C =";
for (@c_files) { print OUT " $_"; }
print OUT "\n${prefix}SOURCES_H =";
for (@h_files) { print OUT " $_"; }
print OUT "\n${prefix}SOURCE_DIRS =";
for (@dirs) { print OUT " $_"; }
print OUT "\n";
if ($pch_file) { print OUT "${prefix}SOURCE_PCH = $pch_file\n"; }
close OUT;

# Handle start tags.
sub handleStart
{
	my $parser = shift; my $startTag = shift;

	if ($startTag eq 'Files') { $inFiles = 1; }
	if ($inFiles and $startTag eq 'File')
	{
		while (@_)
		{
			my $attribute = shift; my $value = shift;
			if ($attribute eq 'RelativePath')
			{
				my $fixedPath = fixPath($value);
				if ($fixedPath eq '') { next; }
				push(@files, $fixedPath);
			}
		}
	}
}

# Handle end tags.
sub handleEnd
{
	my $parser = shift; my $endTag = shift;

	if ($endTag eq 'Files') { $inFiles = 0; }
}

sub fixPath
{
	my $path = shift;

	$path =~ s/\\/\//g; $path =~ s/^\.\///;
	my $fixedPath = '.';
	@pathComponents = split /\//, $path;
	$numComponents = $#pathComponents + 1;
	$componentIndex = 0;
	foreach (@pathComponents)
	{
		++$componentIndex;
		my $component = $_;
		my $componentLower = lc;
		my $fixedComponent;
		my $found = 0;
		my $dirname = "$basedir/$fixedPath";
		my $pathKey = lc "$fixedPath/$component";
		my $isDir = 0;
		if (not $cygwin_mode and exists $pathmap{$pathKey})
		{
			$fixedPath = @pathmap{$pathKey};
			next;
		}
		if ($componentIndex == $numComponents)
		{
			if (not $cygwin_mode and -r "$basedir/$path")
			{
				$fixedComponent = $component;
				$found = 1;
			}
		}
		else { $isDir = 1; }
		if (not $found)
		{
			if (not opendir DIR, "$dirname") {
				if (not $strict_path_mapping)
				{
					warn "Can't open directory '$dirname': $!";
					warn "Source file '$path' ignored!";
					return '';
				}
				else
				{
					die "Can't open directory '$dirname': $!";
				}
			}
			foreach (readdir DIR)
			{
				my $dirComponent = $_;
				my $dirComponentLower = lc;
				if ($componentLower eq $dirComponentLower)
				{
					$fixedComponent = $dirComponent;
					$found = 1;
					last;
				}
			}
			closedir DIR;
			if ($found) { @pathmap{$pathKey} = "$fixedPath/$fixedComponent"; }
		}
		if (not $found) {
			if (not $strict_path_mapping)
			{
				warn "File or directory '$dirname/$component' not found";
				warn "Source file '$path' ignored!";
				return '';
			}
			else
			{
				print STDERR
					"$filename: error: referenced source file '$dirname/$component' ",
					"not found\n";
				die "File or directory '$dirname/$component' not found";
			}
		}
		$fixedPath = "$fixedPath/$fixedComponent";
	}
	$dirname = $fixedPath;
	$dirname =~ s/\/[^\/]*$//;
	$dirname =~ s/^\.\///;
	if (not exists $dirset{$dirname}) { $dirset{$dirname} = 1; }
	$fixedPath =~ s/^\.\///;
	return $fixedPath;
}

# vim:ts=2:sw=2

