#!/usr/bin/perl
#############################################################################
## Crytek Source File
## Copyright (C) 2008, Crytek Studios
##
## Creator: Sascha Demetrio
## Date: Jan 16, 2008
## Description: GNU-make based build system
#############################################################################

use warnings;
use strict;

use Elements;
use Getopt::Long qw(:config no_auto_abbrev bundling);
use IPC::Open2;
use sort 'stable';

# Parse the command line.
my $outputFileName;
my $elementFileName;
my $jobName;
my $ppuNmFileName;
my $ppuMapFileName;
my $ppuSymFileName;
my $optionHelp = 0;
my $optionVerbose = 0;
GetOptions(
    'E|element-file=s' => \$elementFileName,
    'o|header-file=s' => \$outputFileName,
		'j|job=s' => \$jobName,
    'n|ppu-nm=s' => \$ppuNmFileName,
    'm|ppu-map=s' => \$ppuMapFileName,
    's|ppu-sym=s' => \$ppuSymFileName,
    'h|help' => \$optionHelp,
    'v|verbose' => \$optionVerbose
    ) or exit 1;
if ($optionHelp)
{
  print <<EOF;
genexechdr.pl: Generate the header file for a the job exec file.
Synopsis:
  \$PERL genexechdr.pl (Options)
Options:
-E|--element-file FILE
  Specify the element file name.  This option is required.
-o|--header-file FILE
  Specify the output header file to be generated.
  This option is required.
-j|--job NAME
  The name of the job.  This option is required.
-n|--ppu-nm FILE
  File containing the NM output of the PPU executable.  (Either a PPU NM file,
  a PPU map file, or a PPU symbol file must be specified.)
-m|--ppu-map FILE
  File containing the PPU map file.  (Either a PPU NM file, a PPU map file, or
  a PPU symbol file must be specified.)
-s|--ppu-sym FILE
  File containing the PPU symbol file.  (Either a PPU NM file, a PPU map file, or
  a PPU symbol file must be specified.)
-v|--verbose
  Verbose operation.
-h|--help
  Display this help screen and exit.
EOF
  exit 0;
}
if (not defined $elementFileName)
{
  print STDERR "$0: no element file specified!\n";
  exit 1;
}
if (not defined $outputFileName)
{
  print STDERR "$0: no output header file specified!\n";
  exit 1;
}
if (not defined $jobName)
{
	print STDERR "$0: no job name specified!\n";
	exit 1;
}
if (not defined $ppuNmFileName
    and not defined $ppuMapFileName
    and not defined $ppuSymFileName)
{
  print STDERR "$0: no PPU NM|map|sym file specified!\n";
  exit 1;
}

# Load the PPU symbol table.
my $ppuSymTab = PPUSymTab->new();
if (defined $ppuNmFileName)
{
  $ppuSymTab->loadNM($ppuNmFileName);
}
elsif (defined $ppuSymFileName)
{
  $ppuSymTab->loadSym($ppuSymFileName);
}
else
{
  $ppuSymTab->loadMap($ppuMapFileName);
}

sub gen_md5($)
{
    my $bits = shift; 
    my $cmd = 'md5sum -';
    my $result = ''; 
    my $CMDOUT; 
    my $CMDIN;
    my $md5_pid = open2 $CMDOUT, $CMDIN, $cmd
	or die "can not execute command: $!";
    print $CMDIN $bits; close($CMDIN);
    $result = <$CMDOUT>;
    $result =~ s/([A-Fa-f0-9]+)\s+.*\s*/$1/;
    waitpid $md5_pid, 0; 
    return $result; 
}

sub genExecHdrFile ($)
{
  my $unitSet = shift;
  local *OUT;

  # Get the job object.
  my %jobMap = $unitSet->getJobMap();
  if (not defined $jobMap{$jobName})
  {
    print STDERR "$0: unknown job name '$jobName' specified\n";
    exit 1;
  }
  my $job = $jobMap{$jobName};

  my $md5_out = '';
  # Add include guard.
  my $includeGuard = "_EXECHDR_${jobName}_";
  my $include_guard_in = "#ifndef $includeGuard\n#define $includeGuard 1\n\n";
  my $include_guard_out = "\n#endif /* $includeGuard */\n\n";
    
  my %entryElements = ( );  
  foreach my $entryPoint ($job->indirect())
  {
    next if $entryPoint->isMapped();
    my $element = $entryPoint->element();
    my $elementId = $element->id();
    my $elementUUID = $element->uuid();
    $entryElements{$elementId} = $element;
		# Add MD5 for this line.
    my $symbolName = $element->mangledName();
    my $ppuSymbol = $ppuSymTab->get($symbolName);
    my $ppuAddr = $ppuSymbol->addr();
    my $headerLine = "#define \$ADDR_PPU_$elementUUID (0x${ppuAddr}U)\n";
    $md5_out = $md5_out . $headerLine;
  }
  my $md5_outstring = gen_md5($md5_out);

  # Open file reading for MD5 generation.
  # Create empty if not existing

  my $md5_instring = "";
  if(-e $outputFileName)
  {
      open(IN, '<', $outputFileName);
      # Read MD5, it is the commented first line.
      $md5_instring = substr(<IN>,2);
      $md5_instring =~ s/\s*$//;
      close(IN);
  }
  
  # Compare checksums and only write file in case of difference.
  if ($md5_instring ne $md5_outstring)
  {
      # Open/create file for writing
      open(OUT, '>', $outputFileName)
	  or die "can not open output file '$outputFileName': $!";
      print OUT "//$md5_outstring\n";
      my $includeGuard = "_EXECHDR_${jobName}_";
      print OUT $include_guard_in;
      foreach my $element (values %entryElements)
      {
	  my $elementId = $element->id();
	  my $elementUUID = $element->uuid();
	  my $symbolName = $element->mangledName();
	  my $ppuSymbol = $ppuSymTab->get($symbolName);
	  if (not defined $ppuSymbol)
	  {
	      print STDERR
		  "$0: warning: undefined symbol for entry element ID $elementId, ",
		  $element->name(), "\n";
	      print OUT "/* UNDEFINED, EID $elementId, $symbolName */\n";
	      next;
	  }
	  my $ppuAddr = $ppuSymbol->addr();
	  print OUT "#define \$ADDR_PPU_$elementUUID (0x${ppuAddr}U)\n";
      }
      print OUT $include_guard_out;
      close(OUT);
  }
}

my $unitSet = UnitSet->new();
$unitSet->load($elementFileName);
genExecHdrFile($unitSet);

# Tools/genpagehdr.pl
# vim:ts=2:sw=2:expandtab

