#!/usr/bin/perl
#############################################################################
## Crytek Source File
## Copyright (C) 2007, Crytek Studios
##
## Creator: Sascha Demetrio
## Date: Dec 21, 2007
## Description: GNU-make based build system
#############################################################################
	 
use warnings;
use strict;
	 
use Elements;
use Getopt::Long qw(:config no_auto_abbrev bundling);
 
use IPC::Open2;
 
# Parse the command line.
my $elementFileName;
my $fnResolveFileName;
my $targetName;
my $jobName;
my $ppuNmFileName;
my $ppuMapFileName;
my $ppuSymFileName;
my $optionHelp = 0;
my $optionVerbose = 0;
my $linear = 1;
my $bisect = 0;
GetOptions(
    'E|element-file=s' => \$elementFileName,
    'r|fnresolve=s' => \$fnResolveFileName,
    't|target=s' => \$targetName,
    'j|job=s' => \$jobName,
    'n|ppu-nm=s' => \$ppuNmFileName,
    'm|ppu-map=s' => \$ppuMapFileName,
    's|ppu-sym=s' => \$ppuSymFileName,
    'h|help' => \$optionHelp,
    'v|verbose' => \$optionVerbose,
    'l|linear' => \$linear,
    'b|bisect' => \$bisect,
    ) or exit 1;
if ($optionHelp)
{
  print <<EOF;
genfntable.pl: Script executing the post-scan action.
Synopsis:
  \$PERL genfntable.pl (Options)
Options:
-E|--element-file FILE
  Specify the element file name.  This option is required.
-r|--fnresolve FILE
  Specify the output resolver function file to be generated.  This option is
  required.
-t|--target NAME
  The name of the build target.  Typically this is the name of the final
  excutable (e.g. 'PS3Launcher').  This option is required.
-j|--job NAME
  The name of the job.  This option is required.
-n|--ppu-nm FILE
  File containing the NM output of the PPU executable.  (Either a PPU NM file,
  a PPU map file, or a PPU symbol file must be specified.)
-m|--ppu-map FILE
  File containing the PPU map file.  (Either a PPU NM file, a PPU map file, or
  a PPU symbol file must be specified.)
-s|--ppu-sym FILE
  File containing the PPU symbol file.  (Either a PPU NM file, a PPU map file, or
  a PPU symbol file must be specified.)
-v|--verbose
  Verbose operation.
-l|--linear
  Specify that the resolver function should be a linear lookup 
-b|--bisect
  Specify that the resolver function should be a binary search lookup (default)
-h|--help
  Display this help screen and exit.
EOF
  exit 0;
}
if (not defined $elementFileName)
{
  print STDERR "genfntable: no element file specified!\n";
  exit 1;
}
if (not defined $fnResolveFileName)
{
  print STDERR "genfntable: no resolve file name specified!\n";
  exit 1;
}
if (not defined $targetName)
{
  print STDERR "genfntable: no target name specified!\n";
  exit 1;
}
if (not defined $jobName)
{
  print STDERR "genfntable: no job name specified!\n";
  exit 1;
}
if (not defined $ppuNmFileName
    and not defined $ppuMapFileName
    and not defined $ppuSymFileName)
{
  print STDERR "genfntable: no PPU NM|map|sym file specified!\n";
  exit 1;
}
my $spuJobDir = "$targetName/spujob";
 
my $ppuSymTab = PPUSymTab->new();
if (defined $ppuNmFileName)
{
  $ppuSymTab->loadNM($ppuNmFileName);
}
elsif (defined $ppuSymFileName)
{
  $ppuSymTab->loadSym($ppuSymFileName);
}
else
{
  $ppuSymTab->loadMap($ppuMapFileName);
}
	 
# SPU symbol table.
my $spuSymTab = SPUSymTab->new();
 
# NOTE: Currently there's always only one page. When multi-page support is
# added, then we'll have to iterate over all page NM files to gather the SPU
# symbols.
my $pageNum = 0;
 
# The SPU NM files always reside in the same directory as the *.fnc file to be
# generated, so we'll derive the page NM file name from that.
my $jobDir = $fnResolveFileName;
$jobDir =~ s/[^\/]+$//;
 
# NOTE: These must be kept in sync with the SwCache::FnAddrMask() function
# of the code generator (in file SwCache.cpp).
my @variantBits = (
  0x00000001,
  0x00000002,
  0x08000000,
  0x10000000,
  0x20000000,
  0x40000000,
  0x80000000
);

sub gen_md5($)
{
    my $bits = shift; 
    my $cmd = 'md5sum -';
    my $result = ''; 
    my $CMDOUT; 
    my $CMDIN;
    my $md5_pid = open2 $CMDOUT, $CMDIN, $cmd
	or die "can not execute command: $!";
    print $CMDIN $bits; close($CMDIN);
    $result = <$CMDOUT>;
    $result =~ s/([A-Fa-f0-9]+)\s+.*\s*/$1/;
    waitpid $md5_pid, 0; 
    return $result; 
}

# Get the variant mask constant for a given variant.
#
# Conforming with the address notation, the function returns a HEX string
# _without_ a 0x prefix.
#
# Parameters:
# - The variant identifier.  This may be undefined (in which case the function
#   returns the constant string '0'.  If a mapped variant identifier is
#   specified, then the variant mask for the left side (the 'from' variant) is
#   generated.
sub getVariantMask ($)
{
  my $variant = shift;
 
  return '0' if not defined $variant;
  $variant =~ s/=[ML]+$//; # Strip mapped part of the variant.
  $variant =~ /^[ML]*$/ or die "invalid variant identifier '$variant'";
  my $variantLength = length $variant;
  my $mask = 0;
  for (my $i = 0; $i < $variantLength; ++$i)
  {
    my $domain = substr $variant, $i, 1;
    $domain eq 'M' or $domain eq 'L' or die;
    if ($domain eq 'L' and $i <= $#variantBits)
    {
      my $maskBit = $variantBits[$i];
      $mask |= $maskBit;
    }
  }
  my $maskHex = sprintf('%x', $mask);
  return $maskHex;
}
 
# Generate a linear resolver function.
#
# This function initializes the function IDs of the indirect call targets as a
# side effect. The addresses are compared linearily
#
# Parameters:
# - The unit set.
# - The job.
# - The output resolver function file name.
sub genLinearFnResolver ($$$)
{
  my $unitSet = shift;
  my $job = shift;
  my $outputFileName = shift;
  my $fnid = 0;
 
  #my $variantMask = 0;
  #foreach my $bit (@variantBits) { $variantMask |= $bit; }
  #$variantMask = sprintf('%x', $variantMask);

	#read MD5
  my $md5_instring = "";
  if(-e $outputFileName)
  {
      open(IN, '<', $outputFileName);
      # Read MD5, it is the commented first line.
      $md5_instring = substr(<IN>,2);
      $md5_instring =~ s/\s*$//;
      close(IN);
  }
	
	my $md5_out = '';
        my $timestamp = localtime;	
	my @fnResolveCont;
	# This includes the entry point dummy definition.
	my $entryInclude = "#include \"${jobName}_entry.cpp\"";
	push(@fnResolveCont," $entryInclude\n // Call target address resolver function\n");
	push(@fnResolveCont,"// This file was generated automatically by genfntable.pl, DO NOT EDIT!\n");
	my $line = "// Timestamp: $timestamp\n\n";
	push(@fnResolveCont,$line);
	push(@fnResolveCont,"extern \"C\" int \$fn_resolve(unsigned addr)\n {\n");
        push(@fnResolveCont,"  int index;\n   FLAG_NON_EMPTY_FNCT\n   switch (addr)\n   {\n");
 
  my @jobIndirectList = $job->indirect();
  foreach my $entryPoint (@jobIndirectList)
  {
    $entryPoint->setFnid($fnid);
    ++$fnid;
  }

  foreach my $entryPoint (@jobIndirectList)
  {
    my $ppuSymbol = $entryPoint->ppuSymbol();
    die if not defined $ppuSymbol;
    my $ppuSymbolName = $ppuSymbol->name();
 
    my $ppuAddr = $ppuSymbol->addr();
    if (not defined $ppuAddr)
    {
      print STDERR "genfntable: ",
        "no PPU address for symbol '$ppuSymbolName'\n";
      exit 1;
    }
    $ppuAddr =~ s/^0+//;
    if ($ppuAddr eq '') { $ppuAddr = '0'; }
 
    my $element = $entryPoint->element();
    my $elementId = $element->id();
    my $elementName = $element->name();
    my $elementNameSimple = $elementName;
    $elementNameSimple =~ s/\([^)]*\)\s*$//;
 
		push(@fnResolveCont,"  // Element $elementNameSimple\n   // Element ID $elementId\n");
 
    my $variant = $entryPoint->variant();
    my $variantMask = getVariantMask($variant);
    my $variantName = defined($variant) ? $variant : '<default>';
 
    my $fnid;
    my $spuSymbolName;
    if (not $entryPoint->isMapped())
    {
      $fnid = $entryPoint->fnid();
      push(@fnResolveCont,"  // Variant $variantName\n");
    } else {
      my $mappedEntryPoint = $entryPoint->findMapped(\@jobIndirectList);
      if ($mappedEntryPoint->element() != $element)
      {
        die "element mismatch in mapped entry point";
      }
      $fnid = $mappedEntryPoint->fnid();
    }
		my $line = "  case (0x${ppuAddr}U | 0x${variantMask}U):\n     index = $fnid;\n     break;\n";
		push(@fnResolveCont,$line);
		$md5_out = $md5_out . $line;
  }
  push(@fnResolveCont,"  default:\n");
  push(@fnResolveCont,"    __spu_report_resolve_func_mismatch(addr);\n");
  push(@fnResolveCont,"    break;\n  }\n  return index;\n}//stopping here means: pure virtual call (unmapped target)\n\n");
  
  my $md5_outstring = gen_md5($md5_out);
  if ($md5_instring ne $md5_outstring)
  {
	  open(OUT, '>', $outputFileName)
		  or die "can not open resolver output file '$outputFileName': $!";
		print OUT "//$md5_outstring\n";
		print OUT "@fnResolveCont";
		close(OUT) or die;		
  }
}

# Generate a bisecting resolver function.
#
# This function initializes the function IDs of the indirect call targets as a
# side effect. The addresses are compared via binary search
#
# Parameters:
# - The unit set.
# - The job.
# - The output resolver function file name.
sub genBisectFnResolver ($$$)
{
  my $unitSet = shift;
  my $job = shift;
  my $outputFileName = shift;
  my $fnid = 0;
  local *OUT;
 
  #my $variantMask = 0;
  #foreach my $bit (@variantBits) { $variantMask |= $bit; }
  #$variantMask = sprintf('%x', $variantMask);
  #my @sortedElements = sort { @{$b}[0] <=> @{$a}[0] } @items;

  my @jobIndirectList = $job->indirect();
  foreach my $entryPoint (@jobIndirectList)
  {
    $entryPoint->setFnid($fnid);
    ++$fnid;
  }

  my @mapped_addrs = ();
 
  foreach my $entryPoint ($job->indirect())
  {
    my $ppuSymbol = $entryPoint->ppuSymbol();
    die if not defined $ppuSymbol;
    my $ppuSymbolName = $ppuSymbol->name();
 
    my $ppuAddr = $ppuSymbol->addr();
    if (not defined $ppuAddr)
    {
      print STDERR "genfntable: ",
        "no PPU address for symbol '$ppuSymbolName'\n";
      exit 1;
    }
    $ppuAddr =~ s/^0+//;
    if ($ppuAddr eq '') { $ppuAddr = '0'; }

    my $element = $entryPoint->element();
    my $elementId = $element->id();
    my $elementName = $element->name();
    my $elementNameSimple = $elementName;
    $elementNameSimple =~ s/\([^)]*\)\s*$//;

	my $variant = $entryPoint->variant();
	my $variantMask = getVariantMask($variant);
	my $variantName = defined($variant) ? $variant : '<default>';
	
	my $fnid;
	my $spuSymbolName;
	if (not $entryPoint->isMapped())
	{
		$fnid = $entryPoint->fnid();
	} 
	else 
	{
		my $mappedEntryPoint = $entryPoint->findMapped(\@jobIndirectList);
		if ($mappedEntryPoint->element() != $element)
		{
			die "element mismatch in mapped entry point";
		}
		$fnid = $mappedEntryPoint->fnid();
	}

	my $combined = hex($ppuAddr) | hex($variantMask); 
	push @mapped_addrs, [$combined, $ppuAddr, $variantMask, $fnid, $element]; 
  }
  
  my @sorted_addrs = sort { @{$a}[0] <=> @{$b}[0] } @mapped_addrs;

  open(OUT, '>', $outputFileName)
    or die "can not open resolver output file '$outputFileName': $!";
  my $timestamp = localtime;
  my $entryCount = @mapped_addrs;

  print OUT <<EOF;
// Call target address resolver function.
// This file was generated automatically by genfntable.pl, DO NOT EDIT!
// Timestamp: $timestamp
 
extern "C" int \$fn_resolve(unsigned addr)
{
	struct func_map_entry { unsigned addr; int id; }; 

EOF

	if ($entryCount > 0) 
    {
		
       print OUT <<EOF;
	struct func_map_entry func_map[${entryCount}] = {

EOF
	foreach my $mapped (@sorted_addrs)
    {
        my $element = @{$mapped}[4];
		my $elementId = $element->id();
		my $elementName = $element->name();
		my $elementNameSimple = $elementName;
		my $combinedAddr = sprintf("%x",(@{$mapped}[0]));
		my $fnId = @{$mapped}[3]; 
		my $ppuAddr = @{$mapped}[1];
		my $variantMask = @{$mapped}[2];
		$elementNameSimple =~ s/\([^)]*\)\s*$//;
		print OUT <<EOF;
		// PPU : 0x${ppuAddr}U | 0x${variantMask}U
	    // Element $elementNameSimple
		// Element ID $elementId
		{ 0x${combinedAddr}U, $fnId },
EOF
    }

	print OUT <<EOF;
	}; 

	func_map_entry *begin = &func_map[0]; 
	func_map_entry *end   = begin + $entryCount;
	while (begin < end) {
		func_map_entry *midpoint = begin + ((end-begin)>>1);
		if (midpoint->addr < addr) 
			begin = midpoint + 1; 
		else 
			end = midpoint; 
	}
	if (begin->addr == addr)
		return begin->id;	

EOF
	}

	print OUT <<EOF;

	//stopping here means: pure virtual call (unmapped target);
	__spu_report_resolve_func_mismatch(addr);
	return -1;
}

EOF
 
  close(OUT) or die;
}
 
my $unitSet = UnitSet->new();
$unitSet->load($elementFileName);
my %jobMap = $unitSet->getJobMap();
 
if (not defined $jobMap{$jobName})
{
  print STDERR "genfntable: unknown job '$jobName' specified!\n";
  exit 1;
}
my $job = $jobMap{$jobName};
 
if (defined $fnResolveFileName)
{
  $job->setEntryPointSymbols($ppuSymTab, undef);

  die "fnresolve function configured to be linear and bisecting"
	  if ($linear && $bisect);

  if ($linear == 1)
  {
	  genLinearFnResolver($unitSet, $job, $fnResolveFileName);
  }

  if ($bisect == 1)
  {
	  genBisectFnResolver($unitSet, $job, $fnResolveFileName);
  }
}
else
{
  die "unknown option!";
}
 
exit 0;
 
# Tools/genfntable.pl
# vim:ts=2:sw=2:expandtab
