#!/usr/bin/perl
#############################################################################
## Crytek Source File
## Copyright (C) 2008, Crytek Studios
##
## Creator: Sascha Demetrio
## Date: Jan 3, 2008
## Description: GNU-make based build system
#############################################################################

use warnings;
use strict;

use Elements;
use Getopt::Long qw(:config no_auto_abbrev bundling);
use File::Basename;
use Fcntl qw(:flock SEEK_END);

# Parse the command line.
my $elementFileName;
my $pageDescFileName;
my $pageOFileName;
my $targetName;
my $jobName;
my $ppuNmFileName;
my $ppuMapFileName;
my $ppuSymFileName;
my $pageGenTool;
my $optionHelp = 0;
my $optionVerbose = 0;
my $maxPageSize;
my $incredibat;
my $incrediprefix;
GetOptions(
    'incredibat=s' => \$incredibat,    
    'incredi-prefix=s' => \$incrediprefix,    
    'E|element-file=s' => \$elementFileName,
    'd|desc-file=s' => \$pageDescFileName,
    'O|object-file=s' => \$pageOFileName,
    't|target=s' => \$targetName,
    'j|job=s' => \$jobName,
    'n|ppu-nm=s' => \$ppuNmFileName,
    'm|ppu-map=s' => \$ppuMapFileName,
    's|ppu-sym=s' => \$ppuSymFileName,
    'pagegen-tool=s' => \$pageGenTool,
    'h|help' => \$optionHelp,
    'm|max-page-size=s' => \$maxPageSize,
    'v|verbose' => \$optionVerbose
    ) or exit 1;
if ($optionHelp)
{
	print <<EOF;
genpagedesc.pl: Generate a page description file.
Synopsis:
  \$PERL genpagedesc.pl (Options)
Options:
-E|--element-file FILE
  Specify the element file name.  This option is required.
-d|--desc-file FILE
  Specify the output page description file to be generated.
	This option is required.
-O|--object FILE
  Specify the (possibly patched) ELF object file of the page.  This option is
	required.
-t|--target NAME
  The name of the build target.  Typically this is the name of the final
	excutable (e.g. 'PS3Launcher').  This option is required.
-j|--job NAME
  The name of the job.  This option is required.
-n|--ppu-nm FILE
  File containing the NM output of the PPU executable.  (Either a PPU NM file,
  a PPU map file, or a PPU symbol file must be specified.)
-m|--ppu-map FILE
  File containing the PPU map file.  (Either a PPU NM file, a PPU map file, or
  a PPU symbol file must be specified.)
-s|--ppu-sym FILE
  File containing the PPU symbol file.  (Either a PPU NM file, a PPU map file, or
  a PPU symbol file must be specified.)
--pagegen-tool TOOL
  The location of the PageGen tool binary.  This option is required.
-s|max-page-size SIZE (kb)
  The maximum size of a generated code page in kilobytes.
-v|--verbose
  Verbose operation.
-h|--help
  Display this help screen and exit.
EOF
	exit 0;
}
if (not defined $elementFileName)
{
  print STDERR "genpagedesc: no element file specified!\n";
	exit 1;
}
if (not defined $pageDescFileName)
{
	print STDERR "genpagedesc: no page description file specified!\n";
	exit 1;
}
if (not defined $pageOFileName)
{
	print STDERR "genpagedesc: no page object file specified!\n";
	exit 1;
}
if (not defined $targetName)
{
	print STDERR "genpagedesc: no target name specified!\n";
	exit 1;
}
if (not defined $jobName)
{
	print STDERR "genpagedesc: no job name specified!\n";
	exit 1;
}
if (not defined $ppuNmFileName
    and not defined $ppuMapFileName
    and not defined $ppuSymFileName)
{
	print STDERR "genpagedesc: no PPU NM|map|sym file specified!\n";
	exit 1;
}
if (not defined $pageGenTool)
{
	print STDERR "genpagedesc: no page generator tool specified!\n";
	exit 1;
}

#unlink desc-file as the rm-command has been moved into here
unlink($pageDescFileName); 

$pageGenTool =~ s/\\/\//g;
my $spuJobDir = "$targetName/spujob";

# Extract the page ID from the page description file name.
if (not $pageDescFileName =~ /_([0-9]+)\.[a-z]+$/)
{
	print STDERR "genpagedesc: ",
		"can not extract page ID from page description file name ",
		"'$pageDescFileName'\n";
	exit 1;
}
my $pageNum = int $1;

# Load the PPU symbol table.
my $ppuSymTab = PPUSymTab->new();
if (defined $ppuNmFileName)
{
	$ppuSymTab->loadNM($ppuNmFileName);
}
elsif (defined $ppuSymFileName)
{
  $ppuSymTab->loadSym($ppuSymFileName);
}
else
{
  $ppuSymTab->loadMap($ppuMapFileName);
}

# Load the SPU symbol table.
#
# The SPU NM files always reside in the same directory as the *.page (page
# description) file to be # generated, so we'll derive the page NM file name
# from that.
my $spuSymTab = SPUSymTab->new();
my $jobDir = $pageDescFileName;
$jobDir =~ s/[^\/]+$//;
my $pageNMFileName = $pageDescFileName;
$pageNMFileName =~ s/\.[a-z]+$/.nm/;
my $spuTable = $spuSymTab->load($jobName, $jobDir);

# Generate the derived output file names.
my $pagePatchedOFileName = $pageOFileName;
$pagePatchedOFileName =~ s/\.o$/_patched.o/;
my $pagePlainOFileName = $pageOFileName;
$pagePlainOFileName =~ s/_tmp\.o$/.o/;
my $pageCrossPageCallsFileName = $pagePlainOFileName;
$pageCrossPageCallsFileName =~ s/\.o$/_crossbubblecalls.txt/;
my $pageEntryPointsFileName = $pagePlainOFileName;
$pageEntryPointsFileName =~ s/\.o/_entrypoints.txt/;

sub genPageDescFiles ($)
{
	my $unitSet = shift;
	local *OUT;
	# Get the job object.
	my %jobMap = $unitSet->getJobMap();
	if (not defined $jobMap{$jobName})
	{
		print STDERR "genpagedesc: unknown job name '$jobName' specified\n";
		exit 1;
	}
	my $job = $jobMap{$jobName};

	# load and parse the page table file 
	my $pageTable = PageTable->new();
	$pageTable->load("$jobDir/$jobName.pagetable");

	# retrieve the code page 
	my $codePage = @{$pageTable->{CODEPAGES}}[$pageNum];

	# Remove old output files if present.
	unlink $pageDescFileName;
	unlink $pagePatchedOFileName;
	unlink $pageCrossPageCallsFileName;
	unlink $pageEntryPointsFileName;


    # Generate the entry point list of the page.
	my @pageEntryPoints = ( );
	open(OUT, '>', $pageEntryPointsFileName)
		or die "can not open entry points file	'$pageEntryPointsFileName': $!";

	if (@{$pageTable->{CODEPAGES}} == 1)
	{
		# Page ID 0 contains the job entry point.
		my $entryPoint = $job->entryPoint();
		my $entryElement = $entryPoint->element();
		my $entryElementUUID = $entryElement->uuid();
		my $symbolName = $spuSymTab->findSymbol($entryElementUUID);
		if (not defined $symbolName)
		{
			my $entryElementName = $entryElement->name();
			print STDERR "genpagedesc: ",
			"can not find job entry point symbol ",
			"for entry element '$entryElementName' in page 0\n";
			exit 1;
		}
		print OUT $symbolName, ' ', '1', "\n";
 		foreach my $entryPoint ($job->indirect())
 		{
 			next if $entryPoint->isMapped();
 			my $element = $entryPoint->element();
 			my $elementUUID = $element->uuid();
 			my $variant = $entryPoint->variant();
 			my $symbolName = $spuSymTab->findSymbol($elementUUID, $variant);
 			if (not defined $symbolName)
 			{
 				my $elementName = $element->name();
 				print STDERR "genpagedesc: ",
 				"can not find symbol for entry point element '$elementName', ",
 				"variant '$variant' in page $pageNum\n";
 				exit 1;
 			}
 			print OUT $symbolName, ' ', '1', "\n";
 		}
	}
	else
	{
		foreach my $pageMethod (@{$codePage->{METHODS}})
		{
			if ($pageMethod->{ENTRY} != 0)
			{
				my $entryElementUUID = $pageMethod->{UUID};
				my $entryElement =
					$unitSet->{ELEMENT_UUID_MAP}{$entryElementUUID};
				my $entryElementVariant = $pageMethod->{VARIANTNAME};
				if ($entryElementVariant =~ /\$V(.*)_/)
				{
					$entryElementVariant = $1;
				}
				else 
				{
					$entryElementVariant = undef;
				}
				my $symbolName = $spuSymTab->findSymbol($entryElementUUID,
									$entryElementVariant);
				if (not defined $symbolName)
				{
					my $entryElementName = $entryElement->name();
					print STDERR "genpagedesc: ",
					"can not find page entry point symbol ",
					"for entry element ",
					"$entryElementName($entryElementUUID)\n";
					exit 1;
				}
				print OUT $symbolName, ' ', ($pageMethod->{WEAK} ? '1' : '0'), "\n";
			}
		}
	}
	close(OUT) or die;

	# Generate the cross-page calls.
	open(OUT, '>', $pageCrossPageCallsFileName)
		or die "can not open cross page calls file "
		. "'$pageCrossPageCallsFileName': $!";
	foreach my $crossCall (@{$codePage->{CROSSCALLS}})
	{
		my $crossCallElementUUID = $crossCall->{uuid};
		my $crossCallVariant = $crossCall->{variant};
		my $crossCallElement =
			$unitSet->{ELEMENT_UUID_MAP}{$crossCallElementUUID};
		if ($crossCallVariant =~ /\$V(.*)_/)
		{
			$crossCallVariant = $1;
		}
		else 
		{
			$crossCallVariant = undef;
		}
		my $symbolName = $spuSymTab->findSymbol($crossCallElementUUID,
							$crossCallVariant);
		if (not defined $symbolName)
		{
			my $crossCallElementName = $crossCallElement->name();
			print STDERR "genpagedesc: ",
			"can not find job symbol ",
			"for cross page call to '$crossCallElementName'", 
			" [$crossCallElementUUID] in page $pageOFileName\n";
			exit 1;
		}
		print OUT "$symbolName \n";
	}
	close(OUT) or die;
}

sub runPageGenToolLocal ()
{
	# Run the pagegen tool.
	my $commandLine = "$pageGenTool -f $pageOFileName";
	if (defined $maxPageSize)
	{
		$commandLine = $commandLine .  " -p $maxPageSize";
	}
	#system "sh -c \"$commandLine\";";
	system $commandLine;
	if ($? == -1)
	{
		print STDERR "genpagedesc: ",
			"can not execute pagegen tool '$pageGenTool': $!\n";
		exit 1;
	}
	if ($? == -2)
	{
		print STDERR "genpagedesc: ",
			"codepage is larger than maximum allowed codepage size '$pageGenTool': $!\n";
		exit -2;
	}
	elsif ($? != 0)
	{
		print STDERR "genpagedesc: ",
		  "pagegen tool '$pageGenTool' terminated with status code $?\n";
		exit 1;
	}
}

sub runPageGenToolRemote ()
{
    # Run the pagegen tool.
    my $commandLine = "$pageGenTool -f $pageOFileName";
    my $caption = "page generation $pageOFileName";
    if (defined $maxPageSize)
    {
	$commandLine = $commandLine .  " -p $maxPageSize";
    }
    $commandLine = "/group=\"$incrediprefix\" /command "
	. $commandLine;

    # path fixup hack
    $incredibat =~ s/\\\//\//g;
    $incredibat =~ s/\\/\//g;

    my $incredifile;
    open($incredifile, "+>>$incredibat") 
	or die("$0: cannot open $incredibat");

    flock($incredifile, LOCK_EX) or die "flock: $!";
    #print "\@xgSubmit /caption=\"$caption\" " . $commandLine . "\n";
    print $incredifile "\@xgSubmit /caption=\"$caption\" " . $commandLine . "\n";
    flock($incredifile, LOCK_UN);
    close($incredifile);
}

sub removeTempFiles ()
{
	unlink $pageCrossPageCallsFileName;
	unlink $pageEntryPointsFileName;
}

my $unitSet = UnitSet->new();
$unitSet->load($elementFileName);
genPageDescFiles($unitSet);
if (not defined $incredibat)
{
    runPageGenToolLocal();
}
else
{
   runPageGenToolRemote();
}

# Tools/genpagedesc.pl
# vim:ts=2:sw=2:expandtab

