#!/usr/bin/perl
#############################################################################
## Crytek Source File
## Copyright (C) 2008, Crytek Studios
##
## Creator: Sascha Demetrio
## Date: Jan 16, 2008
## Description: GNU-make based build system
#############################################################################

use warnings;
use strict;

use Elements;
use Getopt::Long qw(:config no_auto_abbrev bundling);

# Parse the command line.
my $outputFileName;
my $elementFileName;
my $jobName;
my $ppuNmFileName;
my $ppuMapFileName;
my $ppuSymFileName;
my $optionHelp = 0;
my $optionVerbose = 0;
GetOptions(
    'E|element-file=s' => \$elementFileName,
    'o|header-file=s' => \$outputFileName,
		'j|job=s' => \$jobName,
    'n|ppu-nm=s' => \$ppuNmFileName,
    'm|ppu-map=s' => \$ppuMapFileName,
    's|ppu-sym=s' => \$ppuSymFileName,
    'h|help' => \$optionHelp,
    'v|verbose' => \$optionVerbose
    ) or exit 1;
if ($optionHelp)
{
  print <<EOF;
genpagehdr.pl: Generate the header file for a the page.
Synopsis:
  \$PERL genpagehdr.pl (Options)
Options:
-E|--element-file FILE
  Specify the element file name.  This option is required.
-o|--header-file FILE
  Specify the output header file to be generated.
  This option is required.
-j|--job NAME
  The name of the job.  This option is required.
-n|--ppu-nm FILE
  File containing the NM output of the PPU executable.  (Either a PPU NM file,
  a PPU map file, or a PPU symbol file must be specified.)
-m|--ppu-map FILE
  File containing the PPU map file.  (Either a PPU NM file, a PPU map file, or
  a PPU symbol file must be specified.)
-s|--ppu-sym FILE
  File containing the PPU symbol file.  (Either a PPU NM file, a PPU map file, or
  a PPU symbol file must be specified.)
-v|--verbose
  Verbose operation.
-h|--help
  Display this help screen and exit.
EOF
  exit 0;
}
if (not defined $elementFileName)
{
  print STDERR "$0: no element file specified!\n";
  exit 1;
}
if (not defined $outputFileName)
{
  print STDERR "$0: no output header file specified!\n";
  exit 1;
}
if (not defined $jobName)
{
	print STDERR "$0: no job name specified!\n";
	exit 1;
}
if (not defined $ppuNmFileName
    and not defined $ppuMapFileName
    and not defined $ppuSymFileName)
{
  print STDERR "$0: no PPU NM|map|sym file specified!\n";
  exit 1;
}

# Extract the page ID from the output file name.
if (not $outputFileName =~ /_([0-9]+)\.[a-z]+$/)
{
  print STDERR "$0: ",
    "can not extract page ID from page output file name ",
    "'$outputFileName'\n";
  exit 1;
}
my $pageNum = int $1;

# Load the PPU symbol table.
my $ppuSymTab = PPUSymTab->new();
if (defined $ppuNmFileName)
{
  $ppuSymTab->loadNM($ppuNmFileName);
}
elsif (defined $ppuSymFileName)
{
  $ppuSymTab->loadSym($ppuSymFileName);
}
else
{
  $ppuSymTab->loadMap($ppuMapFileName);
}

sub genPageHdrFile ($)
{
  my $unitSet = shift;
  local *OUT;

  # Get the job object.
  my %jobMap = $unitSet->getJobMap();
  if (not defined $jobMap{$jobName})
  {
    print STDERR "genpagedesc: unknown job name '$jobName' specified\n";
    exit 1;
  }
  my $job = $jobMap{$jobName};

  my %entryElements = ( );
  foreach my $entryPoint ($job->indirect())

  {
    next if $entryPoint->isMapped();
    my $element = $entryPoint->element();
    my $elementId = $element->id();
    $entryElements{$elementId} = $element;
  }

  open(OUT, '>', $outputFileName)
    or die "can not open output file '$outputFileName': $!";
  my $includeGuard = "_PAGEHDR_${jobName}_${pageNum}_";
  print OUT "/* File generated by $0, " . (scalar localtime) . " */\n\n";
  print OUT "#ifndef $includeGuard\n#define $includeGuard 1\n\n";
  foreach my $element (values %entryElements)
  {
    my $elementId = $element->id();
    my $symbolName = $element->mangledName();
    my $ppuSymbol = $ppuSymTab->get($symbolName);
    if (not defined $ppuSymbol)
    {
      print STDERR
          "$0: warning: undefined symbol for entry element ID $elementId, ",
          $element->name(), "\n";
      print OUT "/* UNDEFINED, EID $elementId, $symbolName */\n";
      next;
    }
    my $ppuAddr = $ppuSymbol->addr();
    print OUT "#define \$ADDR_PPU_$elementId (0x${ppuAddr}U)\n";
  }
  print OUT "\n#endif /* $includeGuard */\n\n";
  close(OUT);
}

my $unitSet = UnitSet->new();
$unitSet->load($elementFileName);
genPageHdrFile($unitSet);

# Tools/genpagehdr.pl
# vim:ts=2:sw=2:expandtab

