#!/usr/bin/perl
#############################################################################
## Crytek Source File
## Copyright (C) 2008, Crytek Studios
##
## Creator: Chris Raine
## Date: Nov 5, 2008
## Description: GNU-make based build system
#############################################################################
use warnings;
use strict;
use Elements;
use Getopt::Long qw(:config no_auto_abbrev bundling);
use warnings;

# Parse the command line.
# Scans a job size file and generates a new page table file by filling
# pages until the max page size if reached. The ma

my $textSegAlign = 64;
my $maxPageSize = 32.0;
my $sizeFile ;
my $origPageTable;
my $newPageTable;
my $optionHelp = 0;
my $verboseLevel = 0; 
my $initialPageSize = 0.0;
GetOptions(
	'm|max-page-size=s' => \$maxPageSize,
	's|size-file=s' => \$sizeFile,
	'p|pagetable=s' => \$origPageTable,
	'o|output=s' => \$newPageTable,
	'v|verbose=s' => \$verboseLevel,
	'h|help' => \$optionHelp,
	'i|initial-page-size=s' => \$initialPageSize
	) or exit 1;
if ($optionHelp)
{
	print <<EOF;
genpagetable.pl: scans the output of JobGen for page sizes and
	generates a page defintion file according to the original pagetable.
Synopsis:
  \$PERL genpagetable.pl (Options)
Options:
	-m|--max-page-size SIZE 
	The maximum size of a code page in kb (optional, default 32 kb)
	-s|--size-file FILE
	The path to the size results of the single page mode
	-p|--pagetable FILE
	The path to original page file
	-o|--output FILE 
	The output pagetable file
	-v|--verbose LEVEL 
	The level of verbosity. Default 0
	-i|--initial-page-size
	The initial page size (reserved size for datasegments)
	-h|--help
	Display this help screen and exit.

EOF
		exit 0;
}
if (not defined $sizeFile)
{
	print STDERR 'genpagetable.pl: no size file specified\n'; 
	exit -1; 
}
if (not defined $origPageTable)
{
	print STDERR 'genpagetable.pl: no original page table specified\n'; 
	exit -1; 
}
if (not defined $newPageTable)
{
	print STDERR 'genpagetable.pl: no output file specified\n'; 
	exit -1; 
}

my @items = ();
my @sizes = ();
my %elements = ();

if (not open(SF, '<', $sizeFile))
{
	print STDERR 'genpagetable.pl: could not open file $sizeFile';
	exit -1;
}

$maxPageSize = $maxPageSize * 1024.0;
$initialPageSize = $initialPageSize * 1024.0; 

while (<SF>)
{
	if (/^[0-9a-f]+\s+([0-9a-f]+)\s+(\w)\s+([^\s]+)\s*$/)
	{
		next if ($2 ne 'T' and $2 ne 't');
		my $symbolName = $3;
		my $pageSize = $1;
		my $elementId;
		my $variantSuffix; 
		if ($symbolName =~ /\$E([0-9]+)_/)
		{
			$elementId = $1;
		}
		if (not defined $elementId)
		{
			print STDOUT "Warning: symbol $symbolName has no encoded element id\n" 
				if ($verboseLevel > 2);
			next;
		}
		if ($symbolName =~ /(\$V[^_]+_)/)
		{
			$variantSuffix = $1;
		}
		if (not defined $variantSuffix)
		{
			print STDOUT "Warning: symbol $symbolName has no encoded variant Suffix\n" 
				if ($verboseLevel > 2);
			$variantSuffix = 'undef';
		}
		my $elementKey = join "_", $elementId, $variantSuffix;
		my $elementSize = $textSegAlign + (hex($pageSize)-1) & ~($textSegAlign-1);
		$elements{ $elementKey } = $elementSize;
		print STDOUT "found $symbolName ($pageSize kb)\n"
			if ($verboseLevel > 2);
	}
}



my $pageTable = PageTable->new();
$pageTable->load($origPageTable);

for (my $pageNum=0; $pageNum<@{$pageTable->{CODEPAGES}}; ++$pageNum)
{
	my $codePage = @{$pageTable->{CODEPAGES}}[$pageNum];
	for (my $pageIndex=0; $pageIndex<@{$codePage->{METHODS}};
		 ++$pageIndex)
	{
		my $pageItem = @{$codePage->{METHODS}}[$pageIndex];
		my $elementId = $pageItem->{ID};
		my $variantSuffix = $pageItem->{VARIANTNAME};
		my $elementKey = join "_", $elementId, $variantSuffix; 
		die "required element $pageItem->{NAME} not found in size table!" 
			if (not exists $elements{$elementKey});
		my $sizeItem = $elements{$elementKey};
		die "$pageItem->{NAME} ($sizeItem kb) is larger than the maximum page size ($maxPageSize kb)!" 
			if ($sizeItem >= $maxPageSize);
		push @items, [$sizeItem,
					  $pageItem->{NAME},
					  $pageItem->{ID}, 
					  $pageItem->{VARIANTKEY}];
	}
}
my @sortedElements = sort { @{$b}[0] <=> @{$a}[0] } @items;

if (not open(OP,'>', $newPageTable))
{
	print STDERR 'genpagetable.pl: could not open new page table';
	exit -1;
}

my @result = (); 
my $pageCount = 0;
while (@sortedElements>0)
{
	my $pageSize = $initialPageSize;
	my @page = ();
	while (1)
    {
		my $hasAdded = 0; 
		for (my $itemIndex=0; $itemIndex<@sortedElements; ++$itemIndex)
		{
			my $item = $sortedElements[$itemIndex];
			if (not defined $item)
			{
				next;
			}
 			if ($pageSize+@$item[0] < $maxPageSize)
			{
 				push @page,[@$item[1],@$item[2],@$item[3]];
 				$hasAdded = 1; 
 				$pageSize = $pageSize+@$item[0];
 				delete $sortedElements[$itemIndex];
 				last;
 			}
		}
		if ($hasAdded == 0)
		{
			last;
		}
    }
	print STDOUT "created page $pageCount ($pageSize kb)\n" if ($verboseLevel > 0);
	print OP "[$pageCount]\n";
	foreach my $item (@page)
	{
		print OP "\t@$item[1] '@$item[0]' @$item[2] \n"; 
		print STDOUT "\t @$item[1] '@$item[0]' \n" 
			if ($verboseLevel > 1);
	}
	++$pageCount;
}



