#!/usr/bin/perl
#############################################################################
## Crytek Source File
## Copyright (C) 2007, Crytek Studios
##
## Creator: Sascha Demetrio
## Date: Jul 17, 2007
## Description: GNU-make based build system
#############################################################################

# This script generates the C++ files for an SPU job.
use warnings;
use strict;
use Elements;
use Getopt::Long qw(:config no_auto_abbrev bundling);
use File::Basename;
use Fcntl qw(:flock SEEK_END);
use Env; 

my $perl = '/usr/bin/perl'; 

# Parse the command line.
my $indexFile;
my @entryElementNames = ( );
my $jobName;
my $optionJobExec = 0;
my $optionPageTable = 0;
my $depFile;
my $toolDir;
my $toolBinDir;
my $outputFile;
my $headerFile;
my $optHostSystem;
my $optMakeSystem;
my $optCygwinBaseDir;
my @optConfigVars = ( );
my $optionHelp = 0;
my $optionVerbose = 0;
my $optionVirtualMap;
my $optionCodePageStrategy;
my @optCrycgConfig = ();
my $optionCryCGProfile = 0;
my $incredibat;
my $incrediprefix;
my $sizeMode = 0;
my $useCryCGWrapper; 
my $exec_prefix;
my $responseFile; 
my $optionCodeMirror = 0; 
GetOptions(
    'i|index-file=s' => \$indexFile,
    'incredibat=s' => \$incredibat,    
    'incredi-prefix=s' => \$incrediprefix,    
    'crycg-wrapper=s' => \$useCryCGWrapper, 
    'E|entry=s' => \@entryElementNames,
    'o|output=s' => \$outputFile,
    'H|header=s' => \$headerFile,
    'j|job=s' => \$jobName,
    's|size_mode' => \$sizeMode,    
    'x|job-exec' => \$optionJobExec,
    't|page-table' => \$optionPageTable,
    'd|dep-file=s' => \$depFile,
    't|tool-dir=s' => \$toolDir,
    'h|help' => \$optionHelp,
    'v|verbose' => \$optionVerbose,
    'y|profile' => \$optionCryCGProfile,
    'host-system=s' => \$optHostSystem,
    'make-system=s' => \$optMakeSystem,
    'b|basedir|base-dir=s' => \$optCygwinBaseDir,
    'D|define=s' => \@optConfigVars,
    'V|virtual_map|virtualmap=s' => \$optionVirtualMap,
    'p|codepage_strategy=s' => \$optionCodePageStrategy,
    'C|config-vars=s' => \@optCrycgConfig,
    'perl=s' => \$perl,
    'exec-prefix=s' => \$exec_prefix,
    'response-file=s' => \$responseFile,
    'code-mirror' => \$optionCodeMirror
    ) or exit 1;
if ($optionHelp)
{
  print <<EOF
$0: Script generating an SPU job C++ file.
Synopsis:
  \$PERL $0 (Options)
Options:
-y|--profile
  Use a special CryCG executable with profiling information 
-i|--index-file FILE
  The name of the CCG index file name. (used in crycg mode)
-F|--code-file FILE
  The name of the prelinked code file. (used in cm_backend mode)
-E|--entry NAME
  The name of a job entry point function or method.  Multiple entry points may
  be specified.
-j|--job NAME
  The name of the job.  This option is required.
-x|--job-exec
  If this option is specified, then a job execute file is generated.
-d|--dep-file FILE
  The name of the dependency file.
-t|--tool-dir DIR
  The directory containing the crycg executables.  This is the directory
  containing the 'host-linux' and 'host-win32' subdirectories.
--host-system HOST_SYSTEM
  Specify the host system type.  Must be 'Linux', 'Cygwin', or 'Windows'.
-b|--basedir|--base-dir DIR
  The Cygwin base directory.  This is the directory mounted to '/base' within
  Cygwin.
-D|--define NAME=VALUE
  Config substitution variable to be passed to CryCG.
-v|--verbose
  Verbose operation.
-V|--virtual-map|--virtualmap FILE
  The Filename containing the virtual mapping information of the
  devirtualization process. The filename will be passed to CryCG as
  the -CVirtualMapping option
-h|--help
  Display this help screen and exit.
--perl=PERL
  The path to the perl interpreter which is to be used.
--exec-prefix PREFIX
  An optional prefix string that is prepended to all invokations of
  crycg. 
--response-file FILE 
  If the path to a response file is given, the command line generated
  by genspu.pl will be written into the response file instead of
  executing crycg directly
--code-mirror
  If --code-mirror is attached to the command line, cm_backend will be executed
  instead of crycg. 
EOF
  ;
  exit 0;
}
if (not defined $indexFile)
{
  print STDERR "$0: no index file specified!\n";
  exit 1;
}
if (not @entryElementNames)
{
  print STDERR "$0: no entry element name specified!\n";
  exit 1;
}
if (not defined $jobName)
{
  print STDERR "$0: no job name specified!\n";
  exit 1;
}
if (not defined $outputFile)
{
  print STDERR "$0: no output file name specified!\n";
  exit 1;
}
if (not defined $optHostSystem)
{
  print STDERR "$0: no host system specified!\n";
  exit 1;
}
if (not defined $optMakeSystem)
{
  print STDERR "$0: no make system specified!\n";
  exit 1;
}

# Note: NOT configurable via command line - the Cygwin drive prefix (including
# trailing /).  Cygwin default is '/cygdrive/' - changed to '/' for MinGW
# compatibility.
#my $optCygwinDrivePrefix = '/cygdrive/';
my $optCygwinDrivePrefix = '/';

my $optCygwinMode = 0;
my $optMingWMode = 0;
if ($optHostSystem eq 'Cygwin' and not $optMakeSystem eq 'MinGW') 
{ 
    $optCygwinMode = 1; 
}
if ($optHostSystem eq 'MingW' and $optMakeSystem eq 'MinGW') 
{ 
    $optMingWMode = 1; 
}


# Fix the specified Cygwin dir.
if (defined $optCygwinBaseDir)
{
    $optCygwinBaseDir =~ s#\\#/#g;
    $optCygwinBaseDir =~ s#//+#/#g;
    #while ($optCygwinBaseDir =~ s#[^/]+/\.\./##g) { }
    #$optCygwinBaseDir =~ s#/[^/]+/\.\.$##;
}

# If no tool directory was specified, locate the tool directory and binary
# directory containing the CryCG executable using the command name.  We'll
# assume that this script is _not_ in the system path and the we can get the
# directory containing the script from $0.
if (not defined $toolDir)
{
  my $sepIndex = rindex $0, '/';
  if ($sepIndex == -1) { $sepIndex = rindex $0, '\\'; }
  if ($sepIndex == -1)
  {
    print STDERR "$0: can not locate tool directory!\n";
    exit 1;
  }
  $toolDir = substr $0, 0, $sepIndex;
}
if (not defined $toolBinDir)
{
  if ($^O eq 'linux')
  {
    $toolBinDir = "$toolDir/host-linux";
  }
  elsif ($^O eq 'cygwin')
  {
    $toolBinDir = "$toolDir/host-win32";
  }
  elsif ($^O eq 'msys')
  {
    $toolBinDir = "$toolDir/host-win32";
  }
  else
  {
    print STDERR "$0: ",
      "can not locate tool binary directory for OS '$^O'!\n";
    exit 1;
  }
}
if (not -d $toolDir)
{
  print STDERR "$0: tool directory '$toolDir' does not exist!\n";
  exit 1;
}
if (not -d $toolBinDir)
{
  print STDERR "$0: ",
    "tool binary directory '$toolBinDir' does not exist!\n";
  exit 1;
}

# Locate the CryCG executable.
my $tool;
my @toolOptions = ( );
if (defined $useCryCGWrapper) 
{ 
    $tool = $useCryCGWrapper;
} 
elsif ($optionCodeMirror != 0) 
{ 
    if (-x "$toolBinDir/cm_backend")
    {
        $tool = "$toolBinDir/cm_backend";
    }
    elsif (-f "$toolBinDir/cm_backend.exe")
    {
        $tool = "$toolBinDir/cm_backend.exe";
    }
    else
    {
        print STDERR "$0: ",
        "can not locate generator tool 'cm_backend' or 'cm_backend.exe'!\n";
        exit 1;
    }
} 
else 
{ 
    if ($optionCryCGProfile == 1)
    {
        if (-x "$toolBinDir/crycg_profile")
        {
            $tool = "$toolBinDir/crycg_profile";
        }
        elsif (-f "$toolBinDir/crycg_profile.exe")
        {
            $tool = "$toolBinDir/crycg_profile.exe";
        }
        else
        {
            print STDERR "$0: ",
            "can not locate generator tool 'crycg_profile' or 'crycg_profile.exe'!\n";
            exit 1;
        }
    }
    else
    {
        if (-x "$toolBinDir/crycg")
        {
            $tool = "$toolBinDir/crycg";
        }
        elsif (-f "$toolBinDir/crycg.exe")
        {
            $tool = "$toolBinDir/crycg.exe";
        }
        else
        {
            print STDERR "$0: ",
            "can not locate generator tool 'crycg' or 'crycg.exe'!\n";
            exit 1;
        }
    }
}
if (not $optionVerbose) { push @toolOptions, '-CLogLevel:notice'; }
if (defined $headerFile) { push @toolOptions, "-CHeaderFile:$headerFile"; }
if (defined $optionCodePageStrategy) 
{ 
    push @toolOptions, "-CCodePageStrategy:$optionCodePageStrategy"; 
}
# Locate the CryCG configuration file.
if (-f "$toolDir/crycg.conf")
{
    push @toolOptions, '-c', "$toolDir/crycg.conf";
}
else
{
    print STDERR "$0 WARNING: ",
    "crycg configuration file not found, using default configuration!\n";
}

# Pass on -C options.
foreach my $configVar (@optCrycgConfig)
{
    push @toolOptions, "-C$configVar";
}

# Pass on -D options.
foreach my $configVar (@optConfigVars)
{
    push @toolOptions, "-D$configVar";
}

# Pass on the -V (virtual mapping) file name options if present
if (defined $optionVirtualMap)
{
    push @toolOptions, "-CVirtualMapping:$optionVirtualMap";
}

# Add a suffix to the basename (without extension) of the specified file name.
sub applySuffix ($$)
{
    my $fileName = shift;
    my $suffix = shift;

    my $dotIndex = rindex $fileName, '.';
    if ($dotIndex > 0)
    {
	$fileName =
	    substr($fileName, 0, $dotIndex)
      .$suffix
      .substr($fileName, $dotIndex);
  }
  else
  {
    $fileName .= $suffix;
  }
  return $fileName;
}

# Fix dependencies when running in Cygwin mode.
sub fixDepsCygwin ($$)
{
  my $inFile = shift;
  my $outFile = shift;
  my @lines = ( );
  local *IN;
  local *OUT;

  open(IN, "$inFile")
    or die "Can't open dependency file '$inFile' for fixdeps: $!";
  while (<IN>) { push @lines, $_; }
  close(IN);
  unlink $inFile;
  unlink $outFile;
  open(OUT, ">$outFile")
    or die "Can't open dependency file '$outFile' for fixdeps: $!";
  foreach my $line (@lines)
  {
    # Remove "DIR/.." patterns.
		#while ($line =~ s#[^/]+/\.\./##g) { }
    if (defined $optCygwinBaseDir)
    {
      my $index = index lc $line, lc $optCygwinBaseDir;
      if ($index >= 0)
      {
        # Make sure the base dir is a prefix of a filename.
        if ($index > 0 and (substr $line, $index - 1, 1) ne ' ')
        {
          $index = -1;
        }
      }
      if ($index >= 0)
      {
        my $prefix = substr $line, 0, $index;
        my $suffix = substr $line, $index + length $optCygwinBaseDir, -1;
        $line = $prefix . '/base/' . $suffix;
        $line =~ s#//+#/#g;
      }
    }
    if ($line =~ /^(\s+)([a-z]):\/(.*$)/i)
    {
      my $driveLetter = lc $2;
      $line = "$1$optCygwinDrivePrefix$driveLetter/$3\n";
    }
    elsif ($line =~ /^(\S+:\s+)([a-z]):\/(.*$)/i)
    {
      my $driveLetter = lc $2;
      $line = "$1$optCygwinDrivePrefix$driveLetter/$3\n";
    }
    if ($line =~ /^([a-z]):(.*$)/i)
    {
      my $driveLetter = lc $1;
      $line = "$optCygwinDrivePrefix$driveLetter$2\n";
    }
    $line =~ s/\s+$//;
    print OUT "$line\r\n";
  }
  close(OUT);
}

# Fix dependencies when running in Cygwin mode.
sub fixDepsMingW ($$)
{
  my $inFile = shift;
  my $outFile = shift;
  my @lines = ( );
  local *IN;
  local *OUT;

  open(IN, "$inFile")
    or die "Can't open dependency file '$inFile' for fixdeps: $!";
  while (<IN>) { push @lines, $_; }
  close(IN);
  unlink $inFile;
  unlink $outFile;
  open(OUT, ">$outFile")
    or die "Can't open dependency file '$outFile' for fixdeps: $!";
  foreach my $line (@lines)
  {
      $line =~ s/\r\n/\n/g;
      my $old_line = $line; 
      if ($line =~ m/\s*([a-z]:[^\s]*)\s+.*/i)
      {
	  my $path = $1; 
	  $path =~ s/\\/\//g; 
	  $path =~ s/\/\//\//g; 
	  if ($line =~ m/\s+\\\s+/) {  $line = " $path \\\n"; } 
	  else { $line = " $path \n"; }
      }
      print OUT "$line";
  }
  close(OUT);
}

sub runLocal()
{
    my @toolCommand = split(/ /, $perl); 
    push(@toolCommand, "$toolDir/build.pl");
    if ($optCygwinMode)
    {
        if (defined $optCygwinBaseDir)
        {
            push @toolCommand, '--basedir', $optCygwinBaseDir;
        }
    }
    push @toolCommand, '--host-system', $optHostSystem;
    if (not $optionVerbose) { push @toolCommand, '--silent'; }
    push @toolCommand,
    '--',
    $tool, @toolOptions,
    'gen',
    '-i', $indexFile,
    '-CSPUBreakpoint:'.$entryElementNames[0],
    "-DJOB=$jobName",
    '-o', $outputFile,
    '-u', "$outputFile.u";

    if ($optionCodeMirror != 0)
    {
        my $elementFileName = $ENV{'PS3_ELEMENT_FILE'};
        my $codeFile = dirname ($outputFile); 
        $codeFile = $codeFile . '/' . $jobName . ".code";
        push @toolCommand,
        '-F', $codeFile,    
        '-M', $elementFileName;
    }

    for my $entryElementName (@entryElementNames)
    {
        push @toolCommand, '-e', $entryElementName;
    }
    if ($optionJobExec)
    {
        push @toolCommand, '-j', "$jobName";
    }
    else
    {
        push @toolCommand, "-CJobName:$jobName";
    }
    if (defined $depFile)
    {
        if ($optCygwinMode)
        {
            push @toolCommand, '-d', "${depFile}_";
        }
        else
        {
            push @toolCommand, '-d', $depFile;
        }
    }
    die "cannot generate both jobExec and pageTable at the same time" 
        if ($optionJobExec && $optionPageTable);
    if (!$optionJobExec && $optionPageTable)
    {
        print "generating pagetable for \"$jobName\"\n";
    }
    elsif ($optionJobExec && !$optionPageTable)
    {
        print "generating job \"$jobName\" (exec)\n";
    }
    else
    {
        print "generating code pages for \"$jobName\" \n";
    }

    # If verbose output is selected, then the build.pl wrapper script will echo
    # the command - no need to echo it here!
    if (defined $exec_prefix)
    {
        foreach my $prefix (reverse(split(/ /, $exec_prefix)))
        {
            unshift @toolCommand, $prefix;
        }
    }
    exec @toolCommand or print STDERR "couldn't exec: $!";
} 

sub submitRemotely()
{
    # Run the CryCG tool.
    my $caption = "?";
    my @toolCommand = ();
    push @toolCommand, $tool, @toolOptions,
    'gen',
    '-i', $indexFile,
    '-CSPUBreakpoint:'.$entryElementNames[0],
    "-DJOB=$jobName",
    '-o', $outputFile,
    '-u', "$outputFile.u";
    if ($optionCodeMirror != 0)
    {
        my $elementFileName = $ENV{'PS3_ELEMENT_FILE'};
        my $codeFile = dirname ($outputFile); 
        $codeFile = $codeFile . '/' . $jobName . ".code";
        push @toolCommand,
        '-F', $codeFile,    
        '-M', $elementFileName;
    }
    for my $entryElementName (@entryElementNames)
    {
        push @toolCommand, '-e', $entryElementName;
    }
    if ($optionJobExec)
    {
        push @toolCommand, '-j', "$jobName";
        $caption = "job $jobName";
    }
    else
    {
        push @toolCommand, "-CJobName:$jobName";
        $caption = "pages $jobName";
    }
    if (defined $depFile)
    {
        if ($optCygwinMode)
        {
            push @toolCommand, '-d', "${depFile}_";
        }
        else
        {
            push @toolCommand, '-d', $depFile;
        }
    }
    die "cannot generate both jobExec and pageTable at the same time" 
        if ($optionJobExec && $optionPageTable);
    if (!$optionJobExec && $optionPageTable)
    {
        print "generating pagetable for \"$jobName\"\n";
    }
    elsif ($optionJobExec && !$optionPageTable)
    {
        print "generating job \"$jobName\" (exec)\n";
    }
    else
    {
        print "generating code pages for \"$jobName\" \n";
    }

    # If verbose output is selected, then the build.pl wrapper script will echo
    # the command - no need to echo it here!
    unshift @toolCommand, "/command";
    if (defined $incrediprefix)
    {
        unshift @toolCommand, "/group=\"$incrediprefix\"";
    }
    my $cmdLine = join(' ',@toolCommand);
    $cmdLine =~ s/%/%%/g;

    if ($optionVerbose)
    {
        print "$0: ", join(' ', @toolCommand), "\n";
    }

    # path fixup hack
    $incredibat =~ s/\\\//\//g;
    $incredibat =~ s/\\/\//g;

    my $incredifile;
    open($incredifile, "+>>$incredibat") 
        or die("$0: cannot open $incredibat");

    flock($incredifile, LOCK_EX) or die "flock: $!";
    print $incredifile "\@xgSubmit /caption=\"$caption\" " . $cmdLine . "\n";
    flock($incredifile, LOCK_UN);
    close($incredifile);
}

if (not defined $incredibat)
{
    runLocal();
}
else
{
    submitRemotely();
}

# Tools/genspujob.pl
# vim:ts=2:sw=2:expandtab

