#!/usr/bin/perl
#############################################################################
## Crytek Source File
## Copyright (C) 2007, Crytek Studios
##
## Creator: Sascha Demetrio
## Date: Jul 17, 2007
## Description: GNU-make based build system
#############################################################################

# This script perfroms the post-scan actions.

use warnings;
use strict;

use Elements;
use Getopt::Long qw(:config no_auto_abbrev bundling);

# Parse the command line.
my $elementFileName;
my $makeFileName;
my $targetName;
my $ppuSymbolFileName;
my $ppuSymbolListFileName;
my $ppuSymbolFile;
my $ppuSymbolListFile;
my $ppuElfFileName;
my $optionDebug = 1;
my %optionPrefix = ( );
my $optionHelp = 0;
my $optionVerbose = 0;
my $gen_pages = 0;
my $gen_exec = 0;
my $cm_backend_mode;
GetOptions(
    'E|element-file=s' => \$elementFileName,
    'm|makefile=s' => \$makeFileName,
    't|target=s' => \$targetName,
    'ppu-sym=s' => \$ppuSymbolFileName,
    'ppu-sym-list=s' => \$ppuSymbolListFileName,
    'ppu-sym-file=s' => \$ppuSymbolFile,
    'ppu-sym-list-file=s' => \$ppuSymbolListFile,
    'ppu-elf=s' => \$ppuElfFileName,
    'debug!' => \$optionDebug,
    'P|prefix=s' => \%optionPrefix,
    'h|help' => \$optionHelp,
    'v|verbose' => \$optionVerbose,
    'gen_pages' => \$gen_pages,
    'gen_exec' => \$gen_exec,
    'cm_backend' => \$cm_backend_mode,
	) or exit 1;
if ($optionHelp)
{
	print <<EOF;
postscan.pl: Script executing the post-scan action.
Synopsis:
  \$PERL postscan.pl (Options)
Options:
  -E|--element-file FILE
  Specify the element file name. This option is required.
  -m|--makefile FILE
  Specify the output makefile to be generated. This option is required.
  --cm_backend 
  Specify that postscan.pl should output makefiles compatible with cm_backend. This option is optional.
  -t|--target NAME
  The name of the build target.  Typically this is the name of the final
  excutable (e.g. 'PS3Launcher').  This option is required.
  --ppu-sym FILE
  The name of the PPU symbol file.  If this option is omitted, then no PPU
  symbol file is created.
  --ppu-elf FILE
  The name of the raw linked PPU ELF file.
  --gen_pages
  Generates the makefile that contains the rules for spu code page compilation.
  This option cannot be specified together with the --gen-exec option
  --gen_exec
  Generates the makefile that contains the rules for spu job exec file compilation.
  This option cannot be specified together with the --gen-pages option
  v-P|--prefix PREFIX=SUBST
  Specify a filename prefix to be substituted in the generated makefile.
  -v|--verbose
  Verbose operation.
  -h|--help
  Display this help screen and exit.
EOF
		exit 0;
}

if (not defined $elementFileName)
{
	print STDERR "postscan: no element file specified!\n";
	exit 1;
}
if (not defined $makeFileName)
{
	print STDERR "postscan: no output makefile specified!\n";
	exit 1;
}
if (not defined $targetName)
{
	print STDERR "postscan: no target name specified!\n";
	exit 1;
}
if (!($gen_pages!=0 || $gen_exec!=0))
{
	print STDERR "postscan: postscan must generate either a ";
	print STDERR " a spu exec makefile OR a spu code pages makefile!\n";
	exit 1;
}
if ($gen_pages!=0 && $gen_exec!=0)
{
	print STDERR "postscan: --gen_pages and --gen_exec options are mutually";
	print STDERR " exclusive!\n";
	exit 1;
}

my $spuJobDir = "$targetName/spujob";
my $spuJobRoot = '$(SPU_JOB_ROOT)';

# Stage clean variables names.  For every rule generating output files, the
# resulting files are added to the *_CLEAN variable of the corresponding
# stage.  The script assumes that the C++ code is generated in the post-scan
# step and that the compiling and linking is done in the post-link step (the
# rationale behind moving the compilation into the post-link is that we might
# want to generate some include files from the compiled PPU ELF binary).

# The clean variable collection all generated C++ files.
my $genCleanVar = "SCAN_CLEAN_$targetName";

# The clean variable for compiled output files, not including compiled files
# generated in a combined compile/link rule.
my $compileCleanVar = "LINK_CLEAN_$targetName";

# The clean variable for linker output files.
my $linkCleanVar = "LINK_CLEAN_$targetName";

# Substitute matching filename prefixes.
#
# Parameters:
# - The filename.
sub substPrefix ($)
{
	my $fileName = shift;
	my @matches = ( );

	foreach my $prefix (keys %optionPrefix)
	{
		my $expanded = $optionPrefix{$prefix};
		if (index($fileName, $expanded) == 0) { push @matches, $prefix; }
	}
	if (@matches)
	{
		my $longestMatch = pop @matches;
		my $longestMatchExpanded = $optionPrefix{$longestMatch};
		foreach my $match (@matches)
		{
			my $matchExpanded = $optionPrefix{$match};
			if (length($matchExpanded) > length($longestMatchExpanded))
			{
				$longestMatch = $match;
				$longestMatchExpanded = $matchExpanded;
			}
		}
		$fileName = $longestMatch.substr($fileName, length $longestMatchExpanded);
	}
	$fileName =~ s#/\./#/#g;
	return $fileName;
}

# Substitute a file name extension.
#
# Parameters.
# - The file name.
# - The expected extension (without the dot).
# - The new extension (without the dot).  If this is empty, then the extension
#   dot is stripped.
sub substExt ($$$)
{
	my $fileName = shift;
	my $ext = shift;
	my $newExt = shift;

	my $dotIndex = rindex $fileName, '.'.$ext;
	if ($dotIndex == -1)
	{
		die "expected extension .$ext on file name '$fileName'";
	}
	$fileName = substr($fileName, 0, $dotIndex);
	if ($newExt) { $fileName .= '.'.$newExt; }
	return $fileName;
}

# Generate the CodePage makefile.
#
# Parameters:
# - The unit set.
# - The output makefile name.
sub genPageMakefile ($$)
{
	my $unitSet = shift;
	my $outputFileName = shift;
	my @list = $unitSet->getFunctionElements();
	my $element;
	my @entryList = ( );
	my $totalPageCount = 0;
	local *OUT;

	# Generate a list of all job entry points.  A job entry point is a function
	# or method with an 'entry' attribute and a 'job' sub-attribute.
	my %jobMap = $unitSet->getJobMap();

	# Locate the directory containing the job dependency files.
	my $sepIndex = rindex $elementFileName, '/';
	if ($sepIndex == -1) { $sepIndex = rindex $elementFileName, '\\'; }
	if ($sepIndex == 0) { die "bad element file name '$elementFileName'"; }
	my $dirName = '.';
	if ($sepIndex > 0) { $dirName = substr $elementFileName, 0, $sepIndex; }
	my $dotFilename = "$dirName/job_dependencies.dot"; 
	my $exec_patch_sync = "$dirName/$spuJobDir/exec_patch_sync";
	my $code_page_compile_sync = "$dirName/$spuJobDir/code_page_compile_sync";
	# Generate the makefile.
	open(OUT, '>', $outputFileName)
		or die "can not open output file '$outputFileName': $!";
	die "dependency dot file does not exist!" unless (-e $dotFilename);
	open(DOT, '>>', $dotFilename)
		or die "can not open output file '$dotFilename': $!";
	my $localtime = localtime;
	my $elementFileNameSubst = substPrefix $elementFileName;
	print OUT <<EOF;
# SPU job makefile, generated automatically by postscan.pl
# Time: $localtime
# Element file: $elementFileNameSubst
FLAGS_SUFFIX := _SPU
include \$(MAKE_ROOT)/Lib/setcflags.mk
FLAGS_SUFFIX := _SPUJOB
include \$(MAKE_ROOT)/Lib/setcflags.mk
FLAGS_SUFFIX :=

.SUFFIXES: .c .cpp .o .d .ccg .s .S .page .job .elf .bin .fnc .sym .symlist

COMPILED_CODE_PAGES = 
PATCHED_EXEC_FILES = 
BIN_CODE_PAGES = 
BIN_EXEC = 
EXEC_PATCH_SYNC = $exec_patch_sync
CODE_PAGE_COMPILE_SYNC = $code_page_compile_sync

EOF
   
    foreach my $jobName (keys %jobMap)
	{
		my $job = $jobMap{$jobName};
		
		my $jobDir = substPrefix "$spuJobRoot";
		my $jobExecCPPFile = "$jobDir/$jobName.cpp";
		my $jobLDFlagsVar = "SPUJOB_LDFLAGS_$jobName";
		my $jobPageTable = "$dirName/$spuJobDir/$jobName.pagetable";
		my $jobExecDepFile = substPrefix("$dirName/$spuJobDir/$jobName.d");
		my $jobExecLinkerFile = "$dirName/$spuJobDir/${jobName}_linker.options";								
		my $jobExecOFile = substExt $jobExecCPPFile, 'cpp', 'o';
		my $jobExecHFile = substExt $jobExecCPPFile, 'cpp', 'h';
		my $jobExecNmFile = substExt $jobExecCPPFile, 'cpp', 'nm';
		my $jobFnResolveCPPFile = "$jobDir/${jobName}_fnresolve.cpp";
		my $jobDescFile = substExt $jobExecCPPFile, 'cpp', 'job';
		my $jobPageSizeCleanVar = "SCAN_CLEAN_${jobName}_SIZETABLE_FILES";
		my $jobExecELFFile = substExt $jobExecCPPFile, 'cpp', 'elf';
		my $jobExecELFFileTimeStampFileDep = substExt $jobExecCPPFile, 'cpp', 'tst';
		my $jobExecELFFileTimeStampFileOrig = substExt $jobExecCPPFile, 'cpp', 'tstorig';
		my $symFileDep = '';
		$symFileDep = $ppuSymbolFileName if defined $ppuSymbolFileName;
		# retrieve the indirect list from the jobname
		my $indirectList = $jobMap{$jobName}->indirect();
		my $entryElement = $job->entryElement();

		# Get the entry point element name.
		my $entryElementId = $entryElement->id();

		# parse the page table and extract the number of pages 
		my $pageTable = PageTable->new();
		$pageTable->load($jobPageTable);

		# Get any additional entry points for the job.
		my @entryIndirectList = ( );
		foreach my $entryPoint (@{$job->indirect()})
		{
		    next if $entryPoint->isMapped();
		    my $indirectElement = $entryPoint->element();
		    my $variant = $entryPoint->variant();
		    my $indirectElementId = $indirectElement->id();
		    my $entrySpec = '%'.$indirectElementId;
		    if (defined $variant) { $entrySpec .= ';'.$variant; }
		    push @entryIndirectList, $entrySpec;
		}
		my $entryIndirectIds = join ' ', @entryIndirectList;

		# If the job has indirect call targets, then a address resolution function
		# for these code addresses is needed.  This resolution function is linked
		# against the job executable file.
		my $jobFnResolveOFile = substExt $jobFnResolveCPPFile, 'cpp', 'o';

		# print out the number of pages for the job 
		$totalPageCount += @{$pageTable->{CODEPAGES}};

		# The rule for generating the SPU job page implementation file.
		for(my $pageNum=0; $pageNum<@{$pageTable->{CODEPAGES}}; ++$pageNum)
		{
		    my $jobPageCPPFile = "$jobDir/${jobName}_${pageNum}.cpp";
		    my $jobPageDepFile = "$jobDir/${jobName}_${pageNum}.d";
		    
		    print OUT <<EOF;

$genCleanVar += $jobPageCPPFile

EOF

            # The rule for compiling the SPU job page.  The SPU job C++ files are
            # completely self contained, so there's no point in including a special
	    # dependency file here.  The %.o only depends on the %.cpp.  If the
	    # --debug option was specified on the command line ($optionDebug), then
	    # we'll compile the page a second time with the debug macro defined.
                        my $jobPageOFile = substExt $jobPageCPPFile, 'cpp', 'o';
			my $jobPageHFile = "$jobDir/${jobName}_${pageNum}.h";
			my $jobPageDebugOFile;

			if ($optionDebug)
			{
				$jobPageDebugOFile = $jobPageOFile;
				$jobPageDebugOFile =~ s/\.o$/_debug.o/;
			}
			print DOT "\"$jobPageOFile\" -> \"$jobPageCPPFile\"\n";
			print DOT "\"$jobPageOFile\" -> \"$jobDescFile\"\n";
			print DOT "\"code_page_compile_sync\" -> \"$jobPageOFile\"\n";
			my $jobXFlagsVar = "PROJECT_XFLAGS_SPUJOB_$jobName";
			print OUT <<EOF;
$jobPageOFile: xflags_spu := \$($jobXFlagsVar)
$jobPageOFile: $jobPageCPPFile \$(SPUJOB_PAGE_DEPS) $jobDescFile
\t\$(BUILD_SILENT) \$(RM) '\$\@'
\t\$(BUILD_ECHO) \$(tag_spujob_page_cxx) ${jobName}_${pageNum}.cpp
\t\$(SILENT) if [ -n \"\$(xflags_spu)\" ]; then \\
\t  \$(_BUILD) \\
\t    \$(call COMPILE_CXX_SPUJOB,$jobName/$pageNum,\$<,\$\@,\\
\t      \$(xflags_spu)); \\
\telse \\
\t  \$(_BUILD) \\
\t    \$(call COMPILE_CXX_SPUJOB,$jobName/$pageNum,\$<,\$\@,\\
\t      \$(XFLAGS_SPU)); \\
\tfi
EOF

		      if ($optionDebug)
		      {
				  # For now we'll use hard-coded compile options for compiling the debug
				  # page.  The -D_SPU_DEBUG should remain hard-coded, even if the other
				  # switches are moved into a configurable make variable.
				  print OUT <<EOF;
\t\$(BUILD_SILENT) \$(RM) '$jobPageDebugOFile'
ifeq (\$(MKOPTION_VERBOSE),1)
\t\$(BUILD_ECHO) \$(tag_spujob_page_cxx) ${jobName}_${pageNum}.cpp '(debug)'
endif
\t\$(BUILD) \\
\t  \$(call COMPILE_CXX_SPUJOB,$jobName/$pageNum,\$<,$jobPageDebugOFile,\\
\t     -D_SPU_DEBUG -g -O0)
EOF
			  }

		      print OUT <<EOF;
ifeq (\$(MKOPTION_INCREDIBUILD),1)
  COMPILED_CODE_PAGES += $jobPageOFile
endif

$compileCleanVar += $jobPageOFile
EOF
               if ($optionDebug)
		       {
				   print OUT <<EOF

$compileCleanVar += $jobPageDebugOFile
EOF
 			   }
			   print OUT "\n";
			
            # generate the nm symbol file for the page
            if ($optionDebug) 
            {
			my $jobPageTmpOFile = $jobPageOFile;
			print DOT "\"$jobPageTmpOFile\" -> \"$jobPageTmpOFile\"\n";
			$jobPageTmpOFile =~ s/\.o$/_tmp.o/;
            print OUT <<EOF; 
$jobPageTmpOFile: $jobPageOFile | $code_page_compile_sync
\t\$(BUILD_SILENT) \$(RM) '$jobPageTmpOFile' 
ifeq (\$(MKOPTION_VERBOSE),1)
\t\$(BUILD_ECHO) \$(tag_spujob_page) ${jobName}_${pageNum}.page '(prelink)'
endif
\t\$(BUILD_LINK) \$(LD_SPU) -g -r \\
\t  --unresolved-symbols=ignore-in-object-files -o '$jobPageTmpOFile' \\
\t  '$jobPageOFile' '$jobPageDebugOFile'

$compileCleanVar += $jobPageTmpOFile 

EOF
			}


			# The rule for running the page generator.  The page generator creates a
			# %.page file (which is a text file containing the page
			# description). 
			# The pages are all dependant on all compiled page object
			# files because we need all symbol definitions of all
			# cross call targets during the page description generation
			my $jobPageDescDepFiles = "";
			my $jobPageDescFile = substExt $jobPageCPPFile, 'cpp', 'page';
			my $jobFnResolveCPPFile = "$jobDir/${jobName}_fnresolve.cpp";
			if (not $optionDebug)
			{
			print DOT "\"$jobPageDescFile\" -> \"code_page_compile_sync\"\n";
			print DOT "\"$jobPageDescFile\" -> \"$jobPageOFile\"\n";
			print DOT "\"$jobPageDescFile\" -> \"$jobDescFile\"\n";
				print OUT <<EOF;
ifeq (\$(PROJECT_XFLAGS_SPU_PAGEMODE_$jobName),Single)

$jobPageDescFile: $jobPageOFile $jobDescFile | $code_page_compile_sync
ifeq (\$(MKOPTION_VERBOSE),1)
\t\$(BUILD_ECHO) \$(tag_spujob_page) ${jobName}_${pageNum}.page
endif
\t\$(call SPUJOB_GENPAGEDESC,\$<,\$\@,$jobName,\$(_INCREDI_PREFIX_SIZE))

else 

$jobPageDescFile: $jobPageOFile $jobDescFile | $code_page_compile_sync
ifeq (\$(MKOPTION_VERBOSE),1)
\t\$(BUILD_ECHO) \$(tag_spujob_page) ${jobName}_${pageNum}.page
endif
ifneq (\$(PROJECT_XFLAGS_SPU_PAGEMODE_$jobName),File)
\t\$(call SPUJOB_GENPAGEDESC,\$<,\$\@,$jobName,\$(_INCREDI_PREFIX_SIZE))
else
\t\$(call SPUJOB_GENPAGEDESC,\$<,\$\@,$jobName,\$(_INCREDI_PREFIX_SIZE)); 
endif
endif

EOF
			}
			else
			{
				# Linking step hardcoded for now.
				my $jobPageTmpOFile = $jobPageOFile;
				$jobPageTmpOFile =~ s/\.o$/_tmp.o/;
				print DOT "\"$jobPageDescFile\" -> \"code_page_compile_sync\"\n";
				print DOT "\"$jobPageDescFile\" -> \"$jobPageOFile\"\n";
				print DOT "\"$jobPageDescFile\" -> \"$jobPageTmpOFile\"\n";
				print OUT <<EOF;

ifeq (\$(PROJECT_XFLAGS_SPU_PAGEMODE_$jobName),Single)

$jobPageDescFile: $jobPageOFile $jobDescFile $jobPageDescDepFiles $jobPageTmpOFile | $code_page_compile_sync
ifeq (\$(MKOPTION_VERBOSE),1)
\t\$(BUILD_ECHO) \$(tag_spujob_page) ${jobName}_${pageNum}.page
endif
\t\$(call SPUJOB_GENPAGEDESC,$jobPageTmpOFile,\$\@,$jobName,\$(_INCREDI_PREFIX_SIZE))

else 

$jobPageDescFile: $jobPageOFile $jobDescFile $jobPageDescDepFiles $jobPageTmpOFile | $code_page_compile_sync
ifeq (\$(MKOPTION_VERBOSE),1)
\t\$(BUILD_ECHO) \$(tag_spujob_page) /${jobName}_${pageNum}.page
endif
ifneq (\$(PROJECT_XFLAGS_SPU_PAGEMODE_$jobName),File)
\t\$(call SPUJOB_GENPAGEDESC,$jobPageTmpOFile,\$\@,$jobName,\$(_INCREDI_PREFIX_SIZE))
else
\t\$(call SPUJOB_GENPAGEDESC,$jobPageTmpOFile,\$\@,$jobName,\$(_INCREDI_PREFIX_SIZE)); 
endif
endif

EOF
			}

			print OUT <<EOF;

$linkCleanVar += \\
\t$jobPageDescFile \\
\t${jobPageOFile}.mangled.txt \\
\t\$(patsubst \%.o,\%_crossbubblecalls.txt,$jobPageOFile) \\
\t\$(patsubst \%.o,\%_entrypoints.txt,$jobPageOFile) \\
\t\$(patsubst \%.o,\%_patch.txt,$jobPageOFile) \\
\t\$(patsubst \%.o,\%.log,$jobPageOFile)

EOF


		}

        # Rule for generating the function table file.
        my $jobFncFile = substExt $jobExecCPPFile, 'cpp', 'fnc';
		print DOT "\"$jobFnResolveCPPFile\" -> \"$ppuSymbolFileName\"\n";
		
		print OUT <<EOF;
$jobFnResolveCPPFile: $ppuSymbolFileName
\t\$(call SPUJOB_GENFNRESOLVE,$jobName,\$\@)

$linkCleanVar += \\
\t$jobFncFile \\
\t$jobFnResolveCPPFile

EOF

		# The rule for generating the job description file.  For now, all prefetch
		# information is -1 (no prefetch), so we only need the job name and the
		# entry point.  We _could_ generate the job description files directly -
		# but the final workflow will determine the prefetch pages which can not
		# be done here.
		#
		# NOTE: For now we assume that the entry function receives a single main
		# memory pointer as an argument!
		{
			my $entryPage = -1;
			my $entryElement = $jobMap{$jobName}->entryElement();
			my $entryElementId = $entryElement->{ID};
			my $entryElementName = $entryElement->{NAME};
			my $entryElementMangledName = $entryElement->{MANGLED};
			for(my $pageNum=0; $pageNum<@{$pageTable->{CODEPAGES}}; ++$pageNum)
			{
				my $codePage = @{$pageTable->{CODEPAGES}}[$pageNum];
				for (my $methodNum=0;
					 $methodNum<@{$codePage->{METHODS}}; 
					 ++$methodNum)
				{
					my $methodElement =
						@{$codePage->{METHODS}}[$methodNum];
					if ($methodElement->{MANGLED} eq $entryElementMangledName)
					{
						$entryPage = $pageNum;
					}
				}
			}
			die "can't find code page containing entry element for job $jobName" if ($entryPage == -1);
			print DOT "\"$jobDescFile\" -> \"\$jobExecELFFile\"\n";
			print OUT <<EOF;
$jobDescFile: $jobExecELFFileTimeStampFileDep $spuJobRoot/$jobName.pagetable
\t\$(call SPUJOB_PATCHEXEC,$jobExecELFFile)

PATCHED_EXEC_FILES += $jobDescFile

$linkCleanVar += $jobDescFile

EOF
		}

		# The rule for building the execute BIN file (required for the SN
		# debugger)
		print DOT "\"$jobExecELFFile\" -> \"$jobExecOFile\"\n";
		print DOT "\"$jobExecELFFile\" -> \"$jobFnResolveOFile\"\n";
		print OUT <<EOF;

$jobExecELFFile: ldflags_spu := \$($jobLDFlagsVar)
$jobExecELFFile: $jobExecOFile $jobFnResolveOFile
\t\$(BUILD_SILENT) \$(RM) '\$\@'
\t-\$(call SPUJOB_LINKEXEC,\$< $jobFnResolveOFile,$jobExecELFFile,$jobExecCPPFile.u,\$(ldflags_spu))
\t\$(BUILD_SILENT) touch '$jobExecELFFileTimeStampFileOrig' -r '$jobExecELFFile'

$jobExecELFFileTimeStampFileDep: $jobExecELFFile
\t\$(BUILD_SILENT) touch '$jobExecELFFileTimeStampFileDep' -r '$jobExecELFFileTimeStampFileOrig'
\t\$(BUILD_SILENT) touch '$jobExecELFFile' -r '$jobExecELFFileTimeStampFileOrig'

$linkCleanVar += \\
\t$jobExecELFFile \\
\t$jobExecELFFileTimeStampFileDep \\
\t$jobExecELFFileTimeStampFileOrig

EOF

        # Create the source and object file lists.
        if (@{$pageTable->{CODEPAGES}} > 0)
	    {
			print OUT "${jobName}_SPUJOB_SOURCES =";
			for (my $pageNum=0; $pageNum<@{$pageTable->{CODEPAGES}}; ++$pageNum)
			{
				my $jobPageCPPFile = substPrefix(
					"$spuJobRoot/${jobName}_${pageNum}.cpp");
				print OUT " \\\n\t$jobPageCPPFile";
			}
			print OUT "\n\n${jobName}_SPUJOB_OBJECTS =";
			for (my $pageNum=0; $pageNum<@{$pageTable->{CODEPAGES}}; ++$pageNum)
			{
				my $jobPageOFile = substPrefix(
					"$spuJobRoot/${jobName}_${pageNum}.o");
				print OUT " \\\n\t$jobPageOFile";
			}
			print OUT "\n\n${jobName}_SPUJOB_PAGEDESC_FILES =";
			for (my $pageNum=0; $pageNum<@{$pageTable->{CODEPAGES}}; ++$pageNum)
			{
				my $jobPageDescFile = substPrefix(
					"$spuJobRoot/${jobName}_${pageNum}.page");
				print OUT " \\\n\t$jobPageDescFile";
			}
			print OUT "\n\n";

			# create the target that creates the response file for this job 
			print OUT <<EOF;
$spuJobRoot/${jobName}.Files: $spuJobRoot/${jobName}.pagetable $spuJobRoot/${jobName}.job 
\t\$(SILENT) \$(RM) -f $spuJobRoot/${jobName}.Files
\t\$(SILENT) \$(spu_jobgen_create_list_file) \$(${jobName}_SPUJOB_PAGEDESC_FILES) > $spuJobRoot/${jobName}.Files
\t\$(SILENT) \$(spu_jobgen_create_list_file) \$(${jobName}_SPUJOB_OBJECTS) >> $spuJobRoot/${jobName}.Files
\t\$(SILENT) \$(spu_jobgen_create_list_file) \$(${jobName}_SPUJOB_ELF_FILES) >> $spuJobRoot/${jobName}.Files
\t\$(SILENT) \$(spu_jobgen_create_list_file) $spuJobRoot/${jobName}.elf \$(spu_jobgen_create_list_file) $spuJobRoot/${jobName}.o \\
\t\t\$(spu_jobgen_create_list_file) $spuJobRoot/${jobName}.bin \$(spu_jobgen_create_list_file) $spuJobRoot/${jobName}.pagetable \\
\t\t\$(spu_jobgen_create_list_file) $spuJobRoot/${jobName}.job \$(spu_jobgen_create_list_file) $spuJobRoot/${jobName}.fnc \\
\t\t>> $spuJobRoot/${jobName}.Files

$linkCleanVar += $spuJobRoot/${jobName}.Files

EOF
			

			print OUT "SPUJOB_SOURCES += \$(${jobName}_SPUJOB_SOURCES)\n\n";
			print OUT "SPUJOB_OBJECTS += \$(${jobName}_SPUJOB_OBJECTS)\n\n";
			print OUT "SPUJOB_PAGEDESC_FILES += \$(${jobName}_SPUJOB_PAGEDESC_FILES)\n\n";
			print OUT "SPUJOB_ELF_FILES += \$(${jobName}_SPUJOB_ELF_FILES)\n\n";
			print OUT "SPUJOB_BIN_FILES += \$(${jobName}_SPUJOB_BIN_FILES)\n\n";
		}
	}

        print DOT "}\n";
        close DOT; 

        print OUT <<EOF; 

ifeq (\$(MKOPTION_INCREDIBUILD),1)

$exec_patch_sync: \$(PATCHED_EXEC_FILES) 
\t:

$code_page_compile_sync: \$(COMPILED_CODE_PAGES) | $exec_patch_sync
\t\$(INCREDI_PREFIX_SIZE) xgWait /silent /group=size
\t\$(PERL) \$(MAKE_ROOT)/Tools/invoke_incredibuild.pl --MaxCPUS=128 \$(SHOW_INCREDIBUILD_CONSOLE) --NoLogo --out='\$(INCREDIBAT_LOG)' --command='\$(INCREDIBAT_SPU)' 2>&1 || : 
\t\$(PERL) \$(MAKE_ROOT)/Tools/compiler_output.pl -f -x -d -s -e \$(MAKE_ROOT)/Stage/compile.error '\$(INCREDIBAT_LOG)'
\t\@touch $exec_patch_sync
\t\@touch $code_page_compile_sync
\t\@\$(RM) -f '\$(INCREDIBAT_SPU)'
\techo \$(SPUJOB_LINKPAGE) > \$(SPUJOB_LINKPAGE_CMD_FILE)

else

$exec_patch_sync: \$(PATCHED_EXEC_FILES)
\t\@touch $exec_patch_sync

$code_page_compile_sync: \$(COMPILED_CODE_PAGES)
\t\@touch $code_page_compile_sync
\techo \$(SPUJOB_LINKPAGE) > \$(SPUJOB_LINKPAGE_CMD_FILE)

endif

EOF

        print OUT "\n\nSPUJOB_PAGE_COUNT := $totalPageCount";
	print OUT "\n\nSPUJOB_ENABLE := 1\n\n";

	print OUT "\n\n# vim", ":ts=8", ":sw=8\n\n";
	close OUT;
}

# Generate the spu job exec makefile.
#
# Parameters:
# - The unit set.
# - The output makefile name.
sub genExecMakefile ($$)
{
	my $unitSet = shift;
	my $outputFileName = shift;
	my @list = $unitSet->getFunctionElements();
	my $element;
	my @entryList = ( );
	local *OUT;

	# Generate a list of all job entry points.  A job entry point is a function
	# or method with an 'entry' attribute and a 'job' sub-attribute.
	my %jobMap = $unitSet->getJobMap();

	# Locate the directory containing the job dependency files.
	my $sepIndex = rindex $elementFileName, '/';
	if ($sepIndex == -1) { $sepIndex = rindex $elementFileName, '\\'; }
	if ($sepIndex == 0) { die "bad element file name '$elementFileName'"; }
	my $dirName = '.';
	if ($sepIndex > 0) { $dirName = substr $elementFileName, 0, $sepIndex; }
	my $size_compilation_sync = "$dirName/$spuJobDir/size_compilation_sync";
	my $size_pagedist_sync = "$dirName/$spuJobDir/size_pagedist_sync";
	my $crycg_sync = "$dirName/$spuJobDir/crycg_sync";
	my $crycg_size_sync = "$dirName/$spuJobDir/crycg_size_sync";

	# Generate the makefile.
	my $dotFilename = "$dirName/job_dependencies.dot"; 
	open(DOT, '>', $dotFilename)
		or die "can not open output file '$dotFilename': $!";
	print DOT "digraph G {\n"; 
	print DOT "rankdir=LR\n";
	print DOT "size=\"8,8\"\n";
	print DOT "ratio=fill\n";



	open(OUT, '>', $outputFileName)
		or die "can not open output file '$outputFileName': $!";
	my $localtime = localtime;
	my $elementFileNameSubst = substPrefix $elementFileName;
	print OUT <<EOF;
# SPU job makefile, generated automatically by postscan.pl
# Time: $localtime
# Element file: $elementFileNameSubst

FLAGS_SUFFIX := _SPU
include \$(MAKE_ROOT)/Lib/setcflags.mk
FLAGS_SUFFIX := _SPUJOB
include \$(MAKE_ROOT)/Lib/setcflags.mk
FLAGS_SUFFIX :=

.SUFFIXES: .c .cpp .o .d .ccg .s .S .page .job .elf .bin .fnc .sym .symlist

EOF

		if (defined $ppuSymbolFileName)
        {
		#my $ppuSymListFileName = substExt $ppuSymbolFileName, 'sym', 'symlist';
		my $ppuSymListFileName = $ppuSymbolListFileName;
		print OUT <<EOF;

BUILD_HAS_SPUJOBS:=1

$ppuSymbolFileName: $ppuSymListFileName $ppuElfFileName
\t\$(BUILD_SILENT) \$(RM) '\$\@'
ifeq (\$(MKOPTION_VERBOSE),1)
\t\$(BUILD_ECHO) Creating \$(TARGET).sym
endif
\t\$(call SPUJOB_GENPPUSYM,\$<,\$\@)


COMPILED_SIZE_FILES = 
COMPILED_NM_FILES = 
COMPILED_PAGEDIST_FILES = 
CRYCG_GENERATED_FILES = 
CRYCG_GENERATED_SIZE_FILES = 
CRYCG_SYNC = $crycg_sync
CRYCG_SIZE_SYNC = $crycg_size_sync

$genCleanVar += \\
\t$ppuSymbolFileName \\
\t$ppuSymListFileName

EOF
	}

	foreach my $jobName (keys %jobMap)
	{
		my $job = $jobMap{$jobName};
		my $jobDir = substPrefix "$spuJobRoot";
		my $jobCodeFile = "$jobDir/$jobName.code";
		my $entryElement = $job->entryElement();

		# Get the entry point element name.
		my $entryElementId = $entryElement->id();
		my $entryElementName = $entryElement->mangledName();

		# Get any additional entry points for the job.
		my @entryIndirectList = ();
		my @entryMangledNameList = ();
		my @entryMangledNameVariantList = ();
		foreach my $entryPoint (@{$job->indirect()})
		{
			next if $entryPoint->isMapped();
			my $indirectElement = $entryPoint->element();
			my $variant = $entryPoint->variant();
			my $indirectElementId = $indirectElement->id();
			my $entrySpec = '%'.$indirectElementId;
			if (defined $variant) { $entrySpec .= ';'.$variant; }
      my $cmEntrySpec = '%'.$indirectElement->mangledName().';'.$indirectElementId; 
      if (defined $variant) { $cmEntrySpec .= ';'.$variant }
			push @entryIndirectList, $entrySpec;
			push @entryMangledNameList, $indirectElement->mangledName();
			push @entryMangledNameVariantList, $cmEntrySpec;
		}
		my $entryIndirectIds = join ' ', @entryIndirectList;
		my $entryMangledNames = join ' ', @entryMangledNameList;
		my $entryMangledNameVariants = join ' ',@entryMangledNameVariantList;

    # The linked code files have to be created for each and every job 
    print OUT<<EOF;

$jobCodeFile: \$(PROGRAM_OUTPUT)
\t\@mkdir -p $jobDir
\t\$(_BUILD) \$(call SCAN_LINK_JOB,$jobName,$jobCodeFile,\$(PROJECT_OUTPUT),$entryElementName $entryMangledNames)

$genCleanVar += $jobCodeFile
EOF


		# The relavant size table information (only computed/executed
		# when the code page strategy of the job is set to 'File' mode
		my $jobSizeTable = "$jobDir/$jobName.sizetable";
		my $jobFlowGraph = "$jobDir/$jobName.flow";
		my $jobExecLinkerFile = "$jobDir/${jobName}_linker.options";
		my $jobExecSizeCPPFile = "$jobDir/${jobName}_size.cpp";
		my $jobExecSizeDepFile = "$jobDir/${jobName}_size.d";
		my $jobExecSizeHFile = substExt $jobExecSizeCPPFile, 'cpp', 'h';
		my $jobPageSizeCPPFile = "$jobDir/${jobName}_size_0.cpp";
		my $jobPageSizeOFile = substExt $jobPageSizeCPPFile, 'cpp', 'o';
		my $jobPageSizeDebugOFile = "$jobDir/${jobName}_size_0_debug.o";
		my $jobInputPageTable = "$jobDir/$jobName.pageinfo";
		my $jobPageSizeTable = "$jobDir/${jobName}_size.pagetable";
		my $jobPageSizeDepFile = "$jobDir/${jobName}_size.pagetable.d";
		my $jobPageSizeCleanVar = "SCAN_CLEAN_${jobName}_SIZETABLE_FILES";
		my $jobXFlagsVar = "PROJECT_XFLAGS_SPUJOB_$jobName";
		print DOT "\"$jobPageSizeOFile\" -> \"$crycg_sync\"\n";
		print DOT "\"$jobPageSizeOFile\" -> \"$jobFlowGraph\"\n";
		print DOT "\"$jobInputPageTable\" -> \"$size_compilation_sync\"\n";
		print DOT "\"$jobInputPageTable\" -> \"$jobFlowGraph\"\n";
		print DOT "\"$size_compilation_sync\" -> \"$jobPageSizeOFile\"\n";
		print DOT "\"$size_compilation_sync\" -> \"$jobPageSizeDebugOFile\"\n";
		print DOT "\"$crycg_sync\" -> \"$jobExecSizeCPPFile\"\n";
		print DOT "\"$crycg_sync\" -> \"$jobFlowGraph\"\n";

		print OUT<<EOF;

ifndef PROJECT_XFLAGS_SPUJOBGEN_FLAGS_$jobName
   PROJECT_XFLAGS_SPUJOBGEN_FLAGS_$jobName:=
endif

ifeq (\$(PROJECT_XFLAGS_SPU_DEFAULTDOMAIN_$jobName),LOCAL)
   PROJECT_XFLAGS_SPUJOBGEN_FLAGS_$jobName += DefaultDomainLocal
endif

ifndef PROJECT_XFLAGS_SPU_PAGEMODE_$jobName
   PROJECT_XFLAGS_SPU_PAGEMODE_$jobName := \$(OPTION_CRYCG_DEFAULT_PAGEMODE)
endif

ifndef PROJECT_XFLAGS_SPU_PAGESIZE_$jobName
   PROJECT_XFLAGS_SPU_PAGESIZE_$jobName := \$(OPTION_SPU_PAGESIZE_KB)
endif 

ifndef PROJECT_XFLAGS_SPU_INITIAL_PAGESIZE_$jobName
   PROJECT_XFLAGS_SPU_INITIAL_PAGESIZE_$jobName := \$(OPTION_SPU_INITIAL_PAGESIZE_KB)
endif 

$jobPageSizeCleanVar = 

ifeq (\$(PROJECT_XFLAGS_SPU_PAGEMODE_$jobName),File)

-include $jobExecSizeDepFile
$jobExecSizeCPPFile: $jobCodeFile
ifneq (\$(MKOPTION_INCREDIBUILD),1)
\t\$(BUILD_ECHO) 'generating size estimation file for \"$jobName\"'
endif
\t\$(BUILD_SILENT) mkdir -p '$jobDir'
\t\$(BUILD_SILENT) \$(RM) '\$\@'
\t\$(call SPUJOB_GENEXEC,\$\@,\%$entryElementId,$jobName,$entryIndirectIds,\$(_INCREDI_PREFIX_SIZE))

$jobFlowGraph: $jobExecSizeCPPFile $jobCodeFile
ifneq (\$(MKOPTION_INCREDIBUILD),1)
\t\$(BUILD_ECHO) 'generating initial size estimation pagetable for \"$jobName\"'
endif
\t\$(BUILD_SILENT) \$(RM) '\$\@'
\t\$(call SPUJOB_GENPAGE,$jobExecSizeCPPFile,\%$entryElementId,$jobName,$entryIndirectIds,$jobPageSizeDepFile,Single,\$(PROJECT_XFLAGS_SPUJOBGEN_FLAGS_$jobName) DumpFlowGraph ExternalGlobalVars IsolatedFile DummyHistoryTableDecls PageTableOutputFile:\$\@,-s,\$(_INCREDI_PREFIX_SIZE))

#ifeq (\$(MKOPTION_INCREDIBUILD),1)
 CRYCG_GENERATED_FILES += $jobExecSizeCPPFile $jobFlowGraph
#endif

$jobPageSizeOFile: xflags_spu := \$($jobXFlagsVar)
$jobPageSizeOFile: $jobFlowGraph | $crycg_sync
ifneq (\$(MKOPTION_INCREDIBUILD),1)
\t\$(BUILD_ECHO) 'compiling \"\$(notdir \$(basename $jobPageSizeCPPFile))\" for size estimation'
endif
\t\$(BUILD_SILENT) \$(RM) '\$\@'
\t\$(BUILD_SILENT) \$(RM) '$jobPageSizeDebugOFile'
\t\$(SILENT) if [ -n \"\$(xflags_spu)\" ]; then \\
\t\$(_BUILD) \$(call COMPILE_CXX_SPUJOB,$jobName,$jobPageSizeCPPFile,$jobPageSizeOFile,\$(xflags_spu) -DISOLATED_BUILD); \\
\t\$(_BUILD) \$(call COMPILE_CXX_SPUJOB,$jobName,$jobPageSizeCPPFile,$jobPageSizeDebugOFile,-DISOLATED_BUILD -D_SPU_DEBUG -g -O0); \\
\telse \\
\t\$(_BUILD) \$(call COMPILE_CXX_SPUJOB,$jobName,$jobPageSizeCPPFile,$jobPageSizeOFile,\$(XFLAGS_SPU) -DISOLATED_BUILD); \\
\t\$(_BUILD) \$(call COMPILE_CXX_SPUJOB,$jobName,$jobPageSizeCPPFile,$jobPageSizeDebugOFile,-DISOLATED_BUILD -D_SPU_DEBUG -g -O0); \\
\tfi

$jobInputPageTable: $jobPageSizeOFile $jobFlowGraph | $size_compilation_sync
ifneq (\$(MKOPTION_INCREDIBUILD),1)
\t\$(BUILD_ECHO) 'generating optimal page distribution for \"$jobName\"'
endif
\t\$(BUILD_SILENT) \$(RM) '\$\@'
\t\$(_BUILD) \$(INCREDI_PREFIX_SIZE) \$(call SPUJOB_CODEPAGE_DISTRIBUTE,$jobInputPageTable,$jobFlowGraph,$jobPageSizeOFile $jobPageSizeDebugOFile,\$(PROJECT_XFLAGS_SPU_DISTRIBUTION_STRATEGY_$jobName),\$(PROJECT_XFLAGS_SPU_PAGESIZE_$jobName),\$(PROJECT_XFLAGS_SPU_DISTRIBUTION_FILE_$jobName))

$jobPageSizeCleanVar +=  $jobInputPageTable \\
\t\t$jobSizeTable \\
\t\t$jobPageSizeOFile \\
\t\t$jobPageSizeCPPFile \\
\t\t$jobPageSizeTable \\
\t\t$jobExecSizeCPPFile \\
\t\t$jobFlowGraph

$genCleanVar += \$($jobPageSizeCleanVar)

#ifeq (\$(MKOPTION_INCREDIBUILD),1)
  COMPILED_SIZE_FILES += $jobPageSizeOFile  
  COMPILED_NM_FILES += $jobPageSizeOFile $jobPageSizeDebugOFile
  COMPILED_PAGEDIST_FILES += $jobInputPageTable
#endif

endif

EOF

		# The rule for generating the SPU job execute file.
		my $jobExecDEPS = '${jobName}_DEPS';
		my $jobExecCPPFile = "$jobDir/$jobName.cpp";
		my $jobExecDepFile = substPrefix("$spuJobRoot/$jobName.d");
		my $jobExecHistFile = "$jobDir/${jobName}_history_tables.h";
		print DOT "\"$jobExecCPPFile\" -> \"$jobInputPageTable\"\n";
		print OUT<<EOF;

$jobExecDEPS = $jobCodeFile
ifeq (\$(PROJECT_XFLAGS_SPU_PAGEMODE_$jobName),File)
ifneq (\$(BUILD_WITH_PS3_SDK),1)
$jobExecDEPS += $jobInputPageTable | $size_pagedist_sync
endif
endif 

-include $jobExecDepFile
$jobExecCPPFile: \$($jobExecDEPS)
\t\$(BUILD_SILENT) mkdir -p '$jobDir'
\t\$(BUILD_SILENT) \$(RM) '$jobExecDepFile'
\t\$(BUILD_SILENT) \$(RM) '\$\@'
ifneq (\$(PROJECT_XFLAGS_SPU_PAGEMODE_$jobName),File)
\t\$(call SPUJOB_GENEXEC,\$\@,\%$entryElementId,$jobName,$entryIndirectIds,\$(PROJECT_XFLAGS_SPUJOBGEN_FLAGS_$jobName),\$(_INCREDI_PREFIX_COMPILE))
else
\t\$(call SPUJOB_GENEXEC,\$\@,\%$entryElementId,$jobName,$entryIndirectIds,\$(PROJECT_XFLAGS_SPUJOBGEN_FLAGS_$jobName),\$(_INCREDI_PREFIX_COMPILE))
endif

$genCleanVar += $jobExecCPPFile $jobExecCPPFile.u $jobExecHistFile

ifeq (\$(PROJECT_XFLAGS_SPU_PAGEMODE_$jobName),File)
 #ifeq (\$(MKOPTION_INCREDIBUILD),1)
  CRYCG_GENERATED_SIZE_FILES += $jobExecCPPFile 
 #endif
else
 #ifeq (\$(MKOPTION_INCREDIBUILD),1)
  CRYCG_GENERATED_FILES += $jobExecCPPFile 
 #endif
endif


EOF

    # The rule for compiling the SPU job execute file to an ELF object file.
    # Note that we don't support XFLAGS_SPUJOB for the execute file.
    my $jobExecOFile = substExt $jobExecCPPFile, 'cpp', 'o';
    my $jobExecHFile = substExt $jobExecCPPFile, 'cpp', 'h';
    my $jobExecNmFile = substExt $jobExecCPPFile, 'cpp', 'nm';
    print DOT "\"$jobExecOFile\" -> \"$jobExecCPPFile\"\n";
    print DOT "\"$jobExecOFile\" -> \"$jobExecHFile\"\n";
		print OUT <<EOF;
$jobExecOFile: $jobExecCPPFile $jobExecHFile \$(SPUJOB_JOB_DEPS) | $crycg_size_sync 
\t\$(BUILD_SILENT) \$(RM) '\$\@'
\t\$(BUILD_ECHO) \$(tag_spujob_exec_cxx) ${jobName}.cpp
\t\$(BUILD) \\
\t  \$(call COMPILE_CXX_SPUJOB_NO_PREFIX,$jobName,\$<,\$\@,\$(XFLAGS_SPUJOB))
\t\$(BUILD_LINK) \$(NM_SPU) '\$\@' >'$jobExecNmFile'

$compileCleanVar += $jobExecOFile $jobExecNmFile

EOF

		# The rule for generating the SPU job page implementation file.
		my $jobPageTable = "$jobDir/${jobName}.pagetable";
		my $jobPageDepFile = "$jobDir/${jobName}_pagetable.d";
		my $jobFnResolveCPPFile = "$jobDir/${jobName}_fnresolve.cpp";
		my $codePageStrategy = "PROJECT_XFLAGS_CODEPAGE_STRATEGY_$jobName";
    print DOT "\"$crycg_sync\" -> \"$jobPageTable\"\n";
    print DOT "\"$jobPageTable\" -> \"$jobInputPageTable\"\n";
		print OUT <<EOF;

ifeq (\$(PROJECT_XFLAGS_SPU_PAGEMODE_$jobName),File)
ifneq (\$(BUILD_WITH_PS3_SDK),1)
${jobName}_ADDITIONAL_DEPS = $jobInputPageTable | $size_pagedist_sync
endif
ifndef PROJECT_XFLAGS_SPUJOBGEN_FLAGS_$jobName
PROJECT_XFLAGS_SPUJOBGEN_FLAGS_$jobName = PageTableFile:$jobInputPageTable
else
PROJECT_XFLAGS_SPUJOBGEN_FLAGS_$jobName += PageTableFile:$jobInputPageTable
endif
endif

-include $jobPageDepFile
$jobPageTable: $jobCodeFile \$(${jobName}_ADDITIONAL_DEPS) 
\t\$(BUILD_SILENT) \$(RM) '$jobPageTable'
\t\$(BUILD_SILENT) \$(RM) '$jobPageDepFile'
ifneq (\$(PROJECT_XFLAGS_SPU_PAGEMODE_$jobName),File)
\t\$(call SPUJOB_GENPAGE,$jobExecCPPFile,\%$entryElementId,$jobName,$entryIndirectIds,$jobPageDepFile,\$(PROJECT_XFLAGS_SPU_PAGEMODE_$jobName),\$(PROJECT_XFLAGS_SPUJOBGEN_FLAGS_$jobName),\$(_INCREDI_PREFIX_COMPILE))
else
\t\t\$(call SPUJOB_GENPAGE,$jobExecCPPFile,\%$entryElementId,$jobName,$entryIndirectIds,$jobPageDepFile,\$(PROJECT_XFLAGS_SPU_PAGEMODE_$jobName),\$(PROJECT_XFLAGS_SPUJOBGEN_FLAGS_$jobName),\$(_INCREDI_PREFIX_COMPILE))
endif

ifeq (\$(PROJECT_XFLAGS_SPU_PAGEMODE_$jobName),File)
 #ifeq (\$(MKOPTION_INCREDIBUILD),1)
  CRYCG_GENERATED_SIZE_FILES += $jobPageTable 
 #endif
else
 #ifeq (\$(MKOPTION_INCREDIBUILD),1)
  CRYCG_GENERATED_FILES += $jobPageTable 
 #endif
endif

EOF

			# The rule for generating the exec include files containing the
			# symbol address definitions.
                        print DOT "\"$jobExecHFile\" -> \"$ppuSymbolFileName\"\n";
			print OUT <<EOF;

$jobExecHFile: $ppuSymbolFileName
ifeq (\$(MKOPTION_VERBOSE),1)
\t\$(BUILD_ECHO) \$(tag_spujob_exec_h) ${jobName}.h
endif
\t\$(call SPUJOB_GENEXECHDR,\$\@,$jobName)

$compileCleanVar += $jobExecHFile

EOF

			# If the job has indirect call targets, then a address resolution function
			# for these code addresses is needed.  This resolution function is linked
			# against the job executable file.
			# Otherwise an empty file is created.
			# As the page entry cannot be linked against the job, an int is appended 
			# matching the entry point to fix any linker error.			
 		    my $jobFnResolveOFile = '';
		    $jobFnResolveOFile = substExt $jobFnResolveCPPFile, 'cpp', 'o';
                        print DOT "\"$jobFnResolveOFile\" -> \"$jobFnResolveCPPFile\"\n";
                        print DOT "\"$jobFnResolveOFile\" -> \"$jobExecOFile\"\n";
			print OUT <<EOF;
$jobFnResolveOFile: $jobFnResolveCPPFile $jobExecOFile
\t\$(BUILD_SILENT) \$(RM) '\$\@'
\t\$(BUILD_LINK) cat '$jobExecNmFile' \\
\t  |sed -ne 's/^[ \\t]*U[ \\t]*\\(.*\\)\\s*/int \\1 =0;/p' \\
\t  |sed -e '/\\s*fn_resolve\\s*/d' \\
\t  > '$spuJobRoot/${jobName}_entry.cpp'
\t\$(SILENT) touch $spuJobRoot/${jobName}_entry.cpp -r $jobFnResolveCPPFile
ifeq (\$(MKOPTION_VERBOSE),1)
\t\$(BUILD_ECHO) \$(tag_spujob_exec_cxx) ${jobName}_fnresolve.cpp
endif
\t\$(BUILD) \\
\t  \$(call COMPILE_CXX_SPUJOB_NO_PREFIX,$jobName,\$<,\$\@,\$(XFLAGS_SPUJOB))

$compileCleanVar += $jobFnResolveOFile

EOF

	} # foreach $jobName

	print OUT "SPUJOB_ENABLE := 1\n\n";

	# Create the source and object file lists.
	print OUT "SPUJOB_SOURCES :=";
	foreach my $jobName (keys %jobMap)
	{
		my $jobExecCPPFile = substPrefix(
			"$spuJobRoot/${jobName}.cpp");
		print OUT " \\\n\t$jobExecCPPFile";
	}
	print OUT "\n\nSPUJOB_OBJECTS :=";
	foreach my $jobName (keys %jobMap)
	{
		my $jobExecOFile = substPrefix(
			"$spuJobRoot/${jobName}.o");
		print OUT " \\\n\t$jobExecOFile";
		my $indirectList = $jobMap{$jobName}->indirect();
		if (@$indirectList)
		{
			my $jobFnResolveCPPFile = "$spuJobRoot/${jobName}_fnresolve.cpp";
			my $jobFnResolveOFile = substExt $jobFnResolveCPPFile, 'cpp', 'o';
			print OUT " \\\n\t$jobFnResolveOFile";
		}
	}
	print OUT "\n\nSPUJOB_PAGETABLE_FILES :=";
	foreach my $jobName (keys %jobMap)
	{
		my $jobPageTableFile = substPrefix(
			"$spuJobRoot/${jobName}.pagetable");
		print OUT " \\\n\t$jobPageTableFile";
	}
	print OUT "\n\nSPUJOB_JOBDESC_FILES :=";
	foreach my $jobName (keys %jobMap)
	{
		my $jobDescFile = substPrefix(
			"$spuJobRoot/${jobName}.job");
		print OUT " \\\n\t$jobDescFile";
	}
	print OUT "\n\nSPUJOB_JOBFNC_FILES :=";
	foreach my $jobName (keys %jobMap)
	{
		my $jobFncFile = substPrefix(
			"$spuJobRoot/${jobName}.fnc");
		print OUT " \\\n\t$jobFncFile";
	}
	print OUT "\n\nSPUJOB_FILE_LISTS :=";
	foreach my $jobName (keys %jobMap)
	{
		print OUT " \\\n\t$spuJobRoot/${jobName}.Files";
	}
        foreach my $jobName (keys %jobMap) 
        {
		print OUT "\n\n${jobName}_FILE_LIST := ";
		print OUT "$spuJobRoot/${jobName}.Files\n\n";
	}

        print OUT <<EOF;

ifeq (\$(MKOPTION_INCREDIBUILD),1)

$crycg_sync: \$(CRYCG_GENERATED_FILES)
\t\$(INCREDI_PREFIX_SIZE) xgWait /silent /group=size

$size_compilation_sync: \$(COMPILED_SIZE_FILES) | $crycg_sync
\t\$(INCREDI_PREFIX_SIZE) xgWait /silent /group=size

$size_pagedist_sync: \$(COMPILED_PAGEDIST_FILES) | $size_compilation_sync
\t\$(INCREDI_PREFIX_SIZE) xgWait /silent /group=size

$crycg_size_sync:  \$(COMPILED_SIZE_FILES) \$(COMPILED_PAGEDIST_FILES) \$(CRYCG_GENERATED_FILES) \$(CRYCG_GENERATED_SIZE_FILES) | $size_pagedist_sync
\t\$(PERL) \$(MAKE_ROOT)/Tools/invoke_incredibuild.pl --MaxCPUS=128 \$(SHOW_INCREDIBUILD_CONSOLE) --NoLogo --Log='\$(INCREDIBAT_SPU_LOG)' --command='\$(INCREDIBAT_SPU)' 2>&1 || :
\t\@mkdir -p `dirname $crycg_sync`
\t\@\$(RM) -f '\$(INCREDIBAT_SPU)'
\t\@touch '\$(INCREDIBAT_SPU)'
\t\@touch $crycg_sync
\t\@touch $size_compilation_sync
\t\@touch $size_pagedist_sync
\t\@touch $crycg_size_sync

else

$size_compilation_sync: \$(COMPILED_SIZE_FILES) | $crycg_sync

$crycg_sync: \$(CRYCG_GENERATED_FILES) 

$size_pagedist_sync:  \$(COMPILED_PAGEDIST_FILES) | $size_compilation_sync
 
$crycg_size_sync:  \$(CRYCG_GENERATED_SIZE_FILES) | $size_pagedist_sync
\t\@mkdir -p `dirname $crycg_sync`
\t\@touch $crycg_sync
\t\@touch $size_compilation_sync
\t\@touch $size_pagedist_sync
\t\@touch $crycg_size_sync

endif

$compileCleanVar += $size_compilation_sync $crycg_size_sync $crycg_sync $size_pagedist_sync

EOF


	print OUT "\n\n# vim", ":ts=8", ":sw=8\n\n";
	close OUT;

	if (defined $ppuSymbolListFile)
	{		
		#my $ppuSymListFileName = substExt $ppuSymbolFileName, 'sym', 'symlist';
		my $ppuSymListFileName = $ppuSymbolListFile;
		
		my %ppuSymbols = ( );
		my $doUpdate = 0;

		if (not -r $ppuSymListFileName)
		{
			$doUpdate = 1;
		}
		else
		{
			my $ppuSymListMTime = (stat $ppuSymListFileName)[9];
			my $elementFileMTime = (stat $elementFileName)[9];
			if ($elementFileMTime > $ppuSymListMTime)
			{
				$doUpdate = 1;
			}
		}
		if ($doUpdate)
		{
			foreach my $job (values %jobMap)
			{
				foreach my $entryPoint ($job->getEntryPointList())
				{
					my $element = $entryPoint->element();
					die if not defined $element;
					my $ppuSymbol = $element->mangledName();
					if (not defined $ppuSymbol)
					{
						print STDERR
							"$0: no PPU symbol name for entry point ",
							$element->name(), " (#", $element->id(), ") of job ",
							$job->name(), "\n";
						next;
					}
					$ppuSymbols{$ppuSymbol} = 1;
				}
			}
			open(OUT, '>', $ppuSymListFileName)
				or die "can not open output file '$ppuSymListFileName': $!";
			foreach my $ppuSymbol (keys %ppuSymbols) { print OUT "$ppuSymbol\n"; }
			close(OUT);
		}

	}
}

my $unitSet = UnitSet->new();
$unitSet->load($elementFileName);
if ($gen_pages != 0)
{
#	patchPageTables($unitSet);
	genPageMakefile($unitSet, $makeFileName);
}
if ($gen_exec != 0)
{
	genExecMakefile($unitSet, $makeFileName);
}

# Tools/postscan.pl
# vim:ts=2:sw=2:expandtab


