#!/usr/bin/perl
#
# Script for creating a linked object file masking functions from system
# libraries.

use warnings;
use strict;

use Getopt::Long qw(:config no_auto_abbrev bundling);

my $errorOut = 'error.txt';
my $optOutput;
my $optTmpdir = '/tmp';
my $optPrivate = 'REAL';
my $optVerbose = 0;
my $optPRXOnly = 0;
my $optHelp = 0;
my $NM = 'ppu-lv2-nm';
my $OBJCOPY = 'ppu-lv2-objcopy';
my $LD = 'ppu-lv2-ld';
my $AR = 'ppu-lv2-ar';
GetOptions(
    'o|output=s' => \$optOutput,
    't|tmpdir=s' => \$optTmpdir,
    'p|private=s' => \$optPrivate,
    'prx|prx-only!' => \$optPRXOnly,
    'v|verbose' => \$optVerbose,
    'h|help' => \$optHelp, 
    'n|ppu-nm=s' => \$NM, 
    'c|ppu-objcopy=s' => \$OBJCOPY, 
    'l|ppu-ld=s' => \$LD, 
    'a|ppu-ar=s' => \$AR); 
if ($optHelp)
{
  print <<EOF;
prelink.pl: Script for linking a function substitution object file.
Synopsis:
  \$PERL prelink.pl [(options)] -o (output) (compiled input files)
Options:
-o|--output (output)
  Name of the output file.
-t|--tmpdir (dir)
  Directory for temporary files.  Defaults to /tmp.
-p|--private (name)
  A name prefix to be used for private re-linked symbols.  When linking
  multiple re-linked objects into a single executable, then different private
  names should be specified to avoid link-time name collisions.  The default
  value is 'REAL'.
--prx|--prx-only
  Only scan the stub libraries for the exported symbols.  If this option is not
  specified, then the script will first scan the non-stub libraries and fall
  back to stub libraries only if no static implementation of the exported
  symbols is found.
-v|--verbose
  Verbose operation.
-h|--help
  Display this help screen and exit.
-n|--ppu-nm
  path to the ppu-lv2-nm executable
-c|--ppu-objcopy
  path to the ppu-lv2-objcopy executable 
-l|--ppu-ld
  path to the ppu-lv2-ld executable 
-a|--ppu-ar
  path to the ppu-lv2-ar executable 
EOF
  exit 0
}
if ($#ARGV < 0)
{
  print STDERR "prelink.pl: no input files specified\n";
  exit 1;
}
my @optInputs = @ARGV;
if (not defined $optOutput)
{
  print STDERR "prelink.pl: no output file specified\n";
  exit 1;
}
my $CELL_SDK = $ENV{CELL_SDK};
if (not defined $CELL_SDK or not $CELL_SDK)
{
  print STDERR "prelink.pl: env variable CELL_SDK not defined\n";
  exit 1;
}
my @libPath = ( );
my $libDir = $CELL_SDK . '/target/ppu/lib/fno-exceptions/fno-rtti';
$libDir =~ s/\/\/+/\//g;
$libDir =~ s/^\/([a-zA-Z])\//$1:\//;
$libDir =~ s/^\/cygdrive\/([a-zA-Z])\//$1:\//;
if (not -d $libDir)
{
  print STDERR "prelink.pl: can't find libdir ('$libDir')\n";
  exit 1;
}
push @libPath, $libDir;
$libDir = $CELL_SDK . '/target/ppu/lib';
$libDir =~ s/\/\/+/\//g;
$libDir =~ s/^\/([a-zA-Z])\//$1:\//;
$libDir =~ s/^\/cygdrive\/([a-zA-Z])\//$1:\//;
if (not -d $libDir)
{
  print STDERR "prelink.pl: can't find libdir ('$libDir')\n";
  exit 1;
}
push @libPath, $libDir;
# Extract the list of all real library symbols from the input files.  Real
# library symbols are flagged with a '__PREFIX__' prefix.
my %libSymbols = ( );
foreach my $input (@optInputs)
{
  open(NM, "$NM $input |") or die "can't '$NM' input file '$input'";
  while (<NM>)
  {
    if (/^\s+U\s+\__${optPrivate}__(\S+)\s*$/) 
    { 
	$libSymbols{$1} = 0; 
    }
  }
  close(NM);
}

my %libMap = ( );
my @libObjects = ( );

# Scan for the libraries providing the symbols.
#
# Parameters:
# libDir - The library directory to be scanned.
# scanStubs - Flag indicating if we're scanning stubs or not.
sub scanLibDir ($$)
{
  my $libDir = shift;
  my $scanStubs = shift;

  opendir(LIBDIR, $libDir) or die "can not open '$libDir'";
  foreach my $name (readdir LIBDIR)
  {
    if ($scanStubs)
    {
      next if not $name =~ /_stub\.a$/;
    }
    else
    {
      next if $name =~ /_stub\.a$/;
      next if not $name =~ /\.a$/;
    }
    my $libName = "$libDir/$name";
    my $lib = { };
    $libMap{$libName} = $lib;
    open(NM, "$NM $libName 2>$errorOut |")
      or die "can not '$NM' lib file '$libName";
    my $object;
    my $objectName;
    my $objectMatched = 0;
    while (<NM>)
    {
      next if /^\s*$/;
      if (/^(\w+)\.o:\s*$/)
      {
        $object = [ ];
        $objectName = $1;
        $lib->{$objectName} = $object;
        $objectMatched = 0;
        next;
      }
      if (/^[0-9a-f]+\s+T\s+\.(\w+)\s*$/)
      {
        my $symbol = $1;
        push @$object, $symbol;
        if (defined $libSymbols{$symbol} and $libSymbols{$symbol} == 0)
        {
          $libSymbols{$symbol} = 1;
          print "prelink.pl: '$symbol' found in '$libName'\n" if $optVerbose;
          if (not $objectMatched)
          {
            $objectMatched = 1;
            push @libObjects, [ $libName, $objectName ];
          }
        }
        next;
      }
    }
    close(NM);
  }
  closedir(LIBDIR);
}

# Scan the library directories.
if (not $optPRXOnly)
{
  foreach my $libDir (@libPath) { scanLibDir($libDir, 0); }
}
foreach my $libDir (@libPath) { scanLibDir($libDir, 1); }

# Create a temporary directory.
my $tmpDir = sprintf '%s/prelink_%d', $optTmpdir, time;
if (not -d $tmpDir)
{
  if (-e $tmpDir)
  {
    print STDERR
        "prelink.pl: temporary directory '$tmpDir' exists (not a directory)\n";
    exit 1;
  }
  if (not mkdir $tmpDir)
  {
    system('mkdir', '-p', '--', $tmpDir) == 0
      or die "can not create '$tmpDir': $!";
  }
}

# Extract the library files and rename all symbol exported by the specified
# input files.
my $privateCounter = 0;
my @privateObjects = ( );
foreach my $libObject (@libObjects)
{
  # Extract the referenced object file from the static library.
  my $libName = $libObject->[0];
  my $libBaseName;
  my $object = $libObject->[1];
  if ($libName =~ /\/([^\/]+)\.a$/)
  {
    $libBaseName = $1;
  }
  else
  {
    print STDERR "prelink.pl: runrecognized library name '$libName'\n";
    exit 1;
  }
  if (not -d "$tmpDir/$libBaseName")
  {
    mkdir "$tmpDir/$libBaseName"
      or die "can not create '$tmpDir/$libBaseName': $!";
  }
  my $arCommand = "$AR x $libName $object.o";
  if (system($arCommand) != 0)
  {
    print STDERR
        "prelink.pl: can not extract '$object.o' from '$libName': $!\n";
    exit 1;
  }
  $arCommand = "mv $object.o $tmpDir/$libBaseName/$object.o";
  if (system($arCommand) != 0)
  {
    print STDERR
        "prelink.pl: can not move '$object.o' to '$tmpDir/$libBaseName/$object.o': $!\n";
    exit 1;
  }
  my $objectFile = "$tmpDir/$libBaseName/$object.o";
  if (not -f "$objectFile")
  {
    print STDERR
        "prelink.pl: can not find extracted file '$objectFile'\n";
    exit 1;
  }

  # Rename the exported symbols.
  if (not -d "$tmpDir/private")
  {
    mkdir "$tmpDir/private" or die "can not create '$tmpDir/private': $!";
  }
  $privateCounter += 1;
  my $privateObjectFile = sprintf '%s/priv_%d_%s.o',
      "$tmpDir/private", $privateCounter, $object;
  my $lib = $libMap{$libName};
  my $objectSyms = $lib->{$object};
  my @objcopyCommand = ( $OBJCOPY );
  foreach my $sym (@$objectSyms)
  {
    if (defined $libSymbols{$sym})
    {
      my $renamedSym = sprintf '__%s__%s', $optPrivate, $sym;
      push @objcopyCommand, '--redefine-sym', "$sym=$renamedSym";
      push @objcopyCommand, '--redefine-sym', ".$sym=.$renamedSym";
    }
  }
  push @objcopyCommand, '--', $objectFile, $privateObjectFile;
  if (system(@objcopyCommand) != 0)
  {
    print STDERR "prelink.pl: '$OBJCOPY' failed: $!\n";
    exit 1;
  }
  push @privateObjects, $privateObjectFile;
}

# Link the output file.
my @ldCommand = ( $LD, '-g', '-r', '-o', $optOutput );
foreach my $input (@optInputs)
{
  push @ldCommand, $input;
}
foreach my $privateObject (@privateObjects)
{
  push @ldCommand, $privateObject;
}
if (system(@ldCommand) != 0)
{
  print STDERR "prelink.pl: '$LD' failed: $!\n";
  exit 1;
}

exit 0;

# vim:ts=2:sw=2:expandtab

