#!/usr/bin/perl
#############################################################################
## Crytek Source File
## Copyright (C) 2008, Crytek Studios
##
## Creator: Chris Raine
## Date: Nov 5, 2008
## Description: GNU-make based build system
#############################################################################

# This script parses the nm file given at the command line and ensures that all 
# mangled names with element-id/variant-key mappings match. It emits
# the original cpp file again with the declarations patched. 

use warnings;
use strict;
use Elements;
use Getopt::Long qw(:config no_auto_abbrev bundling);


# Parse the command line.
my $originalSrcFile;
my $patchedSrcFile;
my $nmFile;
my $optionHelp = 0;
GetOptions(
	'i|input-src-file=s' => \$originalSrcFile,
	'o|outpur-src-file=s' => \$patchedSrcFile,
	'n|nm-file=s' => \$nmFile,
	'h|help' => \$optionHelp
	) or exit 1;
if ($optionHelp)
{
	print <<EOF;
mangler.pl: Script to patch c++ mangled identifiers to match their spu
	symbol counterparts
Synopsis:
  \$PERL mangler.pl (Options)
Options:
	-i|--input-src-file FILE
	The path to the input source file. This option is required.
	-o|--outpur-src-file FILE
	The path to the output source file. This option is required.
	-n|--nm-file FILE 
	The path to the nm file containing the mangled symbols of the
	input src file. This option is required
	-h|--help
	Display this help screen and exit.

EOF
		exit 0;
}
if (not defined $originalSrcFile)
{
	print STDERR 'mangler.pl: no input src file specified'; 
	exit -1; 
}
if (not defined $patchedSrcFile)
{
	print STDERR 'mangler.pl: no output src file specified'; 
	exit -1; 
}
if (not defined $nmFile)
{
	print STDERR 'mangler.pl: no symbol file specified'; 
	exit -1; 
}

my %replaceExprs = ();
my @Stubs = ();
my @Symbols = ();
local *IN; 

if (not open(NM, '<', $nmFile))
{
	print STDERR 'mangler.pl: cannot open file $nmFile'; 
	exit -1; 
}
while (<NM>)
{
	my $symbolName;
	my $elementId;
	my $variantId;
	if (/^[0-9a-f]*\s+(\w)\s+([^\s]+)\s*$/)
	{
		if ($1 eq 'U' or $1 eq 'T')
		{
			$symbolName = $2;
		}
		else
		{
			next;
		}
	}
	else 
	{
		next;
	}
	if ($symbolName =~ /\$E([0-9]+)_/)
	{
		$elementId = $1;
	}
	if (not defined $elementId)
	{
		next;
	}
	if ($symbolName =~ /\$V([^_]*)/)
	{
		$variantId = $1;
	}
	else 
	{
		$variantId = ''; 
	}
	if ($symbolName =~ /^__spu_.*(_Z[^\s]+)/)
	{
		my $stub = $1; 
		$stub =~ s/\$/\\\$/g;
		push @Stubs, [$stub, $elementId, $variantId];
	}
	else 
	{
		my $symbol = $symbolName; 
		push @Symbols, [$symbol, $elementId, $variantId];
	}
}

foreach my $stub (@Stubs)
{
	foreach my $symbol (@Symbols)
	{
		if (@$stub[1] eq @$symbol[1] && @$stub[2] eq @$symbol[2])
		{
			$replaceExprs{@$stub[0]} = @$symbol[0];
		}
	}
}

if (not open(IN, '<', $originalSrcFile))
{
	print STDERR 'mangler.pl: cannot open $originalSrcFile for reading';
	exit -1;
}
if (not open(OUT, '>', $patchedSrcFile))
{
	print STDERR 'mangler.pl: cannot open $patchedSrcFile for writing';
	exit -1;
}

while (<IN>)
{
	my $line = $_;
	my $stub; 
	my $symbol;
	while (($stub, $symbol) = each(%replaceExprs))
	{
		$line =~ s/$stub/$symbol/g;
	}
	print OUT $line;
}





