#include "StdAfx.h"
#include "AnimationExportSource.h"
#include "ExportFunctions.h"
#include "SkinningHelpers.h"
#include "PathHelpers.h"
#include "MeshHelpers.h"
#include "UserPropertyHelpers.h"
#include "Export/ISkeletonData.h"
#include "Export/IModelData.h"
#include "Export/IAnimationData.h"
#include "Export/IGeometryFileData.h"
#include "Export/IGeometryData.h"
#include "Export/IMaterialData.h"
#include "Export/IExportContext.h"
#include "Export/ISkinningData.h"
#include "NodeHelpers.h"
#include "interpik.h"
#include "Export/AnimationData.h"

AnimationExportSource::AnimationExportSource(const std::string& filename, const std::set<INode*>& rootBones)
	: m_filename(filename)
	, rootBones(rootBones)
{
}

void AnimationExportSource::GetMetaData(SExportMetaData& metaData) const
{														  
	ExportFunctions::GetMetaData(metaData);
}

std::string AnimationExportSource::GetDCCFileName() const
{
	return std::string(GetCOREInterface()->GetCurFilePath());
}

std::string AnimationExportSource::GetExportDirectory() const
{
	std::string fullPath;

	if (m_filename.empty())
	{
		std::string const dccFilename = GetDCCFileName();
		if (!dccFilename.empty())
		{
			fullPath = dccFilename;
		}
	}
	else if (PathHelpers::IsRelative(m_filename))
	{
		std::string const dccFilename = GetDCCFileName();
		if (!dccFilename.empty())
		{
			fullPath = PathHelpers::Join(PathHelpers::GetDirectory(dccFilename), m_filename);
		}
	}
	else
	{
		fullPath = m_filename;
	}

	if (!fullPath.empty())
	{
		char buffer[MAX_PATH*2];
		int const n = GetFullPathName(fullPath.c_str(), sizeof(buffer), buffer, 0);
		if ((n > 0) && (n < sizeof(buffer)))
		{
			return PathHelpers::GetDirectory(std::string(buffer));
		}
	}

	return std::string();
}

namespace
{
	void RecurseHierarchyAndListBones(ISkeletonData* skeletonData, INode* node, int parentIndex, const IMaterialData* materialData, SkinningHelpers::SkinAccessor& skinAccessor)
	{
		// Check whether we should export this bone.
		bool shouldExportBone = true;
		shouldExportBone = shouldExportBone && !NodeHelpers::IsFootPrint(node); // Hack! Why is this necessary?
		if (!shouldExportBone)
			return;

		// Add the bone.
		int boneIndex = skeletonData->AddBone(node, node->GetName(), parentIndex);

		// Check whether the bone has geometry.
		Mesh* mesh = MeshHelpers::GetMesh(node);
		bool hasGeometry = (mesh ? (mesh->getNumFaces() > 0) : false);
		skeletonData->SetHasGeometry(boneIndex, hasGeometry);

		// Set the transform of the bone.
		Matrix3 initTM;
		skinAccessor.GetBoneInitialPosition(node, initTM);

		Point3 translation, scaling;
		Quat rotation;

		DecomposeMatrix(initTM, translation, rotation, scaling);
		float translationVec[3] = {translation.x, translation.y, translation.z};
		float rotationVec[3];
		QuatToEuler(rotation, rotationVec, RotationValue::kXYZ);
		float scalingVec[3] = {scaling.x, scaling.y, scaling.z};

		skeletonData->SetTranslation(boneIndex, translationVec);
		skeletonData->SetRotation(boneIndex, rotationVec);
		skeletonData->SetScale(boneIndex, scalingVec);

		// Check for phys mesh nodes.
		static const char* physBoneSuffix = " Phys";
		static const char* physParentFrameBoneSuffix = " Phys ParentFrame";
		std::string physBoneName = std::string(node->GetName()) + physBoneSuffix;
		std::string physParentFrameBoneName = std::string(node->GetName()) + physParentFrameBoneSuffix;
		INode* physBone = GetCOREInterface()->GetINodeByName(physBoneName.c_str());
		INode* physParentFrameBone = GetCOREInterface()->GetINodeByName(physParentFrameBoneName.c_str());

		// Set the parent frame matrix if it exists.
		INode* physParentFrameParentBone = (physParentFrameBone ? physParentFrameBone->GetParentNode() : 0);
		if (physParentFrameBone && physParentFrameParentBone)
		{
			Matrix3 frameTM = physParentFrameBone->GetNodeTM(0);
			Matrix3 parentTM = physParentFrameParentBone->GetNodeTM(0);
			Matrix3 relativeFrameTM = frameTM * Inverse(parentTM);

			Point3 translation, scaling;
			Quat rotation;

			DecomposeMatrix(relativeFrameTM, translation, rotation, scaling);
			float translationVec[3] = {translation.x, translation.y, translation.z};
			float rotationVec[3];
			QuatToEuler(rotation, rotationVec, RotationValue::kXYZ);
			float scalingVec[3] = {scaling.x, scaling.y, scaling.z};
			skeletonData->SetParentFrameTranslation(boneIndex, translationVec);
			skeletonData->SetParentFrameRotation(boneIndex, rotationVec);
			skeletonData->SetParentFrameScale(boneIndex, scalingVec);
		}

		// Set physicalization settings.
		if (physBone)
			skeletonData->SetPhysicalized(boneIndex, true);

		// Get joint params.
		Control* physBoneController = (physBone ? physBone->GetTMController() : 0);
		Control* physBoneRotationController = (physBoneController ? physBoneController->GetRotationController() : 0);
		JointParams* physBoneJointParams = (physBoneRotationController ? static_cast<JointParams*>(physBoneRotationController->GetProperty(PROPID_JOINTPARAMS)) : 0);
		DWORD flags = (physBoneJointParams ? physBoneJointParams->flags : 0);
		for (int axis = 0; axis < 3; ++axis)
		{
			if (flags & (JNT_XLIMITED << axis))
			{
				float minimum = (physBoneJointParams ? physBoneJointParams->min[axis] : 0);
				skeletonData->SetLimit(boneIndex, ISkeletonData::Axis(axis), ISkeletonData::LimitMin, minimum);
				float maximum = (physBoneJointParams ? physBoneJointParams->max[axis] : 0);
				skeletonData->SetLimit(boneIndex, ISkeletonData::Axis(axis), ISkeletonData::LimitMax, maximum);
			}
			float springAngle = (physBoneJointParams ? physBoneJointParams->spring[axis] : 0.0f);
			skeletonData->SetSpringAngle(boneIndex, ISkeletonData::Axis(axis), springAngle);
			float springTension = (physBoneJointParams ? physBoneJointParams->stens[axis] : 0.0f);
			skeletonData->SetSpringTension(boneIndex, ISkeletonData::Axis(axis), springTension * 50.0f); // This factor of 50 is bizarre, but it's in the old pipeline and seems to be required for the number to match what appears in the UI.
			float damping = (physBoneJointParams ? physBoneJointParams->damping[axis] : 0.0f);
			skeletonData->SetAxisDamping(boneIndex, ISkeletonData::Axis(axis), damping);
		}

		// Recurse to the node's children.
		for (int childIndex = 0, childCount = node->NumChildren(); childIndex < childCount; ++childIndex)
			RecurseHierarchyAndListBones(skeletonData, node->GetChildNode(childIndex), boneIndex, materialData, skinAccessor);
	}
}

void AnimationExportSource::ReadGeometryFiles(IExportContext* context, IGeometryFileData* geometryFileData)
{
	std::string path = m_filename;
	if (PathHelpers::IsRelative(path))
	{
		std::string maxFilePath = GetCOREInterface()->GetCurFilePath();
		if (maxFilePath.empty())
		{
			return;
		}
		std::string maxFileDirectory = PathHelpers::GetDirectory(maxFilePath);
		path = PathHelpers::Join(maxFileDirectory, path);
	}

	{
		char absoluteBuffer[1024];
		GetFullPathName(path.c_str(), sizeof(absoluteBuffer), absoluteBuffer, 0);
		path = absoluteBuffer;
	}

	IGeometryFileData::SProperties props;
	props.filetypeInt = CRY_FILE_TYPE_CAF;
	props.bDoNotMerge = false;
	string filename = PathHelpers::RemoveExtension(PathHelpers::GetFilename(path));
	geometryFileData->AddGeometryFile(0, filename.c_str(), props);
}

void AnimationExportSource::ReadModels(const IGeometryFileData* geometryFileData, int geometryFileIndex, IModelData* modelData)
{
	int const modelIndex = modelData->AddModel(0, geometryFileData->GetGeometryFileName(geometryFileIndex), -1, false, SHelperData(), "");
	float const translation[] = {0.0f, 0.0f, 0.0f};
	float const rotation[] = {0.0f, 0.0f, 0.0f};
	float const scale[] = {1.0f, 1.0f, 1.0f};
	modelData->SetTranslationRotationScale(modelIndex, translation, rotation, scale);
}

bool AnimationExportSource::ReadMaterials(IExportContext* context, const IGeometryFileData* const geometryFileData, IMaterialData* materialData)
{
	return true;
}

void AnimationExportSource::ReadSkinning(IExportContext* context, ISkinningData* skinningData, const IModelData* modelData, int modelIndex, ISkeletonData* skeletonData)
{
}

bool AnimationExportSource::ReadSkeleton(const IGeometryFileData* geometryFileData, int geometryFileIndex, const IModelData* modelData, int modelIndex, const IMaterialData* materialData, ISkeletonData* skeletonData)
{
	INode* node = static_cast<INode*>(modelData->GetModelHandle(modelIndex));

	SkinningHelpers::SkinAccessor skinAccessor(node);

	std::set<INode*> skeletonRoots = this->rootBones;
	if (skeletonRoots.empty())
		skinAccessor.FindSkeletonRoots(skeletonRoots);
	for (std::set<INode*>::iterator rootPos = skeletonRoots.begin(), rootEnd = skeletonRoots.end(); rootPos != rootEnd; ++rootPos)
		RecurseHierarchyAndListBones(skeletonData, *rootPos, -1, materialData, skinAccessor);

	return true;
}

int AnimationExportSource::GetAnimationCount() const
{
	return 1; // Max only supports 1 animation at a time.
}

std::string AnimationExportSource::GetAnimationName(const IGeometryFileData* geometryFileData, int geometryFileIndex, int animationIndex) const
{
	assert(animationIndex == 0); // Max only supports 1 animation at a time.

	return PathHelpers::RemoveExtension(PathHelpers::GetFilename(m_filename));
}

void AnimationExportSource::GetAnimationTimeSpan(float& start, float& stop, int animationIndex) const
{
	assert(animationIndex == 0); // Max only supports 1 animation at a time.

	Interval animInterval = GetCOREInterface()->GetAnimRange();
	start = TicksToSec(animInterval.Start());
	stop = TicksToSec(animInterval.End());
}

void AnimationExportSource::ReadAnimationFlags(IExportContext* context, IAnimationData* animationData, const IGeometryFileData* const geometryFileData, const IModelData* modelData, int modelIndex, const ISkeletonData* skeletonData, int animationIndex) const
{
}

IAnimationData * AnimationExportSource::ReadAnimation(IExportContext* context, const IGeometryFileData* const geometryFileData, const IModelData* modelData, int modelIndex, const ISkeletonData* skeletonData, int animationIndex, float fps) const
{
	IAnimationData *animationData = new AnimationData(skeletonData->GetBoneCount(), fps, 
																TicksToSec(GetCOREInterface()->GetAnimRange().Start()));

	// Sample the take at the specified fps.
	int firstFrame = int(TicksToSec(GetCOREInterface()->GetAnimRange().Start()) * fps + 0.5f);
	int lastFrame = int(TicksToSec(GetCOREInterface()->GetAnimRange().End()) * fps + 0.5f);
	int frameCount = lastFrame - firstFrame + 1;
	animationData->SetFrameCount(frameCount);

	Matrix3 sceneParentTM;
	sceneParentTM.IdentityMatrix();

	// For each bone sample the transform at each frame.
	std::vector<float> translationArray(frameCount * 3);
	std::fill_n(translationArray.begin(), frameCount * 3, 0.0f);
	std::vector<float> rotationArray(frameCount * 3);
	std::fill_n(rotationArray.begin(), frameCount * 3, 0.0f);
	std::vector<float> scalingArray(frameCount * 3);
	std::fill_n(scalingArray.begin(), frameCount * 3, 1.0f);
	for (int boneIndex = 0, boneCount = skeletonData->GetBoneCount(); boneIndex < boneCount; ++boneIndex)
	{
		INode* node = static_cast<INode*>(skeletonData->GetBoneHandle(boneIndex));
#if 0
		// Sokov: I've commented this code out, because it doesn't work for partial body animations: 
		// parent of the root node of the body part isn't stored in skeletonData. It
		// means we should use data from the scene instead of using data from our 
		// internal skeletonData structure.
		int parentIndex = skeletonData->GetBoneParentIndex(boneIndex);
		INode* const parent = (parentIndex >= 0 ? static_cast<INode*>(skeletonData->GetBoneHandle(parentIndex)) : 0);
#else
		INode* const parent = node->GetParentNode();
#endif

		// Loop through all the frames, sampling the position of the bone.
		for (int frameIndex = firstFrame; frameIndex <= lastFrame; ++frameIndex)
		{
			float frameTimeInSeconds = float(frameIndex) / fps;
			TimeValue frameTime = SecToTicks(frameTimeInSeconds);
			Matrix3 worldNodeTM = node->GetNodeTM(frameTime);
			Matrix3 worldParentTM = (parent ? parent->GetNodeTM(frameTime) : sceneParentTM);
			Matrix3 relativeNodeTM = worldNodeTM * Inverse(worldParentTM);
			// In the special case of the 'locator_locomotion', we should output data in world coordinate.
			if(node->GetName() && stricmp(node->GetName(), "Locator_Locomotion") == 0)
			{
				relativeNodeTM = worldNodeTM;
			}
			Point3 translation, scaling;
			Quat rotation;
			DecomposeMatrix(relativeNodeTM, translation, rotation, scaling);

			float* frameTranslation = &translationArray[3 * (frameIndex - firstFrame)];
			float* frameRotation = &rotationArray[3 * (frameIndex - firstFrame)];
			float* frameScaling = &scalingArray[3 * (frameIndex - firstFrame)];
			frameTranslation[0] = translation.x; frameTranslation[1] = translation.y; frameTranslation[2] = translation.z;
			QuatToEuler(rotation, frameRotation, RotationValue::kXYZ);
			frameScaling[0] = scaling.x; frameScaling[1] = scaling.y; frameScaling[2] = scaling.z;
		}

		// Round scale values to nearest 0.001.
		for (int frameIndex = firstFrame; frameIndex <= lastFrame; ++frameIndex)
		{
			for (int axis = 0; axis < 3; ++axis)
			{
				float& value = scalingArray[(frameIndex - firstFrame) * 3 + axis];
				value = std::floor(value * 1000.0f + 0.5f) / 1000.0f;
			}
		}

		// Convert rotation values to degrees.
		for (int frameIndex = firstFrame; frameIndex <= lastFrame; ++frameIndex)
		{
			for (int axis = 0; axis < 3; ++axis)
				rotationArray[(frameIndex - firstFrame) * 3 + axis] *= 180.0f / 3.14159f;
		}

		// Store the information in the animation data.
		for (int frameIndex = firstFrame; frameIndex <= lastFrame; ++frameIndex)
		{
			animationData->SetFrameData(boneIndex, frameIndex - firstFrame,
				&translationArray[3 * (frameIndex - firstFrame)],
				&rotationArray[3 * (frameIndex - firstFrame)],
				&scalingArray[3 * (frameIndex - firstFrame)]);
		}
	}

	return animationData;
}

bool AnimationExportSource::ReadGeometry(IExportContext* context, IGeometryData* geometry, const IModelData* const modelData, const IMaterialData* const materialData, int modelIndex)
{
	return false;
}

bool AnimationExportSource::ReadGeometryMaterialData(IExportContext* context, IGeometryMaterialData* geometryMaterialData, const IModelData* const modelData, const IMaterialData* const materialData, int modelIndex) const
{
	return false;
}

bool AnimationExportSource::ReadBoneGeometry(IExportContext* context, IGeometryData* geometry, ISkeletonData* skeletonData, int boneIndex, const IMaterialData* const materialData)
{
	return false;
}

bool AnimationExportSource::ReadBoneGeometryMaterialData(IExportContext* context, IGeometryMaterialData* geometryMaterialData, ISkeletonData* skeletonData, int boneIndex, const IMaterialData* const materialData) const
{
	return false;
}

void AnimationExportSource::ReadMorphs(IExportContext* context, IMorphData* morphData, const IModelData* const modelData, int modelIndex)
{
}

bool AnimationExportSource::ReadMorphGeometry(IExportContext* context, IGeometryData* geometry, const IModelData* const modelData, int modelIndex, const IMorphData* const morphData, int morphIndex, const IMaterialData* materialData)
{
	assert(0);
	return false;
}

bool AnimationExportSource::HasValidPosController(const IModelData* modelData, int modelIndex) const
{
	assert(0);
	return false;
}

bool AnimationExportSource::HasValidRotController(const IModelData* modelData, int modelIndex) const
{
	assert(0);
	return false;
}

bool AnimationExportSource::HasValidSclController(const IModelData* modelData, int modelIndex) const
{
	assert(0);
	return false;
}