#include "StdAfx.h"
#include "GeometryExtendedExportSource.h"
#include "ExportFunctions.h"
#include "SkinningHelpers.h"
#include "PathHelpers.h"
#include "MeshHelpers.h"
#include "UserPropertyHelpers.h"
#include "MorpherHelpers.h"
#include "Export/ISkeletonData.h"
#include "Export/IModelData.h"
#include "Export/IAnimationData.h"
#include "Export/IGeometryFileData.h"
#include "Export/IGeometryData.h"
#include "Export/IMaterialData.h"
#include "Export/IExportContext.h"
#include "Export/ISkinningData.h"
#include "Export/IMorphData.h"
#include "Export/IGeometryMaterialData.h"
#include "Export/AnimationData.h"
#include "PropertyHelpers.h"
#include "NodeHelpers.h"
#include "StringHelpers.h"
#include "interpik.h"
#include "shaders.h"


Matrix3 get3dsMaxMatrixFromCryMatrix(const Matrix34& m);
Matrix34 getCryMatrixFrom3dsMaxMatrix(const Matrix3& m);


class NodeRelatives
{
public:
	NodeRelatives()
		: m_pParent(0)
	{
	}

	~NodeRelatives()
	{
	}

	void setParent(ISceneNode* const pNode)
	{
		m_pParent = pNode;
	}

	void addChild(ISceneNode* const pNode)
	{
		m_children.push_back(pNode);
	}

	ISceneNode* getParent() const
	{
		return m_pParent;
	}

	int getChildCount() const
	{
		return (int)m_children.size();
	}

	ISceneNode* getChild(int index) const
	{
		return m_children[index];
	}

private:
	ISceneNode* m_pParent;
	std::vector<ISceneNode*> m_children;
};


class SceneNode3dsMax
	: public ISceneNode
{
public:
	SceneNode3dsMax(INode* pNode)
		: m_pNode(pNode)
	{		
		assert(pNode);
	}

	virtual ~SceneNode3dsMax()
	{
	}

	// ISceneNode

	virtual void setParent(ISceneNode* const pNode)
	{
		m_relatives.setParent(pNode);		
	}

	virtual void addChild(ISceneNode* const pNode)
	{
		m_relatives.addChild(pNode);		
	}

	virtual ISceneNode* getParent() const
	{
		return m_relatives.getParent();
	}

	virtual int getChildCount() const
	{
		return m_relatives.getChildCount();
	}

	virtual ISceneNode* getChild(int index) const
	{
		return m_relatives.getChild(index);
	}

	virtual Matrix34 getWorldTransform() const
	{
		Matrix3 const m = m_pNode->GetNodeTM(0);
		return getCryMatrixFrom3dsMaxMatrix(m);
	}

	virtual void* getNativeSceneNode() const 
	{
		return m_pNode;
	}

private: 
	INode* m_pNode;
	NodeRelatives m_relatives;
};


class SceneNodeVirtual
	: public ISceneNode
{
public:
	SceneNodeVirtual()
	{		
	}

	virtual ~SceneNodeVirtual()
	{
	}

	// ISceneNode

	virtual void setParent(ISceneNode* const pNode)
	{
		m_relatives.setParent(pNode);		
	}

	virtual void addChild(ISceneNode* const pNode)
	{
		m_relatives.addChild(pNode);		
	}

	virtual ISceneNode* getParent() const
	{
		return m_relatives.getParent();
	}

	virtual int getChildCount() const
	{
		return m_relatives.getChildCount();
	}

	virtual ISceneNode* getChild(int index) const
	{
		return m_relatives.getChild(index);
	}

	virtual Matrix34 getWorldTransform() const
	{
		Matrix34 m;
		m.SetIdentity();
		return m;
	}

	virtual void* getNativeSceneNode() const 
	{
		return 0;
	}

private: 
	NodeRelatives m_relatives;
};


static Matrix3 get3dsMaxMatrixFromCryMatrix(const Matrix34& m)
{
	Vec3 const ax = m.GetColumn0();
	Vec3 const ay = m.GetColumn1();
	Vec3 const az = m.GetColumn2();
	Vec3 const tr = m.GetColumn3();

	Matrix3 m3;
	m3.SetRow(0, Point3(ax.x, ax.y, ax.z));
	m3.SetRow(1, Point3(ay.x, ay.y, ay.z));
	m3.SetRow(2, Point3(az.x, az.y, az.z));
	m3.SetRow(3, Point3(tr.x, tr.y, tr.z));

	return m3;	
}

static Matrix34 getCryMatrixFrom3dsMaxMatrix(const Matrix3& m)
{
	Point3 const ax = m.GetRow(0);
	Point3 const ay = m.GetRow(1);
	Point3 const az = m.GetRow(2);
	Point3 const tr = m.GetRow(3);

	return Matrix34::CreateFromVectors(
		Vec3(ax.x, ax.y, ax.z),
		Vec3(ay.x, ay.y, ay.z),
		Vec3(az.x, az.y, az.z),
		Vec3(tr.x, tr.y, tr.z));
}

static void collect3dsMaxNodesRecursively(ISceneNode* const pParent, INode* const pNode)
{
	if (pParent == 0)
	{
		return;
	}

	if (pNode == 0)
	{
		return;
	}

	if (NodeHelpers::IsExcluded(pNode))
	{
		// Skip this node and subtree
		return;				
	}

	SceneNode3dsMax* const p = new SceneNode3dsMax(pNode);
	p->setParent(pParent);
	pParent->addChild(p);

	for (int i = 0; i < pNode->NumberOfChildren(); ++i)
	{
		collect3dsMaxNodesRecursively(p, pNode->GetChildNode(i));
	}
}

static void deleteSceneNodesRecursively(ISceneNode* const pNode)
{
	if (pNode == 0)
	{
		return;
	}

	for (int i = 0; i < pNode->getChildCount(); ++i)
	{
		deleteSceneNodesRecursively(pNode->getChild(i));
	}

	delete pNode;
}

static bool getBoolParameter(const std::string& parameters, const char* const name, bool const defaultValue)
{
	if (parameters.empty() || (name == 0) || (name[0] == 0))
	{
		return defaultValue;
	}

	std::string value;
	if (!PropertyHelpers::GetPropertyValue(parameters, name, value))
	{
		return defaultValue;
	}

	return (!value.empty()) && (atoi(value.c_str()) != 0);
}

static std::string getStringParameter(const std::string& parameters, const char* const name, const std::string& defaultValue)
{
	if (parameters.empty() || (name == 0) || (name[0] == 0))
	{
		return defaultValue;
	}

	std::string value;
	if (!PropertyHelpers::GetPropertyValue(parameters, name, value))
	{
		return defaultValue;		
	}

	return value;
}

GeometryExtendedExportSource::GeometryExtendedExportSource(const std::string& parameters, const std::vector<INode*>& nodes)
{
	m_bExportFilePerNode = getBoolParameter(parameters, "ExportFilePerNode", false);
	m_bMergeAllNodes = getBoolParameter(parameters, "MergeAllNodes", true);

	if (m_bExportFilePerNode)
	{
		for (size_t i = 0; i < nodes.size(); ++i)
		{
			SceneNodeVirtual* const pRoot = new SceneNodeVirtual();
			m_scenesRootNode.push_back(pRoot);

			collect3dsMaxNodesRecursively(pRoot, nodes[i]);
		}
	}
	else
	{
		SceneNodeVirtual* const pRoot = new SceneNodeVirtual();
		m_scenesRootNode.push_back(pRoot);

		for (size_t i = 0; i < nodes.size(); ++i)
		{
			collect3dsMaxNodesRecursively(pRoot, nodes[i]);
		}
	}
}

GeometryExtendedExportSource::~GeometryExtendedExportSource()
{
	for (size_t i = 0; i < m_scenesRootNode.size(); ++i)
	{
		deleteSceneNodesRecursively(m_scenesRootNode[i]);
	}
}

void GeometryExtendedExportSource::GetMetaData(SExportMetaData& metaData) const
{														  
	ExportFunctions::GetMetaData(metaData);
}

std::string GeometryExtendedExportSource::GetDCCFileName() const
{
	return std::string(GetCOREInterface()->GetCurFilePath());
}

std::string GeometryExtendedExportSource::GetExportDirectory() const
{
	return PathHelpers::GetDirectory(GetDCCFileName());
}

namespace
{
	void RecurseHierarchyAndListBones(ISkeletonData* skeletonData, INode* node, int parentIndex, const IMaterialData* materialData, SkinningHelpers::SkinAccessor& skinAccessor)
	{
		// Check whether we should export this bone.
		bool shouldExportBone = true;
		shouldExportBone = shouldExportBone && !NodeHelpers::IsFootPrint(node); // Hack! Why is this necessary?
		if (!shouldExportBone)
			return;

		// Add the bone.
		int boneIndex = skeletonData->AddBone(node, node->GetName(), parentIndex);

		// Check whether the bone has geometry.
		Mesh* mesh = MeshHelpers::GetMesh(node);
		bool hasGeometry = (mesh ? (mesh->getNumFaces() > 0) : false);
		skeletonData->SetHasGeometry(boneIndex, hasGeometry);

		// Set the transform of the bone.
		Matrix3 initTM;
		skinAccessor.GetBoneInitialPosition(node, initTM);

		Point3 translation, scaling;
		Quat rotation;

		DecomposeMatrix(initTM, translation, rotation, scaling);
		float translationVec[3] = {translation.x, translation.y, translation.z};
		float rotationVec[3];
		QuatToEuler(rotation, rotationVec, RotationValue::kXYZ);
		float scalingVec[3] = {scaling.x, scaling.y, scaling.z};

		skeletonData->SetTranslation(boneIndex, translationVec);
		skeletonData->SetRotation(boneIndex, rotationVec);
		skeletonData->SetScale(boneIndex, scalingVec);

		// Check for phys mesh nodes.
		static const char* physBoneSuffix = " Phys";
		static const char* physParentFrameBoneSuffix = " Phys ParentFrame";
		std::string physBoneName = std::string(node->GetName()) + physBoneSuffix;
		std::string physParentFrameBoneName = std::string(node->GetName()) + physParentFrameBoneSuffix;
		INode* physBone = GetCOREInterface()->GetINodeByName(physBoneName.c_str());
		INode* physParentFrameBone = GetCOREInterface()->GetINodeByName(physParentFrameBoneName.c_str());

		// Set the parent frame matrix if it exists.
		INode* physParentFrameParentBone = (physParentFrameBone ? physParentFrameBone->GetParentNode() : 0);
		if (physParentFrameBone && physParentFrameParentBone)
		{
			Matrix3 frameTM = physParentFrameBone->GetNodeTM(0);
			Matrix3 parentTM = physParentFrameParentBone->GetNodeTM(0);
			Matrix3 relativeFrameTM = frameTM * Inverse(parentTM);

			Point3 translation, scaling;
			Quat rotation;

			DecomposeMatrix(relativeFrameTM, translation, rotation, scaling);
			float translationVec[3] = {translation.x, translation.y, translation.z};
			float rotationVec[3];
			QuatToEuler(rotation, rotationVec, RotationValue::kXYZ);
			float scalingVec[3] = {scaling.x, scaling.y, scaling.z};
			skeletonData->SetParentFrameTranslation(boneIndex, translationVec);
			skeletonData->SetParentFrameRotation(boneIndex, rotationVec);
			skeletonData->SetParentFrameScale(boneIndex, scalingVec);
		}

		// Set physicalization settings.
		if (physBone)
			skeletonData->SetPhysicalized(boneIndex, true);

		// Get joint params.
		Control* physBoneController = (physBone ? physBone->GetTMController() : 0);
		Control* physBoneRotationController = (physBoneController ? physBoneController->GetRotationController() : 0);
		JointParams* physBoneJointParams = (physBoneRotationController ? static_cast<JointParams*>(physBoneRotationController->GetProperty(PROPID_JOINTPARAMS)) : 0);
		DWORD flags = (physBoneJointParams ? physBoneJointParams->flags : 0);
		for (int axis = 0; axis < 3; ++axis)
		{
			if (flags & (JNT_XLIMITED << axis))
			{
				float minimum = (physBoneJointParams ? physBoneJointParams->min[axis] : 0);
				skeletonData->SetLimit(boneIndex, ISkeletonData::Axis(axis), ISkeletonData::LimitMin, minimum);
				float maximum = (physBoneJointParams ? physBoneJointParams->max[axis] : 0);
				skeletonData->SetLimit(boneIndex, ISkeletonData::Axis(axis), ISkeletonData::LimitMax, maximum);
			}
			float springAngle = (physBoneJointParams ? physBoneJointParams->spring[axis] : 0.0f);
			skeletonData->SetSpringAngle(boneIndex, ISkeletonData::Axis(axis), springAngle);
			float springTension = (physBoneJointParams ? physBoneJointParams->stens[axis] : 0.0f);
			skeletonData->SetSpringTension(boneIndex, ISkeletonData::Axis(axis), springTension * 50.0f); // This factor of 50 is bizarre, but it's in the old pipeline and seems to be required for the number to match what appears in the UI.
			float damping = (physBoneJointParams ? physBoneJointParams->damping[axis] : 0.0f);
			skeletonData->SetAxisDamping(boneIndex, ISkeletonData::Axis(axis), damping);
		}

		// Recurse to the node's children.
		for (int childIndex = 0, childCount = node->NumChildren(); childIndex < childCount; ++childIndex)
			RecurseHierarchyAndListBones(skeletonData, node->GetChildNode(childIndex), boneIndex, materialData, skinAccessor);
	}

	SHelperData GetNodeHelperData(INode* const node)
	{
		SHelperData helperData;
		helperData.m_eHelperType = SHelperData::eHelperType_UNKNOWN;

		Object* const der_obj = node ? node->GetObjectRef() : 0;

		if (!der_obj)
		{
			return helperData;
		}

		Object* const obj = der_obj->Eval(0).obj;
		const SClass_ID sid = obj->SuperClassID();
		const Class_ID cid = obj->ClassID();

		if (sid == HELPER_CLASS_ID)
		{
			if (cid==Class_ID(POINTHELP_CLASS_ID, 0))	
			{
				helperData.m_eHelperType = SHelperData::eHelperType_Point;
			}
			else if(cid==Class_ID(DUMMY_CLASS_ID, 0))
			{
				helperData.m_eHelperType = SHelperData::eHelperType_Dummy;

				const Box3 box = ((const DummyObject *)obj)->GetBox();

				helperData.m_boundBoxMin[0] = box.pmin.x;
				helperData.m_boundBoxMin[1] = box.pmin.y;
				helperData.m_boundBoxMin[2] = box.pmin.z;

				helperData.m_boundBoxMax[0] = box.pmax.x;
				helperData.m_boundBoxMax[1] = box.pmax.y;
				helperData.m_boundBoxMax[2] = box.pmax.z;
			}
		}

		return helperData;
	}


	void RecurseHierarchyAndListModels(IModelData* const modelData, ISceneNode* const pNode, int parentModelIndex)
	{
		if (pNode == 0)
		{
			return;				
		}

		if (pNode->getNativeSceneNode())
		{
			INode* const node = static_cast<INode*>(pNode->getNativeSceneNode());

			if (NodeHelpers::IsExcluded(node))
			{
				return;				
			}

			ObjectState objectState = node->EvalWorldState(0);
			SClass_ID const superClassID = objectState.obj->SuperClassID();

			if ((superClassID == GEOMOBJECT_CLASS_ID) || (superClassID == HELPER_CLASS_ID))
			{
				Point3 translation;
				Quat rotation;
				Point3 scaling;
				{
					ISceneNode* const pParent = pNode->getParent();
					Matrix34 const tm = pNode->getWorldTransform();
					Matrix34 const relativeNodeTM = (pParent ?  pParent->getWorldTransform().GetInverted() * tm : tm);

					DecomposeMatrix(get3dsMaxMatrixFromCryMatrix(relativeNodeTM), translation, rotation, scaling);
				}

				float const translationVec[3] = {translation.x, translation.y, translation.z};
				float rotationVec[3];
				QuatToEuler(rotation, rotationVec, RotationValue::kXYZ);
				float const scalingVec[3] = {scaling.x, scaling.y, scaling.z};

				bool const bGeometry = (superClassID == GEOMOBJECT_CLASS_ID);
				SHelperData const helperData = (superClassID == HELPER_CLASS_ID) ? GetNodeHelperData(node) : SHelperData();
				std::string const prop = UserPropertyHelpers::GetNodeProperties(node);

				int const modelIndex = modelData->AddModel(node, node->GetName(), parentModelIndex, bGeometry, helperData, prop);

				modelData->SetTranslationRotationScale(modelIndex, translationVec, rotationVec, scalingVec);

				parentModelIndex = modelIndex;
			}
		}

		// Recurse to the node's children.
		for (int childIndex = 0; childIndex < pNode->getChildCount(); ++childIndex)
		{
			RecurseHierarchyAndListModels(modelData, pNode->getChild(childIndex), parentModelIndex);
		}
	}

	bool RecurseHierarchyAndListRootMaterials(IExportContext* context, std::set<Mtl*>& mtls, void* ptr, bool bNative3dsMaxNode)
	{
		bool everythingOk = true;

		if (ptr == 0)
		{
			return everythingOk;
		}

		ISceneNode* const pNode = bNative3dsMaxNode ? 0 : static_cast<ISceneNode*>(ptr);
		INode* const node = bNative3dsMaxNode ? static_cast<INode*>(ptr) : static_cast<INode*>(pNode->getNativeSceneNode());

		if (node)
		{
			ObjectState objectState = node->EvalWorldState(0);
			SClass_ID const superClassID = objectState.obj->SuperClassID();

			if (superClassID == GEOMOBJECT_CLASS_ID)
			{
				Mtl* const mtl = node->GetMtl();

				if (mtl == 0)
				{
					const char* name = node->GetName();
					name = ((name==0) || (name[0]==0)) ? "<unknown>" : name;
					Log(context, IExportContext::MessageSeverity_Error, "Geometry node '%s' has no material assigned.", name);
					everythingOk = false;				
				}
				else
				{
					mtls.insert(mtl);
				}
			}

			// Recurse through any bones this model may have.
			SkinningHelpers::SkinAccessor skinAccessor(node);
			std::set<INode*> skeletonRoots;
			skinAccessor.FindSkeletonRoots(skeletonRoots);
			for (std::set<INode*>::iterator rootPos = skeletonRoots.begin(); rootPos != skeletonRoots.end(); ++rootPos)
			{
				everythingOk = RecurseHierarchyAndListRootMaterials(context, mtls, *rootPos, true) && everythingOk;
			}
		}

		// Recurse to the node's children.
		if (bNative3dsMaxNode)
		{
			for (int childIndex = 0; childIndex < node->NumberOfChildren(); ++childIndex)
			{
				everythingOk = RecurseHierarchyAndListRootMaterials(context, mtls, node->GetChildNode(childIndex), true) && everythingOk;
			}
		}
		else
		{
			for (int childIndex = 0; childIndex < pNode->getChildCount(); ++childIndex)
			{
				everythingOk = RecurseHierarchyAndListRootMaterials(context, mtls, pNode->getChild(childIndex), false) && everythingOk;
			}
		}

		return everythingOk;
	}

	int getIndexOfBlockParameter( IParamBlock2* pBlock, const MCHAR* name )
	{
		for (int i=0, count = pBlock->NumParams(); i<count; ++i)
		{
			const int id = pBlock->IndextoID(i);
			const ParamDef& paramDef = pBlock->GetParamDef(id);

			if (paramDef.int_name && (strcmp(paramDef.int_name, name) == 0))
			{
				return i;
			}
		}
		return -1;
	}

	std::string GetMaterialProperties(Mtl* mtl)
	{
		std::string properties = "";

		if (mtl->ClassID() != Class_ID(DMTL_CLASS_ID, 0))
		{
			return properties;
		}

		StdMat2* pStdMat = (StdMat2*)mtl;
		Shader* pShader = pStdMat->GetShader();

		const Class_ID CrytekShaderClassID(0x6c0b48e3, 0x2f836597);

		if ((pShader == 0) || (pShader->ClassID() != CrytekShaderClassID))
		{
			return properties;
		}

		IParamBlock2* pBlock = pShader->GetParamBlock(0);
		if (pBlock == 0)
		{
			return properties;
		}
		
		const int idxPhysicalize = getIndexOfBlockParameter(pBlock, "physicalizeMaterial");
		const int idxSurfaceName = getIndexOfBlockParameter(pBlock, "surfaceName");

		if ((idxPhysicalize >= 0) && (idxSurfaceName >= 0))
		{
			properties += "__phys";
			
			const bool physicalize = (pBlock->GetInt( pBlock->IndextoID(idxPhysicalize) ) != 0);
			
			if(!physicalize)
			{
				properties += "None";
			}
			else
			{
				const char* surfaceName = pBlock->GetStr( pBlock->IndextoID(idxSurfaceName) );
				if (surfaceName == 0)
				{
					surfaceName = "";
				}

				if(stricmp(surfaceName, "Physical Proxy (NoDraw)") == 0)
				{
					properties += "ProxyNoDraw";
				}
				else if(stricmp(surfaceName, "No Collide") == 0)
				{
					properties += "NoCollide";
				}
				else if(stricmp(surfaceName, "Obstruct") == 0)
				{
					properties += "Obstruct";
				}
				else
				{
					properties += "Default";
				}
			}
		}

		return properties;
	}

	bool AddRootMaterial(IExportContext* context, IMaterialData* materialData, Mtl* mtl)
	{
		assert(materialData);

		if (mtl == 0)
		{
			Log(context, IExportContext::MessageSeverity_Error, "Unexpected NULL material.");
			return false;
		}

		std::string name = mtl->GetName();

		assert(!name.empty());

		const int subMtlCount = mtl->NumSubMtls();

		if (subMtlCount <= 0)
		{
			// Simple non-multimaterial. Handle it as a multimaterial with single submaterial.			
			const int subMtlIndex = 0;
			materialData->AddMaterial(name.c_str(), subMtlIndex, mtl, GetMaterialProperties(mtl).c_str());
		}
		else
		{
			for (int subMtlIndex = 0; subMtlIndex < subMtlCount; ++subMtlIndex)
			{
				Mtl* const subMtl = mtl->GetSubMtl(subMtlIndex);

				if (subMtl == 0)
				{
					// This material ID (subMtlIndex) does not exist in this material. So we will not create material slot.
					continue;
				}

				// We don't allow to have recursive multisubmaterials
				if (subMtl->NumSubMtls() > 0)
				{
					Log(context, IExportContext::MessageSeverity_Error, "Recursive multisubmaterial found. It's not allowed.");
					return false;
				}

				// Add material slot to our "library" of materials.
				// Note: we don't store real submaterial's name, because we don't need it. But we can, if it needed for debug/error messages.
				// string childName(mtl->GetSubMtlSlotName(subMtlIndex));
				// while (!childName.empty() && strchr(" (0123456789)", childName[0]) != 0)
				//    childName = childName.substr(1);

				materialData->AddMaterial(name.c_str(), subMtlIndex, mtl, GetMaterialProperties(subMtl).c_str());
			}
		}

		return true;
	}
}

void GeometryExtendedExportSource::ReadGeometryFiles(IExportContext* context, IGeometryFileData* geometryFileData)
{
	for (size_t i = 0; i < m_scenesRootNode.size(); ++i)
	{
		assert(m_scenesRootNode[i]);

		std::string name;

		if (m_bExportFilePerNode)
		{
			const ISceneNode* const root = m_scenesRootNode[i];
			INode* const node = static_cast<INode*>(root->getChild(0)->getNativeSceneNode());
			assert(node);
			name = node->GetName();
		}
		else
		{
			name = PathHelpers::RemoveExtension(PathHelpers::GetFilename(GetDCCFileName()));
		}

		std::replace(name.begin(), name.end(), ' ', '_');

		IGeometryFileData::SProperties prop;
		prop.bDoNotMerge = !m_bMergeAllNodes;
		prop.filetypeInt = CRY_FILE_TYPE_CGF;

		geometryFileData->AddGeometryFile(m_scenesRootNode[i], name.c_str(), prop);
	}
}

void GeometryExtendedExportSource::ReadModels(const IGeometryFileData* geometryFileData, int geometryFileIndex, IModelData* modelData)
{
	ISceneNode* const geometryFileNode = static_cast<ISceneNode*>(geometryFileData->GetGeometryFileHandle(geometryFileIndex));
	RecurseHierarchyAndListModels(modelData, geometryFileNode, -1);
}

bool GeometryExtendedExportSource::ReadMaterials(IExportContext* context, const IGeometryFileData* const geometryFileData, IMaterialData* materialData)
{
	// Find all the materials used by the models.
	typedef std::set<Mtl*> MaterialSet;
	MaterialSet rootMtls;
	for (int geometryFileIndex = 0; geometryFileIndex < geometryFileData->GetGeometryFileCount(); ++geometryFileIndex)
	{
		ISceneNode* const geometryFileNode = static_cast<ISceneNode*>(geometryFileData->GetGeometryFileHandle(geometryFileIndex));
		bool const ok = RecurseHierarchyAndListRootMaterials(context, rootMtls, geometryFileNode, false);
		if (!ok)
		{
			return false;
		}
	}

	// Remember all the materials.
	for (MaterialSet::iterator materialPos = rootMtls.begin(); materialPos != rootMtls.end(); ++materialPos)
	{
		const bool ok = AddRootMaterial(context, materialData, *materialPos);
		if (!ok)
		{
			return false;
		}
	}

	return true;
}

void GeometryExtendedExportSource::ReadSkinning(IExportContext* context, ISkinningData* skinningData, const IModelData* const modelData, int modelIndex, ISkeletonData* skeletonData)
{
	// Build a table mapping INode* to bone indices.
	std::map<INode*, int> boneIndexMap;
	for (int boneIndex = 0, boneCount = skeletonData->GetBoneCount(); boneIndex < boneCount; ++boneIndex)
	{
		INode* bone = (INode*)skeletonData->GetBoneHandle(boneIndex);
		boneIndexMap.insert(std::make_pair(bone, boneIndex));
	}

	INode* node = static_cast<INode*>(modelData->GetModelHandle(modelIndex));

	SkinningHelpers::SkinAccessor skinAccessor(node);

	// Find the bone links for all vertices.
	int vertexCount = skinAccessor.GetVertexCount();
	skinningData->SetVertexCount(vertexCount);
	for (int vertexIndex = 0; vertexIndex < vertexCount; ++vertexIndex)
	{
		int linkCount = skinAccessor.GetNumBoneLinksForVertex(vertexIndex);
		for (int linkIndex = 0; linkIndex < linkCount; ++linkIndex)
		{
			INode* bone;
			float weight;
			skinAccessor.GetBoneLinkInfoForVertex(vertexIndex, linkIndex, bone, weight);

			std::map<INode*, int>::const_iterator boneIndexPos = boneIndexMap.find(bone);
			int boneIndex = (boneIndexPos != boneIndexMap.end() ? (*boneIndexPos).second : -1);
			if (boneIndex >= 0)
				skinningData->AddWeight(vertexIndex, boneIndex, weight);
		}
	}
}

bool GeometryExtendedExportSource::ReadSkeleton(const IGeometryFileData* const geometryFileData, int geometryFileIndex, const IModelData* const modelData, int modelIndex, const IMaterialData* materialData, ISkeletonData* skeletonData)
{
	INode* node = static_cast<INode*>(modelData->GetModelHandle(modelIndex));

	SkinningHelpers::SkinAccessor skinAccessor(node);

	std::set<INode*> skeletonRoots;
	skinAccessor.FindSkeletonRoots(skeletonRoots);
	for (std::set<INode*>::iterator rootPos = skeletonRoots.begin(), rootEnd = skeletonRoots.end(); rootPos != rootEnd; ++rootPos)
		RecurseHierarchyAndListBones(skeletonData, *rootPos, -1, materialData, skinAccessor);

	return skinAccessor.HasSkin();
}

int GeometryExtendedExportSource::GetAnimationCount() const
{
	return 0;
}

std::string GeometryExtendedExportSource::GetAnimationName(const IGeometryFileData* geometryFileData, int geometryFileIndex, int animationIndex) const
{
	assert(animationIndex == 0); // Max only supports 1 animation at a time.

	return PathHelpers::RemoveExtension(PathHelpers::GetFilename(GetDCCFileName()));
}

void GeometryExtendedExportSource::GetAnimationTimeSpan(float& start, float& stop, int animationIndex) const
{
	assert(animationIndex == 0); // Max only supports 1 animation at a time.

	Interval animInterval = GetCOREInterface()->GetAnimRange();
	start = TicksToSec(animInterval.Start());
	stop = TicksToSec(animInterval.End());
}

void GeometryExtendedExportSource::ReadAnimationFlags(IExportContext* context, IAnimationData* animationData, const IGeometryFileData* const geometryFileData, const IModelData* modelData, int modelIndex, const ISkeletonData* skeletonData, int animationIndex) const
{
}

IAnimationData * GeometryExtendedExportSource::ReadAnimation(IExportContext* context, const IGeometryFileData* const geometryFileData, const IModelData* modelData, int, const ISkeletonData* skeletonData, int animationIndex, float fps) const
{
	int modelCount = modelData->GetModelCount();
	IAnimationData *animationData = NULL;

	for (int modelIndex = 0; modelIndex < modelCount; ++modelIndex)
	{
		INode* node = static_cast<INode*>(modelData->GetModelHandle(modelIndex));
		Control *pControl = node->GetTMController();
		if (pControl != 0 && pControl->ClassID() == Class_ID(PRS_CONTROL_CLASS_ID,0)
		&& (pControl->GetPositionController() != NULL ||
				pControl->GetRotationController() != NULL ||
				pControl->GetScaleController() != NULL))
		{
			if (animationData == NULL)
				animationData = new NonSkeletalAnimationData(modelCount);
			
			// Position controller
			if (pControl->GetPositionController() != NULL)
			{
				Control *pControlPos = pControl->GetPositionController();
				if (pControlPos->ClassID() == Class_ID(TCBINTERP_POSITION_CLASS_ID,0))
				{
					IKeyControl* ikey = GetKeyControlInterface(pControlPos);
					int frameCount = ikey->GetNumKeys();
					animationData->SetFrameCountPos(modelIndex, frameCount);
					for (int frameIndex = 0; frameIndex < frameCount; ++frameIndex)
					{
						ITCBPoint3Key key;		
						float translation[3];
						ikey->GetKey(frameIndex, &key); 
						translation[0] = key.val.x;
						translation[1] = key.val.y;
						translation[2] = key.val.z;
						animationData->SetFrameDataPos(modelIndex, frameIndex, translation);
						animationData->SetFrameTimePos(modelIndex, frameIndex, TicksToSec(key.time));
						IAnimationData::TCB tcb;
						tcb.tension = key.tens; tcb.continuity = key.cont; tcb.bias = key.bias;
						animationData->SetFrameTCBPos(modelIndex, frameIndex, tcb);
						IAnimationData::Ease ease;
						ease.in = key.easeIn; ease.out = key.easeOut;
						animationData->SetFrameEaseInOutPos(modelIndex, frameIndex, ease);
					}
				}
				else
					assert(0);
			}

			// Rotation controller
			if (pControl->GetRotationController() != NULL)
			{
				Control *pControlRot = pControl->GetRotationController();
				if (pControlRot->ClassID() == Class_ID(TCBINTERP_ROTATION_CLASS_ID,0))
				{
					IKeyControl* ikey = GetKeyControlInterface(pControlRot);
					int frameCount = ikey->GetNumKeys();
					animationData->SetFrameCountRot(modelIndex, frameCount);
					for (int frameIndex = 0; frameIndex < frameCount; ++frameIndex)
					{
						ITCBRotKey key;		
						float rotation[3];
						ikey->GetKey(frameIndex, &key); 
						QuatToEuler(Quat(key.val), rotation, RotationValue::kXYZ);
						rotation[0] = RAD2DEG(rotation[0]);
						rotation[1] = RAD2DEG(rotation[1]);
						rotation[2] = RAD2DEG(rotation[2]);
						animationData->SetFrameDataRot(modelIndex, frameIndex, rotation);
						animationData->SetFrameTimeRot(modelIndex, frameIndex, TicksToSec(key.time));
						IAnimationData::TCB tcb;
						tcb.tension = key.tens; tcb.continuity = key.cont; tcb.bias = key.bias;
						animationData->SetFrameTCBRot(modelIndex, frameIndex, tcb);
						IAnimationData::Ease ease;
						ease.in = key.easeIn; ease.out = key.easeOut;
						animationData->SetFrameEaseInOutRot(modelIndex, frameIndex, ease);
					}
				}
				else
					assert(0);
			}

			// Scale controller
			if (pControl->GetScaleController() != NULL)
			{
				Control *pControlScl = pControl->GetScaleController();
				if (pControlScl->ClassID() == Class_ID(TCBINTERP_SCALE_CLASS_ID,0))
				{
					IKeyControl* ikey = GetKeyControlInterface(pControlScl);
					int frameCount = ikey->GetNumKeys();
					animationData->SetFrameCountScl(modelIndex, frameCount);
					for (int frameIndex = 0; frameIndex < frameCount; ++frameIndex)
					{
						ITCBScaleKey key;		
						float scale[3];
						ikey->GetKey(frameIndex, &key); 
						scale[0] = key.val.s.x;
						scale[1] = key.val.s.y;
						scale[2] = key.val.s.z;
						animationData->SetFrameDataScl(modelIndex, frameIndex, scale);
						animationData->SetFrameTimeScl(modelIndex, frameIndex, TicksToSec(key.time));
						IAnimationData::TCB tcb;
						tcb.tension = key.tens; tcb.continuity = key.cont; tcb.bias = key.bias;
						animationData->SetFrameTCBScl(modelIndex, frameIndex, tcb);
						IAnimationData::Ease ease;
						ease.in = key.easeIn; ease.out = key.easeOut;
						animationData->SetFrameEaseInOutScl(modelIndex, frameIndex, ease);
					}
				}
				else
					assert(0);
			}
		}
	}

	return animationData;
}

static void ConstructMaterialIDToIndexMap(std::vector<int>& mtlIDToIndexMap, const Mtl* const mtl, const IMaterialData* const materialData)
{
	mtlIDToIndexMap.resize(0);

	if (mtl == 0)
	{
		return;
	}

	for (int materialIndex = 0, materialCount = materialData->GetMaterialCount(); materialIndex < materialCount; ++materialIndex)
	{
		const Mtl* const pMtl = static_cast<Mtl*>( materialData->GetHandle(materialIndex) );

		if (mtl == pMtl)
		{
			const int id = materialData->GetID(materialIndex);
			const int oldSize = int(mtlIDToIndexMap.size());
			const int minSize = id+1;
			if (oldSize < minSize)
			{
				mtlIDToIndexMap.resize(minSize, -1);	
			}
			mtlIDToIndexMap[id] = materialIndex;
		}
	}
}

namespace
{
	struct MeshStreams
	{
		std::vector<MeshHelpers::Vector3> positions;
		std::vector<MeshHelpers::Vector3> normals;
		std::vector<MeshHelpers::TextureCoordinate> textureCoordinates;
		std::vector<MeshHelpers::VertexColour> vertexColours;
		std::vector<MeshHelpers::Triangle> triangles;
	};

	bool CheckMaterialDataForNode(IExportContext* context, INode* node, Mtl* const mtl, const std::vector<int>& mtlIDToIndexMap)
	{
		assert(node);
		assert(mtl);

		const char* nodeName = node->GetName();
		nodeName = (nodeName && nodeName[0]) ? nodeName : "<unknown>";

		Mesh* mesh = MeshHelpers::GetMesh(node);
		if (!mesh)
		{
			Log(context, IExportContext::MessageSeverity_Error, "Strange 'no mesh' case when checking materials for node '%s'", nodeName);
			return false;
		}

		for (int triangleIndex = 0, triangleCount = mesh->getNumFaces(); triangleIndex < triangleCount; ++triangleIndex)
		{
			const int mtlID = mesh->faces[triangleIndex].getMatID();

			if (mtlID < 0)
			{
				Log(context, IExportContext::MessageSeverity_Error, "Unexpected materialID value: %d", mtlID);
				return false;
			}

			if ((mtlID >= mtlIDToIndexMap.size()) || (mtlIDToIndexMap[mtlID] < 0))
			{
				const char* matName = mtl->GetName();
				matName = (matName && matName[0]) ? matName : "<unknown>";

				Log(
					context,
					IExportContext::MessageSeverity_Error, 
					"Node '%s' uses submaterialID %d but this submaterialID is missing in material '%s'", 
					nodeName, mtlID+1, matName);

				return false;
			}
		}

		return true;
	}

	bool ReadGeometryForNode(IExportContext* context, MeshStreams& streams, INode* node, Mtl* const mtl, const std::vector<int>& mtlIDToIndexMap)
	{
		assert(node);
		assert(mtl);

		const char* nodeName = node->GetName();
		nodeName = (nodeName && nodeName[0]) ? nodeName : "<unknown>";

		Mesh* mesh = MeshHelpers::GetMesh(node);
		if (!mesh)
		{
			Log(context, IExportContext::MessageSeverity_Error, "Strange 'no mesh' case when reading geometry for node '%s'", nodeName);
			return false;
		}

		bool ok = CheckMaterialDataForNode(context, node, mtl, mtlIDToIndexMap);
		if (!ok)
		{
			return false;
		}

		MeshHelpers::MeshData meshData;

		ok = MeshHelpers::AssignTriangleIndices(mesh, streams.triangles, meshData);
		if (!ok)
		{
			Log(context, IExportContext::MessageSeverity_Error, "Unexpected empty mesh.");
			return false;
		}

		Matrix3 objectOffsetTM;
		objectOffsetTM.IdentityMatrix();
		MeshHelpers::GetObjectOffsetTM(objectOffsetTM, node);

		MeshHelpers::CalculatePositions(mesh, meshData, objectOffsetTM, streams.positions);
		MeshHelpers::CalculateNormals(mesh, meshData, streams.normals);
		MeshHelpers::CalculateTextureCoordinates(meshData, streams.textureCoordinates);
		MeshHelpers::CalculateVertexColours(meshData, streams.vertexColours);

		return true;
	}

	void WriteGeometryData(MeshStreams& streams, IGeometryData* geometry, const std::vector<int>& mtlIDToIndexMap)
	{
		for (int i = 0, count = int(streams.positions.size()); i < count; ++i)
		{
			geometry->AddPosition(streams.positions[i].x, streams.positions[i].y, streams.positions[i].z);
		}
		for (int i = 0, count = int(streams.normals.size()); i < count; ++i)
		{
			geometry->AddNormal(streams.normals[i].x, streams.normals[i].y, streams.normals[i].z);
		}
		for (int i = 0, count = int(streams.textureCoordinates.size()); i < count; ++i)
			geometry->AddTextureCoordinate(streams.textureCoordinates[i].u, streams.textureCoordinates[i].v);
		for (int i = 0, count = int(streams.vertexColours.size()); i < count; ++i)
			geometry->AddVertexColour(streams.vertexColours[i].r, streams.vertexColours[i].g, streams.vertexColours[i].b, streams.vertexColours[i].a);
		for (int i = 0, count = int(streams.triangles.size()); i < count; ++i)
		{
			int indices[12] = {
				streams.triangles[i].v[0].p, streams.triangles[i].v[0].n, streams.triangles[i].v[0].t, streams.triangles[i].v[0].c,
				streams.triangles[i].v[1].p, streams.triangles[i].v[1].n, streams.triangles[i].v[1].t, streams.triangles[i].v[1].c,
				streams.triangles[i].v[2].p, streams.triangles[i].v[2].n, streams.triangles[i].v[2].t, streams.triangles[i].v[2].c,
			};
			int materialID = streams.triangles[i].mtlID;
			int materialIndex = (materialID >= 0 && materialID < mtlIDToIndexMap.size() ? mtlIDToIndexMap[materialID] : -1);
			assert(materialIndex>=0);
			geometry->AddPolygon(indices, materialIndex);
		}
	}

	bool ReadGeometryDataForNode(IExportContext* context, IGeometryData* geometry, INode* node, Mtl* const mtl, const std::vector<int>& mtlIDToIndexMap)
	{
		assert(node);
		assert(mtl);

		bool success = true;

		MeshStreams streams;

		success = success && ReadGeometryForNode(context, streams, node, mtl, mtlIDToIndexMap);
		if (success)
		{
			WriteGeometryData(streams, geometry, mtlIDToIndexMap);
		}

		return success;
	}
}

bool GeometryExtendedExportSource::ReadGeometry(IExportContext* context, IGeometryData* geometry, const IModelData* const modelData, const IMaterialData* const materialData, int modelIndex)
{
	INode* const node = static_cast<INode*>(modelData->GetModelHandle(modelIndex));
	if (node == 0)
	{
		Log(context, IExportContext::MessageSeverity_Error, "Unexpected NULL node.");
		return false;
	}

	Mtl* const mtl = node->GetMtl();
	if (mtl == 0)
	{
		Log(context, IExportContext::MessageSeverity_Error, "Unexpected NULL material.");
		return false;
	}

	std::vector<int> mtlIDToIndexMap;
	ConstructMaterialIDToIndexMap(mtlIDToIndexMap, mtl, materialData);

	return ReadGeometryDataForNode(context, geometry, node, mtl, mtlIDToIndexMap);
}

bool GeometryExtendedExportSource::ReadGeometryMaterialData(IExportContext* context, IGeometryMaterialData* geometryMaterialData, const IModelData* const modelData, const IMaterialData* const materialData, int modelIndex) const
{
	INode* node = static_cast<INode*>(modelData->GetModelHandle(modelIndex));
	if (node == 0)
	{
		Log(context, IExportContext::MessageSeverity_Error, "Unexpected NULL node.");
		return false;
	}

	Mesh* mesh = MeshHelpers::GetMesh(node);
	if (!mesh)
		return false;

	Mtl* mtl = node->GetMtl();
	if (mtl == 0)
	{
		Log(context, IExportContext::MessageSeverity_Error, "Unexpected NULL material.");
		return false;
	}

	std::vector<int> mtlIDToIndexMap;
	ConstructMaterialIDToIndexMap(mtlIDToIndexMap, mtl, materialData);

	bool ok = CheckMaterialDataForNode(context, node, mtl, mtlIDToIndexMap);
	if (!ok)
	{
		return false;
	}

	std::vector<MeshHelpers::Triangle> triangles;
	MeshHelpers::MeshData meshData;

	ok = MeshHelpers::AssignTriangleIndices(mesh, triangles, meshData);
	if (!ok)
	{
		Log(context, IExportContext::MessageSeverity_Error, "Unexpected empty mesh.");
		return false;
	}

	assert(triangles.size() == meshData.faceCount);

	for (int triangleIndex = 0, triangleCount = int(triangles.size()); triangleIndex < triangleCount; ++triangleIndex)
	{
		const int mtlID = triangles[triangleIndex].mtlID;
		const int materialIndex = ((mtlID >=0) && (mtlID < mtlIDToIndexMap.size())) ? mtlIDToIndexMap[mtlID] : -1;
		if (materialIndex < 0)
		{
			Log(context, IExportContext::MessageSeverity_Error, "Unexpected bad materialID.");
			return false;			
		}
		else
		{
			geometryMaterialData->AddUsedMaterialIndex(materialIndex);
		}
	}

	return true;
}

bool GeometryExtendedExportSource::ReadBoneGeometry(IExportContext* context, IGeometryData* geometry, ISkeletonData* skeletonData, int boneIndex, const IMaterialData* const materialData)
{
	INode* node = static_cast<INode*>(skeletonData->GetBoneHandle(boneIndex));
	Mtl* mtl = (node ? node->GetMtl() : 0);

	std::vector<int> mtlIDToIndexMap;
	ConstructMaterialIDToIndexMap(mtlIDToIndexMap, mtl, materialData);

	return ReadGeometryDataForNode(context, geometry, node, mtl, mtlIDToIndexMap);
}

bool GeometryExtendedExportSource::ReadBoneGeometryMaterialData(IExportContext* context, IGeometryMaterialData* geometryMaterialData, ISkeletonData* skeletonData, int boneIndex, const IMaterialData* const materialData) const
{
	INode* node = static_cast<INode*>(skeletonData->GetBoneHandle(boneIndex));
	if (node == 0)
	{
		Log(context, IExportContext::MessageSeverity_Error, "Unexpected NULL node.");
		return false;
	}

	Mesh* mesh = MeshHelpers::GetMesh(node);
	if (!mesh)
		return false;

	Mtl* mtl = node->GetMtl();
	if (mtl == 0)
	{
		Log(context, IExportContext::MessageSeverity_Error, "Unexpected NULL material.");
		return false;
	}

	std::vector<int> mtlIDToIndexMap;
	ConstructMaterialIDToIndexMap(mtlIDToIndexMap, mtl, materialData);

	bool ok = CheckMaterialDataForNode(context, node, mtl, mtlIDToIndexMap);
	if (!ok)
	{
		return false;
	}

	std::vector<MeshHelpers::Triangle> triangles;
	MeshHelpers::MeshData meshData;

	ok = MeshHelpers::AssignTriangleIndices(mesh, triangles, meshData);
	if (!ok)
	{
		Log(context, IExportContext::MessageSeverity_Error, "Unexpected empty mesh.");
		return false;
	}

	assert(triangles.size() == meshData.faceCount);

	for (int triangleIndex = 0, triangleCount = int(triangles.size()); triangleIndex < triangleCount; ++triangleIndex)
	{
		const int mtlID = triangles[triangleIndex].mtlID;
		const int materialIndex = ((mtlID >=0) && (mtlID < mtlIDToIndexMap.size())) ? mtlIDToIndexMap[mtlID] : -1;
		if (materialIndex < 0)
		{
			Log(context, IExportContext::MessageSeverity_Error, "Unexpected bad materialID.");
			return false;			
		}
		else
		{
			geometryMaterialData->AddUsedMaterialIndex(materialIndex);
		}
	}

	return true;
}

void GeometryExtendedExportSource::ReadMorphs(IExportContext* context, IMorphData* morphData, const IModelData* const modelData, int modelIndex)
{
	INode* node = static_cast<INode*>(modelData->GetModelHandle(modelIndex));
	MorphR3* morpher = (node ? MorpherHelpers::GetMorpher(node) : 0);

	morphData->SetHandle(morpher);
	for (int channelIndex = 0, channelCount = (morpher ? int(morpher->chanBank.size()) : 0); channelIndex < channelCount; ++channelIndex)
	{
		if (morpher->chanBank[channelIndex].mActive)
			morphData->AddMorph((void*)INT_PTR(channelIndex), morpher->chanBank[channelIndex].mName.data());
	}
}

bool GeometryExtendedExportSource::ReadMorphGeometry(IExportContext* context, IGeometryData* geometry, const IModelData* const modelData, int modelIndex, const IMorphData* const morphData, int morphIndex, const IMaterialData* materialData)
{
	INode* node = static_cast<INode*>(modelData->GetModelHandle(modelIndex));
	if (node == 0)
	{
		Log(context, IExportContext::MessageSeverity_Error, "Unexpected NULL node.");
		return false;
	}

	MorphR3* morpher = static_cast<MorphR3*>(morphData->GetHandle());
	int morphChannelIndex = int(INT_PTR(morphData->GetMorphHandle(morphIndex)));

	Mtl* mtl = node->GetMtl();
	if (mtl == 0)
	{
		Log(context, IExportContext::MessageSeverity_Error, "Unexpected NULL material.");
		return false;
	}

	std::vector<int> mtlIDToIndexMap;
	ConstructMaterialIDToIndexMap(mtlIDToIndexMap, mtl, materialData);

	bool success = true;
	MeshStreams streams;
	success = success && ReadGeometryForNode(context, streams, node, mtl, mtlIDToIndexMap);
	MorpherHelpers::DeformPositions(streams.positions, morpher, morphChannelIndex);

	if (success)
	{
		WriteGeometryData(streams, geometry, mtlIDToIndexMap);
	}

	return success;
}

static bool IsThereValidController(Control *pControl, int classId)
{
	IKeyControl* ikey = NULL;
	bool result = (pControl != 0
		&& pControl->ClassID() == Class_ID(classId,0)
		&& (ikey = GetKeyControlInterface(pControl))
		&& ikey->GetNumKeys() > 0);

	return result;
}

bool GeometryExtendedExportSource::HasValidPosController(const IModelData* modelData, int modelIndex) const
{
	INode* node = static_cast<INode*>(modelData->GetModelHandle(modelIndex));
	Control* pControl = node->GetTMController();
	if (pControl == 0 || pControl->ClassID() != Class_ID(PRS_CONTROL_CLASS_ID,0))
		return false;

	return IsThereValidController(pControl->GetPositionController(), TCBINTERP_POSITION_CLASS_ID);
}

bool GeometryExtendedExportSource::HasValidRotController(const IModelData* modelData, int modelIndex) const
{
	INode* node = static_cast<INode*>(modelData->GetModelHandle(modelIndex));
	Control* pControl = node->GetTMController();
	if (pControl == 0 || pControl->ClassID() != Class_ID(PRS_CONTROL_CLASS_ID,0))
		return false;

	return IsThereValidController(pControl->GetRotationController(), TCBINTERP_ROTATION_CLASS_ID);
}

bool GeometryExtendedExportSource::HasValidSclController(const IModelData* modelData, int modelIndex) const
{
	INode* node = static_cast<INode*>(modelData->GetModelHandle(modelIndex));
	Control* pControl = node->GetTMController();
	if (pControl == 0 || pControl->ClassID() != Class_ID(PRS_CONTROL_CLASS_ID,0))
		return false;

	return IsThereValidController(pControl->GetScaleController(), TCBINTERP_SCALE_CLASS_ID);
}
