#include "StdAfx.h"
#include "MeshHelpers.h"

namespace MeshHelpers
{
	enum {VERTEX_COLOUR_CHANNEL = 0};
	enum {VERTEX_ALPHA_CHANNEL = MAP_ALPHA};
}


struct FaceCorner
{
	void set(int a_faceIndex, int a_cornerIndex)
	{
		m_faceIndex = a_faceIndex;
		m_cornerIndex = a_cornerIndex;
	}

	static bool less_vertexThenFaceThenCorner(Mesh* pMesh, const FaceCorner& v0, const FaceCorner& v1)
	{
		const Face& face0 = pMesh->faces[v0.m_faceIndex];
		const Face& face1 = pMesh->faces[v1.m_faceIndex];

		const int vertexIndex0 = face0.v[v0.m_cornerIndex];
		const int vertexIndex1 = face1.v[v1.m_cornerIndex];

		if (vertexIndex0 != vertexIndex1)
		{
			return vertexIndex0 < vertexIndex1;
		}

		if (v0.m_faceIndex != v1.m_faceIndex)
		{
			return v0.m_faceIndex < v1.m_faceIndex;
		}

		return v0.m_cornerIndex < v1.m_cornerIndex;
	}

	bool sameVertexIndex(Mesh* pMesh, const FaceCorner& v) const
	{
		const Face& face0 = pMesh->faces[m_faceIndex];
		const Face& face1 = pMesh->faces[v.m_faceIndex];

		const int vertexIndex0 = face0.v[m_cornerIndex];
		const int vertexIndex1 = face1.v[v.m_cornerIndex];

		return vertexIndex0 == vertexIndex1;
	}

	int m_faceIndex;
	int m_cornerIndex;
};

struct FaceCornerLess
{
	FaceCornerLess(Mesh* a_mesh) 
		: mesh(a_mesh)
	{
	}

	bool operator()(const FaceCorner& l, const FaceCorner& r) const
	{
		return FaceCorner::less_vertexThenFaceThenCorner(mesh, l, r);
	}

private:
	Mesh* mesh;
};

struct FaceListLess
{
	FaceListLess(const std::vector<int>& a_chains) 
		: chains(a_chains)
	{
	}

	bool operator()(int l, int r)
	{
		int p0 = chains[l];
		int p1 = chains[r];

		for(;;)
		{
			const int face0 = chains[p0++];
			const int face1 = chains[p1++];

			if(face0 != face1)
			{
				return face0 < face1;
			}

			if(face0 < 0)
			{
				return false;
			}
		}
	}

private:
	const std::vector<int>& chains;
};



Mesh* MeshHelpers::GetMesh(INode* node)
{
	Object* worldState;
	if (!node || !(worldState = node->EvalWorldState(0).obj))
		return 0;

	Object* obj = worldState->Eval(0).obj;
	if (!obj->CanConvertToType(Class_ID(TRIOBJ_CLASS_ID, 0)))
		return 0;
// FIXME: should we "delete tri;"?
	TriObject* tri = (TriObject*)obj->ConvertToType(0, Class_ID(TRIOBJ_CLASS_ID, 0));
	Mesh* mesh = &tri->mesh;

	return mesh;
}

void MeshHelpers::GetObjectOffsetTM(Matrix3& objectOffsetTM, INode* node)
{
	Point3 pos = node->GetObjOffsetPos();
	objectOffsetTM.PreTranslate(pos);
	Quat quat = node->GetObjOffsetRot();
	PreRotateMatrix(objectOffsetTM, quat);
	ScaleValue scaleValue = node->GetObjOffsetScale();
	ApplyScaling(objectOffsetTM, scaleValue);
}


bool MeshHelpers::AssignTriangleIndices(Mesh* mesh, std::vector<MeshHelpers::Triangle>& triangles, MeshHelpers::MeshData& meshData)
{
	meshData.clear();
	triangles.clear();

	if((mesh->getNumFaces() <= 0) || (mesh->getNumVerts() <= 0))
	{
		return false;
	}

	meshData.faceCount = mesh->getNumFaces();

	triangles.resize(meshData.faceCount);

	// Store the material IDs.
	for (int triangleIndex = 0; triangleIndex < meshData.faceCount; ++triangleIndex)
	{
		triangles[triangleIndex].mtlID = mesh->faces[triangleIndex].getMatID();
	}

	// Assign the indices to the vertex positions.
	// TODO: check ranges of indices
	meshData.positionCount = mesh->getNumVerts();
	for (int faceIndex = 0; faceIndex < meshData.faceCount; ++faceIndex)
	{
		const Face& face = mesh->faces[faceIndex];
		for (int vertex = 0; vertex < 3; ++vertex)
		{
			triangles[faceIndex].v[vertex].p = face.v[vertex];
		}
	}

	// Fill helper arrays for computation of normals.
	{
		std::vector<FaceCorner> faceCorners;
		faceCorners.resize(3*meshData.faceCount);

		for (int faceIndex = 0; faceIndex < meshData.faceCount; ++faceIndex)
		{
			for (int cornerIndex = 0; cornerIndex < 3; ++cornerIndex)
			{
				faceCorners[faceIndex*3 + cornerIndex].set(faceIndex, cornerIndex);
			}
		}

		FaceCornerLess fcLess(mesh);
		std::sort(faceCorners.begin(), faceCorners.end(), fcLess);

		meshData.facelistChainsForNormals.reserve(3*meshData.faceCount + 3*meshData.faceCount*2);   // usually needs more space, so it will grow
		meshData.facelistChainsForNormals.resize(3*meshData.faceCount);	  // start of array stores indices of heads of lists (for each facecorner)

		int startOfVertex = 0;
		for (int i = 1, count = (int)faceCorners.size(); i <= count; ++i)
		{
			if ( (i==count) || (!faceCorners[i].sameVertexIndex(mesh, faceCorners[i-1])) )
			{
				// End of block of facecorners with "same vertex" found. Form a facelist for each facecorner sharing this vertex.
				for (int j = startOfVertex; j < i; ++j)
				{
					const int idx = faceCorners[j].m_faceIndex*3 + faceCorners[j].m_cornerIndex;
					meshData.facelistChainsForNormals[idx] = (int)meshData.facelistChainsForNormals.size();

					const Face& faceMain = mesh->faces[ faceCorners[j].m_faceIndex ];
					for (int k = startOfVertex; k < i; ++k)
					{
						const Face& face = mesh->faces[ faceCorners[k].m_faceIndex ];
						if ((k==j) || (faceMain.smGroup & face.smGroup))
						{
							meshData.facelistChainsForNormals.push_back( faceCorners[k].m_faceIndex );
							// TODO: use face size or corner angle when computing smoothed normals
						}	
					}
					// note: each facelist is sorted by face index, because faceCorners[] is sorted by vertex index and then by face index.

					// add "end of list" code
					meshData.facelistChainsForNormals.push_back( -1 );
				}
				startOfVertex = i;
			}
		}

		{
			meshData.facelistHeadsForNormals.resize(3*meshData.faceCount);	// allocate maximum possible number of normals. we will downsize it later

			std::vector<int> sortedFacelists(3*meshData.faceCount);
			for (int i = 0, count = (int)sortedFacelists.size(); i < count; ++i)
			{
				sortedFacelists[i] = i;		// facecorner id
			}
			FaceListLess less(meshData.facelistChainsForNormals);
			std::sort(sortedFacelists.begin(), sortedFacelists.end(), less);

			meshData.normalCount = 0;
			for (int i = 0, count = int(sortedFacelists.size()); i < count; ++i)
			{
				if ((i == 0) || less(sortedFacelists[i - 1], sortedFacelists[i]))
				{
					meshData.facelistHeadsForNormals[meshData.normalCount++] = meshData.facelistChainsForNormals[ sortedFacelists[i] ];
				}
				const int faceIdx = sortedFacelists[i] / 3;
				const int cornerIdx = sortedFacelists[i] - (faceIdx * 3);
				triangles[faceIdx].v[cornerIdx].n = meshData.normalCount - 1;
			}

			meshData.facelistHeadsForNormals.resize(meshData.normalCount);
		}
	}

	// Loop through the texture faces, assigning the indices.
	{
		meshData.textureCoordinates.set(mesh->numTVerts, mesh->tVerts, mesh->tvFace);
		if (meshData.textureCoordinates.vertexCount > 0)
		{
			for (int faceIndex = 0; faceIndex < meshData.faceCount; ++faceIndex)
			{
				const TVFace& tface = meshData.textureCoordinates.faces[faceIndex];
				for (int corner = 0; corner < 3; ++corner)
				{
					triangles[faceIndex].v[corner].t = tface.t[corner];
				}
			}
		}
	}

	// Loop through the color and alpha faces, assigning the indices.

	if (mesh->mapSupport(VERTEX_COLOUR_CHANNEL))
	{
		const int channel = VERTEX_COLOUR_CHANNEL;
		meshData.color.set(mesh->getNumMapVerts(channel), mesh->mapVerts(channel), mesh->mapFaces(channel));
	}

	if (mesh->mapSupport(VERTEX_ALPHA_CHANNEL))
	{
		const int channel = VERTEX_ALPHA_CHANNEL;
		meshData.alpha.set(mesh->getNumMapVerts(channel), mesh->mapVerts(channel), mesh->mapFaces(channel));
	}

	// Check if we have both color and alpha arrays and they both use same indices
	meshData.colorAndAlphaAreMatching = false;
	if ((meshData.color.vertexCount > 0) && (meshData.color.vertexCount == meshData.alpha.vertexCount))
	{
		for (int faceIndex = 0; faceIndex < meshData.faceCount; ++faceIndex)
		{
			const TVFace& faceColor = meshData.color.faces[faceIndex];
			const TVFace& faceAlpha = meshData.alpha.faces[faceIndex];
			for (int corner = 0; corner < 3; ++corner)
			{
				if (faceColor.t[corner] != faceAlpha.t[corner])
				{
					goto nonMatchingFaceFound;
				}
			}
		}
		meshData.colorAndAlphaAreMatching = true;
nonMatchingFaceFound: ;
	}
		
	// Assigning the indices.

	if (meshData.colorAndAlphaAreMatching 
		|| ((meshData.color.vertexCount > 0) && (meshData.alpha.vertexCount <= 0)))
	{
		meshData.vertexColorAndAlphaCount = meshData.color.vertexCount;

		for (int faceIndex = 0; faceIndex < meshData.faceCount; ++faceIndex)
		{
			const TVFace& faceColor = meshData.color.faces[faceIndex];
			for (int corner = 0; corner < 3; ++corner)
			{
				triangles[faceIndex].v[corner].c = faceColor.t[corner];
			}
		}
	}
	else if ((meshData.color.vertexCount <= 0) && (meshData.alpha.vertexCount > 0))
	{
		meshData.vertexColorAndAlphaCount = meshData.alpha.vertexCount;

		for (int faceIndex = 0; faceIndex < meshData.faceCount; ++faceIndex)
		{
			const TVFace& faceAlpha = meshData.alpha.faces[faceIndex];
			for (int corner = 0; corner < 3; ++corner)
			{
				triangles[faceIndex].v[corner].c = faceAlpha.t[corner];
			}
		}
	}
	else if ((meshData.color.vertexCount > 0) && (meshData.alpha.vertexCount > 0))
	{
		// We have both colors and alphas but they are using non-matching face arrays

		meshData.vertexColorAndAlphaCount = meshData.faceCount * 3;

		for (int faceIndex = 0; faceIndex < meshData.faceCount; ++faceIndex)
		{
			for (int corner = 0; corner < 3; ++corner)
			{
				triangles[faceIndex].v[corner].c = faceIndex*3 + corner;
			}
		}
	}
	else
	{
		meshData.vertexColorAndAlphaCount = 0;
	}

	return true;
}

void MeshHelpers::CalculatePositions(Mesh* mesh, const MeshHelpers::MeshData& meshData, Matrix3& objectOffsetTM, std::vector<MeshHelpers::Vector3>& positions)
{
	positions.resize(meshData.positionCount);

	for (int positionIndex = 0; positionIndex < meshData.positionCount; ++positionIndex)
	{
		Point3 pt = objectOffsetTM.PointTransform(mesh->verts[positionIndex]);
		positions[positionIndex].x = pt.x;
		positions[positionIndex].y = pt.y;
		positions[positionIndex].z = pt.z;
	}
}

void MeshHelpers::CalculateNormals(Mesh* mesh, const MeshHelpers::MeshData& meshData, std::vector<MeshHelpers::Vector3>& normals)
{
	assert(meshData.facelistHeadsForNormals.size() == meshData.normalCount);

	normals.resize(meshData.normalCount);

	for (int i = 0; i < meshData.normalCount; ++i)
	{
		Point3 normal;
		normal.Set(0,0,0);

		int p = meshData.facelistHeadsForNormals[i];

		for (;;)
		{
			const int faceIndex = meshData.facelistChainsForNormals[p++];

			if (faceIndex < 0)
			{
				break;
			}

			const Face& face = mesh->faces[faceIndex];
			const Point3& v0 = mesh->verts[face.v[0]];
			const Point3& v1 = mesh->verts[face.v[1]];
			const Point3& v2 = mesh->verts[face.v[2]];

			Point3 faceNormal = (v1 - v0) ^ (v2 - v1);
			normal += Normalize(faceNormal);
		}

		normal = Normalize(normal);

		normals[i].x = normal.x;
		normals[i].y = normal.y;
		normals[i].z = normal.z;
	}
}

void MeshHelpers::CalculateTextureCoordinates(const MeshHelpers::MeshData& meshData, std::vector<MeshHelpers::TextureCoordinate>& textureCoordinates)
{
	textureCoordinates.resize(meshData.textureCoordinates.vertexCount);

	for (int textureCoordinateIndex = 0; textureCoordinateIndex < meshData.textureCoordinates.vertexCount; ++textureCoordinateIndex)
	{
		textureCoordinates[textureCoordinateIndex].u = meshData.textureCoordinates.vertices[textureCoordinateIndex].x;
		textureCoordinates[textureCoordinateIndex].v = meshData.textureCoordinates.vertices[textureCoordinateIndex].y;
	}
}

void MeshHelpers::CalculateVertexColours(const MeshHelpers::MeshData& meshData, std::vector<MeshHelpers::VertexColour>& vertexColours)
{
	vertexColours.resize(meshData.vertexColorAndAlphaCount);

	if (meshData.colorAndAlphaAreMatching)
	{
		for (int i = 0; i < meshData.color.vertexCount; ++i)
		{
			vertexColours[i].r = meshData.color.vertices[i].x;
			vertexColours[i].g = meshData.color.vertices[i].y;
			vertexColours[i].b = meshData.color.vertices[i].z;

			vertexColours[i].a = meshData.alpha.vertices[i].x;
		}
	}
	else if ((meshData.color.vertexCount > 0) && (meshData.alpha.vertexCount <= 0))
	{
		for (int i = 0; i < meshData.color.vertexCount; ++i)
		{
			vertexColours[i].r = meshData.color.vertices[i].x;
			vertexColours[i].g = meshData.color.vertices[i].y;
			vertexColours[i].b = meshData.color.vertices[i].z;

			vertexColours[i].a = 1.0f;
		}
	}
	else if ((meshData.color.vertexCount <= 0) && (meshData.alpha.vertexCount > 0))
	{
		for (int i = 0; i < meshData.alpha.vertexCount; ++i)
		{
			vertexColours[i].r = 1.0f;
			vertexColours[i].g = 1.0f;
			vertexColours[i].b = 1.0f;

			vertexColours[i].a = meshData.alpha.vertices[i].x;
		}
	}
	else if ((meshData.color.vertexCount > 0) && (meshData.alpha.vertexCount > 0))
	{
		for (int faceIndex = 0; faceIndex < meshData.faceCount; ++faceIndex)
		{
			const TVFace& faceColor = meshData.color.faces[faceIndex];
			const TVFace& faceAlpha = meshData.alpha.faces[faceIndex];
			for (int corner = 0; corner < 3; ++corner)
			{
				const int colorIndex = faceColor.t[corner];
				vertexColours[faceIndex * 3 + corner].r = meshData.color.vertices[ colorIndex ].x;
				vertexColours[faceIndex * 3 + corner].g = meshData.color.vertices[ colorIndex ].y;
				vertexColours[faceIndex * 3 + corner].b = meshData.color.vertices[ colorIndex ].z;

				const int alphaIndex = faceAlpha.t[corner];
				vertexColours[faceIndex * 3 + corner].a = meshData.alpha.vertices[ alphaIndex ].x;
			}
		}
	}
}
