#include "StdAfx.h"
#include "SkinningHelpers.h"
#include "ModifierHelpers.h"
#include <iskin.h>

SkinningHelpers::SkinAccessor::SkinAccessor(INode* meshNode)
: meshNode(meshNode)
{
	// Find the skin modifier for the node.
	Modifier* skinModifier = ModifierHelpers::FindModifierSupportingInterface(meshNode, I_SKIN);
	this->skin = (skinModifier ? (ISkin*)skinModifier->GetInterface(I_SKIN) : 0);
	skinContext = (this->skin && meshNode ? this->skin->GetContextInterface(meshNode) : 0);
}

void SkinningHelpers::SkinAccessor::FindSkeletonRoots(std::set<INode*>& skeletonRoots)
{
	// Loop through the bones that are listed in the skin modifier and find the ultimate root
	// of the skeleton.
	for (int boneIndex = 0, boneCount = (this->skin ? this->skin->GetNumBones() : 0); boneIndex < boneCount; ++boneIndex)
	{
		INode* bone = (this->skin ? this->skin->GetBone(boneIndex) : 0);
		INode* root;
		for (root = bone;
			root->GetParentNode() && root->GetParentNode() != GetCOREInterface()->GetRootNode();
			root = root->GetParentNode());

		skeletonRoots.insert(root);
	}
}

void SkinningHelpers::SkinAccessor::GetBoneInitialPosition(INode* bone, Matrix3& tm)
{
	// Ask the skin modifier for the bind-pose position of the node. If the node was added
	// to the hierarchy after the skin modifier was applied, it might not have an initial
	// position registered with the modifier. In this case we try to find a parent that
	// does have an initial position and then apply the difference between the initial
	// transform and the current transform of the parent to get a reasonable initial
	// transform.
	tm = bone->GetNodeTM(0);
	for (INode* ancestor = bone; ancestor; ancestor = ancestor->GetParentNode())
	{
		Matrix3 tmParentInitial;
		if (this->skin && SKIN_OK == this->skin->GetBoneInitTM(ancestor, tmParentInitial))
		{
			Matrix3 tmParentNow = ancestor->GetNodeTM(0);
			tm = (tm * Inverse (tmParentNow)) * tmParentInitial;
			break;
		}
	}
}

int SkinningHelpers::SkinAccessor::GetVertexCount()
{
	return (this->skinContext ? this->skinContext->GetNumPoints() : 0);
}

int SkinningHelpers::SkinAccessor::GetNumBoneLinksForVertex(int vertexIndex)
{
	return (this->skinContext ? this->skinContext->GetNumAssignedBones(vertexIndex) : 0);
}

void SkinningHelpers::SkinAccessor::GetBoneLinkInfoForVertex(int vertexIndex, int linkIndex, INode*& bone, float& weight)
{
	bone = (this->skin && this->skinContext ? this->skin->GetBone(this->skinContext->GetAssignedBone(vertexIndex, linkIndex)) : 0);
	weight = (this->skinContext ? this->skinContext->GetBoneWeight(vertexIndex, linkIndex) : 0.0f);
}

bool SkinningHelpers::SkinAccessor::HasSkin() const
{
	return this->skin && this->skinContext;
}
