#include "StdAfx.h"
#include "mayacryshaderutil.h"
#include "MayaCryUtils.h"
#include "MayaUtils.h"

float grayScale (const MColor& color)
{
	return (color.r+color.g+color.b)/3;
}

CryIRGB makeCryIRGB(unsigned char r, unsigned char g, unsigned char b)
{
	CryIRGB result;
	result.r = r;
	result.g = g;
	result.b = b;
	return result;
}

CMayaCryShaderUtil::CMayaCryShaderUtil(const MObject& objShader):
	m_objShader (objShader)
{
	MStatus status;
	status = m_fnLambert.setObject (objShader);
	if (!status)
		throw status;

	initName();

	m_fOpacity   = 1 - grayScale(m_fnLambert.transparency ());

	m_fSelfIllum = grayScale (m_fnLambert.incandescence ());
	m_fSpecLevel = 0;
	m_fSpecShininess = 0;

	// the base color of the material (diffuse)
	MColor rgbColor (1,1,1);

	// Fixup: Maya sets the color to 0 if the diffuse map is present
	if (!isTexturePresent (ID_DI))
		rgbColor = m_fnLambert.color ();

	// construct final output diffuse
	m_rgbDiffuse  = MayaToCryIRGB (rgbColor * m_fnLambert.diffuseCoeff ());

	// no specular reflections in the lambert material
	m_rgbSpecular	=	makeCryIRGB(0,0,0);
	m_rgbAmbient  = MayaToCryIRGB (m_fnLambert.ambientColor ());

	MFnReflectShader fnReflect (objShader, &status);
	if (status)
	{
		//m_fSpecLevel = fnReflect.reflectivity ();
		m_rgbSpecular = MayaToCryIRGB (fnReflect.specularColor ());
		m_fSpecLevel = fnReflect.reflectivity();
		MFnPhongShader fnPhong (objShader, &status);
		if (status)
		{
			//m_fSpecLevel = 1;
			m_fSpecShininess = fnPhong.cosPower () / 100;
		}
		else
		{
			// this is not a phong, but it is a reflect shader
			MFnBlinnShader fnBlinn (objShader, &status);
			if (status)
			{
				m_fSpecShininess = 1 - fnBlinn.eccentricity();
				m_fSpecLevel = fnBlinn.specularRollOff();
			}
		}
	}
}


//////////////////////////////////////////////////////////////////////////
// constructs the Cry-compatible shader name (s_*([#]Templ*)[/Phys])
void CMayaCryShaderUtil::initName ()
{
	m_strName =m_fnLambert.name ().asChar ();
	MString strTemplate = getAttrValue<MString> (m_objShader, "cryTemplate");
	if (strTemplate.length() > 0)
	{
		m_strName += "(";
		m_strName += strTemplate.asChar();
		m_strName += ")";
	}
	MString strPhyicsMaterial = getAttrValue<MString>(m_objShader, "cryPhysicsMaterial");
	if (strPhyicsMaterial.length() > 0)
	{
		m_strName += "/";
		m_strName += strPhyicsMaterial.asChar();
	}
	MStatus statusTry;
	MString strSortValue = getAttrValue<MString> (m_objShader, "crySortValue","",&statusTry);
	if (statusTry)
	{
		if (strSortValue.length() > 0)
		{
			// try to convert the value to an integer; if it doesn't work, then output a warning
			for (const char* p = strSortValue.asChar(); *p; ++p)
			{
				if (!(*p >= '0' && *p <= '9'))
				{
					std::cerr << "Warning: crySortValue \"" << strSortValue.asChar() << "\" is not a valid decimal number\n";
					std::cerr.flush();
					break;
				}
			}

			m_strName += "[";
			m_strName += strSortValue.asChar();
			m_strName += "]";
		}
	}
	else
	{
		int nSortValue = getAttrValue<int> (m_objShader, "crySortValue", 0);
		if (nSortValue)
		{
			char szBuf[16];
			sprintf (szBuf, "[%d]", nSortValue);
			m_strName += szBuf;
		}
	}
}

CMayaCryShaderUtil::~CMayaCryShaderUtil(void)
{
}


// get the texture that is identified by the nMap constant (specular, diffuse, etc.)
// into the given texture structure
bool CMayaCryShaderUtil::getTextureInfo (TextureMap3& dst, int nMap)
{
	memset (&dst, 0, sizeof(dst));
	dst.uoff_ctrlID = dst.urot_ctrlID = dst.uscl_ctrlID = dst.voff_ctrlID = dst.vrot_ctrlID = dst.vscl_ctrlID = dst.wrot_ctrlID = -1;
	dst.uscl_val = dst.vscl_val = 1.0f;

	const char* szAttrName = getTextureAttrName (nMap);
	MObject objTexture = findConnected (m_objShader, szAttrName, true);
	if (objTexture.isNull ())
		return false;

	float fAmount = getTextureAmount(nMap);

	MFnDependencyNode fnTexture (objTexture);

	if (objTexture.apiType () == MFn::kBump)
	{
		MStatus status;
		if (!fnTexture.findPlug ("bumpDepth", &status).getValue (fAmount) || !status)
		{
			std::cerr << "Cannot determine the bumpDepth for \"" << fnTexture.name ().asChar () << "\": " << status << "\n";
			std::cerr.flush();
			return false;
		}
		Log ("Resolving bump node \"%s\" for \"%s\"", fnTexture.name ().asChar (), szAttrName);
		objTexture = findConnected (objTexture, "bumpValue", true);
		if (objTexture.isNull ())
		{
			Log ("Failed to resolve bump node bumpValue");
			return false;
		}
		if (!fnTexture.setObject (objTexture))
		{
			Log ("bumpValue is not a depnode (it's a %s)", objTexture.apiTypeStr ());
			return false;
		}
	}

	dst.Amount = (int)floor(fAmount * 100);// TODO: allow the user to specify this via Hypershader interface

	Log ("Texture \"%s\" found (file \"%s\") for material \"%s\"", szAttrName, fnTexture.name ().asChar (), getNameStr());

	memset( dst.Reserved,0,sizeof(dst.Reserved) ); // ?

	MStatus status;
	MString mstrFileName;
	if (!fnTexture.findPlug ("fileTextureName", &status).getValue (mstrFileName) || !status)
	{
		std::cerr << "Cannot read texture file name from texture file object " << fnTexture.name().asChar () << "\n";
		std::cerr.flush ();
		return false;
	}
	std::string strFileName = mstrFileName.asChar ();
	if (!strFileName[0])
	{
		std::cerr << "Empty texture file name in texture file object " << fnTexture.name ().asChar () << "\n";
		std::cerr.flush ();
		return false;
	}

	FixupPath (strFileName);

	if (!GetRelativePath (dst.name, sizeof(dst.name), strFileName.c_str()))
		std::cerr << "Cannot get relative path from texture file name (maybe it is too long): " << strFileName << " (the maximum is " << (sizeof(dst.name)-1) << " characters)\n";

	int nFilterType;
	if (fnTexture.findPlug ("filterType",&status).getValue (nFilterType) && status)
	{
		//Log("Texture %s filter type: %d", fnTexture.name().asChar (), nFilterType);
		if (nFilterType == 0) // None
			dst.flags |= TEXMAP_NOMIPMAP;
	}

	dst.type = TEXMAP_ENVIRONMENT; // TODO: other types

	dst.utile = dst.vtile = true;

	// TODO: continue exporting UVGen etc.

	return true;
}


//////////////////////////////////////////////////////////////////////////
// returns the fade value of the texture: 1 if the texture if full-range
// applied, 0 if it's invisible. This works only for some (e.g. bump -
// through a work node, diffuse  -a special attribute of shader)
float CMayaCryShaderUtil::getTextureAmount (int nMap)
{
	MStatus status;
	float fResult = 1;

	switch (nMap)
	{
	case ID_DI:
		fResult = m_fnLambert.diffuseCoeff (&status);
		if (!status)
			fResult = 1;
		break;
	}

	return fResult;
}

const char* CMayaCryShaderUtil::getTextureAttrName (int nMap)
{
	switch (nMap)
	{
	case ID_AM: // ambient
		return "ambientColor";
	case ID_DI:// diffuse
		return "color";
	case ID_SP:// specular
		return "specularColor";
	case ID_SI: // self-illumination
		return "incandescence";	// Not implemented in material 746
	case ID_OP: // opacity
		return "transparency";	// ??
	case ID_BU: // bump 
		return "normalCamera";
	case ID_SS:	// shininess strength
		return "reflectionSpecularity"; // ??
	case ID_FI: // filter color
		return "transparency"; // ??
	case ID_RL: // reflection
		return "reflectedColor";
	case ID_RR: // refraction
		return "refractedColor";  // no such attribute in Maya?
	case ID_DP: // displacement 
	case ID_SH: // shininesNs
	default:
		return "unsupportedTexture"; // unsupported
	}
}


// returns true if the given texture map is present at the corresponding shader connection
bool CMayaCryShaderUtil::isTexturePresent (int nMap)
{
	TextureMap3 tex;
	return getTextureInfo (tex, nMap);
}