/*
ScPl - A plotting library for .NET

DataSourceAdapter.cs
Copyright (C) 2004
Felix Livni

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
   
2. Redistributions in binary form must reproduce the following text in 
   the documentation and / or other materials provided with the 
   distribution: 
   
   "This product includes software developed as part of 
   the ScPl plotting library project available from: 
   http://www.netcontrols.org/scpl/" 

------------------------------------------------------------------------

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

$Id: DataSourceAdapter.cs,v 1.2 2004/04/07 21:33:26 livni Exp $

*/

using System;
using System.Collections;
using System.ComponentModel;
using System.Data;

namespace scpl
{
	/// <summary>
	/// Summary description for DataSourceAdapter.
	/// </summary>
	public class DataSourceAdapter : ISequenceAdapter
	{
		#region Member Variables
		protected PointD[] points_;
		protected double minX_, minY_, maxX_, maxY_;
		#endregion

		#region Constructors
		protected DataSourceAdapter(object datasource, string dataMember, string dataValueYField, string dataValueXField, double start, double step)
		{
			// rules
			// datasource can be:
			//		a dataset (requires dataMember)
			//		an IListSource (e.g. datatable)
			//		an IList 
			//		a dataview
			// x,y values can be one of (for now):
			//		byte
			//		int
			//		float
			//		double
			// if dataValueYField is not present, then it is an IList or we use column0 of a DataTable
			// if dataValueXField is present, then it overrides start, step
			// if step is not present, then defaults to 1
			// if start is not present, then defaults to 0

			if(datasource is DataSet)
			{
				DataTable dt;
				if(dataMember != null)
				{
					// TODO error check
					dt = ((DataSet)datasource).Tables[dataMember];
				}
				else
				{
					// TODO error check
					dt = ((DataSet)datasource).Tables[0];
				}
				Init(((IListSource)dt).GetList(), dataValueYField, dataValueXField, start, step);
			}
			else if(datasource is DataTable)
			{
				Init(((IListSource)datasource).GetList(), dataValueYField, dataValueXField, start, step);
			}
			else if(datasource is IList)
			{
				Init((IList)datasource, dataValueYField, dataValueXField, start, step);
			}
			else
			{
				throw new Exception("datasource is not of allowed type. type = " + datasource.GetType());
			}

			minX_ = minX_ = points_[0].X;
			minY_ = maxY_ = points_[0].Y;
			foreach(PointD p in points_)
			{
				if(p == null)
					continue;

				if(p.X < minX_)
					minX_ = p.X;
				if(p.X > maxX_)
					maxX_ = p.X;
				if(p.Y < minY_)
					minY_ = p.Y;
				if(p.Y > maxY_)
					maxY_ = p.Y;
			}
		}

		public DataSourceAdapter(object datasource, string dataMember, string dataValueYField, double start, double step)
			: this(datasource, dataMember, dataValueYField, null, start, step)
		{

		}

		public DataSourceAdapter(object datasource, string dataMember, string dataValueYField, double start)
			: this(datasource, dataMember, dataValueYField, null, start, 1)
		{

		}

		public DataSourceAdapter(object datasource, string dataMember, string dataValueYField)
			: this(datasource, dataMember, dataValueYField, null, 0.0, 1.0)
		{

		}

		public DataSourceAdapter(object datasource, string dataMember, string dataValueYField, string dataValueXField)
			: this(datasource, dataMember, dataValueYField, dataValueXField, 0.0, 1.0)
		{

		}

		public DataSourceAdapter(object datasource, double start, double step)
			: this(datasource, null, null, null, start, step)
		{

		}

		public DataSourceAdapter(object datasource, double start)
			: this(datasource, null, null, null, start, 1.0)
		{

		}

		public DataSourceAdapter(object datasource)
			: this(datasource, null, null, null, 0.0, 1.0)
		{

		}
		#endregion

		#region Init(...) Methods
		private void Init(IList list, string dataValueYField, string dataValueXField, double start, double step)
		{
			points_ = new PointD[list.Count];
			int pointsIndex=0;
			foreach(object o in list)
			{
				double x, y;
				if( o is DataRow || o is DataRowView )
				{
					DataRow row;
					if(o.GetType().ToString().Equals("System.Data.DataRowView"))
						row = ((DataRowView)o).Row;
					else
						row = (DataRow)o;
					if( dataValueXField != null )
					{
						object xValue = row[dataValueYField];
						if( xValue == null )
						{
							throw new Exception("No such dataValueXField = " + dataValueXField);
						}
						
						if( xValue == DBNull.Value )
						{
							continue;
						}

						x = double.Parse(row[dataValueXField].ToString());
					}
					else
					{
						// TODO error check
						x = start + (pointsIndex * step);
					}

					if( dataValueYField != null )
					{
						object yValue = row[dataValueYField];
						if( yValue == null )
						{
							throw new Exception("No such dataValueYField = " + dataValueYField);
						}

						if( yValue == DBNull.Value )
						{
							continue;
						}
												
						y = double.Parse(row[dataValueYField].ToString());
					}
					else
					{
						// TODO error check
						y = double.Parse(row[0].ToString());
					}
				}
				else if(o is Int32 || o is Int64 || o is Double || o is Single)
				{
					// TODO error check
					x = start + (pointsIndex * step);
					y = double.Parse(o.ToString());
				}
				else
				{
					throw new Exception("ERROR: can't add point of type: " + o.GetType().ToString());
				}
				points_[pointsIndex++] = new PointD(x, y);
			}
		}
		#endregion

		#region Suggest Axes
		/// <summary>
		/// Instances a default abscissa axis.
		/// </summary>
		/// <returns>An instance of the abscissa Axis.</returns>
		public Axis SuggestXAxis()
		{
			// ERROR CHECK
			double range = maxX_ - minX_;
				
			// NOTE: I'm not sure about this logic, but it is copied for ArrayAdapter
			if ( range > 0.0F )
			{
				range *= 0.08F;
			}
			else 
			{
				range = 0.01F;
			}
			
			return new LinearAxis(minX_ - range, maxX_ + range);
		}
		/// <summary>
		/// Instances a default ordinate axis.
		/// </summary>
		/// <returns>An instance of the ordinate Axis.</returns>
		public Axis SuggestYAxis()
		{
			// ERROR CHECK
			double range = maxY_ - minY_;
			
			if ( range > 0.0F )
			{
				range *= 0.08F;
			}
			else
			{
				range = 0.01F;
			}

			return new LinearAxis( minY_ - range, maxY_ + range);
		}

		#endregion

		#region Properties
		/// <summary>
		/// Indexer for returning the datapoint.
		/// </summary>
		public virtual PointD this[int index]
		{	
			get
			{
				return points_[index];
			}
		}

		public int Count
		{
			get 
			{ 
				return points_.Length;
			}
		}

		#endregion

	}
}
