/*
ScPl - A plotting library for .NET

LinearAxis.cs
Copyright (C) 2003
Matt Howlett

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
   
2. Redistributions in binary form must reproduce the following text in 
   the documentation and / or other materials provided with the 
   distribution: 
   
   "This product includes software developed as part of 
   the ScPl plotting library project available from: 
   http://www.netcontrols.org/scpl/" 

------------------------------------------------------------------------

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

$Id: LinearAxis.cs,v 1.32 2004/06/24 11:10:31 mhowlett Exp $

*/

using System.Drawing;
using System.Collections;
using System;
using System.Text;
using System.Diagnostics;

namespace scpl
{
	/// <summary>
	/// 
	/// </summary>
	public class LinearAxis : Axis, System.ICloneable
	{
		#region Clone Implementation
		/// <summary>
		/// Deep copy of LinearAxis.
		/// </summary>
		/// <returns>A copy of the LinearAxis Class</returns>
		public override object Clone()
		{
			LinearAxis a = new LinearAxis();
			// ensure that this isn't being called on a derived type. If it is, then oh no!
			if (this.GetType() != a.GetType())
			{
				throw new System.Exception( "Clone not defined in derived type. Help!" );
			}
			this.DoClone( this, a );
			return a;
		}

		/// <summary>
		/// Helper method for Clone.
		/// </summary>
		protected void DoClone( LinearAxis b, LinearAxis a )
		{
			Axis.DoClone( b, a );

			a.numberSmallTicks_ = b.numberSmallTicks_;
			a.largeTickValue_ = b.largeTickValue_;
			a.largeTickStep_ = b.largeTickStep_;

			a.offset_ = b.offset_;
			a.scale_ = b.scale_;
		}
		#endregion
		#region Constructors
		/// <summary>
		/// Constructor, cloning an existing Axis.
		/// </summary>
		/// <param name="a">The Axis to clone.</param>
		public LinearAxis( Axis a )
			: base( a )
		{
		}

		/// <summary>
		/// Default parameterless constructor.
		/// </summary>
		public LinearAxis()
			: base()
		{
		}

		/// <summary>
		/// Constructor that takes only world min and max values.
		/// </summary>
		/// <param name="worldMin">The minimum world coordinate.</param>
		/// <param name="worldMax">The maximum world coordinate.</param>
		public LinearAxis( double worldMin, double worldMax )
			: base( worldMin, worldMax )
		{
		}
		#endregion
		#region DrawTicks
		/// <summary>
		/// Draw the ticks
		/// </summary>
		/// <param name="g">The drawing surface on which to draw.</param>
		/// <param name="physicalMin"> </param>
		/// <param name="physicalMax"> </param>
		/// <param name="boundingBox"> </param>
		/// <param name="labelOffset"> </param>
		protected override void DrawTicks(
			Graphics g, 
			PointF physicalMin, 
			PointF physicalMax, 
			out object labelOffset,
			out object boundingBox )
		{

			PointF tLabelOffset;
			RectangleF tBoundingBox;

			this.InitOffsetAndBounds( out labelOffset, out boundingBox );

			// large ticks X position is in l1
			ArrayList l1 = this.LargeTickPositions;
			string strFormat = this.NumberFormat;

			labelOffset = new PointF( 0.0f, 0.0f );
			boundingBox = null;

			if (l1.Count > 0)
			{
				for (int i = 0; i < l1.Count; ++i)
				{
					double labelNumber = (double)l1[i];

					StringBuilder label = new StringBuilder();
					label.AppendFormat(strFormat, labelNumber);

					this.DrawTick( g, ((double)l1[i]-this.offset_)/this.scale_, 
						this.LargeTickSize, label.ToString(),
						new Point(0,0), physicalMin, physicalMax, 
						out tLabelOffset, out tBoundingBox );
					
					this.UpdateOffsetAndBounds( ref labelOffset, ref boundingBox, 
						tLabelOffset, tBoundingBox );

				}
			}

			ArrayList stp = this.SmallTickPositions;

			for (int i = 0; i < stp.Count; ++i)
			{
				this.DrawTick( g, ((double)stp[i]-this.offset_)/this.scale_, 
					this.SmallTickSize, "", 
					new Point(0,0), physicalMin, physicalMax, 
					out tLabelOffset, out tBoundingBox );

				// assume bounding box and label offset unchanged by small tick bounds.
			}

		}
		#endregion
		#region SmallTickPositions
		/// <summary>
		/// The ArrayList containing the positions of the small ticks.
		/// </summary>
		public override ArrayList SmallTickPositions
		{
			get
			{
				double adjustedMax = this.AdjustedWorldValue( WorldMax );
				double adjustedMin = this.AdjustedWorldValue( WorldMin );

				ArrayList toRet = new ArrayList();

				ArrayList l1 = this.LargeTickPositions;

				double bigTickSpacing = this.DetermineTickSpacing();
				int nSmall = this.DetermineNumberSmallTicks( bigTickSpacing );
				double smallTickSpacing = bigTickSpacing / (double)nSmall;

				// if there is at least one big tick
				if (l1.Count > 0)
				{
					double pos1 = (double)l1[0];
					while (pos1 > adjustedMin)
					{
						pos1 -= smallTickSpacing;
						toRet.Add( pos1 );
					}
				}

				for (int i = 0; i < l1.Count; ++i )
				{
					for (int j = 1; j < nSmall; ++j )
					{
						double pos = (double)l1[i] + ((double)j) * smallTickSpacing;
						if (pos <= adjustedMax)
						{
							toRet.Add( pos );
						}
					}
				}
				return toRet;
			}
		}
		#endregion
		#region AdjustedWorldValue
		public double AdjustedWorldValue( double v )
		{
			return v * this.scale_ + this.offset_;
		}
		#endregion
		#region LargeTickPositions
		/// <summary>
		/// Get the large tick positions for this axis.
		/// </summary>
		public override ArrayList LargeTickPositions
		{
			get
			{
				// (1) error check

				if ( (worldMin_ == null) || (worldMax_ == null) )
				{
					throw new System.Exception( "world extent of axis not set." );
				}
				
				double adjustedMax = this.AdjustedWorldValue( WorldMax );
				double adjustedMin = this.AdjustedWorldValue( WorldMin );

				// (2) determine distance between large ticks.

				double tickDist;

				if (this.largeTickStep_ != null)
				{
					tickDist = (double)this.largeTickStep_;
				}
				else
				{
					tickDist = this.DetermineTickSpacing();
				}

				// (3) determine starting position.
			
				double first = 0.0f;

				if (this.largeTickValue_ != null) 
				{
					first = (double)this.largeTickValue_;

					// TODO: make more efficient
					while (first < adjustedMin)
					{
						first += tickDist;
					}

					while (first >= (adjustedMin + tickDist))
					{
						first -= tickDist;
					}
				}

				else
				{
					if( adjustedMin > 0.0 )
					{
						double nToFirst = Math.Floor(adjustedMin / tickDist) + 1.0f;
						first = nToFirst * tickDist;
					}
					else
					{
						double nToFirst = Math.Floor(-adjustedMin/tickDist) - 1.0f;
						first = -nToFirst * tickDist;
					}

					// could miss one, if first is just below zero.
					if ((first - tickDist) >= adjustedMin * slightlySmallerMultiply_)
					{
						first -= tickDist;
					}
				}


				// (4) now make list of large tick positions.
				
				ArrayList positions = new ArrayList();

				double position = first;
				while ((position <= adjustedMax/slightlySmallerMultiply_) && (position >= adjustedMin*slightlySmallerMultiply_))
				{
					positions.Add( position );
					position += tickDist;
				}

				return positions;
			}
		}
		private const double slightlySmallerMultiply_ = 0.9999;
		#endregion
		#region DetermineTickSpacing
		private double DetermineTickSpacing( )
		{
			// error check.

			if ( worldMin_ == null || worldMax_ == null )
			{
				throw new System.Exception( "world extent of axis not set." );
			}

			if (this.largeTickStep_ != null)
			{
				if ( (double)this.largeTickStep_ <= 0.0f )
				{
					throw new System.Exception( "can't have negative tick step - reverse WorldMin WorldMax instead." );
				}

				return (double)this.largeTickStep_;
			}

			double adjustedMax = this.AdjustedWorldValue( WorldMax );
			double adjustedMin = this.AdjustedWorldValue( WorldMin );

			double range = adjustedMax - adjustedMin;

			if ( range > 0.0 )
			{
				// the 3.0 is magic enough that I'll hard code it here for now.
				double approxTickDist = range / 3.0f;

				// normalize tick distance down to value between 0.1 -> 1.0.
				int tenMulCount = 0;
				while( approxTickDist >= 1.0f )
				{
					approxTickDist /= 10.0f;
					tenMulCount -= 1;
				}
				while( approxTickDist < 0.1f )
				{
					approxTickDist *= 10.0f;
					tenMulCount += 1;
				}

				// now calulate a nice round tick dist.
				double roundTickDist = 0.0f;

				if( approxTickDist < 0.2f )
				{
					roundTickDist = 0.1f;
				}
				else if (approxTickDist < 0.5f)
				{
					roundTickDist = 0.2f;
				}
				else
				{
					roundTickDist = 0.5f;
				}

				// now renormalize..
				if ( tenMulCount > 0 )
				{
					for( int i=0; i<tenMulCount; ++i )
					{
						roundTickDist /= 10.0f;
					}
				}
				else if (tenMulCount < 0)
				{
					for( int i=0; i<-tenMulCount; ++i )
					{
						roundTickDist *= 10.0f;
					}
				}
				return roundTickDist;
			}
			else
			{
				return 0.0f;
			}
		}
		#endregion
		#region DetermineNumberSmallTicks
		private int DetermineNumberSmallTicks( double bigTickDist )
		{
			if (this.numberSmallTicks_ != null)
			{
				return (int)this.numberSmallTicks_+1;
			}

			if( bigTickDist>0.0f)
			{
				while( bigTickDist >= 1.0f )
				{
					bigTickDist /= 10.0f;
				}
				while( bigTickDist < 0.1f )
				{
					bigTickDist *= 10.0f;
				}

				if( bigTickDist < 0.15f )
				{
					return 5;
				}
				else if( bigTickDist < 0.3f )
				{
					return 2;
				}
				else
				{
					return 5;
				}
			}
			else
			{
				return 0;
			}
		}
		#endregion
		#region LargeTickStep
		/// <summary>
		/// If set, gives the distance between large ticks.
		/// </summary>
		public double LargeTickStep
		{
			set
			{
				largeTickStep_ = value;
			}
			get
			{
				return (double)largeTickStep_;
			}
		}
		/// <summary>
		/// If set, gives the distance between large ticks.
		/// </summary>
		private object largeTickStep_;
		#endregion
		#region LargeTickValue
		/// <summary>
		/// If set, a large tick will be placed at this position, and other large ticks will 
		/// be placed relative to this position.
		/// </summary>
		public double LargeTickValue
		{
			set
			{
				largeTickValue_ = value;
			}
			get
			{
				return (double)largeTickValue_;
			}
		}
		/// <summary>
		/// If set, a large tick will be placed at this position, and other large ticks will 
		/// be placed relative to this position.
		/// </summary>
		private object largeTickValue_ = null;
		#endregion
		#region NumberSmallTicks
		/// <summary>
		/// The number of small ticks between large ticks.
		/// </summary>
		public int NumberSmallTicks
		{
			set
			{
				numberSmallTicks_ = value;
			}
			get
			{
				return (int)numberSmallTicks_;
			}
		}
		/// <summary>
		/// The number of small ticks between large ticks.
		/// </summary>
		private object numberSmallTicks_ = null;
		#endregion
		#region Scale
		/// <summary>
		/// Affects 
		/// </summary>
		public double Scale
		{
			get
			{
				return scale_;
			}
			set
			{
				scale_ = value;
			}
		}
		private double scale_ = 1.0;
		#endregion
		#region Offset
		public double Offset
		{
			get
			{
				return offset_;
			}
			set
			{
				offset_ = value;
			}
		}
		private double offset_ = 0.0;
		#endregion
	}
}
